#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliMode.EventMgr import EventHandlerMode, TriggerOnCountersMode, \
                             TriggerOnLoggingMode, TriggerOnCustomMode
from EventLib import triggerTypeTaccToCli, maintenanceOperTaccToCli
from CliSavePlugin.IntfCliSave import IntfConfigMode

CliSave.GlobalConfigMode.addCommandSequence( 'Event.config' )

class EventHandlerConfigMode( EventHandlerMode, CliSave.Mode ):
   def __init__( self, param ):
      EventHandlerMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( EventHandlerConfigMode,
                                       after=[ IntfConfigMode ] )
EventHandlerConfigMode.addCommandSequence( 'EventHandler.config' )

class EventTriggerOnCountersMode( TriggerOnCountersMode, CliSave.Mode ):
   def __init__( self, param ):
      TriggerOnCountersMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

EventHandlerConfigMode.addChildMode( EventTriggerOnCountersMode, after=[] )
EventTriggerOnCountersMode.addCommandSequence( 'EventHandler.config.counters' )

class EventTriggerOnLoggingMode( TriggerOnLoggingMode, CliSave.Mode ):
   def __init__( self, param ):
      TriggerOnLoggingMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

EventHandlerConfigMode.addChildMode( EventTriggerOnLoggingMode, after=[] )
EventTriggerOnLoggingMode.addCommandSequence( 'EventHandler.config.logging' )

class EventTriggerOnCustomMode( TriggerOnCustomMode, CliSave.Mode ):
   def __init__( self, param ):
      TriggerOnCustomMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

EventHandlerConfigMode.addChildMode( EventTriggerOnCustomMode, after=[] )
EventTriggerOnCustomMode.addCommandSequence( 'EventHandler.config.custom' )

@CliSave.saver( 'Event::Config', 'sys/event/config' )
def saveEvents( entity, root, requireMounts, options ):
   saveAll = options.saveAll

   # check if a built-in event handler is modified
   # by comparing its attributes with default
   def checkBuiltinEventModified ( handlerName, event ):
      if handlerName in entity.defaultBuiltinEvent:
         builtinEvent = entity.defaultBuiltinEvent.get( handlerName )
         if ( builtinEvent.triggerType == event.triggerType and
              builtinEvent.operstatus == event.operstatus and
              builtinEvent.ip == event.ip and
              builtinEvent.ip6 == event.ip6 and
              builtinEvent.intfName == event.intfName and
              builtinEvent.actionKind == event.actionKind and
              builtinEvent.command == event.command and
              builtinEvent.delay == event.delay and
              builtinEvent.repeatInterval == event.repeatInterval and
              builtinEvent.maxRepeatActionCount == event.maxRepeatActionCount and
              builtinEvent.asynchronous == event.asynchronous and
              builtinEvent.timeout == event.timeout and
              builtinEvent.terminateActionOnTimeout == \
                 event.terminateActionOnTimeout and
              builtinEvent.thresholdCount == event.thresholdCount and
              builtinEvent.threshold == event.threshold and
              builtinEvent.pollInterval == event.pollInterval and
              builtinEvent.countersCondition == event.countersCondition and
              builtinEvent.logRegex == event.logRegex and
              builtinEvent.scriptTrigger == event.scriptTrigger and
              builtinEvent.maintenanceUnitName == event.maintenanceUnitName and
              builtinEvent.maintenanceOper == event.maintenanceOper and
              builtinEvent.maintenanceStage == event.maintenanceStage and
              builtinEvent.maintenanceBgpPeer == event.maintenanceBgpPeer and
              builtinEvent.vrfName == event.vrfName and
              builtinEvent.metricName == event.metricName and
              builtinEvent.builtinHandler == event.builtinHandler and
              builtinEvent.builtinHandlerDisabled == event.builtinHandlerDisabled and
              builtinEvent.hasSubhandlers == event.hasSubhandlers and
              builtinEvent.runUponConfig == event.runUponConfig ):
            return False
      return True

   for handlerName, event in sorted( entity.event.items() ):

      if event.builtinHandler:
         if event.builtinHandlerDisabled: # pylint: disable=no-else-continue
            # built-in handlers are enabled by default and can be disabled
            # with aconfig command
            cmds = root[ 'Event.config' ]
            cmds.addCommand( 'no event-handler %s' % handlerName )
            continue
         elif not saveAll and not checkBuiltinEventModified( handlerName, event ) :
            continue

      mode = root[ EventHandlerConfigMode ].getOrCreateModeInstance( handlerName )
      cmds = mode[ 'EventHandler.config' ]

      # Trigger type
      if event.triggerType in [ 'onBoot', 'onStartupConfig' ]:
         cmds.addCommand( "trigger %s" % triggerTypeTaccToCli[ event.triggerType ] )
      elif event.triggerType == 'onIntf':
         args = ' '.join( [ x for x in [ 'operstatus', 'ip', 'ip6' ]
                                    if getattr( event, x ) ] )
         cmds.addCommand( "trigger %s %s %s" \
                          % ( triggerTypeTaccToCli[ event.triggerType ],
                              event.intfName.split()[0], args ) )
      elif event.triggerType == 'vmTracerVm':
         cmds.addCommand( "trigger vm-tracer vm" )
      elif event.triggerType == 'onCounters':
         mode1 = mode[ EventTriggerOnCountersMode ].getOrCreateModeInstance( 
            handlerName )
         cmds1 = mode1[ 'EventHandler.config.counters' ]
         if event.pollInterval != entity.defaultPollInterval or saveAll:
            cmds1.addCommand( "poll interval %d" % event.pollInterval )
         if event.countersCondition != entity.defaultCountersCondition or saveAll:
            cmds1.addCommand( "condition %s" % event.countersCondition )         
         if event.hasSubhandlers != entity.defaultHasSubhandlers:
            cmds1.addCommand( "granularity per-source" )
      elif event.triggerType == 'onLogging':
         mode1 = mode[ EventTriggerOnLoggingMode ].getOrCreateModeInstance(
            handlerName )
         cmds1 = mode1[ 'EventHandler.config.logging' ]
         if event.pollInterval != entity.defaultLogPoll or saveAll:
            cmds1.addCommand( "poll interval %d" % event.pollInterval )
         if event.logRegex != entity.defaultLogRegex or saveAll:
            cmds1.addCommand( "regex %s" % event.logRegex )
      elif event.triggerType == 'onCustomCondition':
         mode1 = mode[ EventTriggerOnCustomMode ].getOrCreateModeInstance(
            handlerName )
         cmds1 = mode1[ 'EventHandler.config.custom' ]
         if event.pollInterval != entity.defaultPollInterval or saveAll:
            cmds1.addCommand( "poll interval %d" % event.pollInterval )
         if event.scriptTrigger != entity.defaultScriptTrigger or saveAll:
            if '\n' in event.scriptTrigger:
               lines = [ "condition bash" ]
               commandMultiline = event.scriptTrigger + "EOF"
               for line in commandMultiline.split( "\n" ):
                  lines.append( '   ' + line )
               cmds1.addCommand( "\n".join( lines ) )
            else:
               cmds1.addCommand( "condition bash %s" % event.scriptTrigger )
      elif event.triggerType == 'onMaintenance':
         maintenanceUnit = ""
         if event.maintenanceBgpPeer:
            if event.vrfName != 'default':
               maintenanceUnit = "bgp {} vrf {}".format( event.maintenanceBgpPeer,
                                                         event.vrfName )
            else:
               maintenanceUnit = "bgp %s" % ( event.maintenanceBgpPeer )
         elif event.intfName:
            maintenanceUnit = "interface %s" % event.intfName
         else:
            maintenanceUnit = "unit %s" % event.maintenanceUnitName

         maintHookPoints = ( "begin", "end", "all" )
         stage = ""
         if event.maintenanceStage in maintHookPoints:
            stage = event.maintenanceStage
         else:
            hookPoint, maintStage = event.maintenanceStage.split( '_' )
            stage = f"{hookPoint} stage {maintStage}"
         cmds.addCommand( "trigger %s %s %s %s"
                          % ( triggerTypeTaccToCli[ event.triggerType ],
                              maintenanceOperTaccToCli[ event.maintenanceOper ],
                              maintenanceUnit, stage ) )
      elif event.triggerType == 'none':
         pass
      else:
         assert False

      # Run Upon Config
      if event.runUponConfig != entity.defaultRunUponConfig or saveAll:
         cmds.addCommand( "%strigger on-config disabled" \
                          % ( 'no ' if event.runUponConfig else '' ) )

      # Action
      if event.actionKind == 'actionKindBash':
         if '\n' in event.command:
            lines = [ "action bash" ]
            commandMultiline = event.command + "EOF"
            for line in commandMultiline.split( "\n" ):
               lines.append( '   ' + line )
            cmds.addCommand( "\n".join( lines ) )
         elif event.command != entity.defaultCommand or saveAll:
            cmds.addCommand( "action bash %s" % event.command )
      elif event.actionKind == 'actionKindLog':
         cmds.addCommand( "action log" )
      else:
         assert False

      # Device-health action
      if event.metricName != entity.defaultMetricName:
         cmds.addCommand( "action increment device-health metric %s" % \
                          event.metricName )
      elif saveAll:
         cmds.addCommand( "no action increment device-health metric" )

      # Delay
      if event.delay != entity.defaultDelay or saveAll:
         cmds.addCommand( "delay %d" % event.delay )

      # Repeat Interval and action count
      if event.maxRepeatActionCount != entity.defaultMaxRepeatActionCount or saveAll:
         cmds.addCommand( "repeat interval %d count %d" % ( event.repeatInterval,
            event.maxRepeatActionCount ) )
      elif event.repeatInterval != entity.defaultRepeatInterval or saveAll:
         cmds.addCommand( "repeat interval %d" % event.repeatInterval )

      # Asynchronous
      if event.asynchronous != entity.defaultAsynchronous or saveAll:
         cmds.addCommand( "%sasynchronous" \
                          % ( '' if event.asynchronous else 'no ' ) )

      # Timeout
      if event.timeout != entity.defaultTimeout or \
            event.terminateActionOnTimeout != \
            entity.defaultTerminateActionOnTimeout or \
            saveAll:
         cmd = "timeout %d" % event.timeout
         if event.terminateActionOnTimeout:
            cmd += " terminate"
         cmds.addCommand( cmd )

      # threshold
      if event.threshold != entity.defaultThreshold or saveAll:
         cmds.addCommand( "threshold %d count %d" % \
                             ( int( event.threshold ), event.thresholdCount ) )
