# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string


import CliSave
import Url

from EventMonUtils import qtTableConfigs
from EventMonCliTables import allEventsDict

CliSave.GlobalConfigMode.addCommandSequence( 'EventMon.global' )

# Table name and its cli token are not necessary the same. Some cli token can map 
# to multiple tables e.x. ('nat' -> {'nat', 'sfe'}), we use TableStatus.active to 
# determine whether a particular table is the active one.
tableNameToCliTokenMapping = { "sfe" : "nat",
                               "backup-activation" : "backupactivation" }
cliTokenToTableNameMapping = { "nat" : "sfe",
                               "backupactivation" : "backup-activation" }

@CliSave.saver( 'EventMon::Config', 'eventMon/config', 
                 requireMounts = ( 'eventMon/sfe/status', ) )
def saveEventMonConfig( cfg, root, requireMounts, options ):
   def tableIsActive( tableName ):
      tableStatus = requireMounts[ f'eventMon/{ tableName }/status' ]
      return tableStatus.active

   cmds = root[ 'EventMon.global' ]
   saveAll = options.saveAll

   if cfg.agentEnabled != cfg.defaultAgentEnabled:
      noToken = '' if cfg.agentEnabled else 'no '
      cmds.addCommand( noToken + 'event-monitor' )
   elif saveAll:
      cmds.addCommand( 'default event-monitor' )

   for tableName, tableConfig in qtTableConfigs( cfg ):
      # BUG842873: cli token can be mapped to multiple tables
      if tableName in cliTokenToTableNameMapping:
         mappedTable = cliTokenToTableNameMapping[ tableName ]
         if tableIsActive(  mappedTable ): # the mapped table is active
            continue
      if tableName in tableNameToCliTokenMapping:
         if not tableIsActive( tableName ): # table is not active, skip
            continue
         tableName = tableNameToCliTokenMapping[ tableName ]

      if tableConfig.initialEnabledValue:
         if not tableConfig.enabled or saveAll:
            noToken = '' if tableConfig.enabled else 'no '
            cmds.addCommand( noToken + 'event-monitor %s' % tableName )
      else:
         if tableConfig.enabled or saveAll:
            noToken = '' if tableConfig.enabled else 'no '
            cmds.addCommand( noToken + 'event-monitor %s' % tableName ) 

   if ( cfg.foreverLogFileName != cfg.defaultForeverLogFileName or
        cfg.foreverLogDirectory != cfg.defaultForeverLogDirectory ):
      baseFilePath = "%s/%s" % ( cfg.foreverLogDirectory, cfg.foreverLogFileName )
      cmds.addCommand( 'event-monitor backup path %s' \
                       % Url.filenameToUrl( baseFilePath ) )
   elif saveAll:
      cmds.addCommand( 'no event-monitor backup path' )

   for tableName, tableConfig in qtTableConfigs( cfg ):
      if tableName in cliTokenToTableNameMapping:
         mappedTable = cliTokenToTableNameMapping[ tableName ]
         if tableIsActive(  mappedTable ):
            continue
      if tableName in tableNameToCliTokenMapping:
         if not tableIsActive( tableName ):
            continue
         tableName = tableNameToCliTokenMapping[ tableName ]

      if tableConfig.foreverLogOverride:
         baseFilePath = "%s/%s" % ( tableConfig.foreverLogDirectory,
                                    tableConfig.foreverLogFileName )
         cmds.addCommand( 'event-monitor %s backup path %s' \
                   % ( tableName, Url.filenameToUrl( baseFilePath ) ) )
      elif saveAll:
         cmds.addCommand( 'no event-monitor %s backup path' % tableName )

   # Max table size can be set for both qt and shaq configs so iterate over all.
   for tableName, tableConfig in cfg.table.items():
      if tableName not in allEventsDict:
         continue
      if tableConfig.maxTableSize.val != tableConfig.maxTableSize.defaultVal \
         or saveAll:
         cmds.addCommand( f'event-monitor {tableName} max-size ' +
                         f'{tableConfig.maxTableSize.val}' )
      

   if cfg.foreverLogMaxSize.val != cfg.foreverLogMaxSize.defaultVal or saveAll:
      cmds.addCommand( 'event-monitor backup max-size %d' \
                        % cfg.foreverLogMaxSize.val )

   if cfg.bufferSize.val != cfg.bufferSize.defaultVal or saveAll:
      cmds.addCommand( 'event-monitor buffer max-size %d' \
                        % cfg.bufferSize.val )
