# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import io
import Tracing
import CliSave
from CliMode.FlowTracker import TrackerModeBase
from CliMode.FlowExporter import ExporterModeBase
from CliMode.FlowCongestion import CongestionModeBase
from CliMode.FlowGroups import (
      GroupsModeBase,
      GroupModeBase,
)
from FlowTrackerCliUtil import (
      ftrCapabilitiesPathPrefix,
      ftrTypeDfw,
      ftrTypeHardware,
      ftrTypeInbandTelemetry,
      ftrTypeKwStr,
      ftrTypeMirrorOnDrop,
      ftrTypeSampled,
      ftrTypeCpuQueue,
      congestionSupportedForType,
      getSampledFlowTrackingCliConfigPath,
      getHardwareFlowTrackingCliConfigPath,
      getInbandTelemetryFlowTrackingCliConfigPath,
      getMirrorOnDropFlowTrackingCliConfigPath,
      getCpuQueueFlowTrackingCliConfigPath,
      getDfwFlowTrackingCliConfigPath,
      showCongestion,
      showExporter,
      showFtr,
      showGroup,
)
from CliSavePlugin.FlowTrackingCliSave import FlowTrackingCliSaveMode
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'FlowTrackerCliSave' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

SwCapabilities = TacLazyType( 'FlowTracking::Capabilities' )

#---------------------------------------------------------------------------------
# Cli savers
#---------------------------------------------------------------------------------

class TrackerCliSaveMode( TrackerModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t0( 'TrackerCliSaveMode: ', param )
      TrackerModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return False

FlowTrackingCliSaveMode.addChildMode( TrackerCliSaveMode,
                                      before=[ 'FlowTracking.trailer' ] )
TrackerCliSaveMode.addCommandSequence( 'FlowTracker.config' )

class CongestionSaveMode( CongestionModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t0( 'FlowTrackerCongestionSaveMode: ', param )
      CongestionModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return False

TrackerCliSaveMode.addChildMode( CongestionSaveMode, after=[ 'FlowTracker.config' ] )
CongestionSaveMode.addCommandSequence( 'FlowTracker.congestionConfig' )

class ExporterSaveMode( ExporterModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t0( 'FlowTrackerExporterSaveMode: ', param )
      ExporterModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return False

TrackerCliSaveMode.addChildMode( ExporterSaveMode, after=[ CongestionSaveMode ] )
ExporterSaveMode.addCommandSequence( 'FlowTracker.exporterConfig' )

class GroupsSaveMode( GroupsModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t0( 'FlowTrackerGroupsSaveMode: ', param )
      GroupsModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

TrackerCliSaveMode.addChildMode( GroupsSaveMode, after=[ ExporterSaveMode ] )
GroupsSaveMode.addCommandSequence( 'FlowTracker.groupsConfig' )

class GroupSaveMode( GroupModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t0( 'FlowTrackerGroupSaveMode: ', param )
      GroupModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return False

GroupsSaveMode.addChildMode( GroupSaveMode, after=[ 'FlowTracker.groupsConfig' ] )
GroupSaveMode.addCommandSequence( 'FlowTracker.groupConfig' )

mplsHwCapsPath = 'routing/hardware/mpls/capability'

def saveFlowTrackerConfig( ftrType, trConfig, root, requireMounts, options ):
   trName = trConfig.name
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   t0( 'saveFlowTrackerConfig', trConfig, trConfig.parent,
       root, options, trName )
   ftrTypeStr = ftrTypeKwStr[ ftrType ]
   ftrMode = root[ FlowTrackingCliSaveMode ].getOrCreateModeInstance( ftrTypeStr )
   trMode = ftrMode[ TrackerCliSaveMode ].getOrCreateModeInstance(
                                                      ( ftrTypeStr, trName ) )
   t1( 'ftrMode', ftrMode, 'trMode', trMode )
   trCmds = trMode[ 'FlowTracker.config' ]
   trOutput = io.StringIO()
   mplsHwCaps = requireMounts[ mplsHwCapsPath ]
   ftrTypeCaps = requireMounts.get( ftrCapabilitiesPathPrefix + ftrType )
   ftrSwCaps = SwCapabilities( ftrType )
   showFtr( trOutput, trName, trConfig, ftrType, mplsHwCaps,
            ftrTypeCaps=ftrTypeCaps, ftrTypeSwCaps=ftrSwCaps,
            detail=( saveAll or saveAllDetail ) )
   for cmd in trOutput.getvalue().splitlines():
      trCmds.addCommand( cmd )

   if trConfig.congestionConfig:
      congestionMode = trMode[ CongestionSaveMode ].getOrCreateModeInstance(
                                                      ( ftrTypeStr, trName ) )
      congestionCmds = congestionMode[ 'FlowTracker.congestionConfig' ]
      congOutput = io.StringIO()
      showCongestion( congOutput, trConfig.congestionConfig, ftrSwCaps,
                      detail=( saveAll or saveAllDetail ) )
      for cmd in congOutput.getvalue().splitlines():
         congestionCmds.addCommand( cmd )
   elif congestionSupportedForType( ftrSwCaps ) and ( saveAll or saveAllDetail ):
      cmd = 'no record export on congestion'
      trCmds.addCommand( cmd )

   for expName, expConfig in trConfig.expConfig.items():
      expMode = trMode[ ExporterSaveMode ].getOrCreateModeInstance(
                                             ( ftrTypeStr, trName, expName ) )
      expCmds = expMode[ 'FlowTracker.exporterConfig' ]
      output = io.StringIO()
      showExporter( output, expName, expConfig, ftrTypeStr, ftrTypeCaps, ftrSwCaps,
                    saveAll=saveAll, saveAllDetail=saveAllDetail )
      for cmd in output.getvalue().splitlines():
         expCmds.addCommand( cmd )

   grpsMode = trMode[ GroupsSaveMode ].getOrCreateModeInstance(
      ( ftrTypeStr, trName ) )
   if trConfig.fgConfig:
      seqnoToGroupMap = {}
      for groupConfig in trConfig.fgConfig.values():
         seqnoToGroupMap[ groupConfig.seqno ] = groupConfig
      for groupSeqno in sorted( seqnoToGroupMap ):
         groupConfig = seqnoToGroupMap[ groupSeqno ]
         groupName = groupConfig.fgName
         grpMode = grpsMode[ GroupSaveMode ].getOrCreateModeInstance(
                              ( ftrTypeStr, trName, groupSeqno, groupName ) )
         grpCmds = grpMode[ 'FlowTracker.groupConfig' ]
         output = io.StringIO()
         showGroup( output, groupName, groupConfig,
                    saveAll=saveAll, saveAllDetail=saveAllDetail )
         for cmd in output.getvalue().splitlines():
            grpCmds.addCommand( cmd )

@CliSave.saver( 'FlowTracking::FlowTrackerConfig',
                getSampledFlowTrackingCliConfigPath(),
                attrName='flowTrackerConfig',
                requireMounts=(
                   mplsHwCapsPath,
                   ftrCapabilitiesPathPrefix + ftrTypeSampled,
                ) )
def saveSampledFlowTrackerConfig( trConfig, root, requireMounts, options ):
   saveFlowTrackerConfig( ftrTypeSampled, trConfig, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::FlowTrackerConfig',
                getHardwareFlowTrackingCliConfigPath(),
                attrName='flowTrackerConfig',
                requireMounts=(
                   mplsHwCapsPath,
                   ftrCapabilitiesPathPrefix + ftrTypeHardware,
                ) )
def saveHardwareFlowTrackerConfig( trConfig, root, requireMounts, options ):
   saveFlowTrackerConfig( ftrTypeHardware, trConfig, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::FlowTrackerConfig',
                getInbandTelemetryFlowTrackingCliConfigPath(),
                attrName='flowTrackerConfig',
                requireMounts=(
                   mplsHwCapsPath,
                   ftrCapabilitiesPathPrefix + ftrTypeInbandTelemetry,
                ) )
def saveInbandTelemetryFlowTrackerConfig( trConfig, root, requireMounts, options ):
   saveFlowTrackerConfig(
      ftrTypeInbandTelemetry, trConfig, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::FlowTrackerConfig',
                getMirrorOnDropFlowTrackingCliConfigPath(),
                attrName='flowTrackerConfig',
                requireMounts=(
                   mplsHwCapsPath,
                   ftrCapabilitiesPathPrefix + ftrTypeMirrorOnDrop,
                ) )
def saveMirrorOnDropFlowTrackerConfig( trConfig, root, requireMounts, options ):
   saveFlowTrackerConfig(
      ftrTypeMirrorOnDrop, trConfig, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::FlowTrackerConfig',
                getCpuQueueFlowTrackingCliConfigPath(),
                attrName='flowTrackerConfig',
                requireMounts=(
                   mplsHwCapsPath,
                   ftrCapabilitiesPathPrefix + ftrTypeCpuQueue,
                ) )
def saveCpuQueueFlowTrackerConfig( trConfig, root, requireMounts, options ):
   saveFlowTrackerConfig(
      ftrTypeCpuQueue, trConfig, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::FlowTrackerConfig',
                getDfwFlowTrackingCliConfigPath(),
                attrName='flowTrackerConfig',
                requireMounts=(
                   mplsHwCapsPath,
                   ftrCapabilitiesPathPrefix + ftrTypeDfw,
                ) )
def saveDfwFlowTrackerConfig( trConfig, root, requireMounts, options ):
   saveFlowTrackerConfig( ftrTypeDfw, trConfig, root, requireMounts, options )
