# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing
import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliMode.FlowTracking import FlowTrackingModeBase
from FlowTrackerCliUtil import (
   ftrCapabilitiesPathPrefix,
   mplsHwCapabilitiesPath,
   bridgingHwCapabilitiesPath,
   ftrTypeCpuQueue,
   ftrTypeDfw,
   ftrTypeHardware,
   ftrTypeKwStr,
   ftrTypeInbandTelemetry,
   ftrTypeMirrorOnDrop,
   ftrTypeSampled,
   egressStr,
   getCpuQueueFlowTrackingCliConfigPath,
   getSampledFlowTrackingCliConfigPath,
   getHardwareFlowTrackingCliConfigPath,
   getInbandTelemetryFlowTrackingCliConfigPath,
   getMirrorOnDropFlowTrackingCliConfigPath,
   getDfwFlowTrackingCliConfigPath,
   showFlowTracking,
   showFlowTrackingTrailer,
   sampleModeFiltered,
)
from IntfRangePlugin.DpsIntf import DpsAutoIntfType
from IntfRangePlugin.TunnelIntfRange import TunnelAutoIntfType
from IntfRangePlugin.EthIntf import EthPhyAutoIntfType
from IntfRangePlugin.LagIntf import LagAutoIntfType
from IntfRangePlugin.SwitchIntf import SwitchAutoIntfType
from Intf.IntfRange import allIntfNames

traceHandle = Tracing.Handle( 'FlowTrackingCliSave' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

#---------------------------------------------------------------------------------
# Cli savers
#---------------------------------------------------------------------------------

class FlowTrackingCliSaveMode( FlowTrackingModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t0( 'FlowTrackingCliSaveMode:', param )
      FlowTrackingModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

# Used to generate a block of flow tracking global level config commands.

CliSave.GlobalConfigMode.addChildMode( FlowTrackingCliSaveMode,
                              before=[ IntfConfigMode ] )
FlowTrackingCliSaveMode.addCommandSequence( 'FlowTracking.global' )
FlowTrackingCliSaveMode.addCommandSequence( 'FlowTracking.trailer',
                              after=[ 'FlowTracking.global' ] )
IntfConfigMode.addCommandSequence( 'FlowTracker.config' )

def saveFlowTrackingGlobalConfig( ftrType, entity, root, requireMounts, options ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   ftrTypeCaps = requireMounts.get( ftrCapabilitiesPathPrefix + ftrType )
   mplsHwCaps = requireMounts.get( mplsHwCapabilitiesPath )
   bridgingHwCaps = requireMounts.get( bridgingHwCapabilitiesPath )
   lines = showFlowTracking( config=entity, ftrType=ftrType, ftrTypeCaps=ftrTypeCaps,
                             cliSave=True, saveAll=saveAll,
                             saveAllDetail=saveAllDetail,
                             mplsHwCaps=mplsHwCaps, bridgingHwCaps=bridgingHwCaps )

   if lines is not None:
      mode = root[ FlowTrackingCliSaveMode ].getOrCreateModeInstance(
                                                      ftrTypeKwStr[ ftrType ] )
      cmds = mode[ 'FlowTracking.global' ]
      t0( 'saveFlowTrackingGlobalConfig:', entity, root, options )
      for cmd in lines:
         cmds.addCommand( cmd )

   # Associating flow tracker per interface is not supported for some types.
   if ftrType in { ftrTypeCpuQueue, ftrTypeMirrorOnDrop }:
      # BUG857332 - add swCap for flow tracker per intf support
      return
   # interface flow tracker config
   cfgEgressIntfNames = []
   if saveAll or saveAllDetail:
      intfTypes = [ intf.tagLong for intf in [ EthPhyAutoIntfType,
                                                LagAutoIntfType,
                                                SwitchAutoIntfType,
                                                TunnelAutoIntfType,
                                                DpsAutoIntfType ] ]
      cfgIntfNames = list( allIntfNames( requireMounts,
                                   explicitIntfTypes=intfTypes,
                                   includeSubIntf=True ) )
      cfgEgressIntfNames = cfgIntfNames
   else:
      cfgIntfNames = list( entity.flowTrackerIntfConfig )
      cfgEgressIntfNames = list( entity.flowTrackerEgressIntfConfig )

   t1( "cfgIntfNames", cfgIntfNames )
   for intfName in cfgIntfNames:
      ftrIntf = entity.flowTrackerIntfConfig.get( intfName )
      if ftrType == ftrTypeInbandTelemetry:
         continue
      if ftrIntf:
         sampleMode = ""
         if ftrIntf.sampleMode == sampleModeFiltered:
            sampleMode = " filtered"
         cmd = "flow tracker {} {}{}".format( ftrTypeKwStr[ ftrType ],
                                              ftrIntf.trackerName,
                                              sampleMode )
      else:
         cmd = "no flow tracker %s" % ftrTypeKwStr[ ftrType ]

      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      intfCmds = mode[ 'FlowTracker.config' ]
      intfCmds.addCommand( cmd )
   if ftrType != ftrTypeSampled:
      return
   for intfName in cfgEgressIntfNames:
      ftrIntf = entity.flowTrackerEgressIntfConfig.get( intfName )
      if ftrIntf:
         cmd = "flow tracker {} {} {}".format( ftrTypeKwStr[ ftrType ],
                                               egressStr,
                                               ftrIntf.trackerName )
      else:
         cmd = "no flow tracker {} {}".format( ftrTypeKwStr[ ftrType ],
                                               egressStr )
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      intfCmds = mode[ 'FlowTracker.config' ]
      intfCmds.addCommand( cmd )

@CliSave.saver( 'FlowTracking::Config',
                getSampledFlowTrackingCliConfigPath(),
                requireMounts=( mplsHwCapabilitiesPath,
                                bridgingHwCapabilitiesPath, ) )
def saveSampledFlowTrackingGlobalConfig( entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfig(
      ftrTypeSampled, entity, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::Config',
                getHardwareFlowTrackingCliConfigPath() )
def saveHardwareFlowTrackingGlobalConfig( entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfig(
      ftrTypeHardware, entity, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::Config',
                getInbandTelemetryFlowTrackingCliConfigPath() )
def saveInbandTelemetryFlowTrackingGlobalConfig(
      entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfig(
      ftrTypeInbandTelemetry, entity, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::MirrorOnDropConfig',
                getMirrorOnDropFlowTrackingCliConfigPath(),
                requireMounts=( ftrCapabilitiesPathPrefix + ftrTypeMirrorOnDrop, ) )
def saveMirrorOnDropFlowTrackingGlobalConfig(
      entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfig(
      ftrTypeMirrorOnDrop, entity, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::Config',
                getCpuQueueFlowTrackingCliConfigPath(),
                requireMounts=( ftrCapabilitiesPathPrefix + ftrTypeCpuQueue, ) )
def saveCpuQueueFlowTrackingGlobalConfig(
      entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfig(
      ftrTypeCpuQueue, entity, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::Config',
                getDfwFlowTrackingCliConfigPath(),
                requireMounts=( ftrCapabilitiesPathPrefix + ftrTypeDfw, ) )
def saveDfwFlowTrackingGlobalConfig( entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfig( ftrTypeDfw, entity, root, requireMounts, options )

def saveFlowTrackingGlobalConfigTrailer(
      ftrType, entity, root, requireMounts, options ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   lines = showFlowTrackingTrailer( config=entity, ftrType=ftrType, cliSave=True,
                             saveAll=saveAll, saveAllDetail=saveAllDetail )

   if lines is not None:
      mode = root[ FlowTrackingCliSaveMode ].getOrCreateModeInstance(
                                                      ftrTypeKwStr[ ftrType ] )
      cmds = mode[ 'FlowTracking.trailer' ]
      t0( 'saveFlowTrackingGlobalConfigTrailer:', entity, root, options )
      for cmd in lines:
         cmds.addCommand( cmd )

@CliSave.saver( 'FlowTracking::Config',
                getSampledFlowTrackingCliConfigPath() )
def saveSampledFlowTrackingGlobalConfigTrailer(
      entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfigTrailer(
      ftrTypeSampled, entity, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::Config',
                getHardwareFlowTrackingCliConfigPath() )
def saveHardwareFlowTrackingGlobalConfigTrailer(
      entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfigTrailer(
      ftrTypeHardware, entity, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::Config',
                getInbandTelemetryFlowTrackingCliConfigPath() )
def saveInbandTelemetryFlowTrackingGlobalConfigTrailer(
      entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfigTrailer(
      ftrTypeInbandTelemetry, entity, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::MirrorOnDropConfig',
                getMirrorOnDropFlowTrackingCliConfigPath() )
def saveMirrorOnDropFlowTrackingGlobalConfigTrailer(
      entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfigTrailer(
      ftrTypeMirrorOnDrop, entity, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::Config',
                getCpuQueueFlowTrackingCliConfigPath() )
def saveCpuQueueFlowTrackingGlobalConfigTrailer(
      entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfigTrailer(
      ftrTypeCpuQueue, entity, root, requireMounts, options )

@CliSave.saver( 'FlowTracking::Config', getDfwFlowTrackingCliConfigPath() )
def saveDfwFlowTrackingGlobalConfigTrailer(
      entity, root, requireMounts, options ):
   saveFlowTrackingGlobalConfigTrailer(
      ftrTypeDfw, entity, root, requireMounts, options )
