#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
import Tracing
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.TrafficPolicyCliSave import TrafficPolicySaver
from FlowspecCliLib import ( flowspecCapable,
                             getFlowspecCmdFrom,
                             getFlowspecIntfInterfaceSetCmdFrom )
from CliMode.TrafficPolicy import ( ActionsModeBase,
                                    MatchRuleModeBase,
                                    TrafficPolicyModeBase )
from CliMode.Flowspec import (
   FlowspecModeBase,
   FlowspecPoliciesModeBase,
   POLICYMAP_FEATURE,
)
from TrafficPolicyLib import ActionType, _denyActionToCmd, _policeActionToCmd
from RouteMapLib import getExtCommValue, commValueToPrint, CommunityType

from TypeFuture import TacLazyType
from Toggles import (
   RoutingLibToggleLib,
   PolicyMapToggleLib,
)

t0 = Tracing.trace0

UnresolvedNexthopAction = TacLazyType( "PolicyMap::PbrUnresolvedNexthopAction" )

#----------------------------------------------------------------------------
# Returns the list of interface names which support flowspec.
#----------------------------------------------------------------------------
def allFlowspecIntfNames( requireMounts ):
   assert requireMounts
   intfConfigDir = requireMounts[ 'interface/config/all' ]
   return [ intfName for intfName in intfConfigDir.intfConfig
            if flowspecCapable( intfName ) ]

#----------------------------------------------------------------------------
# CliSave saver to convert flowspec/cliConfig and produce the running-config
#----------------------------------------------------------------------------
IntfConfigMode.addCommandSequence( 'Flowspec.intf' )

@CliSave.saver( 'Flowspec::CliConfig', 'flowspec/cliConfig',
                requireMounts=( 'interface/config/all', ) )
def saveFlowspecConfig( flowspecConfig, root, requireMounts, options ):
   configuredIntf = []
   t0( "saveAll", options.saveAll, "saveAllDetail", options.saveAllDetail )
   if options.saveAll:
      configuredIntf = allFlowspecIntfNames( requireMounts )
   else:
      configuredIntf = flowspecConfig.enabledIntf
   for intf in configuredIntf:
      intfMode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = intfMode[ 'Flowspec.intf' ]
      intf = flowspecConfig.enabledIntf.get( intf, None )
      ipVersion = intf.ipVersion if intf else None
      if ipVersion is None:
         ipVersion = Tac.Value( 'Flowspec::IpVersion' )
      command = getFlowspecCmdFrom( ipVersion )
      sourceVrf = intf.sourceVrfOverride if intf else None
      if sourceVrf:
         command += f' source vrf {sourceVrf} override'
      if options.saveAll or ipVersion:
         cmds.addCommand( command )

   if RoutingLibToggleLib.toggleBgpFlowspecInterfaceSetEnabled():
      if not options.saveAll:
         configuredIntf = flowspecConfig.intfInterfaceSetConfig.keys()
      for intf in configuredIntf:
         intfConfig = flowspecConfig.intfInterfaceSetConfig.get( intf, None )
         intfMode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
         cmds = intfMode[ 'Flowspec.intf' ]
         command = getFlowspecIntfInterfaceSetCmdFrom( intfConfig )
         cmds.addCommand( command )

class FlowspecSaveMode( FlowspecModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FlowspecModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( FlowspecSaveMode,
                                       after=[ IntfConfigMode ] )
FlowspecSaveMode.addCommandSequence( 'Flowspec.Flowspec' )
FlowspecSaveMode.addCommandSequence( 'Flowspec.NexthopConfig' )
FlowspecSaveMode.addCommandSequence( 'Flowspec.hardwarePersistent' )

@CliSave.saver( 'Pbr::PbrConfig', 'flowspec/input/cli' )
def saveFlowspecModeConfig( flowspecPbrConfig, root, requireMounts, options ):
   dropDelayed = UnresolvedNexthopAction.dropDelayed
   dropImmediate = UnresolvedNexthopAction.dropImmediate

   flowspecMode = root[ FlowspecSaveMode ].getSingletonInstance()
   cmds = flowspecMode[ 'Flowspec.NexthopConfig' ]

   if flowspecPbrConfig.pbrUnresolvedNexthopAction == dropDelayed:
      cmds.addCommand( 'next-hop unresolved drop' )
   elif flowspecPbrConfig.pbrUnresolvedNexthopAction == dropImmediate:
      cmds.addCommand( 'next-hop unresolved drop immediate' )
   elif options.saveAll:
      cmds.addCommand( 'no next-hop unresolved' )

FlowspecSaveMode.addCommandSequence( 'Flowspec.CountersPollInterval' )

@CliSave.saver( 'Pbr::PbrConfig', 'flowspec/input/cli' )
def saveCountersPollIntervalConfig( flowspecPbrConfig, root, requireMounts,
                                    options ):
   flowspecMode = root[ FlowspecSaveMode ].getSingletonInstance()
   cmds = flowspecMode[ 'Flowspec.CountersPollInterval' ]
   if interval := flowspecPbrConfig.counterPollInterval:
      cmds.addCommand(
         f'counters poll interval {int( interval )} seconds' )
   elif options.saveAll and \
        PolicyMapToggleLib.toggleFlowspecSharkCountersEnabled():
      cmds.addCommand( 'no counters poll interval' )

FlowspecSaveMode.addCommandSequence( 'Flowspec.InterfaceSetCountLimit' )

@CliSave.saver( 'Flowspec::CliConfig', 'flowspec/cliConfig' )
def saveInterfaceSetcountLimit( flowspecConfig, root, requireMounts, options ):
   if RoutingLibToggleLib.toggleBgpFlowspecInterfaceSetEnabled():
      flowspecMode = root[ FlowspecSaveMode ].getSingletonInstance()
      cmds = flowspecMode[ 'Flowspec.InterfaceSetCountLimit' ]
      if flowspecConfig.interfaceSetCountLimit != 0:
         intSetCountLimCmd = 'interface-set count limit ' + \
            str( flowspecConfig.interfaceSetCountLimit )
         cmds.addCommand( intSetCountLimCmd )
      elif options.saveAll:
         cmds.addCommand( 'no interface-set count limit' )

@CliSave.saver( 'Flowspec::CliConfig', 'flowspec/cliConfig' )
def saveHardwarePersistent( flowspecConfig, root, requireMounts, options ):
   if PolicyMapToggleLib.toggleFlowspecConfigDrivenEnabled():
      flowspecPresent = flowspecConfig.isConfigDriven
      flowspecMode = root[ FlowspecSaveMode ].getSingletonInstance()
      cmds = flowspecMode[ 'Flowspec.hardwarePersistent' ]
      if flowspecPresent:
         cmds.addCommand( 'hardware persistent' )
      elif options.saveAll:
         cmds.addCommand( 'no hardware persistent' )

class FlowspecPoliciesSaveMode( FlowspecPoliciesModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FlowspecPoliciesModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class FlowspecPolicySaveMode( TrafficPolicyModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TrafficPolicyModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, None )

   @property
   def configPolicyKeyword( self ):
      return 'policy'

   def instanceKey( self ):
      return self.trafficPolicyName

class FlowspecPolicyMatchRuleSaveMode( MatchRuleModeBase, CliSave.Mode ):
   def __init__( self, param ):
      MatchRuleModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, None )

   def instanceKey( self ):
      return self.prio

   @classmethod
   def useInsertionOrder( cls ):
      # because `instanceKey` is overridden with prio
      return True

class FlowspecPolicyActionsSaveMode( ActionsModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ActionsModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, None )

   def skipIfEmpty( self ):
      return True

class FlowspecPolicySaver( TrafficPolicySaver ):
   ''' Flowspec definition of the traffic-policy saver '''
   trafficPoliciesMode = FlowspecPoliciesSaveMode
   trafficPolicyMode = FlowspecPolicySaveMode
   defaultActionsMode = None
   matchRuleMode = FlowspecPolicyMatchRuleSaveMode
   actionsRuleMode = FlowspecPolicyActionsSaveMode
   packetTypeMatchMode = None

   def trafficPolicyModeCmds( self, policyMode ):
      return policyMode[ 'Flowspec.policyConfig' ]

   def matchModeCmds( self, matchMode ):
      return matchMode[ 'Flowspec.matchConfig' ]

   def actionModeCmds( self, actionsMode ):
      return actionsMode[ 'Flowspec.actionConfig' ]

   def defaultActionsModeCmds( self, defaultActionsMode ):
      return None

   def actionsToCmds( self, actions ):
      # Flowspec provides a custom action mapping as it defines the
      # redirect route-target action now known to traffic-policy.
      actionCmds = []
      for actionType, action in actions.items():
         if actionType == ActionType.deny:
            cmd = _denyActionToCmd( action )
            actionCmds.append( cmd )
         elif actionType == ActionType.police:
            cmd = _policeActionToCmd( action )
            actionCmds.append( cmd )
         elif actionType == 'setDscp':
            cmd = f'set dscp {action.dscp}'
            actionCmds.append( cmd )
         elif actionType == ActionType.setNexthop:
            nh = action.nexthop.keys()[ 0 ].stringValue
            cmd = f'redirect next-hop {nh}'
            actionCmds.append( cmd )
         elif actionType == ActionType.redirectVrfRouteTarget:
            rtExtComm = getExtCommValue( action.vrfRt )
            rtStr = commValueToPrint( rtExtComm,
                                      CommunityType.communityTypeExtended,
                                      asdotConfigured=True )
            assert rtStr.startswith( 'rt ' )
            cmd = f'redirect route-target {rtStr[3:]}'
            actionCmds.append( cmd )
         else:
            raise NotImplementedError( "Unsupported action" )
      return sorted( actionCmds )

   def save( self ):
      # Need to override this method from the base type as it assumes the
      # policies configuration is off of global config mode. The flowspec policies
      # are an additional level down: global->flow-spec->policies.
      allPolicies = []
      if self.entity.pmapType is not None:
         allPolicies = sorted( self.entity.pmapType.pmap )
      # we have some config, go to the right mode
      flowspecMode = self.root[ FlowspecSaveMode ].getSingletonInstance()
      policiesMode = \
         flowspecMode[ self.trafficPoliciesMode ].getSingletonInstance()

      # Output any traffic policies
      for policy in allPolicies:
         self.saveTrafficPolicy( policy, policiesMode )

if RoutingLibToggleLib.toggleBgpFlowspecStaticPolicyEnabled():
   FlowspecSaveMode.addCommandSequence( 'Flowspec.Policies' )
   FlowspecSaveMode.addChildMode( FlowspecPoliciesSaveMode )
   FlowspecPolicySaveMode.addCommandSequence( 'Flowspec.policyConfig' )
   FlowspecPoliciesSaveMode.addChildMode( FlowspecPolicySaveMode )
   FlowspecPolicyMatchRuleSaveMode.addCommandSequence( 'Flowspec.matchConfig' )
   FlowspecPolicySaveMode.addChildMode( FlowspecPolicyMatchRuleSaveMode,
                                        after=[ 'Flowspec.policyConfig' ] )
   FlowspecPolicyActionsSaveMode.addCommandSequence( 'Flowspec.actionConfig' )
   FlowspecPolicyMatchRuleSaveMode.addChildMode( FlowspecPolicyActionsSaveMode,
                                                 after=[ 'Flowspec.matchConfig' ] )

@CliSave.saver( 'Flowspec::FlowspecPolicyConfig',
                'flowspec/policy/config' )
def saveFlowspecPolicyConfig( entity, root, requireMounts, options ):
   if not RoutingLibToggleLib.toggleBgpFlowspecStaticPolicyEnabled():
      return
   cliDumper = FlowspecPolicySaver( entity, root, requireMounts, options,
                                    POLICYMAP_FEATURE )
   cliDumper.save()
