# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Arnet
import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliMode.Generator import (
   GeneratorsModeBase,
   ProfilesModeBase,
)
from EoamTypes import (
   enumToRateUnitMap,
   FeatureEnabledEnum,
)
from GeneratorCliUtils import (
   generatorTypeKwLinkQualification,
)

class GeneratorsMode( GeneratorsModeBase, CliSave.Mode ):
   def __init__( self, param ):
      GeneratorsModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class ProfilesMode( ProfilesModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ProfilesModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( GeneratorsMode, after=[ IntfConfigMode ] )
GeneratorsMode.addCommandSequence( 'Eoam.featureConfig' )
GeneratorsMode.addChildMode( ProfilesMode, before=[ 'Eoam.featureConfig' ] )
ProfilesMode.addCommandSequence( 'Eoam.profileConfig' )
@CliSave.saver( 'FlowGenerator::FeatureConfig', 'generator/featureConfig' )
def saveFeatureConfig( entity, root, requireMounts, options ):
   generatorMode = root[ GeneratorsMode ].getOrCreateModeInstance(
      generatorTypeKwLinkQualification )
   cmds = generatorMode[ 'Eoam.featureConfig' ]
   if FeatureEnabledEnum.linkQualification in entity.featureEnabled:
      cmds.addCommand( 'no disabled' )
   elif options.saveAll or options.saveAllDetail:
      cmds.addCommand( 'disabled' )

@CliSave.saver( 'Sat::SatProfilesConfig', 'sat/profile/cliConfig' )
def saveSatProfileConfig( entity, root, requireMounts, options ):
   for profileName in sorted( entity.txConfig ):
      profile = entity.txConfig.get( profileName )
      generatorMode = root[ GeneratorsMode ].getOrCreateModeInstance(
         generatorTypeKwLinkQualification )
      profileMode = generatorMode[ ProfilesMode ].\
            getOrCreateModeInstance(
            ( generatorTypeKwLinkQualification, profileName ) )
      cmds = profileMode[ 'Eoam.profileConfig' ]

      if profile.packetRateAndUnit:
         cmds.addCommand( 'traffic rate %d %s' % \
                          ( profile.packetRateAndUnit.packetRateValue,
                            enumToRateUnitMap[ profile.packetRateAndUnit.\
                                               packetRateUnit ] ) )
      elif options.saveAll:
         cmds.addCommand( 'no traffic rate' )

      if profile.pktSize:
         cmds.addCommand( 'packet size %d bytes' % profile.pktSize )
      elif options.saveAll:
         cmds.addCommand( 'no packet size' )

      if profile.duration:
         cmds.addCommand( 'traffic duration %d seconds' % profile.duration )
      elif options.saveAll:
         cmds.addCommand( 'no traffic duration' )

GeneratorsMode.addCommandSequence( 'Eoam.profileIntf', after=[ ProfilesMode ],
                                   before=[ 'Eoam.featureConfig' ] )
@CliSave.saver( 'Sat::IntfConfig', 'sat/intf/cliConfig' )
def saveIntfSatProfileConfig( entity, root, requireMounts, options ):
   for intf in Arnet.sortIntf( entity.intfList ):
      intfInfo = entity.intfList.get( intf )
      mode = root[ GeneratorsMode ].getOrCreateModeInstance(
         generatorTypeKwLinkQualification )
      cmds = mode[ 'Eoam.profileIntf' ]

      if intfInfo.profile:
         cmds.addCommand( f'interface {intf} profile {intfInfo.profile}' )
