#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.Gnmi import MgmtGnmiMode, GnmiTransportMode, GrpctunnelTransportMode
import AuthnUserPriorityCli
from IpLibConsts import DEFAULT_VRF

class MgmtGnmiSaveMode( MgmtGnmiMode, CliSave.Mode ):
   def __init__( self, param ):
      MgmtGnmiMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class GnmiTransportSaveMode( GnmiTransportMode, CliSave.Mode ):
   def __init__( self, param ):
      GnmiTransportMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class GrpctunnelTransportSaveMode( GrpctunnelTransportMode, CliSave.Mode ):
   def __init__( self, param ):
      GrpctunnelTransportMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( MgmtGnmiSaveMode )
MgmtGnmiSaveMode.addCommandSequence( "Mgmt.gnmi" )

MgmtGnmiSaveMode.addChildMode( GnmiTransportSaveMode )
GnmiTransportSaveMode.addCommandSequence( 'Mgmt.gnmi.transport' )

MgmtGnmiSaveMode.addChildMode( GrpctunnelTransportSaveMode )
GrpctunnelTransportSaveMode.addCommandSequence( 'Mgmt.gnmi.grpctunnel' )

@CliSave.saver( "Gnmi::Config", "mgmt/gnmi/config",
      requireMounts = ( 'mgmt/octa/config', ) )
def saveGnmi( gnmiConfig, root, requireMounts, options ):
   octaConfig = requireMounts[ 'mgmt/octa/config' ]
   parentMode = root[ MgmtGnmiSaveMode ].getSingletonInstance()
   cmds = parentMode[ "Mgmt.gnmi" ]
   if octaConfig.enabled:
      cmds.addCommand( "provider eos-native" )
   elif options.saveAll:
      cmds.addCommand( "no provider eos-native" )
   if gnmiConfig.setPersistence:
      cmds.addCommand( "operation set persistence" )
   elif options.saveAll:
      cmds.addCommand( "no operation set persistence" )

   for name in gnmiConfig.endpoints:
      mode = parentMode[ GnmiTransportSaveMode ].getOrCreateModeInstance(
         name )
      cmds = mode[ 'Mgmt.gnmi.transport' ]
      endpoint = gnmiConfig.endpoints[ name ]

      if not endpoint.enabled:
         cmds.addCommand( "shutdown" )
      elif options.saveAll:
         cmds.addCommand( "no shutdown" )

      if endpoint.sslProfile != '':
         cmds.addCommand( "ssl profile %s" % endpoint.sslProfile )
      elif options.saveAll:
         cmds.addCommand( 'no ssl profile' )

      if endpoint.port != endpoint.portDefault or options.saveAll:
         cmds.addCommand( "port %s" % endpoint.port )

      if ( endpoint.connectionLimit != endpoint.connectionLimitDefault or
          options.saveAll ):
         cmds.addCommand( "connection limit %s" % endpoint.connectionLimit )

      if endpoint.vrfName != DEFAULT_VRF or options.saveAll:
         cmds.addCommand( "vrf %s" % endpoint.vrfName )

      if endpoint.serviceAcl != '':
         cmds.addCommand( 'ip access-group %s' % endpoint.serviceAcl )
      elif options.saveAll:
         cmds.addCommand( 'no ip access-group' )

      if endpoint.qosDscp:
         cmds.addCommand( "qos dscp %s" % endpoint.qosDscp )

      if endpoint.authorization:
         cmds.addCommand( "authorization requests" )

      if endpoint.accounting:
         cmds.addCommand( "accounting requests" )
      elif options.saveAll:
         cmds.addCommand( 'no accounting requests' )

      if endpoint.useNotifSendTime:
         cmds.addCommand( "notification timestamp send-time" )
      elif options.saveAll:
         cmds.addCommand( "no notification timestamp" )

      if endpoint.listenAddrs:
         addrs = [ str( i ) for i in sorted( endpoint.listenAddrs ) ]
         cmds.addCommand( "listen-addresses %s" % " ".join( addrs ) )
      elif options.saveAll:
         cmds.addCommand( "no listen-addresses" )

      if ( endpoint.authnUsernamePriority.values() !=
            AuthnUserPriorityCli.defaultAuthnUserPriority ):
         elms = [ elm.source if elm.source !=
               AuthnUserPriorityCli.usernameSourceType.x509Oid else
               elm.x509Oid for elm in endpoint.authnUsernamePriority.values() ]
         CLITokens = ( [ AuthnUserPriorityCli.authnUsernamePriorityToCLI[ elm ]
            if elm in AuthnUserPriorityCli.authnUsernamePriorityToCLI else
            f"x509-{elm}" for elm in elms ] )
         cmds.addCommand( "authentication username priority %s" % " ".join(
            CLITokens ) )
      elif options.saveAll:
         cmds.addCommand( f"authentication username priority "
               f"{AuthnUserPriorityCli.X509_SPIFFE} "
               f"{AuthnUserPriorityCli.METADATA} "
               f"{AuthnUserPriorityCli.X509_COMMON_NAME}" )

   for name in gnmiConfig.grpctunnelClients:
      mode = parentMode[ GrpctunnelTransportSaveMode ].getOrCreateModeInstance(
                           name )
      cmds = mode[ 'Mgmt.gnmi.grpctunnel' ]
      client = gnmiConfig.grpctunnelClients[ name ]

      if not client.enabled:
         cmds.addCommand( 'shutdown' )
      elif options.saveAll:
         cmds.addCommand( 'no shutdown' )

      if client.vrf != DEFAULT_VRF:
         cmds.addCommand( f'vrf {client.vrf}' )
      elif options.saveAll:
         cmds.addCommand( 'no vrf' )

      if client.tunnelSslProfile != '':
         cmds.addCommand( f'tunnel ssl profile {client.tunnelSslProfile}' )
      elif options.saveAll:
         cmds.addCommand( 'no tunnel ssl profile' )

      if client.gnmiSslProfile != '':
         cmds.addCommand( f'gnmi ssl profile {client.gnmiSslProfile}' )
      elif options.saveAll:
         cmds.addCommand( 'no gnmi ssl profile' )

      if ( client.destinationHostAndPort.hostname != '' and
           client.destinationHostAndPort.port != 0 ):
         cmds.addCommand( 'destination {} port {}'.format(
            client.destinationHostAndPort.hostname,
            client.destinationHostAndPort.port ) )
      elif options.saveAll:
         cmds.addCommand( 'no destination' )

      if ( client.localInterfaceAndPort.hostname != '' and
           client.localInterfaceAndPort.port != 0 ):
         cmds.addCommand( 'local interface {} port {}'.format(
            client.localInterfaceAndPort.hostname,
            client.localInterfaceAndPort.port ) )
      elif client.localInterfaceAndPort.hostname != '':
         cmds.addCommand( 'local interface {}'.format(
            client.localInterfaceAndPort.hostname ) )
      elif options.saveAll:
         cmds.addCommand( 'no local interface' )

      if client.targetId != '':
         cmds.addCommand( f'target {client.targetId}' )
      elif options.saveAll:
         cmds.addCommand( 'no target' )

