# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpAddr
from CliMode.IcmpErrorResponderMode import (
      IcmpErrExtBaseMode,
      IcmpErrExtVrfBaseMode )
from CliSavePlugin.RouterGeneralCliSave import (
   getRouterGeneralVrfMode,
   RouterGeneralCliSaveMode,
   RouterGeneralVrfCliSaveMode )

from Toggles.IcmpResponderToggleLib import (
      toggleIcmpExtendedErrorResponderEnabled,
      toggleIcmpExtendedErrorResponderIpv6Enabled,
      toggleIcmpExtendedErrorResponderNdVrfEnabled )
import CliSave
from TypeFuture import TacLazyType

TristateBool = TacLazyType( 'Ark::TristateBoolean' )
NodeIdCodePoint = TacLazyType( 'IcmpResponder::ErrorResponder::NodeIdCodePoint' )

class IcmpErrExtCliSaveMode( IcmpErrExtBaseMode, CliSave.Mode ):
   def enterCmd( self ):
      return 'icmp error extensions'

   def __init__( self, param ):
      IcmpErrExtBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class IcmpErrExtVrfCliSaveMode( IcmpErrExtVrfBaseMode, CliSave.Mode ):
   def enterCmd( self ):
      return 'icmp error extensions'

   def __init__( self, param ):
      IcmpErrExtVrfBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

if toggleIcmpExtendedErrorResponderEnabled():
   RouterGeneralCliSaveMode.addChildMode( IcmpErrExtCliSaveMode,
                                          after=[ 'config' ] )
   IcmpErrExtCliSaveMode.addCommandSequence( 'icmpErrExt' )

   RouterGeneralVrfCliSaveMode.addChildMode( IcmpErrExtVrfCliSaveMode,
                                             after=[ 'vrfConfig' ] )
   IcmpErrExtVrfCliSaveMode.addCommandSequence( 'icmpErrExt' )


@CliSave.saver( 'IcmpResponder::ErrorResponder::Config',
                'icmp/errorResponder/config' )
def saveIcmpErrorResponderConfig( entity, root, requireMounts, options ):
   if not toggleIcmpExtendedErrorResponderEnabled():
      return

   hasGlobalConfigNonDefault = entity.enabled or entity.addHostName
   hasVrfConfigNonDefault = entity.vrfConfig
   hasRouterGeneralConfigNonDefault = ( hasGlobalConfigNonDefault or
                                        hasVrfConfigNonDefault )
   saveAll = options.saveAll

   if saveAll or hasRouterGeneralConfigNonDefault:
      mode = root[ RouterGeneralCliSaveMode ].getSingletonInstance()

      # No need to make this conditional on hasGlobalConfigNonDefault
      # show running-config all should at least show defaults, because mode-enter
      # doesn't mean enable. This is how other router-general config works.
      globalMode = mode[ IcmpErrExtCliSaveMode ].getSingletonInstance()
      globalCmds = globalMode[ 'icmpErrExt' ]

      if toggleIcmpExtendedErrorResponderNdVrfEnabled():
         if entity.enabled:
            if entity.enabled == TristateBool.valueSet( True ):
               globalCmds.addCommand( 'extensions vrf all' )
            elif entity.enabled == TristateBool.valueSet( False ):
               globalCmds.addCommand( 'extensions vrf all disabled' )
         elif saveAll:
            globalCmds.addCommand( 'no extensions vrf all' )

      if entity.addHostName:
         globalCmds.addCommand( 'params include hostname' )
      elif saveAll:
         globalCmds.addCommand( 'no params include hostname' )

      if entity.nodeIdCodePoint != NodeIdCodePoint.invalid:
         globalCmds.addCommand( f'nodeid codepoint {entity.nodeIdCodePoint}' )
      elif saveAll:
         globalCmds.addCommand( 'no nodeid codepoint' )

      for vrfName in entity.vrfConfig:
         rtrGeneralVrfMode = getRouterGeneralVrfMode( mode, vrfName )
         vrfConfig = entity.vrfConfig[ vrfName ]
         vrfMode = rtrGeneralVrfMode[ IcmpErrExtVrfCliSaveMode ].\
               getOrCreateModeInstance( vrfName )
         vrfCmds = vrfMode[ 'icmpErrExt' ]
         if vrfConfig.enabled == TristateBool.valueSet( True ):
            vrfCmds.addCommand( 'extensions' )
         elif vrfConfig.enabled == TristateBool.valueSet( False ):
            vrfCmds.addCommand( 'extensions disabled' )
         elif saveAll:
            vrfCmds.addCommand( 'no extensions' )

         if vrfConfig.v4AclName != '':
            vrfCmds.addCommand( f'ip access-group {vrfConfig.v4AclName}' )
         elif saveAll:
            vrfCmds.addCommand( 'no ip access-group' )

         if toggleIcmpExtendedErrorResponderIpv6Enabled():
            if vrfConfig.v6AclName != '':
               vrfCmds.addCommand( f'ip6 access-group {vrfConfig.v6AclName}' )
            elif saveAll:
               vrfCmds.addCommand( 'no ip6 access-group' )

         if IpAddr( vrfConfig.nodeIdConfig.v4Address ):
            addr = vrfConfig.nodeIdConfig.v4Address
            vrfCmds.addCommand( f'nodeid ipv4 address {addr}' )
         elif saveAll:
            vrfCmds.addCommand( 'no nodeid ipv4 address' )

         if vrfConfig.nodeIdConfig.v6Address:
            addr = vrfConfig.nodeIdConfig.v6Address
            vrfCmds.addCommand( f'nodeid ipv6 address {addr}' )
         elif saveAll:
            vrfCmds.addCommand( 'no nodeid ipv6 address' )

         if vrfConfig.nodeIdConfig.intfId:
            intfName = vrfConfig.nodeIdConfig.intfId
            vrfCmds.addCommand( f'nodeid interface {intfName}' )
         elif saveAll:
            vrfCmds.addCommand( 'no nodeid interface' )
