#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=superfluous-parens
# pylint: disable=consider-using-f-string

import Tac
import CliSave
from CliSavePlugin import IntfCliSave
from CliSavePlugin.IgmpVrfMode import RouterIgmpBaseConfigMode
from CliSavePlugin.IgmpVrfMode import RouterIgmpVrfConfigMode
from CliSavePlugin.MrouteCliSave import RouterMulticastBaseConfigMode
from CliSavePlugin.MrouteCliSave import RouterMulticastAfConfigMode
from IpLibConsts import DEFAULT_VRF
from McastCommonCliLib import AddressFamily
from McastCommonCliLib import mcastRoutingSupported
from RoutingIntfUtils import allRoutingProtocolIntfNames
from Toggles import IgmpHostProxyToggleLib

IntfCliSave.IntfConfigMode.addCommandSequence( 'IgmpHostProxy.intf', \
      after=[ 'Ira.ipIntf' ])

@CliSave.saver( 'IgmpHostProxy::AllVrfConfig',
      'igmphostproxy/allvrfconfig',
      requireMounts=( 'routing/hardware/status', ) )
def saveIgmpHostProxyConfig( igmpHostProxyVrfConfig, root, requireMounts,
      options ):
   saveVrfConfig( igmpHostProxyVrfConfig, root, requireMounts, options )
   saveIhpAclProcessingRule( igmpHostProxyVrfConfig, root, requireMounts, options )

def saveIhpAclProcessingRule( igmpHostProxyVrfConfig, root, requireMounts,
      options ):
   if not IgmpHostProxyToggleLib.toggleIgmpHostProxyAclEnabled():
      return
   
   if not mcastRoutingSupported( None,
         requireMounts[ 'routing/hardware/status' ] ):
      return

   def getCmd():
      rootMode = root[ RouterMulticastBaseConfigMode ].getSingletonInstance()
      mode = rootMode[ RouterMulticastAfConfigMode ].getOrCreateModeInstance( (
          DEFAULT_VRF, AddressFamily.ipv4 ) )
      cmds = mode[ 'Multicast.vrf.af.config' ]
      return cmds
   
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   cmd = 'host-proxy access-list rules priority order sequence-number'
   cmdAll = 'no host-proxy access-list rules priority order sequence-number'
   if igmpHostProxyVrfConfig.aclRuleOrderSequenceNum:
      cmds = getCmd()
      cmds.addCommand( cmd )
   elif saveAll or saveAllDetail:
      cmds = getCmd()
      cmds.addCommand( cmdAll )

def saveVrfConfig( igmpHostProxyVrfConfig, root, requireMounts, options ):
   if not mcastRoutingSupported( None,
         requireMounts[ 'routing/hardware/status' ] ):
      return

   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   matchAllCmd = 'host-proxy match mroute all'
   matchIifCmd = 'host-proxy match mroute iif'

   # Handling default VRF separately. This require separate handling
   # to just incline with the cliSave behaviour of ssm aware command present
   # under router igmp

   if DEFAULT_VRF in igmpHostProxyVrfConfig.vrfConfig and \
         igmpHostProxyVrfConfig.vrfConfig[ DEFAULT_VRF ].iifAware is True:
      mode = root[ RouterIgmpBaseConfigMode ].getSingletonInstance()
      cmds = mode[ 'Igmp.config' ]
      cmds.addCommand( matchIifCmd )
   elif saveAll or saveAllDetail:
      mode = root[ RouterIgmpBaseConfigMode ].getSingletonInstance()
      cmds = mode[ 'Igmp.config' ]
      cmds.addCommand( matchAllCmd )

   for vrfName, vrfConfig in igmpHostProxyVrfConfig.vrfConfig.items():
      if vrfName == DEFAULT_VRF:
         continue
      parentMode = root[ RouterIgmpBaseConfigMode ].getSingletonInstance()
      mode = parentMode[ RouterIgmpVrfConfigMode ].getOrCreateModeInstance(
            vrfName )
      cmds = mode[ 'Igmp.vrf.config' ]
      if vrfConfig.iifAware:
         cmds.addCommand( matchIifCmd )
      elif saveAll or saveAllDetail:
         cmds.addCommand( matchAllCmd )

def saveIntfConfig( igmpHostProxyIntfConfig, root, saveAll, saveAllDetail ):
   intf = igmpHostProxyIntfConfig
   mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( intf.intfId )
   cmds = mode[ 'IgmpHostProxy.intf' ]
   starSource = '0.0.0.0'

   if intf.defaultHostProxy:
      cmds.addCommand( "ip igmp host-proxy" )

   # pylint: disable-next=too-many-nested-blocks
   if intf.groupSourceList or intf.aclList or intf.deletedGroupSourceList:   
      if intf.groupSourceList:
         for group, groupSource in intf.groupSourceList.items():
            if groupSource.includeSourceList:
               for incSource in groupSource.includeSourceList:
                  cmd = f"ip igmp host-proxy {group} include {incSource}"
                  cmds.addCommand( cmd )
            if groupSource.excludeSourceList:
               for excSource in groupSource.excludeSourceList:
                  if ( excSource != '0.0.0.0' ):
                     cmd = f"ip igmp host-proxy {group} exclude {excSource}"
                     cmds.addCommand( cmd )
                  else:
                     cmd = "ip igmp host-proxy %s" % group
                     cmds.addCommand( cmd )
            if not groupSource.includeSourceList and not \
                  groupSource.excludeSourceList:
               cmd = "ip igmp host-proxy %s" % group
               cmds.addCommand( cmd )
   
      if intf.aclList:
         for aclName in intf.aclList:
            cmd = "ip igmp host-proxy access-list %s" % aclName
            cmds.addCommand( cmd )

      if intf.deletedGroupSourceList:
         for group, groupSource in intf.deletedGroupSourceList.items():
            if starSource in groupSource.excludeSourceList:
               cmd = "no ip igmp host-proxy %s" % group
               cmds.addCommand( cmd )
               continue
            if groupSource.includeSourceList:
               for incSource in groupSource.includeSourceList:
                  cmd = f"no ip igmp host-proxy {group} include {incSource}"
                  cmds.addCommand( cmd )
            if groupSource.excludeSourceList:
               for excSource in groupSource.excludeSourceList:
                  if ( excSource != '0.0.0.0' ):
                     cmd = "no ip igmp host-proxy %s exclude %s" % \
                           ( group, excSource )
                     cmds.addCommand( cmd )

   if ( intf.unsolicitedReportIntvl.val != \
         intf.unsolicitedReportIntvl.defaultValue ):
      cmds.addCommand( 'ip igmp host-proxy report-interval %d' %
                       ( intf.unsolicitedReportIntvl.val ) )
   elif saveAll:
      cmds.addCommand( 'no ip igmp host-proxy report-interval' )

   if ( intf.igmpVersion.val != intf.igmpVersion.defaultValue ):
      cmds.addCommand( 'ip igmp host-proxy version %s' %
                       ( intf.igmpVersion.val ) )
   elif saveAll:
      cmds.addCommand( 'no ip igmp host-proxy version' )

@CliSave.saver( 'IgmpHostProxy::Config', 'igmphostproxy/config',
                requireMounts = ( 'routing/hardware/status',
                                  'interface/config/all',
                                  'interface/status/all' ) )
def saveConfig( igmpHostProxyConfig, root, requireMounts, options ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   # Save the default config only if the platform supports multicast routing
   if not mcastRoutingSupported( None,
                                 requireMounts[ 'routing/hardware/status' ] ):
      saveAll = False
      saveAllDetail = False

   if saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( requireMounts,
                                                  includeEligible=True )
   elif saveAll:
      # Routing configuration is allowed on switchports as well. 
      # Save configuration on all routing protocol interfaces and switchports
      # with non-default config.
      cfgIntfNames = set( allRoutingProtocolIntfNames( requireMounts ) )
      cfgIntfNames.update( igmpHostProxyConfig.intfConfig )
   else:
      cfgIntfNames = igmpHostProxyConfig.intfConfig

   for intfName in cfgIntfNames:
      intfConfig = igmpHostProxyConfig.intfConfig.get( intfName )
      if not intfConfig:
         if saveAll:
            intfConfig = \
                  Tac.newInstance( 'IgmpHostProxy::IntfConfig', \
                  intfName )
            mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( \
                  intfConfig.intfId )
            cmds = mode[ 'IgmpHostProxy.intf' ]

            cmds.addCommand( 'no ip igmp host-proxy' )
      else:
         saveIntfConfig( intfConfig, root, saveAll, saveAllDetail )
