# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from RoutingIntfUtils import allIpIntfNames
import Tac, Arnet

CliSave.GlobalConfigMode.addCommandSequence( 'Ipv6RouterAdvt.config' )
IntfConfigMode.addCommandSequence( 'Ipv6RouterAdvt.intf6', after=[ 'Ira.intf6' ] )

CliSave.GlobalConfigMode.addCommandSequence( 'Ipv6RouterAdvt.NdPrefixProfile',
                                             before=[ IntfConfigMode ] )

#-----------------------------------------------------------------------
# Saver methods
#----------------------------------------------------------------------

def _addPrefixOptions( cmdStr, prefix ):
   if prefix.noAdvertise:
      cmdStr += ' no-advertise'
   else:
      def printValidLifetime():
         if prefix.validLifetime == 0xffffffff:
            return ' infinite'
         else:
            return f' {prefix.validLifetime}'
      validLifetimePrinted = False
      prefixDefault = Tac.Type( 'RouterAdvt::PrefixDefault' )
      if prefix.validLifetime != prefixDefault.validLifetime:
         validLifetimePrinted = True
         cmdStr += printValidLifetime()
      if prefix.preferredLifetime != prefixDefault.preferredLifetime:
         # Cli syntax requires validLifetime to be specified before
         # preferredLifetime
         if not validLifetimePrinted:
            cmdStr += printValidLifetime()
         if prefix.preferredLifetime == 0xffffffff:
            cmdStr += ' infinite'
         else:
            cmdStr += f' {prefix.preferredLifetime}'
      if prefix.noOnLink:
         cmdStr += ' no-onlink'
      if prefix.noAutoConfig:
         cmdStr += ' no-autoconfig'
   return cmdStr

def saveIntfConfig( entity, root, options ):

   saveAll = options.saveAll
   if entity.name.startswith( "Internal" ):
      # Internal interface configuration. Abort
      # BUG944 - need a more general way of doing this
      return
   
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Ipv6RouterAdvt.intf6' ]

   if str( entity.intfId ).startswith( 'Management' ):
      # For management interfaces, the default is ra suppress all.
      # So, display 'no ipv6 nd ra suppress' if raSuppress is not set.
      if not entity.raSuppress:
         cmds.addCommand( 'no ipv6 nd ra disabled' )
      elif not entity.raSuppressRouterSolicitationResponse:
         cmds.addCommand( 'ipv6 nd ra disabled' )
      elif saveAll:
         cmds.addCommand( 'ipv6 nd ra disabled all' )
   else:
      if entity.raSuppress:
         cmdStr = 'ipv6 nd ra disabled'
         if entity.raSuppressRouterSolicitationResponse:
            cmdStr += ' all'
         cmds.addCommand( cmdStr )
      elif saveAll:
         cmds.addCommand( "no ipv6 nd ra disabled" )

   def _raIntervalValueStr():
      valueStr = f'msec {entity.raIntervalMaxMsec}'
      if entity.raIntervalMinMsec:
         valueStr += f' {entity.raIntervalMinMsec}'
      return valueStr
   
   def _addCmd( attr, cmdStr ):
      if attr:
         cmds.addCommand( cmdStr )
      elif saveAll:
         cmds.addCommand( 'no ' + cmdStr )

   if entity.raIntervalMaxMsec != entity.raIntervalMaxDefault or saveAll:
      cmds.addCommand( 'ipv6 nd ra interval ' + _raIntervalValueStr() )

   if entity.raLifetime != entity.raLifetimeDefault or saveAll:
      cmds.addCommand( f'ipv6 nd ra lifetime {entity.raLifetime}' )

   _addCmd( entity.raMtuSuppress, 'ipv6 nd ra mtu suppress' )
   _addCmd( entity.raNsIntervalUnspecified, 'ipv6 nd ra ns-interval unspecified' )
   _addCmd( entity.managedConfigFlag, 'ipv6 nd managed-config-flag' )
   _addCmd( entity.otherConfigFlag, 'ipv6 nd other-config-flag' )

   if entity.reachableTime != entity.reachableTimeDefault or saveAll:
      cmds.addCommand( f'ipv6 nd reachable-time {entity.reachableTime}' )

   if entity.routerPreference != 'medium' or saveAll:
      cmds.addCommand( 'ipv6 nd router-preference ' + entity.routerPreference )

   prefixStrList = sorted( prefix.stringValue for prefix in entity.prefixes )
   for prefixStr in prefixStrList:
      cmdStr = 'ipv6 nd prefix ' + prefixStr
      prefix = entity.prefixes[ Arnet.Ip6AddrWithMask( prefixStr ) ]
      if prefix.profile:
         cmdStr += f' profile {prefix.profile}'
         cmds.addCommand( cmdStr )
      else:
         cmds.addCommand( _addPrefixOptions( cmdStr, prefix ) )

   if entity.dnsServersLifetime != entity.dnsServersLifetimeDefault or saveAll:
      cmds.addCommand( 
         f'ipv6 nd ra dns-servers lifetime {entity.dnsServersLifetime}' )
   
   dnsServerStrList = sorted( dnsServer.stringValue
                              for dnsServer in entity.dnsServers )
   for dnsServerStr in dnsServerStrList:
      cmdStr = 'ipv6 nd ra dns-server ' + dnsServerStr
      dnsServerEntry = entity.dnsServers[ Arnet.Ip6Addr( dnsServerStr ) ]
      if dnsServerEntry.lifetime != dnsServerEntry.lifetimeDefault:
         cmdStr += f' lifetime {dnsServerEntry.lifetime}'
      cmds.addCommand( cmdStr )
      
   if entity.dnsSuffixesLifetime != entity.dnsSuffixesLifetimeDefault or saveAll:
      cmds.addCommand( 
         f'ipv6 nd ra dns-suffixes lifetime {entity.dnsSuffixesLifetime}' )
   dnsSuffixesList = sorted( dnsSuffix for dnsSuffix in entity.dnsSuffixes )
   for dnsSuffix in dnsSuffixesList:
      cmdStr = 'ipv6 nd ra dns-suffix ' + dnsSuffix
      dnsSuffixEntry = entity.dnsSuffixes[  dnsSuffix  ]
      if dnsSuffixEntry.lifetime != dnsSuffixEntry.lifetimeDefault:
         cmdStr += f' lifetime {dnsSuffixEntry.lifetime}'
      cmds.addCommand( cmdStr )

   if entity.hopLimit != entity.hopLimitDefault or saveAll:
      cmds.addCommand( f'ipv6 nd ra hop-limit {entity.hopLimit}' )

   if entity.networkBootFileUrl:
      url = entity.networkBootFileUrl
      cmdStr = f'ipv6 nd ra experimental network-boot {url}'
      cmds.addCommand( cmdStr )

   for prefix in sorted( entity.raRouteInfoOptionConfig ):
      config = entity.raRouteInfoOptionConfig[ prefix ]
      cmdStr = (
         f'ipv6 nd ra route-information {config.prefix.stringValue} '
         f'preference {config.preference}'
      )
      if config.lifetime == 0xffffffff:
         cmdStr += ' lifetime infinite'
      else:
         cmdStr += f' lifetime {config.lifetime} seconds'
      cmds.addCommand( cmdStr )

   if entity.pref64:
      cmdStr = f'ipv6 nd ra pref64 {entity.pref64.prefix}'
      if entity.pref64.lifetime is not None:
         cmdStr += f' lifetime {entity.pref64.lifetime} seconds'
      elif saveAll:
         raIntervalMaxSec = entity.raIntervalMaxMsec // 1000
         cmdStr += f' lifetime {3 * raIntervalMaxSec } seconds'
      cmds.addCommand( cmdStr )

@CliSave.saver( 'RouterAdvt::Config', 'routing6/routerAdvt/config',
                requireMounts = ( 'interface/config/all', 'interface/status/all' ) )
def saveIp6Config( entity, root, requireMounts, options ):

   cmds = root[ 'Ipv6RouterAdvt.config' ]

   # Ra Consistency Knob
   if entity.raConsistency:
      cmds.addCommand( 'ipv6 nd ra consistency-check default' )
   elif options.saveAll:
      cmds.addCommand( 'no ipv6 nd ra consistency-check default' )

   # IntfConfig
   if options.saveAllDetail:
      cfgIntfNames = allIpIntfNames( requireMounts, includeEligible=True )
   elif options.saveAll:
      # IPv6 configuration is allowed on switchport interfaces as well. 
      # Display config on all ip interfaces and switchports with non-default
      # config.
      cfgIntfNames = set( allIpIntfNames( requireMounts ) )
      cfgIntfNames.update( entity.intf )
   else:
      cfgIntfNames = entity.intf

   for profile in sorted( entity.prefixProfiles ):
      cmdStr = f'ipv6 nd prefix profile {profile}'
      root[ 'Ipv6RouterAdvt.NdPrefixProfile' ].addCommand(
            _addPrefixOptions( cmdStr, entity.prefixProfiles[ profile ] ) )

   for intfName in cfgIntfNames:
      if options.intfFilter and intfName not in options.intfFilter:
         continue
      intfConfig = entity.intf.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'RouterAdvt::IntfConfig', intfName )
            if 'Management' in intfName:
               intfConfig.raSuppress = True
               intfConfig.raSuppressRouterSolicitationResponse = True
         else:
            continue
      saveIntfConfig( intfConfig, root, options )

