# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave, Arnet # pylint: disable-msg=W0611
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.IraCliSaveCommon import (
      saveOptimizePrefixConfig,
      IPV6,
)
# Need to import this to get the ConfigTag.config config sequence
import CliSavePlugin.ConfigTagCliSave # pylint: disable=unused-import
from IpLibConsts import DEFAULT_VRF
from RoutingConsts import (
      defaultStaticRouteName,
      defaultStaticRoutePreference,
      defaultStaticRouteTag,
      defaultStaticRouteMetric,
      defaultStaticRouteWeight
)
from RoutingIntfUtils import allIpIntfNames
import Tac
import os
import Toggles.IraToggleLib

IntfConfigMode.addCommandSequence( 'Ira.intf6', after=[ 'Arnet.intf',
   'Arnet.l2l3barrier', 'Ira.ipIntf' ] )

CliSave.GlobalConfigMode.addCommandSequence( 'Ira.routes6',
                                             after=[ 'ConfigTag.config',
                                                IntfConfigMode ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.routing6',
                                             after=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.ip6AggregateAddr',
                                             after=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.ip6EcmpNexthopIndex',
                                             after=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.ip6nd',
                                             after=[ IntfConfigMode ] )
#-----------------------------------------------------------------------
# Saver methods
#----------------------------------------------------------------------
@CliSave.saver( 'Routing6::VrfConfigDir', 'routing6/vrf/config' )
def saveVrfRouting( entity, root, requireMounts, options ):
   for vrfname in sorted( entity.vrfConfig ):
      saveRoutingAttribute( entity.vrfConfig[ vrfname ], root, options, vrfname )

def saveRoutingAttribute( entity, root, options, vrfName ):
   saveAll = options.saveAll
   vrfString = ' vrf %s' % vrfName if vrfName else ''
   if entity.routing:
      root[ 'Ira.routing6' ].addCommand( "ipv6 unicast-routing%s" % vrfString )
   elif saveAll:
      root[ 'Ira.routing6' ].addCommand( "no ipv6 unicast-routing%s" % vrfString )

@CliSave.saver( 'Routing6::VrfRouteListConfigDir', 'routing6/vrf/routelistconfig',
      requireMounts=( 'configTag/configTagIdState', ),
      commandTagSupported=True )
def saveVrfRoutingTables( entity, root, requireMounts, options ):
   for vrfname in sorted( entity.vrfConfig ):
      saveStaticRoutes( entity.vrfConfig[ vrfname ], root, requireMounts,
            options, vrfname )

routing6Consts = Tac.Value( "Ira::RoutingConsts", defaultStaticRoutePreference,
                            defaultStaticRouteTag, defaultStaticRouteName,
                            defaultStaticRouteMetric, defaultStaticRouteWeight )

@CliSave.saver( 'Routing6::Config', 'routing6/config' )
def saveRouting( entity, root, requireMounts, options ):
   saveRoutingAttribute( entity, root, options, vrfName=None )

@CliSave.saver( 'Routing6::RouteListConfig', 'routing6/routelistconfig',
      requireMounts=( 'configTag/configTagIdState', ),
      commandTagSupported=True )
def saveRoutingTable( entity, root, requireMounts, options ):
   saveStaticRoutes( entity, root, requireMounts, options, vrfName=None )

def saveStaticRoutes( entity, root, requireMounts, options, vrfName ):
   saveAll = options.saveAll
   commandTagIdState = requireMounts[ 'configTag/configTagIdState' ]
   vrfString = ' vrf %s' % vrfName if vrfName else ''
   routeCmd = 'ipv6 route%s ' % vrfString
   vrfName = vrfName.strip() if vrfName else 'default'
   saver = Tac.Value( "Ira::StaticRoutesCliSaver" )
   cmds = saver.save( entity, saveAll, routeCmd, routing6Consts,
         vrfName, commandTagIdState )
   for cmd in cmds.split( '\n' ):
      if cmd != '':
         root[ 'Ira.routes6' ].addCommand( cmd )

@CliSave.saver( 'Routing6::Hardware::Config', 'routing6/hardware/config',
      requireMounts = ( 'routing6/config', 'routing6/hardware/status', ) )
def saveIp6RoutingHardwareConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   # Save Aggregate prefix config
   for aggPrefix in sorted( entity.aggregateRoute ):
      cmd = \
         'ipv6 hardware fib aggregate-address %s summary-only software-forward' % \
         ( aggPrefix.stringValue )
      root[ 'Ira.ip6AggregateAddr' ].addCommand( cmd )

   routing6Config = requireMounts[ 'routing6/config' ]
   if not entity.icmpUnreachable:
      cmd = 'ipv6 icmp rate-limit-unreachable 0'
      root[ 'Ira.routing6' ].addCommand( cmd )
   elif ( saveAll and routing6Config.routing ) or options.saveAllDetail:
      cmd = 'no ipv6 icmp rate-limit-unreachable 0'
      root[ 'Ira.routing6' ].addCommand( cmd )

   routing6HwStatus = requireMounts[ 'routing6/hardware/status' ]
   saveOptimizePrefixConfig( entity, root, options, IPV6, routing6HwStatus )

@CliSave.saver( 'Routing6::Config', 'routing6/config' )
def saveIpv6IcmpRedirectConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   if saveAll and entity.sendRedirects:
      root[ 'Ira.routing' ].addCommand( "ipv6 icmp redirect")
   elif not entity.sendRedirects:
      root[ 'Ira.routing' ].addCommand( "no ipv6 icmp redirect")

@CliSave.saver( 'Routing6::Hardware::Config', 'routing6/hardware/config',
      requireMounts=( 'routing6/hardware/status', ) )
def saveFibFilterConfig( entity, root, requireMounts, options ):
   routingHwStatus = requireMounts[ 'routing6/hardware/status' ]
   if not routingHwStatus.fibSuppressionSupported:
      return
   saveAll = options.saveAll
   if( entity.fibSuppression ): # pylint: disable=superfluous-parens
      root[ 'Ira.routing' ].addCommand(
                        "ipv6 fib compression redundant-specifics filter" )
   elif( saveAll ): # pylint: disable=superfluous-parens
      root[ 'Ira.routing' ].addCommand(
                        "no ipv6 fib compression redundant-specifics filter" )

@CliSave.saver( 'Ip6::Config', 'ip6/config',
                requireMounts = ( 'interface/config/all', 'interface/status/all',
                                  'routing6/hardware/status' ) )
def saveIp6Config( entity, root, requireMounts, options ):
   routingHwStatus = requireMounts[ 'routing6/hardware/status' ]

   def enhancedDadSupported():
      # Treat as supported in breadth tests; skip if kernel doesn't support
      return ( 'CLISAVE_TEST_STARTUP_CONFIG' in os.environ ) or \
         os.path.isfile( '/proc/sys/net/ipv6/conf/all/enhanced_dad' )

   if enhancedDadSupported() and \
      ( entity.enhancedDad != entity.enhancedDadDefault or options.saveAll ):
      cmds = root[ 'Ira.ip6nd' ]
      if entity.enhancedDad:
         cmds.addCommand( 'ipv6 nd enhanced-dad default' )
      else:
         cmds.addCommand( 'no ipv6 nd enhanced-dad default' )

   # IntfConfig
   if options.saveAllDetail:
      cfgIntfNames = allIpIntfNames( requireMounts, includeEligible=True )
   elif options.saveAll:
      # IP configuration is allowed on switchport interfaces as well.
      # Display config on all ip interfaces and switchports with non-default
      # config.
      ipIntfNames = allIpIntfNames( requireMounts )
      cfgIntfNames = set( ipIntfNames )
      cfgIntfNames.update( entity.intf )
   else:
      cfgIntfNames = entity.intf

   for intfName in cfgIntfNames:
      if options.intfFilter and intfName not in options.intfFilter:
         continue
      intfConfig = entity.intf.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Ip6::IntfConfig', intfName )
            intfConfig.l3Config = Tac.newInstance( 'L3::Intf::Config', intfName )
         else:
            continue
      saveIntfConfig( intfConfig, root, options, routingHwStatus )

def saveIntfConfig( entity, root, options, routingHwStatus ):
   saveAll = options.saveAll
   if entity.intfId.startswith( "Internal" ):
      # Internal interface configuration. Abort
      # BUG944 - need a more general way of doing this
      return

   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Ira.intf6' ]

   if entity.vrf != DEFAULT_VRF:
      assert entity.vrf != ''

   if entity.enable:
      cmds.addCommand( "ipv6 enable" )
   elif saveAll:
      cmds.addCommand( "no ipv6 enable" )

   acceptDad = Tac.Type( "Ip6::AcceptDad" )

   if entity.acceptDad != entity.acceptDadDefault or saveAll:
      if entity.acceptDad == acceptDad.dadDisabled:
         cmds.addCommand( "ipv6 nd dad disabled" )
      elif entity.acceptDad == acceptDad.dadGlobal:
         if saveAll:
            cmds.addCommand( "default ipv6 nd dad" )
      elif entity.acceptDad == acceptDad.dadEnabled:
         cmds.addCommand( "ipv6 nd dad" )
      elif entity.acceptDad == acceptDad.dadStrict:
         cmds.addCommand( "ipv6 nd dad strict" )

   if entity.acceptUnsolicitedNa:
      cmds.addCommand( "ipv6 nd na unsolicited accept" )
   elif saveAll:
      cmds.addCommand( "no ipv6 nd na unsolicited accept" )

   if entity.addrSource == Tac.Type( "Ip6::AddrSource" ).slaac:
      cmds.addCommand( "ipv6 address auto-config" )
   elif len( entity.addr ) == 0 and saveAll:
      cmds.addCommand( "no ipv6 address" )
   else:
      for addr, addrInfo in entity.addr.items():
         if entity.useNsSourceAddr and entity.nsSourceAddr == addr.address:
            cmds.addCommand( "ipv6 address %s ns source" % ( addr.stringValue ) )
         elif addrInfo.deprecated:
            cmds.addCommand( "ipv6 address %s source least-preferred" %
                  ( addr.stringValue ) )
         else:
            cmds.addCommand( "ipv6 address %s" % ( addr.stringValue ) )

   if entity.acceptRaDefRtr:
      cmds.addCommand( "ipv6 nd ra rx accept default-route" )
   elif saveAll:
      cmds.addCommand( "no ipv6 nd ra rx accept default-route" )

   if entity.acceptRaRtrPref:
      cmds.addCommand( "ipv6 nd ra rx accept route-preference" )
   elif saveAll:
      cmds.addCommand( "no ipv6 nd ra rx accept route-preference" )

   if not entity.linkLocalAddrWithMask.address.isZero:
      cmds.addCommand( "ipv6 address %s link-local" % \
                     ( entity.linkLocalAddrWithMask.stringValue ) )

   if not entity.attachedRoutes:
      cmds.addCommand( 'no ipv6 attached-routes' )
   elif saveAll:
      cmds.addCommand( 'ipv6 attached-routes' )

   if entity.nsRetransmitInterval != entity.nsRetransmitIntervalDefault:
      cmds.addCommand( 'ipv6 nd ns-interval %u' %( entity.nsRetransmitInterval ) )

   if entity.urpf.mode != 'disable':
      if entity.urpf.mode == 'strict':
         cmd = 'ipv6 verify unicast source reachable-via rx'
      elif entity.urpf.mode == 'loose':
         cmd = 'ipv6 verify unicast source reachable-via any'
         if not entity.urpf.allowDefaultRoute and \
               Toggles.IraToggleLib.toggleUrpfIgnoreDefaultRouteEnabled() and \
               routingHwStatus.urpfLooseNonDefaultSupported:
            cmd += ' non-default'
         if entity.urpf.lookupVrf != "" and \
               Toggles.IraToggleLib.toggleSplitVrfUrpfResolutionEnabled() and \
               routingHwStatus.urpfLooseLookupVrfSupported:
            cmd += ' lookup-vrf ' + entity.urpf.lookupVrf
      else:
         # TODO BUG642564: Use entity.urpf.allowDefaultRoute under the strict mode
         # when strictDefault mode is removed.
         cmd = 'ipv6 verify unicast source reachable-via rx allow-default'

      if entity.urpf.acl != "":
         cmd += ' exception-list ' + entity.urpf.acl
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( "no ipv6 verify unicast" )

@CliSave.saver( 'Routing6::Hardware::Config', 'routing6/hardware/config',
                requireMounts = ( 'routing6/hardware/status', ) )
def saveResilientEcmp6Config( entity, root, requireMounts, options ):
   routingHwStatus = requireMounts[ 'routing6/hardware/status' ]
   if not routingHwStatus.resilientEcmpSupported:
      return
   cmds = root['Ira.routing6']
   if not entity.resilientEcmpPrefix and not entity.policyBasedResilientEcmp:
      if options.saveAll or options.saveAllDetail:
         cmds.addCommand( 'no ipv6 hardware fib ecmp resilience' )
      return
   for prefix, ecmpConfig in entity.resilientEcmpPrefix.items() :
      orderedKw = ' ordered' if ecmpConfig.orderedVias else ''
      cmds.addCommand(
            "ipv6 hardware fib ecmp resilience %s capacity %d redundancy %d%s"
               % ( prefix.stringValue , ecmpConfig.capacity, \
                      ecmpConfig.redundancy, orderedKw ) )
   if entity.policyBasedResilientEcmp:
      # ordered keyword is not supported for policy based resilient ecmp
      ecmpConfig = entity.policyBasedResilientEcmp
      cmds.addCommand(
         "ipv6 hardware fib ecmp resilience marked capacity %d redundancy %d"
         % ( ecmpConfig.capacity, ecmpConfig.redundancy ) )
