# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from IpLibTypes import vrfNameSortKey
import Tac
import collections
from itertools import chain

IPV4 = 4
IPV6 = 6

OptimizeProfileType = Tac.Type( 'Routing::Hardware::OptimizeProfileType' )

def saveOptimizePrefixConfig( entity, root, options, ipVersion, rhs ):
   saveAll = options.saveAll
   if ipVersion == IPV4:
      optimizePrefixRoot = root[ 'Ira.routing' ]
      ipVersionStr = 'ip'
      urpfExactMatchSupported = rhs.urpfExactMatchSupported
      prefixLenSupportedInExactMatch = rhs.prefixLenSupportedInExactMatch
   else:
      optimizePrefixRoot = root[ 'Ira.routing6' ]
      ipVersionStr = 'ipv6'
      urpfExactMatchSupported = False
      prefixLenSupportedInExactMatch = rhs.prefixLenSupportedInExactMatch

   # Generating the CLI commands looking at routing/hardware/config is not
   # starightforward for VRF specific optimize configurations. This is because
   # we have an entry for each VRF in vrfOptimizeProfileName or
   # vrfPrefixLensInExactMatch collection. We need to figure out which all VRFs
   # have the same configuration. Our CLI accepts multiple VRFs in the same
   # command for simplicity, so we need to generate commands having multiple
   # VRFs with the same configuration. For profiles this is easy, we only need
   # to figure out all the VRFs have the same profile. For
   # vrfPrefixLensInExactMatch collection, we need to figure out all VRFs have
   # the same 'prefix-length a expand b compress c' configuration. We do this
   # by generating the command for each VRF, putting them in a dictionary and
   # finally iterating over all the keys of that dictionary.
   # optimizeProfileConfig and optimizePrefixLenConfig are those dictionaries
   # whose key is the setting and the value is the list of VRFs having that
   # configuration.
   globalOptimizeProfile = None
   globalOptimizePrefixLenKey = None
   optimizeProfileConfig = collections.defaultdict( list )
   optimizePrefixLenConfig = collections.defaultdict( list )

   vrfs = sorted( chain(
      entity.vrfPrefixLensInExactMatch, entity.vrfOptimizeProfileName ),
                  key=vrfNameSortKey )

   # optimizeConfigurations is the list of tuples which indicate the global
   # configuration and configurations for the VRFs.
   # - first item being the VRF (empty string for global setting)
   # - second item being the profile (empty string if no profile exists)
   # - third item being the PrefixLensInExactMatch object (None if no such
   #   object).
   optimizeConfigurations = [
      ( '', entity.globalOptimizeProfileName, entity.globalPrefixLensInExactMatch ),
   ]
   for vrf in vrfs:
      optimizeConfigurations.append(
         ( vrf, entity.vrfOptimizeProfileName.get( vrf ),
           entity.vrfPrefixLensInExactMatch.get( vrf ) ) )

   optimizeProfileSettingList = rhs.optimizeProfileSettingList
   optimizeProfileSettingColl = None
   if optimizeProfileSettingList:
      optimizeProfileSettingColl = optimizeProfileSettingList.optimizeProfileSetting

   # Iterate over optimizeConfigurations and add to
   # - optimizeProfileConfig dictionary if a profile is applied.
   # - optimizePrefixLenConfig dictionary if a prefix length configuration is
   #   applied.
   for vrf, optimizeProfileName, prefixLensInExactMatch in optimizeConfigurations:
      if ( rhs.optimizeProfileSettingList and
           optimizeProfileSettingColl and
           optimizeProfileName and
           optimizeProfileName in optimizeProfileSettingColl ):
         profileName = optimizeProfileName
         if profileName == OptimizeProfileType.urpfInternet:
            profileName = "urpf-internet"
         if vrf:
            optimizeProfileConfig[ profileName ].append( vrf )
         else:
            globalOptimizeProfile = profileName
      elif prefixLenSupportedInExactMatch:
         prefixes = ''
         expandPrefixes = ''
         compressPrefixes = ''
         # Iterating over sorted collection to make sure it is deterministic
         # when we have multiple prefix lengths optimized.
         for prefixLen in sorted( prefixLensInExactMatch.prefixLen.keys() ):
            hwPrefixLen = prefixLensInExactMatch.prefixLen[ prefixLen ]
            if hwPrefixLen == prefixLen:
               prefixes = '%s %d' % ( prefixes, prefixLen )
            elif hwPrefixLen > prefixLen:
               expandPrefixes = '%s %d' % ( expandPrefixes, prefixLen )
            else:
               compressPrefixes = '%s %d' % ( compressPrefixes, prefixLen )
         urpf = ''
         if urpfExactMatchSupported and entity.urpfExactMatchEnabled:
            urpf = 'urpf'
         if prefixes:
            key = ( prefixes, expandPrefixes, compressPrefixes, urpf )
            if vrf:
               optimizePrefixLenConfig[ key ].append( vrf )
            else:
               globalOptimizePrefixLenKey = key

   def addProfileCommand( profile, vrfs=None ):
      vrfCmd = ''
      if vrfs:
         # Sorting the VRFs to make sure the order of VRFs in the generated
         # command is deterministic.
         vrfCmd = ' vrf %s' % ' '.join( sorted( vrfs, key=vrfNameSortKey ) )
      cmd = '%s hardware fib optimize%s prefixes profile %s' % (
            ipVersionStr, vrfCmd, profile )
      optimizePrefixRoot.addCommand( cmd )

   def addPrefixLenCommand( optimizePrefixLenKey, vrfs=None ):
      vrfCmd = ''
      if vrfs:
         # Sorting the VRFs to make sure the order of VRFs in the generated
         # command is deterministic.
         vrfCmd = ' vrf %s' % ' '.join( sorted( vrfs, key=vrfNameSortKey ) )
      prefixes, expandPrefixes, compressPrefixes, urpf = optimizePrefixLenKey
      if prefixes:
         cmd = '%s hardware fib optimize%s prefix-length%s' % (
               ipVersionStr, vrfCmd, prefixes )
         if expandPrefixes:
            cmd = '%s expand %s' % ( cmd, expandPrefixes )
         if compressPrefixes:
            cmd = '%s compress %s' % ( cmd, compressPrefixes )
         if urpf:
            cmd = '%s %s' % ( cmd, urpf )
         optimizePrefixRoot.addCommand( cmd )

   if globalOptimizeProfile or globalOptimizePrefixLenKey:
      if globalOptimizeProfile:
         addProfileCommand( globalOptimizeProfile )
      else:
         addPrefixLenCommand( globalOptimizePrefixLenKey )
   elif saveAll or options.saveAllDetail:
      # NOTE: 'prefixes' keyword is not present in IPv6 REM command. IPv4
      # REM supports an option disable-vrf because of which it needs to have
      # 'no ip hardware fib optimize prefixes' as opposed to IPv6 having
      # only 'no ipv6 hardware fib optimize'.
      if ipVersion == IPV4:
         cmd = 'no ip hardware fib optimize prefixes'
      else:
         cmd = 'no ipv6 hardware fib optimize'
      optimizePrefixRoot.addCommand( cmd )

   for profile, vrfs in sorted( optimizeProfileConfig.items(),
                                key=lambda p: p[ 0 ] ):
      addProfileCommand( profile, vrfs )
   for optimizePrefixLenKey, vrfs in sorted( optimizePrefixLenConfig.items(),
                                             key=lambda p: p[ 0 ] ):
      addPrefixLenCommand( optimizePrefixLenKey, vrfs )
