# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.IraCliSave import RouterKernelCliSaveMode
from CliMode.Ira import VrfDefinitionBaseMode, VrfDefaultsBaseMode

class VrfDefinitionCliSaveMode( VrfDefinitionBaseMode, CliSave.Mode ):
   def __init__( self, vrfName ):
      VrfDefinitionBaseMode.__init__( self, vrfName )
      CliSave.Mode.__init__( self, vrfName )

# VRF definition needs to come before interfaces so that the interfaces can
# use the VRFs

CliSave.GlobalConfigMode.addChildMode( VrfDefinitionCliSaveMode,
                                       before=[ IntfConfigMode ] )
VrfDefinitionCliSaveMode.addCommandSequence( 'vrfDef' )

class VrfDefaultsCliSaveMode( VrfDefaultsBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      VrfDefaultsBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

# Place the defaults for all VRFs just ahead of the VRF definitions
CliSave.GlobalConfigMode.addChildMode( VrfDefaultsCliSaveMode,
      before=[ VrfDefinitionCliSaveMode ] )
VrfDefaultsCliSaveMode.addCommandSequence( 'vrfDefaults' )

CliSave.GlobalConfigMode.addCommandSequence( 'Ira.icmpParam',
                                             after=[ IntfConfigMode ] )

class VrfAfMode( CliSave.Mode ):
   def enterCmd( self ):
      # FIXFIXFIX BUG29310.  need to create an overridable
      # exitCmd() in the CliSaveMode stuff, so that we can insert the
      # other config commands for this mode in between
      return 'address-family %s\n   exit-address-family' % self.param_

VrfDefinitionCliSaveMode.addChildMode( VrfAfMode )
VrfAfMode.addCommandSequence( 'vrfAfDef' )

@CliSave.saver( 'Ip::AllVrfConfig', 'ip/vrf/config' )
def saveVrfConfig( entity, root, requireMounts, options ):
   for vrfName in sorted( entity.vrf ):
      vrf = entity.vrf[ vrfName ]
      mode = root[ VrfDefinitionCliSaveMode ].getOrCreateModeInstance( vrfName )
      # Set the same deprecated value for all vrfs
      cmds = mode[ 'vrfDef' ]

      #  extract description from Sysdb and print it as CLI configuration
      if vrf.description != '':
         cmds.addCommand( 'description %s' % vrf.description )
      elif options.saveAll:
         cmds.addCommand( 'no description' )

      # extract expected route count from Sysdb and print it as CLI configuration
      if vrf.expectedRouteCount.isSet:
         cmds.addCommand( 'route expected count %d' % vrf.expectedRouteCount.value )
      elif options.saveAll or options.saveAllDetail:
         cmds.addCommand( 'no route expected count' )

         # VRF AF mode is not supported at this time, see comment in
         # IraVrfCli.py for why.
   for vrfName in sorted( entity.kernelNetConfig ):
      mode = root[ RouterKernelCliSaveMode ].getOrCreateModeInstance( vrfName )
      cmds = mode[ 'Router.kernel' ]
      # extract reservedPorts from Sysdb and print it as CLI configuration
      reservedPorts = entity.kernelNetConfig[ vrfName ].reservedPorts
      if reservedPorts != '\n':
         cmds.addCommand( 'reserved-ports %s' % reservedPorts )
      elif options.saveAll or options.saveAllDetail:
         cmds.addCommand( 'no reserved-ports' )
   # The default value of icmpBurstSize is 1000000
   cmds = root[ 'Ira.icmpParam' ]
   if entity.icmpBurstSize != 1000000:
      cmds.addCommand( 'ip icmp send limit burst %s packets'
         % entity.icmpBurstSize )
   elif options.saveAll or options.saveAllDetail:
      cmds.addCommand( 'no ip icmp send limit burst' )

   # The default value of icmpMsgsPerSec is 1000000
   if entity.icmpMsgsPerSec != 1000000:
      cmds.addCommand( 'ip icmp send limit rate %s pps'
         % entity.icmpMsgsPerSec )
   elif options.saveAll or options.saveAllDetail:
      cmds.addCommand( 'no ip icmp send limit rate' )


#      for af in sorted( vrf.addressFamily ):
#         if vrf.addressFamily[ af ]:
#            mode[ VrfAfMode ].getOrCreateModeInstance( af )

@CliSave.saver( 'Ip::VrfDefaultsConfig', 'ip/vrf/defaults/config' )
def saveVrfDefaultsConfig( entity, root, requireMounts, options ):

   if entity.expectedRouteCount.isSet:
      mode = root[ VrfDefaultsCliSaveMode ].getOrCreateModeInstance( entity )
      cmds = mode[ 'vrfDefaults' ]
      cmds.addCommand( 'route expected count %d' % entity.expectedRouteCount.value )
   elif options.saveAll or options.saveAllDetail:
      mode = root[ VrfDefaultsCliSaveMode ].getOrCreateModeInstance( entity )
      cmds = mode[ 'vrfDefaults' ]
      cmds.addCommand( 'no route expected count' )

@CliSave.saver( 'Tac::Dir', 'ip/vrf/routeDistinguisherInputDir/config' )
def saveVrfRdConfig( entity, root, requireMounts, options ):
   vrfDefinitionRouteDistinguisherInput = entity.get( 'vrfDefinition' )
   if not vrfDefinitionRouteDistinguisherInput:
      return
   for vrfName in sorted( vrfDefinitionRouteDistinguisherInput.routeDistinguisher ):
      rd = vrfDefinitionRouteDistinguisherInput.routeDistinguisher[ vrfName ]
      # Not sure if mode is created here instead of saveVrfConfig() but if it is
      # the case, the correct deprecated mode will be set later by saveVrfConfig()
      mode = root[ VrfDefinitionCliSaveMode ].getOrCreateModeInstance( vrfName )
      cmds = mode[ 'vrfDef' ]
      assert rd != 'INVALID'
      cmds.addCommand( 'rd %s' % ( rd ) )
