# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict

from CliSavePlugin.SystemL1 import SystemL1SaveMode
from Fru.ModuleRange import moduleListToModuleRangeString
from TypeFuture import TacLazyType
import CliSave

MountConstants = TacLazyType( 'L1Profile::MountConstants' )

SystemL1SaveMode.addCommandSequence( 'L1Profile.Apply' )

@CliSave.saver( typename='L1Profile::CliConfig',
                pathPrefix=MountConstants.cliConfigPath() )
def saveL1ProfileApply( cliConfig, root, requireMounts, options ):
   if not cliConfig.cardProfile:
      return

   mode = root[ SystemL1SaveMode ].getOrCreateModeInstance( None )
   cmds = mode[ 'L1Profile.Apply' ]

   # In order to not bloat the user config, card ranges need to be compressed. For
   # example:
   #  module Linecard1 profile X -+
   #                              |--> module Linecard1-2 profile X
   #  module Linecard2 profile X -+
   #
   # To aid processing, cards are first grouped together by their configured
   # card profiles.
   cardProfileToCardSlotIds = defaultdict( set )
   for cardSlotId, cardProfile in cliConfig.cardProfile.items():
      cardProfileToCardSlotIds[ cardProfile.name ].add( cardSlotId )

   # Next, we iterate over each configured card profile, compress all cards
   # configured with that profile into a canonical range and save the resulting
   # command.
   for cardProfile, cardSlotIds in sorted( cardProfileToCardSlotIds.items() ):
      cardProfileRanges = moduleListToModuleRangeString( cardSlotIds )
      for cardProfileRange in cardProfileRanges:
         if cardProfileRange == 'FixedSystem':
            cardProfileRange = 'switch'
         cmds.addCommand( f'module {cardProfileRange} profile {cardProfile}' )
