# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac
import CliSave
import EthIntfUtil
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliMode.L2Protocol import L2ProtocolModeBase, L2ProtocolProfileModeBase
from EbraLib import ( l2ProtocolFromMacAddress, tacL2PtTagFormat, tacL2PtAction,
                      l2PtProtocolToCliToken, l2PtForwardTarget )

#-------------------------------------------------------------------------------
# Object used for saving commands in "l2-protocol" mode.
#-------------------------------------------------------------------------------
class L2ProtocolMode( L2ProtocolModeBase, CliSave.Mode ):
   def __init__( self, param ):
      L2ProtocolModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( L2ProtocolMode )
L2ProtocolMode.addCommandSequence( 'Ebra.l2protocol' )

#-------------------------------------------------------------------------------
# Object used for saving commands in "forwarding profile <profileName>" mode.
#-------------------------------------------------------------------------------
class L2ProtocolProfileMode( L2ProtocolProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      L2ProtocolProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )
L2ProtocolMode.addChildMode( L2ProtocolProfileMode )
L2ProtocolProfileMode.addCommandSequence( 'Ebra.forwardingProfile' )

@CliSave.saver( 'Ebra::L2Pt::L2PtProfileConfig',
                'l2protocolforwarding/profileconfig',
                requireMounts=( 'cli/config', ) )
def saveConfig( l2PtProfileConfig, root, requireMounts, options ):
   if not l2PtProfileConfig.profile and not CliSave.hasComments( 'l2-protocol',
                                                                 requireMounts ):
      return
   profileRoot = root[ L2ProtocolMode ].getOrCreateModeInstance( None )
   for profileName, profile in l2PtProfileConfig.profile.items():
      l2ProtocolProfileMode = profileRoot[ L2ProtocolProfileMode
                                         ].getOrCreateModeInstance( profileName )
      cmds = l2ProtocolProfileMode[ 'Ebra.forwardingProfile' ]
      # For IS-IS forward, there are 5 different DMAC ranges handled in
      # L2Pt code, which results in 4 redundant entries in running-config.
      # Filter out the redundant entries here.
      processedProtocols = []
      for protocolInfo in profile.seqToProtocolInfo.values():
         addrConfig = protocolInfo.l2ProtocolMatch.addrConfig
         if addrConfig:
            address = protocolInfo.l2ProtocolMatch.addr
         else:
            protocol = l2PtProtocolToCliToken[ l2ProtocolFromMacAddress(
               protocolInfo.l2ProtocolMatch.addr ) ]
         tagFormat = protocolInfo.l2ProtocolMatch.tagFormat
         action = protocolInfo.action.action
         if addrConfig or ( protocol, tagFormat ) not in processedProtocols:
            if addrConfig:
               cmd = f'mac-address {address} '
            else:
               cmd = '%s ' % protocol # pylint: disable=consider-using-f-string
            if tagFormat != tacL2PtTagFormat.all:
               cmd += '%s ' % tagFormat # pylint: disable=consider-using-f-string
            if action == tacL2PtAction.trap:
               # pylint: disable-next=consider-using-f-string
               cmd += '%s %s' % ( tacL2PtAction.forward, l2PtForwardTarget() )
            else:
               cmd += '%s' % action # pylint: disable=consider-using-f-string
            cmds.addCommand( cmd )
            if not addrConfig:
               processedProtocols.append( ( protocol, tagFormat ) )

IntfConfigMode.addCommandSequence( 'Ebra.IntfL2ProtocolForwardingProfile' )

@CliSave.saver( 'Ebra::L2Pt::L2PtIntfConfig',
                'l2protocolforwarding/intfconfig',
                requireMounts=( 'interface/config/eth/phy/slice', ) )
def saveIntfConfigs( intfConfig, root, requireMounts, options ):
   if options.saveAll:
      # Get all eligible physical ports
      phyIntfConfigDir = requireMounts[ 'interface/config/eth/phy/slice' ]
      phyIntfNames = EthIntfUtil.allPhyEthernetInterfaceNames( phyIntfConfigDir )
   else:
      phyIntfNames = intfConfig.intfToProfile

   for intfName in phyIntfNames:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ 'Ebra.IntfL2ProtocolForwardingProfile' ]
      profileName = intfConfig.intfToProfile.get( intfName )
      if not profileName:
         if options.saveAll:
            cmds.addCommand( 'no l2-protocol forwarding profile' )
      else:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'l2-protocol forwarding profile %s' % profileName )
