#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements saving the Lag CLI, including Port-Channel interfaces
# and interface-specific configuration.
#-------------------------------------------------------------------------------
import CliSave
import Tac
from CliSavePlugin.IntfCliSave import IntfConfigMode
from RoutingIntfUtils import allIpIntfNames

perlinkMode = Tac.Type( "Bfd::PerlinkMode" )

def isManagement( intfName ):
   return intfName.startswith( "Management" )

def isInternal( intfName ):
   return intfName.startswith( "Internal" )

def intfSupportsPerlink( intfName, ethLagIntfConfigDir ):
   # Helper method to check if intfName supports per-link or not. Only LAG
   # interfaces supports BFD per-link feature. ethLagIntfConfigDir points to
   # 'interface/config/eth/lag' in sysdb.
   return ( intfName in ethLagIntfConfigDir.intfConfig and not \
         isManagement( intfName ) and not isInternal( intfName ) )

@CliSave.saver( 'Bfd::ConfigIntf', 'bfd/config/intf',
                requireMounts = ( 'interface/config/all',
                                  'interface/status/all',
                                  'interface/config/eth/lag' ) )
def saveBfdConfig( entity, root, requireMounts, options ):
   cfgIntfNames = []
   if options.saveAllDetail:
      cfgIntfNames = allIpIntfNames( requireMounts, includeEligible=True )
   elif options.saveAll:
      intfNames = allIpIntfNames( requireMounts )
      cfgIntfNames = set( intfNames )
      cfgIntfNames.update( entity.perLink,
                           entity.neighborAddrIntfConfig,
                           entity.rfc7130Interop )
   else:
      cfgIntfNames = set()
      cfgIntfNames.update( entity.perLink,
                           entity.neighborAddrIntfConfig,
                           entity.rfc7130Interop )

   # Interface config
   for intfName in cfgIntfNames:
      perLink = entity.perLink.get( intfName, perlinkMode.none )
      neighborAddrIntfConfig = entity.neighborAddrIntfConfig.get( intfName, None )
      rfc7130Interop = entity.rfc7130Interop.get( intfName, False )
      if perLink != perlinkMode.none or neighborAddrIntfConfig or \
         rfc7130Interop or options.saveAll:
         _saveBfdIntfConfig( intfName, perLink, neighborAddrIntfConfig,
                             rfc7130Interop, root, requireMounts, options )

def _saveBfdIntfConfig( intfName, perLink, neighborAddrIntfConfig,
                        rfc7130Interop, root, requireMounts, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
   cmds = mode[ 'Bfd.intf' ]
   ethLagIntfConfigDir = requireMounts[ 'interface/config/eth/lag' ]
   if intfSupportsPerlink( intfName, ethLagIntfConfigDir ):
      if neighborAddrIntfConfig:
         if not neighborAddrIntfConfig.neighborAddrV4.isAddrZero:
            # pylint: disable-next=consider-using-f-string
            cmds.addCommand( "bfd neighbor %s" %
                          neighborAddrIntfConfig.neighborAddrV4.stringValue )
         if not neighborAddrIntfConfig.neighborAddrV6.isAddrZero:
            # pylint: disable-next=consider-using-f-string
            cmds.addCommand( "bfd neighbor %s" %
                          neighborAddrIntfConfig.neighborAddrV6.stringValue )
         if options.saveAll and neighborAddrIntfConfig.neighborAddrV4.isAddrZero \
            and neighborAddrIntfConfig.neighborAddrV6.isAddrZero:
            cmds.addCommand( "no bfd neighbor" )            
      elif options.saveAll:
         cmds.addCommand( "no bfd neighbor" )
      
      if perLink == perlinkMode.legacy:
         cmds.addCommand( "bfd per-link" )
      elif perLink == perlinkMode.rfc7130:
         cmds.addCommand( "bfd per-link rfc-7130" )
      elif options.saveAll:
         cmds.addCommand( "no bfd per-link" )

      if rfc7130Interop:
         cmds.addCommand( "bfd rfc-7130 interop" )
      elif options.saveAll:
         cmds.addCommand( "no bfd rfc-7130 interop" )
