# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.LdpMode import (
   GrHelperMode,
   GrSpeakerMode,
   LdpMode,
   MldpMode,
)
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.Security import mgmtSecurityConfigPath
from CliSavePlugin.Security import SecurityConfigMode
from IpLibConsts import DEFAULT_VRF
from collections import defaultdict
from itertools import groupby
import natsort
import ReversibleSecretCli
import Tac
MplsLabel = Tac.Type( 'Arnet::MplsLabel' )
LdpProtoParam = Tac.Type( 'Ldp::LdpProtoParam' )
LdpLabelLocalTerminationMode = Tac.Type( 'Ldp::LdpLabelLocalTerminationMode' )

# -------------------------------------------------------------------------------
# Object used for saving commands in "mpls-ldp" mode.
# -------------------------------------------------------------------------------
class LdpConfigMode( LdpMode, CliSave.Mode ):
   def __init__( self, param ):
      LdpMode.__init__( self )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( LdpConfigMode,
                                       after=[ 'Mpls.config', SecurityConfigMode ] )

LdpConfigMode.addCommandSequence( 'Ldp.config' )
IntfConfigMode.addCommandSequence( 'Ldp.intf', after=[ 'Ira.ipIntf' ] )

class MldpConfigMode( MldpMode, CliSave.Mode ):
   def __init__( self, param ):
      MldpMode.__init__( self )
      CliSave.Mode.__init__( self, param )

LdpConfigMode.addChildMode( MldpConfigMode, after=[ 'Ldp.config' ] )
MldpConfigMode.addCommandSequence( 'Mldp.config' )

class GrHelperConfigMode( GrHelperMode, CliSave.Mode ):
   def __init__( self, param ):
      GrHelperMode.__init__( self )
      CliSave.Mode.__init__( self, param )

class GrSpeakerConfigMode( GrSpeakerMode, CliSave.Mode ):
   def __init__( self, param ):
      GrSpeakerMode.__init__( self )
      CliSave.Mode.__init__( self, param )

LdpConfigMode.addChildMode( GrHelperConfigMode, after=[ 'Ldp.config' ] )
GrHelperConfigMode.addCommandSequence( 'GrHelper.config' )
LdpConfigMode.addChildMode( GrSpeakerConfigMode, after=[ 'Ldp.config' ] )
GrSpeakerConfigMode.addCommandSequence( 'GrSpeaker.config' )

def saveIntfConfig( root, ldpIntfConfigColl, options ):
   for intfName in ldpIntfConfigColl:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ "Ldp.intf" ]
      igpSync = ldpIntfConfigColl[ intfName ].igpSync
      if igpSync == "on":
         cmds.addCommand( "mpls ldp igp sync" )
      elif igpSync == "off":
         cmds.addCommand( "no mpls ldp igp sync" )
      elif options.saveAll:
         cmds.addCommand( "default mpls ldp igp sync" )

def saveIntfLdpEnabled( root, ldpEnabledInterfaces, options ):
   for intfName in ldpEnabledInterfaces:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ "Ldp.intf" ]
      cmds.addCommand( "mpls ldp interface" )

# pylint: disable-msg=C0322
@CliSave.saver( 'Ldp::LdpConfigColl', 'mpls/ldp/ldpConfigColl',
                requireMounts=( 'interface/config/all',
                                'interface/status/all',
                                'acl/cpconfig/cli',
                                mgmtSecurityConfigPath, ) )
def saveLdpConfig( entity, root, requireMounts, options ):
   config = entity.config.get( 'default' )
   if config is None:
      return

   ldpServiceAclConfig = None
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   serviceAclVrfConfig = aclCpConfig.cpConfig[ 'ip' ].serviceAcl
   if DEFAULT_VRF in serviceAclVrfConfig:
      ldpServiceAclConfig = serviceAclVrfConfig[ DEFAULT_VRF ].service.get( 'ldp' )

   if ( config.enabled or config.routerId != '0.0.0.0'
         or config.routerIdIntfId != ''
         or config.ipTransportAddrIntfId != ''
         or config.md5Password
         or config.encodedPassword
         or config.activePasswordIndex
         or config.activePasswordIndexForNeighbor
         or config.linkReadyTimeout != config.linkReadyTimeoutDefault
         or config.linkReadyDelay != config.linkReadyDelayDefault
         or config.ldpProtoParamSetting
         or list( config.staticTarget ) != []
         or config.fecFilterPrefixListName != ''
         or config.helloHoldTime != LdpProtoParam.defaultBasicHelloHoldTime
         or config.helloInterval != LdpProtoParam.defaultBasicHelloInterval
         or config.targetedHelloHoldTime != LdpProtoParam.defaultTargetHelloHoldTime
         or config.targetedHelloInterval != LdpProtoParam.defaultTargetHelloInterval
         or config.mLdpEnabled != False
         or list( config.staticP2mpFecBindingColl ) != []
         or ( ldpServiceAclConfig and ldpServiceAclConfig.aclName )
         or config.grOperFlag.value
         or config.helloRedundancy
         or config.ldpLabelLocalTerminationMode != (
            LdpLabelLocalTerminationMode.implicitNull )
         or config.entropyLabel is True
         or config.allowLuTunnelRedist is True
         or config.endOfLib
         or options.saveAll ):
      mode = root[ LdpConfigMode ].getSingletonInstance()
   else:
      saveIntfConfig( root, config.intfConfigColl, options )
      saveIntfLdpEnabled( root, config.ldpEnabledIntf, options )
      return

   if config.routerId != '0.0.0.0':
      mode[ 'Ldp.config' ].addCommand( 'router-id ' + config.routerId )
   elif config.routerIdIntfId != '':
      mode[ 'Ldp.config' ].addCommand( 'router-id interface '
                                       + config.routerIdIntfId )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no router-id' )

   if config.ipTransportAddrIntfId != '':
      mode[ 'Ldp.config' ].addCommand( 'transport-address interface ' +
                                       config.ipTransportAddrIntfId )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no transport-address' )

   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   if config.md5Password:
      mode[ 'Ldp.config' ].addCommand(
         ReversibleSecretCli.getCliSaveCommand( 'password {}',
                                                securityConfig,
                                                config.md5Password ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no password' )

   for index in sorted( config.encodedPassword ):
      cmd = ReversibleSecretCli.getCliSaveCommand(
         f'authentication index {index} password {{}}',
         securityConfig,
         config.encodedPassword[ index ] )
      mode[ 'Ldp.config' ].addCommand( cmd )

   if config.activePasswordIndex:
      mode[ 'Ldp.config' ].addCommand( 'authentication index {} active'.format(
         config.activePasswordIndex ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no authentication index active' )

   for ipAddr in natsort.natsorted( config.activePasswordIndexForNeighbor ):
      index = config.activePasswordIndexForNeighbor[ ipAddr ]
      mode[ 'Ldp.config' ].addCommand(
            f'neighbor {ipAddr} authentication index {index} active' )

   if config.linkReadyTimeout == 0:
      mode[ 'Ldp.config' ].addCommand( 'no igp sync holddown' )
   elif config.linkReadyTimeout == Tac.endOfTime:
      mode[ 'Ldp.config' ].addCommand( 'igp sync holddown until-established' )
   elif config.linkReadyTimeout != config.linkReadyTimeoutDefault:
      mode[ 'Ldp.config' ].addCommand( 'igp sync holddown ' +
              str( int( config.linkReadyTimeout ) ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'igp sync holddown ' +
              str( int( config.linkReadyTimeoutDefault ) ) )

   if config.linkReadyDelay == 0:
      mode[ 'Ldp.config' ].addCommand( 'no igp sync delay' )
   elif config.linkReadyDelay != config.linkReadyDelayDefault:
      mode[ 'Ldp.config' ].addCommand( 'igp sync delay ' +
              str( int( config.linkReadyDelay ) ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'igp sync delay ' +
              str( int( config.linkReadyDelayDefault ) ) )

   # Following are for hidden commands. Only show when it's configured
   if config.ldpProtoParamSetting:
      mode[ 'Ldp.config' ].addCommand( 'proto-param ' +
                                       hex( config.ldpProtoParamSetting ) )

   for targetIpAddr in natsort.natsorted( config.staticTarget ):
      mode[ 'Ldp.config' ].addCommand( 'neighbor %s targeted' % str( targetIpAddr ) )

   if config.fecFilterPrefixListName != '':
      mode[ 'Ldp.config' ].addCommand( 'fec filter prefix-list %s' %
                                                     config.fecFilterPrefixListName )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no fec filter' )

   # Link hello configuration
   if config.helloHoldTime != LdpProtoParam.defaultBasicHelloHoldTime:
      if config.helloHoldTime == LdpProtoParam.infiniteHelloHoldTime:
         mode[ 'Ldp.config' ].addCommand( 'discovery hello hold-time infinite' )
      else:
         mode[ 'Ldp.config' ].addCommand( 'discovery hello hold-time ' +
                                          str( config.helloHoldTime ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'discovery hello hold-time ' +
                                    str( LdpProtoParam.defaultBasicHelloHoldTime ) )

   if config.helloInterval != LdpProtoParam.defaultBasicHelloInterval:
      mode[ 'Ldp.config' ].addCommand( 'discovery hello interval ' +
                                       str( config.helloInterval ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'discovery hello interval ' +
                                    str( LdpProtoParam.defaultBasicHelloInterval ) )

   # Targeted hello configuration
   if config.targetedHelloHoldTime != LdpProtoParam.defaultTargetHelloHoldTime:
      if config.targetedHelloHoldTime == LdpProtoParam.infiniteHelloHoldTime:
         mode[ 'Ldp.config' ].addCommand(
            'discovery targeted-hello hold-time infinite' )
      else:
         mode[ 'Ldp.config' ].addCommand( 'discovery targeted-hello hold-time ' +
                                          str( config.targetedHelloHoldTime ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'discovery targeted-hello hold-time ' +
                                    str( LdpProtoParam.defaultTargetHelloHoldTime ) )

   if config.targetedHelloInterval != LdpProtoParam.defaultTargetHelloInterval:
      mode[ 'Ldp.config' ].addCommand( 'discovery targeted-hello interval ' +
                                       str( config.targetedHelloInterval ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'discovery targeted-hello interval ' +
                                    str( LdpProtoParam.defaultTargetHelloInterval ) )

   # Service ACL
   if ldpServiceAclConfig and ldpServiceAclConfig.aclName:
      mode[ 'Ldp.config' ].addCommand( 'ip access-group %s' %
                                       ldpServiceAclConfig.aclName )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no ip access-group' )

   # P2mp
   if config.mLdpEnabled or config.staticP2mpFecBindingColl:
      mldpMode = mode[ MldpConfigMode ].getSingletonInstance()
      if config.mLdpEnabled:
         mldpMode[ 'Mldp.config' ].addCommand( 'no shutdown' )
      else:
         mldpMode[ 'Mldp.config' ].addCommand( 'shutdown' )
      staticP2mpFecs = defaultdict( set )
      for p2mpId, entry in config.staticP2mpFecBindingColl.items():
         staticP2mpFecs[ p2mpId.rootIp ].add( ( p2mpId.gOpaqueId,
                                                entry.egressVlanId ) )
      for rootIp, gOpaqueIds in natsort.natsorted( staticP2mpFecs.items() ):
         # itertools magic: Organize the opaque IDs into groups of
         # ( key, ( opaque-id, vlan-id ) ) where each group consists of consecutive
         # opaque-ids with the same vlan-id and the same key.
         # v[0]=position, v[1][0]=opaque-id, v[1][1]=vlan-id
         # Non-consecutive opaque-ids result in different keys and thus groups.
         for _, gOpaqueIdsGroup in groupby( enumerate( sorted( gOpaqueIds ) ),
                                            key=lambda v: ( v[ 1 ][ 1 ],
                                                            v[ 1 ][ 0 ] - v[ 0 ] ) ):
            gOpaqueIdsGroup = list( gOpaqueIdsGroup ) # generator to list
            if len( gOpaqueIdsGroup ) == 1:
               # Use the 'single' version of the command when there is only one
               # opaque-id in the group
               opaqueId, vlanId = gOpaqueIdsGroup[ 0 ][ 1 ]
               if vlanId:
                  mldpMode[ 'Mldp.config' ].addCommand(
                     f'p2mp-lsp root {rootIp} opaque-id {opaqueId} vlan {vlanId}' )
               else:
                  mldpMode[ 'Mldp.config' ].addCommand(
                     f'p2mp-lsp root {rootIp} opaque-id {opaqueId}' )
            else:
               # With two or more opaque-ids in the group, use the 'range' version.
               # Because the opaque-ids in the group are consecutive, we can use
               # the first (0) and last (-1) item to calculate the range size.
               # secondary index 1 is for the opaque-id and vlan-id (index 0 is the
               # group key)
               opaqueId, vlanId = gOpaqueIdsGroup[ 0 ][ 1 ]
               size = gOpaqueIdsGroup[ -1 ][ 1 ][ 0 ] - opaqueId + 1
               if vlanId:
                  mldpMode[ 'Mldp.config' ].addCommand(
                     f'p2mp-lsp root {rootIp} opaque-id range {opaqueId} {size} '
                     f'vlan {vlanId}' )
               else:
                  mldpMode[ 'Mldp.config' ].addCommand(
                     f'p2mp-lsp root {rootIp} opaque-id range {opaqueId} {size}' )
   elif options.saveAll:
      mldpMode = mode[ MldpConfigMode ].getSingletonInstance()
      mldpMode[ 'Mldp.config' ].addCommand( 'shutdown' )

   # Graceful restart configuration
   if config.grOperFlag.grFlagHelper:
      ldpGrHelperMode = mode[ GrHelperConfigMode ].getSingletonInstance()
      if config.grMaxRecoveryTimeout != LdpProtoParam.defaultGrMaxRecoveryTimeout:
         ldpGrHelperMode[ 'GrHelper.config' ].addCommand(
            'timer recovery %d' % config.grMaxRecoveryTimeout )
      if ( config.grNeighborLivenessTimeout !=
           LdpProtoParam.defaultGrNeighborLivenessTimeout ):
         ldpGrHelperMode[ 'GrHelper.config' ].addCommand(
            'timer neighbor-liveness %d' % config.grNeighborLivenessTimeout )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no graceful-restart role helper' )
   if config.grOperFlag.grFlagSpeaker:
      ldpGrSpeakerMode = mode[ GrSpeakerConfigMode ].getSingletonInstance()
      if config.grHoldingTimeout != LdpProtoParam.defaultGrHoldingTimeout:
         ldpGrSpeakerMode[ 'GrSpeaker.config' ].addCommand(
            'timer state-holding %d' % config.grHoldingTimeout )
      if config.grReconnectTimeout != LdpProtoParam.defaultGrReconnectTimeout:
         ldpGrSpeakerMode[ 'GrSpeaker.config' ].addCommand(
            'timer reconnect %d' % config.grReconnectTimeout )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no graceful-restart role speaker' )

   if config.helloRedundancy:
      if config.helloRedundancyTimeout == Tac.endOfTime:
         mode[ 'Ldp.config' ].addCommand(
               'neighbor hello-redundancy duration infinite' )
      elif ( config.helloRedundancyTimeout ==
             LdpProtoParam.defaultHelloRedundancyTimeout ):
         mode[ 'Ldp.config' ].addCommand( 'neighbor hello-redundancy' )
      else:
         mode[ 'Ldp.config' ].addCommand( 'neighbor hello-redundancy duration %d' %
               config.helloRedundancyTimeout )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'neighbor hello-redundancy none' )

   if config.onlyLdpEnabledIntfs:
      mode[ 'Ldp.config' ].addCommand( 'interface disabled default' )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no interface disabled default' )

   if config.ldpLabelLocalTerminationMode == (
      LdpLabelLocalTerminationMode.explicitNull ):
      mode[ 'Ldp.config' ].addCommand( 'label local-termination explicit-null' )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'label local-termination implicit-null' )

   if config.entropyLabel:
      mode[ 'Ldp.config' ].addCommand( 'entropy-label' )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no entropy-label' )

   if config.allowLuTunnelRedist:
      if config.rcfForLuTunnelRedist != "":
         mode[ 'Ldp.config' ].addCommand(
                  'tunnel source-protocol bgp ipv4 labeled-unicast rcf %s()' %
                  config.rcfForLuTunnelRedist )
      else:
         mode[ 'Ldp.config' ].addCommand(
                  'tunnel source-protocol bgp ipv4 labeled-unicast' )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand(
                  'no tunnel source-protocol bgp ipv4 labeled-unicast' )

   if config.endOfLib:
      mode[ 'Ldp.config' ].addCommand( 'neighbor end-of-lib' )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no neighbor end-of-lib' )

   # This should be the last line
   if config.enabled:
      mode[ 'Ldp.config' ].addCommand( 'no shutdown' )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'shutdown' )

   # Save interface-level parameters:
   #     * "mpls ldp igp sync"
   #     * "mpls ldp interface"
   saveIntfConfig( root, config.intfConfigColl, options )
   saveIntfLdpEnabled( root, config.ldpEnabledIntf, options )

def Plugin( entityManager ):
   pass
