# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.LineSystem import LineSystemMode, PortRangeMode
import CliSave

# Define sequence names

lineSystemPortConfigSeq = "LineSystem.portConfig"

# "line system" commands will appear as a child of the global config mode, with
# "port" commands being a child of the "line system" config mode. Running-config
# should look like:
#
# line system
#   port X
#     booster ...
#     ...
#   !
#   port Y
# !
#
# Save blocks for line-system commands should be arranged like:
#
# +--------------------------------------------------
# | line system  (start of "line system" mode collection)
# |
# | +------------------------------------------------
# | |  port X  (start of "port X" mode collection)
# | |
# | | #----------------------------------------------
# | | |  booster ...  (start of a "LineSystem.portConfig" command sequence)
# | | |  ...
# | | #----------------------------------------------
# | +------------------------------------------------
# |
# | +------------------------------------------------
# | |  port Y  (start of "port Y" mode collection)
# | |
# | | #----------------------------------------------
# | | |  ...
# ...

class LineSystemCliSaveMode( LineSystemMode, CliSave.Mode ):
   def __init__( self, param ):
      LineSystemMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( LineSystemCliSaveMode )

class PortRangeCliSaveMode( PortRangeMode, CliSave.Mode ):
   def __init__( self, param ):
      PortRangeMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

LineSystemCliSaveMode.addChildMode( PortRangeCliSaveMode )
PortRangeCliSaveMode.addCommandSequence( lineSystemPortConfigSeq )

@CliSave.saver( "Tac::Dir", "hardware/line/system/cli/config/slice" )
def saveLineSystemCliConfig( baseDir, root, sysdbRoot, options ):
   # Used for command formatting in for-loops
   cliToken = { "booster": "booster", "preAmp": "pre-amp" }
   lsModeBase = root[ LineSystemCliSaveMode ].getSingletonInstance()

   for cliConfigDir in baseDir.values():
      for cliConfig in cliConfigDir.lineSystemCliConfig.values():
         # Want all ports to fall under single "line system" header, so use an
         # arbitarary constant string as identifier. This should only happen if
         # there is at least one port config object, since we don't want the
         # line-system header appearing otherwise.
         portCliSaveMode = lsModeBase[ PortRangeCliSaveMode ]

         # Retrieve port config
         port = cliConfig.name
         portNum = port.lower().strip( "port" )
         portMode = portCliSaveMode.getOrCreateModeInstance( portNum )
         portCmds = portMode[ lineSystemPortConfigSeq ]

         # Generate config commands for the port. "saveAll" and "saveAllDetail"
         # will behave the same for these commands, since it doesn't seem
         # reasonable to generate save-blocks for all possible ports.
         for ctrlPathName in [ "booster", "preAmp" ]:
            cmdToken = cliToken[ ctrlPathName ]
            ctrlPath = getattr( cliConfig, ctrlPathName )
            # [ no ] ( booster | pre-amp ) disabled
            if not ctrlPath.enabled:
               portCmds.addCommand( f"{cmdToken} disabled" )
            elif options.saveAll or options.saveAllDetail:
               portCmds.addCommand( f"no {cmdToken} disabled" )
            # [ no ] ( booster | pre-amp ) transmitter disabled
            if ctrlPath.txDisable:
               portCmds.addCommand( f"{cmdToken} transmitter disabled" )
            elif options.saveAll or options.saveAllDetail:
               portCmds.addCommand( f"no {cmdToken} transmitter disabled" )
