#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliMode.LoopProtect import MonitorLoopProtectMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from Vlan import vlanSetToCanonicalString

TRANSMIT_INTERVAL_DEFAULT = 5
DISABLED_TIME_DEFAULT = 604800
RATE_LIMIT_DEFAULT = 1000

class MonitorLoopProtectSaveMode( MonitorLoopProtectMode, CliSave.Mode ):
   def __init__( self, param ):
      MonitorLoopProtectMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

# Used to generate a block of loop protection config level commands
CliSave.GlobalConfigMode.addChildMode( MonitorLoopProtectSaveMode,
                                       after=[ IntfConfigMode ] )
MonitorLoopProtectSaveMode.addCommandSequence( 'LoopProtect.config' )
IntfConfigMode.addCommandSequence( 'LoopProtect.intf' )

def atDefault( config ):
   notIntfConfig = not config.globalIntfConfig or (
      config.globalIntfConfig.disabledTime == DISABLED_TIME_DEFAULT and
      config.globalIntfConfig.transmitInterval == TRANSMIT_INTERVAL_DEFAULT )
   notTxConfig = not config.transmitConfig or \
       config.transmitConfig.rateLimit == RATE_LIMIT_DEFAULT
   return not config.globalEnabled and notIntfConfig and notTxConfig and not \
      config.vlanEnabled and not config.intfExclude

@CliSave.saver( 'LoopProtect::Config::Config', 'loopprotect/config' )
def saveLoopProtectConfig( entity, root, requireMounts, options ):
   if atDefault( entity ) and not options.saveAll:
      return

   mode = root[ MonitorLoopProtectSaveMode ].getOrCreateModeInstance( entity )
   cmds = mode[ 'LoopProtect.config' ]
   saveAll = options.saveAll

   if entity.globalEnabled:
      cmds.addCommand( 'no shutdown' )

   if entity.vlanEnabled:
      vlanStr = vlanSetToCanonicalString( entity.vlanEnabled )
      cmds.addCommand( 'protect vlan %s' % vlanStr )

   transmitConf = entity.transmitConfig
   if transmitConf:
      if transmitConf.rateLimit != RATE_LIMIT_DEFAULT or saveAll:
         cmds.addCommand( 'rate-limit %s' % transmitConf.rateLimit )
   elif saveAll:
      cmds.addCommand( 'rate-limit %s' % RATE_LIMIT_DEFAULT )

   intfConf = entity.globalIntfConfig
   if intfConf:
      if intfConf.transmitInterval != TRANSMIT_INTERVAL_DEFAULT or saveAll:
         cmds.addCommand( 'transmit-interval %s' % intfConf.transmitInterval )
      if intfConf.disabledTime != DISABLED_TIME_DEFAULT or saveAll:
         cmds.addCommand( 'disabled-time %s' % intfConf.disabledTime )
   elif saveAll:
      cmds.addCommand( 'transmit-interval %s' % TRANSMIT_INTERVAL_DEFAULT )
      cmds.addCommand( 'disabled-time %s' % DISABLED_TIME_DEFAULT )

   for intfId in entity.intfExclude:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
      cmds = mode[ 'LoopProtect.intf' ]
      cmds.addCommand( 'no loop-protection' )
