#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import CliSave
from RoutingIntfUtils import allRoutingProtocolIntfNames
from McastCommonCliLib import mcastRoutingSupported
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.MrouteCliSave import getCliSaveVersion
import Tac

IntfConfigMode.addCommandSequence( 'McastCommon.intf', after=[ 'Ira.ipIntf' ] )

AddressFamily = Tac.Type( "Arnet::AddressFamily" )
LegacyConfig = Tac.Type( "McastCommon::LegacyConfig" )
VlanIntfId = Tac.Type( "Arnet::VlanIntfId" )
sourceInjectDefaultPref = Tac.Value( "HostInject::Constants" ).defaultPref

def saveMcastCommonConfig( mcastHostInjectConfig, root, requireMounts, options, af ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   # Save the default config only if the platform supports multicast routing
   if af == AddressFamily.ipv4:
      hwStatus = requireMounts[ 'routing/hardware/status' ]
   else:
      hwStatus = requireMounts[ 'routing6/hardware/status' ]
   if not mcastRoutingSupported(
         None,
         hwStatus ):
      saveAll = False
      saveAllDetail = False

   mcastLegacyConfig = requireMounts[ 'routing/multicast/legacyconfig' ]

   if saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( requireMounts,
                                                  includeEligible=True )
   elif saveAll:
      # Routing configuration is allowed on switchports as well.
      # Save configuration on all routing protocol interfaces and switchports
      # with non-default config.
      rpIntfNames = allRoutingProtocolIntfNames( requireMounts )
      intfIdSet = [ i.intfId for i in mcastHostInjectConfig.config ]
      cfgIntfNames = rpIntfNames + intfIdSet
   else:
      cfgIntfNames = [ i.intfId for i in mcastHostInjectConfig.config ]

   for intfId in cfgIntfNames:
      cKey = Tac.newInstance( "HostInject::DefaultConfigKey", intfId )
      ic = mcastHostInjectConfig.config.get( cKey )
      if not ic and not saveAll:
         continue
      saveIntfConfig( ic, intfId, root, saveAll, saveAllDetail,
                      mcastLegacyConfig, af )

def saveIntfConfig( ic, intfName, root, saveAll, saveAllDetail,
                    mcastLegacyConfig, af ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
   cmds = mode[ 'McastCommon.intf' ]

   version = getCliSaveVersion( mcastLegacyConfig.version, saveAll )
   legacy = version != LegacyConfig.ipMode

   if af == AddressFamily.ipv4:
      if legacy:
         cmdPrefix = 'ip multicast '
      else:
         cmdPrefix = 'multicast %s ' % af # pylint: disable=consider-using-f-string
   elif af == AddressFamily.ipv6:
      cmdPrefix = 'multicast %s ' % af # pylint: disable=consider-using-f-string

   saveAllPrefix = 'no multicast %s ' % af # pylint: disable=consider-using-f-string

   if VlanIntfId.isVlanIntfId( Arnet.IntfId( intfName ) ):
      cmd = 'source route export'
      if ic:
         if ic.pref != sourceInjectDefaultPref:
            # pylint: disable-next=consider-using-f-string
            cmds.addCommand( cmdPrefix + cmd + ' %d' % ic.pref )
         else:
            cmds.addCommand( cmdPrefix + cmd )
      elif saveAll:
         cmds.addCommand( saveAllPrefix + cmd )

@CliSave.saver( 'HostInject::HostInjectConfig', 'routing/attached-host/mcast/config',
                requireMounts = ( 'routing/hardware/status',
                                  'interface/config/all', 'interface/status/all',
                                  'l3/config',
                                  'routing/multicast/legacyconfig' ) )
def saveMcastCommonIpv4Config( mcastHostInjectConfig, root, requireMounts, options ):

   saveMcastCommonConfig( mcastHostInjectConfig, root, requireMounts, options,
                          AddressFamily.ipv4 )

@CliSave.saver( 'HostInject::HostInjectConfig',
                'routing6/attached-host/mcast/config',
                requireMounts = ( 'routing6/hardware/status',
                                  'interface/config/all', 'interface/status/all',
                                  'l3/config',
                                  'routing/multicast/legacyconfig' ) )
def saveMcastCommonIpv6Config( mcastHostInjectConfig, root, requireMounts, options ):

   saveMcastCommonConfig( mcastHostInjectConfig, root, requireMounts, options,
                          AddressFamily.ipv6 )
