# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliSavePlugin import IntfCliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.IraVrfCliSave import VrfDefinitionCliSaveMode
from CliSave import GlobalConfigMode
import Tac
from socket import IPPROTO_UDP, IPPROTO_TCP
from CliMode.NatMode import NatPoolMode, NatPortOnlyPoolMode, NatProfileMode, \
                            NatSyncMode
from CliMode.NatMode import NatFlowMode, NatFlowPolicyMode, \
                            NatFlowMatchMode, NatFlowActionMode
from CliMode.NatServiceList import NatServiceListMode
from IpLibConsts import DEFAULT_VRF
from RoutingIntfUtils import allRoutingProtocolIntfNames
from TypeFuture import TacLazyType

GlobalConfigMode.addCommandSequence( 'Nat.config',
                                     before=[ VrfDefinitionCliSaveMode ] )

IntfCliSave.IntfConfigMode.addCommandSequence( 'Nat.intf', after=[ 'Ira.ipIntf' ] )

ipProtoMap = { IPPROTO_TCP: 'tcp',
               IPPROTO_UDP: 'udp' }

DefaultValue = TacLazyType( "Ip::Nat::DefaultValue" )

class NatPoolConfigMode( NatPoolMode, CliSave.Mode ):
   def __init__( self, param ):
      NatPoolMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

NatPoolConfigMode.addCommandSequence( 'Nat.poolConfig' )
GlobalConfigMode.addChildMode( NatPoolConfigMode, after=[ IntfConfigMode ] )

class NatPortOnlyPoolConfigMode( NatPortOnlyPoolMode, CliSave.Mode ):
   def __init__( self, param ):
      NatPortOnlyPoolMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

NatPortOnlyPoolConfigMode.addCommandSequence( 'Nat.poolConfig' )
GlobalConfigMode.addChildMode( NatPortOnlyPoolConfigMode, after=[ IntfConfigMode ] )

class NatSyncConfigMode( NatSyncMode, CliSave.Mode ):
   def __init__( self, param ):
      NatSyncMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

NatSyncConfigMode.addCommandSequence( 'Nat.syncConfig' )
GlobalConfigMode.addChildMode( NatSyncConfigMode, after=[ NatPoolConfigMode ] )

class NatServiceListConfigMode( NatServiceListMode, CliSave.Mode ):
   def __init__( self, param ):
      NatServiceListMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

NatServiceListConfigMode.addCommandSequence( 'Nat.serviceListConfig' )
GlobalConfigMode.addChildMode( NatServiceListConfigMode, after=[ IntfConfigMode ] )

class NatProfileConfigMode( NatProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      NatProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

NatProfileConfigMode.addCommandSequence( 'Nat.profileConfig' )
GlobalConfigMode.addChildMode( NatProfileConfigMode, before=[ IntfConfigMode ] )

class NatFlowConfigMode( NatFlowMode, CliSave.Mode ):
   def __init__( self, param ):
      NatFlowMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

GlobalConfigMode.addChildMode( NatFlowConfigMode, after=[ IntfConfigMode ] )
NatFlowConfigMode.addCommandSequence( 'Nat.natFlow' )

class NatFlowPolicyConfigMode( NatFlowPolicyMode, CliSave.Mode ):
   def __init__( self, param ):
      NatFlowPolicyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

NatFlowConfigMode.addChildMode( NatFlowPolicyConfigMode,
                                after=[ 'Nat.natFlow' ] )
NatFlowPolicyConfigMode.addCommandSequence( 'Nat.natFlowPolicy' )

class NatFlowMatchConfigMode( NatFlowMatchMode, CliSave.Mode ):
   def __init__( self, param ):
      NatFlowMatchMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

NatFlowPolicyConfigMode.addChildMode( NatFlowMatchConfigMode,
                                      after=[ 'Nat.natFlowPolicy' ] )
NatFlowMatchConfigMode.addCommandSequence( 'Nat.natFlowMatch' )

class NatFlowActionConfigMode( NatFlowActionMode, CliSave.Mode ):
   def __init__( self, param ):
      NatFlowActionMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

NatFlowMatchConfigMode.addChildMode( NatFlowActionConfigMode,
                                     after=[ 'Nat.natFlowMatch' ] )
NatFlowActionConfigMode.addCommandSequence( 'Nat.natFlowAction' )

@CliSave.saver( 'Ip::Nat::Config', 'ip/nat/config',
                requireMounts=( 'ip/nat/status',
                                'ip/nat/hwCapabilities',
                                'interface/config/all',
                                'interface/status/all' ) )
def saveNatConfig( natConfig, root, requireMounts, options ):

   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   settings = natConfig.settings

   if saveAllDetail:
      rpIntfNames = allRoutingProtocolIntfNames( requireMounts,
                                                 includeEligible=True )
      cfgIntfNames = set( rpIntfNames )
      cfgIntfNames.update( natConfig.intfConfig )
      cfgIntfNames = sorted( cfgIntfNames )
   elif saveAll:
      # Routing configuration is allowed on switchports as well.
      # Save configuration on all routing protocol interfaces and switchports
      # with non-default config.
      rpIntfNames = allRoutingProtocolIntfNames( requireMounts )
      cfgIntfNames = set( rpIntfNames )
      cfgIntfNames.update( natConfig.intfConfig )
      cfgIntfNames = sorted( cfgIntfNames )
   else:
      cfgIntfNames = natConfig.intfConfig

   for intfName in cfgIntfNames:
      intfConfig = natConfig.intfConfig.get( intfName )
      if intfConfig:
         saveIntfConfig( intfConfig, root, saveAll, saveAllDetail )

   cmds = root[ 'Nat.config' ]

   tcpSynAckRequired = settings.tcpSynAckRequired
   if not tcpSynAckRequired or saveAll:
      pktType = 'syn-ack' if tcpSynAckRequired else 'syn'
      cmd = "ip nat tcp flow creation trigger %s" % pktType
      cmds.addCommand( cmd )

   natAddressSelectionHash = settings.natAddressSelectionHash
   if not natAddressSelectionHash or saveAll:
      algorithm = 'hash' if natAddressSelectionHash else 'random'
      cmd = "ip nat translation address selection algorithm %s" % algorithm
      cmds.addCommand( cmd )
      
   sourceIpOnlyHash = settings.sourceIpOnlyHash
   if sourceIpOnlyHash or saveAll:
      cmd = "ip nat translation address selection hash field source-ip"
      if not sourceIpOnlyHash:
         cmd = "no " + cmd
      cmds.addCommand( cmd )

   ipAddressTranslationAny = settings.ipAddressTranslationAny
   if ipAddressTranslationAny or saveAll:
      cmd = "ip nat translation address selection any"
      if not ipAddressTranslationAny:
         cmd = "no " + cmd
      cmds.addCommand( cmd )

   if DEFAULT_VRF in natConfig.vrfConfig:
      natVrfConfig = natConfig.vrfConfig[ DEFAULT_VRF ]
      saveNatVrfConfig( natVrfConfig, root, saveAll, saveAllDetail )
   for natVrf in natConfig.vrfConfig:
      if natVrf != DEFAULT_VRF:
         natVrfConfig = natConfig.vrfConfig[ natVrf ]
         saveNatVrfConfig( natVrfConfig, root, saveAll, saveAllDetail )

   timeoutConfig = natConfig.timeoutConfig

   if timeoutConfig.tcpTimeout != DefaultValue.tcpTimeoutDefault or saveAll:
      cmds.addCommand(
         'ip nat translation tcp-timeout %d' % timeoutConfig.tcpTimeout )
   
   unestablishedTcpSave = timeoutConfig.unestablishedTcpTimeout != \
           DefaultValue.unestablishedTcpTimeoutDefault
   if unestablishedTcpSave or saveAll:
      cmds.addCommand(
         'ip nat translation tcp-timeout unestablished %d' %
         timeoutConfig.unestablishedTcpTimeout )

   if timeoutConfig.udpTimeout != DefaultValue.udpTimeoutDefault or saveAll:
      cmds.addCommand(
         'ip nat translation udp-timeout %d' % timeoutConfig.udpTimeout )
   
   if timeoutConfig.icmpTimeout != DefaultValue.icmpTimeoutDefault or saveAll:
      cmds.addCommand(
         'ip nat translation icmp-timeout %d' % timeoutConfig.icmpTimeout )

   for poolName in natConfig.poolConfig:
      pool = natConfig.poolConfig[ poolName ]
      savePoolConfig( pool, root, saveAll, saveAllDetail )

   for serviceListName in natConfig.serviceListConfig:
      serviceList = natConfig.serviceListConfig[ serviceListName ]
      saveServiceListConfig( serviceList, root, saveAll, saveAllDetail )

   if settings.connLimit != DefaultValue.connLimitDefault or saveAll:
      cmds.addCommand( 'ip nat translation max-entries %d' % settings.connLimit )
   if settings.connLimitLowMarkPerc != DefaultValue.connLimitLowMarkPercDefault \
      or saveAll:
      cmds.addCommand( 'ip nat translation low-mark %d' %
         settings.connLimitLowMarkPerc )

   if settings.symmetricConnLimit != DefaultValue.connLimitDefault or saveAll:
      cmds.addCommand( 'ip nat translation max-entries %d symmetric' %
                       settings.symmetricConnLimit )
   if settings.symmetricConnLimitLowMarkPerc != \
         DefaultValue.connLimitLowMarkPercDefault or saveAll:
      cmds.addCommand( 'ip nat translation low-mark %d symmetric' %
                       settings.symmetricConnLimitLowMarkPerc )

   if settings.fullConeConnLimit != DefaultValue.connLimitDefault or saveAll:
      cmds.addCommand( 'ip nat translation max-entries %d full-cone' %
                       settings.fullConeConnLimit )
   if settings.fullConeConnLimitLowMarkPerc != \
         DefaultValue.connLimitLowMarkPercDefault or saveAll:
      cmds.addCommand( 'ip nat translation low-mark %d full-cone' %
                       settings.fullConeConnLimitLowMarkPerc )

   if settings.allHostsConnLimit != DefaultValue.connLimitDefault or saveAll:
      cmds.addCommand( 'ip nat translation max-entries %d host' %
            settings.allHostsConnLimit )

   if settings.allHostsLowMarkPerc != DefaultValue.connLimitLowMarkPercDefault \
      or saveAll:
      cmds.addCommand( 'ip nat translation low-mark %d host' %
            settings.allHostsLowMarkPerc )

   if settings.allPoolsLowMarkPerc != DefaultValue.connLimitLowMarkPercDefault \
      or saveAll:
      cmds.addCommand( 'ip nat translation low-mark %d pool' %
            settings.allPoolsLowMarkPerc )

   for hostIp in natConfig.hostConnectionConfig:
      cmds.addCommand( 'ip nat translation max-entries %d %s' %
            ( natConfig.hostConnectionConfig[ hostIp ].connLimit, hostIp ) )

   for poolName in natConfig.poolConnectionConfig:
      cmds.addCommand( 'ip nat translation max-entries %d pool %s' %
            ( natConfig.poolConnectionConfig[ poolName ].connLimit, poolName ) )

   if settings.counterEnabled != DefaultValue.counterEnabledDefault or saveAll:
      if settings.counterEnabled:
         cmds.addCommand( 'ip nat translation counters' )
      else:
         cmds.addCommand( 'no ip nat translation counters' )

   if ( settings.loggingEnabled != DefaultValue.loggingEnabledDefault or saveAll ):
      if settings.loggingEnabled:
         cmds.addCommand( 'ip nat logging' )
      else:
         cmds.addCommand( 'no ip nat logging' )

   if ( settings.staticAclSharingEnabled !=
           DefaultValue.staticAclSharingEnabledDefault
        or saveAll ):
      if settings.staticAclSharingEnabled:
         cmds.addCommand( 'hardware nat static access-list resource sharing' )
      else:
         cmds.addCommand( 'no hardware nat static access-list resource sharing' )

   natHwCapabilities = requireMounts[ 'ip/nat/hwCapabilities' ]
   if natHwCapabilities.natFragmentCommandSupported:
      staticFragment = natConfig.fragmentConfig.staticFragment
      dynamicFragment = natConfig.fragmentConfig.dynamicFragment
      staticFragmentDefault = natHwCapabilities.natFragmentDefault.staticFragment
      dynamicFragmentDefault = natHwCapabilities.natFragmentDefault.dynamicFragment

      displayStatic = \
         ( staticFragmentDefault and staticFragment == 'fragmentDisabled' ) or \
         ( not staticFragmentDefault and staticFragment == 'fragmentEnabled' )
      displayDynamic = \
         ( dynamicFragmentDefault and dynamicFragment == 'fragmentDisabled' ) or \
         ( not dynamicFragmentDefault and dynamicFragment == 'fragmentEnabled' )

      if not displayStatic and not displayDynamic:
         if saveAll:
            cmds.addCommand( 'default hardware nat fragment' )
      else:
         staticCmd = "hardware nat static fragment"
         if staticFragmentDefault:
            staticCmd += " disabled"
         dynamicCmd = "hardware nat dynamic fragment"
         if dynamicFragmentDefault:
            dynamicCmd += " disabled"

         if displayStatic and displayDynamic:
            if staticFragmentDefault == dynamicFragmentDefault:
               fragmentCmd = 'hardware nat fragment'
               if staticFragmentDefault:
                  fragmentCmd += " disabled"
               cmds.addCommand( fragmentCmd )
            else:
               cmds.addCommand( staticCmd )
               cmds.addCommand( dynamicCmd )
         elif displayStatic:
            cmds.addCommand( staticCmd )
         else:
            cmds.addCommand( dynamicCmd )

   if settings.vrfLeakEnabled != DefaultValue.vrfLeakEnabledDefault or saveAll:
      if settings.vrfLeakEnabled:
         cmds.addCommand( 'hardware nat vrf leak' )
      else:
         cmds.addCommand( 'no hardware nat vrf leak' )

   socketSize = settings.natNetlinkSocketBufSize
   if socketSize or saveAll:
      if socketSize:
         cmds.addCommand( 'ip nat kernel buffer size %d' % socketSize )
      else:
         cmds.addCommand( 'no ip nat kernel buffer size' )

   natFlowSave = NatFlowSaver( natConfig, root )
   natFlowSave.save()

class NatFlowSaver:
   flowMode = NatFlowConfigMode
   flowPolicyMode = NatFlowPolicyConfigMode
   flowMatchMode = NatFlowMatchConfigMode
   flowActionMode = NatFlowActionConfigMode

   def __init__( self, natConfig, root ):
      self.natConfig = natConfig
      self.root = root

   def flowPolicyModeCmds( self, flowPolicyMode ):
      return flowPolicyMode[ 'Nat.natFlowPolicy' ]

   def flowMatchModeCmds( self, flowMatchMode ):
      return flowMatchMode[ 'Nat.natFlowMatch' ]

   def flowActionModeCmds( self, flowActionMode ):
      return flowActionMode[ 'Nat.natFlowAction' ]

   def saveNatFlowConfig( self ):
      policiesMode = self.root[ self.flowMode ].getSingletonInstance()
      for policyName in self.natConfig.natFlow:
         self.savePolicy( policyName, policiesMode )

   def savePolicy( self, policyName, policiesMode ):
      policy = self.natConfig.natFlow[ policyName ]
      if not policy:
         return
      param = ( self.flowMode, policyName )
      policyMode = policiesMode[ self.flowPolicyMode ].\
                   getOrCreateModeInstance( param )
      for matchName, match in sorted( policy.match.items() ):
         self.saveMatch( policyMode, policyName, matchName, match )

   def saveMatch( self, policyMode, policyName, matchName, match ):
      param = ( self.flowPolicyMode, policyName, matchName )
      matchMode = policyMode[ self.flowMatchMode ].\
                  getOrCreateModeInstance( param )
      cmds = self.flowMatchModeCmds( matchMode )
      if match.flow.src:
         cmd = 'source ' + match.flow.src.stringValue
         cmds.addCommand( cmd )
      if match.flow.dst:
         cmd = 'destination ' + match.flow.dst.stringValue
         cmds.addCommand( cmd )
      if match.flow.protocol:
         cmd = 'protocol ' + ipProtoMap.get( match.flow.protocol,
                                             str( match.flow.protocol ) )
         if match.flow.srcPortStart:
            cmd += ' source ' + str( match.flow.srcPortStart )
         if match.flow.srcPortEnd > match.flow.srcPortStart:
            cmd += ' ' + str( match.flow.srcPortEnd )
         if match.flow.dstPortStart:
            cmd += ' destination ' + str( match.flow.dstPortStart )
         if match.flow.dstPortEnd > match.flow.dstPortStart:
            cmd += ' ' + str( match.flow.dstPortEnd )
         cmds.addCommand( cmd )
      if match.flow.priority:
         cmd = 'priority ' + str( match.flow.priority )
         cmds.addCommand( cmd )

      for actionName, action in sorted( match.action.items() ):
         self.saveAction( matchMode, policyName, matchName, actionName, action )

   def saveAction( self, matchMode, policyName, matchName, actionName, action ):
      param = ( self.flowMatchMode, policyName, matchName, actionName )
      actionMode = matchMode[ self.flowActionMode ].\
                   getOrCreateModeInstance( param )
      cmds = self.flowActionModeCmds( actionMode )
      if action.addr.src:
         cmd = 'nat source ' + action.addr.src.ip
         if action.addr.src.port:
            cmd += ' port ' + str( action.addr.src.port )
         cmds.addCommand( cmd )
      if action.addr.dst:
         cmd = 'nat destination ' + action.addr.dst.ip
         if action.addr.dst.port:
            cmd += ' port ' + str( action.addr.dst.port )
         cmds.addCommand( cmd )
      for intfName, intf in sorted( action.intf.items() ):
         cmd = 'output interface ' + intfName
         if intf.mac != "00:00:00:00:00:00":
            cmd += ' mac ' + intf.mac
         if intf.vlan:
            cmd += ' vlan ' + str( intf.vlan )
         cmds.addCommand( cmd )

   def save( self ):
      self.saveNatFlowConfig()


def savePoolConfig( pool, root, saveAll, saveAllDetail ):
   if pool.portOnly:
      mode = root[ NatPortOnlyPoolConfigMode ].getOrCreateModeInstance( pool.name )
   else:
      mode = root[ NatPoolConfigMode ].getOrCreateModeInstance(
            ( pool.name, pool.prefixLen ) )
   cmds = mode[ 'Nat.poolConfig' ]

   for poolRange in pool.poolRange:
      portRange = ''
      if poolRange.startPort != 0 and poolRange.endPort != 0:
         portRange = ' ' + str(poolRange.startPort) + ' ' + str(poolRange.endPort)

      if pool.portOnly:
         cmd = 'port range' + portRange
      else:
         cmd = 'range ' + poolRange.startIp + ' ' + poolRange.endIp + portRange
      cmds.addCommand( cmd )

   if pool.portOnly:
      return

   if pool.poolUtilizationLogThreshold != \
         DefaultValue.poolUtilizationLogThresholdDefault:
      if pool.poolUtilizationLogThreshold == 0:
         cmds.addCommand( 'no utilization threshold' )
      else:
         cmds.addCommand( 'utilization threshold %d action log' %
                       pool.poolUtilizationLogThreshold )
   else:
      if saveAll:
         cmds.addCommand( 'utilization threshold %d action log' %
           DefaultValue.poolUtilizationLogThresholdDefault )

def saveServiceListConfig( serviceList, root, saveAll, saveAllDetail ):
   mode = root[ NatServiceListConfigMode ].getOrCreateModeInstance(
          serviceList.name  )
   cmds = mode[ 'Nat.serviceListConfig' ]

   if serviceList.serviceFtp:
      cmd = 'service ftp'
      cmds.addCommand( cmd )

def saveIntfConfig( natIntfConfig, root, saveAll, saveAllDetail ):

   intf = natIntfConfig
   if intf.profile:
      mode = root[ NatProfileConfigMode ].getOrCreateModeInstance(
                                             ( intf.name, intf.vrfName ) )
      cmds = mode[ 'Nat.profileConfig' ]
   else:
      mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( intf.name )
      cmds = mode[ 'Nat.intf' ]

   def prepareCmd( target, lAddr, acl, gAddr, proto, group, prefix, ingress,
                   table=None, comment='' ):
      cmd = 'ip nat %s' % target
      if not group:
         # Static NAT
         if table == 'egress' or ( target == 'destination' and not ingress ):
            cmd += ' egress'
         elif target == 'source' and ingress:
            cmd += ' ingress'
      else:
         # Twice NAT
         if ingress:
            cmd += ' ingress'
      cmd += ' static'
      cmd += ' %s' % lAddr.ip
      if lAddr.port != 0:
         cmd += ' %s' % lAddr.port
      if acl:
         cmd += ' access-list %s' % acl
      cmd += ' %s' % gAddr.ip
      if gAddr.port != 0:
         cmd += ' %s' % gAddr.port
      if proto != 0:
         cmd += ' protocol %s' % ipProtoMap[ proto ]
      if group != 0:
         cmd += ' group %s' % group
      if prefix != 0:
         cmd += ' prefix-length %s' % prefix
      if comment:
         cmd += ' comment %s' % comment
      return cmd

   for _lAddr, snat in sorted( intf.staticNat.items() ):
      cmd = prepareCmd( snat.target,
                        snat.staticKey.localAddress, snat.staticKey.acl,
                        snat.globalAddress, snat.protocol, 0, snat.prefixLen,
                        snat.ingress, _lAddr.table, snat.comment )
      cmds.addCommand( cmd )

   for group, tnat in sorted( intf.twiceNat.items() ):
      if tnat.srcIpAndPortOld.ip != '0.0.0.0':
         cmd = prepareCmd( 'source',
                           tnat.srcIpAndPortOld, None, tnat.srcIpAndPortNew,
                           tnat.protocol, group, 0, tnat.ingress,
                           comment=tnat.comment )
         cmds.addCommand( cmd )

      if tnat.dstIpAndPortOld.ip != '0.0.0.0':
         cmd = prepareCmd( 'destination',
                           tnat.dstIpAndPortOld, None, tnat.dstIpAndPortNew,
                           tnat.protocol, group, 0, tnat.ingress,
                           comment=tnat.comment )
         cmds.addCommand( cmd )

   for dynamicNat in sorted( intf.dynamicNat.values() ):
      cmd = 'ip nat ' + dynamicNat.target + ' dynamic'
      cmd += ' access-list ' + dynamicNat.acl
      assert bool( dynamicNat.pool ) != bool( dynamicNat.overload )
      if dynamicNat.pool != '':
         cmd += ' pool ' + dynamicNat.pool
         if dynamicNat.target == "source":
            cmd += ' full-cone' if dynamicNat.fullCone else ''
            cmd += ' address-only' if dynamicNat.addrOnly else ''
         if dynamicNat.group:
            cmd += ' group %d' % dynamicNat.group
         if dynamicNat.serviceList:
            cmd += ' service-list ' + dynamicNat.serviceList
      else:
         cmd += ' overload'
         if dynamicNat.group:
            cmd += ' group %d' % dynamicNat.group
      if dynamicNat.priority:
         cmd += ' priority %s' % dynamicNat.priority
      if dynamicNat.comment:
         cmd += ' comment %s' % dynamicNat.comment
      cmds.addCommand( cmd )

   # requires manual sorting until TACC supports ordered sets
   for policy in sorted( intf.flowPolicy ):
      cmd = 'ip nat flow policy ' + policy
      cmds.addCommand( cmd )

   if intf.fwdMissTranslationInfo != Tac.Value( 'Ip::Nat::FwdMissTranslationInfo' ):
      cmd = 'translation miss match peer-criteria action forward interface ' + \
            intf.fwdMissTranslationInfo.fwdIntfId + \
            ' peer-address ' + intf.fwdMissTranslationInfo.fwdIp
      cmds.addCommand( cmd )

   # In profile mode, need to also configure the nat forwarding mode for the intfs
   if intf.profile:
      for intfId in intf.profileIntf:
         mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( intfId )
         cmds = mode[ 'Nat.intf' ]
         cmd = 'ip nat service-profile ' + intf.name
         cmds.addCommand( cmd )

def saveNatVrfConfig( natVrfConfig, root, saveAll, saveAllDetail ):
   insideVrf = natVrfConfig.name
   for srcOld, snat in sorted( natVrfConfig.staticNat.items() ):
      if snat.target == 'source':
         localAddr = srcOld
         globalAddr = snat.srcAddressNew
      else:
         localAddr = snat.srcAddressNew
         globalAddr = srcOld
      cmd = 'ip nat %s static %s %s inside-vrf %s' % ( snat.target, localAddr,
                                                       globalAddr, insideVrf )
      if snat.outsideVrf == DEFAULT_VRF:
         cmds = root[ 'Nat.config' ]
      else:
         vrfName = snat.outsideVrf
         mode = root[ VrfDefinitionCliSaveMode ].getOrCreateModeInstance( vrfName )
         cmds = mode[ 'vrfDef' ]
      cmds.addCommand( cmd )

   if natVrfConfig.sourcePolicer != Tac.Value( 'Ip::Nat::VrfPolicer' ):
      cmd = 'ip nat source inside-vrf %s rate %d burst %d' % (
            insideVrf, natVrfConfig.sourcePolicer.rate,
            natVrfConfig.sourcePolicer.burst )
      cmds = root[ 'Nat.config' ]
      cmds.addCommand( cmd )

@CliSave.saver( 'Ip::Nat::Sync::Config', 'ip/nat/sync/config',
                requireMounts=( 'ip/nat/sync/status',
                                'interface/config/all',
                                'interface/status/all' ) )
def saveNatSyncConfig( natSyncConfig, root, requireMounts, options ):
   if natSyncConfig.description  or \
         natSyncConfig.sysname != natSyncConfig.defaultSysname or \
         natSyncConfig.shutdown != natSyncConfig.shutdownDefault or \
         natSyncConfig.remote != '0.0.0.0' or \
         natSyncConfig.localIntfId or \
         natSyncConfig.peerExpiryPeriod != natSyncConfig.peerExpiryPeriodDefault or \
         natSyncConfig.portRange.startPort != 0 or \
         natSyncConfig.portRange.endPort != 0 or \
         not natSyncConfig.portRangeSplitEnabled or \
         options.saveAll:
      mode = root[ NatSyncConfigMode ].getOrCreateModeInstance(  'nat-sync'  )
      cmds = mode[ 'Nat.syncConfig' ]

   if natSyncConfig.description:
      cmd = 'description %s' % natSyncConfig.description
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no description' )

   if natSyncConfig.sysname != natSyncConfig.defaultSysname:
      cmd = 'sysname %s' % natSyncConfig.sysname
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no sysname' )

   if natSyncConfig.peerExpiryPeriod != natSyncConfig.peerExpiryPeriodDefault:
      cmd = 'expiry-interval %s' % natSyncConfig.peerExpiryPeriod
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no expiry-interval' )

   if natSyncConfig.shutdown != natSyncConfig.shutdownDefault:
      cmd = 'shutdown'
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no shutdown' )

   if natSyncConfig.remote != '0.0.0.0':
      cmd = 'peer-address %s' % natSyncConfig.remote
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no peer-address' )

   if natSyncConfig.localIntfId:
      cmd = 'local-interface %s' % natSyncConfig.localIntfId
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no local-interface' )

   if natSyncConfig.portRange.startPort != 0 or \
      natSyncConfig.portRange.endPort != 0:
      cmd = 'port-range %s %s' % ( natSyncConfig.portRange.startPort,
                                   natSyncConfig.portRange.endPort )
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no port-range' )

   if not natSyncConfig.portRangeSplitEnabled:
      cmd = 'port-range split disabled'
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'port-range split' )
