# Copyright (c) 2007, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import Arnet
import CliSave
import NtpLib
from IpLibConsts import DEFAULT_VRF
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.Security import mgmtSecurityConfigPath
from CliSavePlugin.Security import SecurityConfigMode
from RoutingIntfUtils import allIpIntfNames
import ReversibleSecretCli
import functools
from operator import attrgetter

# CLI save block 'Mgmt.Security' need to be loaded first
# for password encryption configuration in Ntp
CliSave.GlobalConfigMode.addCommandSequence( 'Ntp.config', 
                                             after=[ SecurityConfigMode ] )
IntfConfigMode.addCommandSequence( 'Ntp.intfConfig' )

@CliSave.saver( 'Ntp::Config', 'sys/time/ntp/config',
                requireMounts=(
                   'interface/config/all',
                   'interface/status/all',
                   'acl/cpconfig/cli',
                   'sys/time/clock/config',
                   mgmtSecurityConfigPath, ) )
def saveConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Ntp.config' ]

   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   for key in sorted( entity.symmetricKey.values(), key=attrgetter( 'keyId' ) ):
      cmd = ReversibleSecretCli.getCliSaveCommand(
         f"ntp authentication-key {key.keyId:d} {key.keyType} {{}}",
         securityConfig,
         key.secret )
      cmds.addCommand( cmd )

   if entity.trustedKeys:
      cmd = "ntp trusted-key %s" % entity.trustedKeys
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmd = "no ntp trusted-key"
      cmds.addCommand( cmd )

   if entity.authMode == NtpLib.authModeAll:
      cmd = "ntp authenticate"
      cmds.addCommand( cmd )
   elif entity.authMode == NtpLib.authModeServers:
      cmd = "ntp authenticate servers"
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmd = "no ntp authenticate"
      cmds.addCommand( cmd )

   if entity.defaultSourceIntf.intf:
      cmd = 'ntp local-interface'

      if entity.defaultSourceIntf.vrf != DEFAULT_VRF:
         cmd = f"{cmd} vrf {entity.defaultSourceIntf.vrf}"
      cmd = f"{cmd} {entity.defaultSourceIntf.intf}"
      cmds.addCommand( cmd )

   if entity.dscpValue != entity.dscpValueInvalid:
      cmds.addCommand( "ntp qos dscp %d" % entity.dscpValue )

   servers = entity.server.values()
   def _cmpServers( a, b ):
      return NtpLib.compareVrfAndHost( a.vrfAndHost, b.vrfAndHost )
   for server in sorted( servers, key=functools.cmp_to_key( _cmpServers ) ):
      serverCmd = "ntp server"
      if server.vrf != DEFAULT_VRF:
         serverCmd += " vrf %s" % server.vrf
      serverCmd += " %s" % server.ipOrHost
      for option in [ "prefer", "refresh", "burst", "iburst" ]:
         if getattr( server, option ):
            serverCmd += " " + option
      for option in [ "version", "minpoll", "maxpoll" ]:
         value = getattr( server, option )
         if value:
            serverCmd += f" {option} {value}"
      if server.sourceIntf:
         sourceToken = "source" if server.sourceUsed else "local-interface"
         serverCmd += f" {sourceToken} {server.sourceIntf}"
      if server.sourceAddr:
         serverCmd += " source-address " + str( server.sourceAddr )
      if server.keyId != 0:
         serverCmd += " key %d" % server.keyId
      cmds.addCommand( serverCmd )

   if entity.serverModeEnabledDefault:
      cmds.addCommand( "ntp serve all" )
   elif options.saveAll:
      cmds.addCommand( "no ntp serve all" )

   for serveVrf in sorted( entity.serveVrfName ):
      serveVrfCmd = "ntp serve all vrf " + serveVrf
      cmds.addCommand( serveVrfCmd )

   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   for aclType in ( 'ip', 'ipv6' ):
      serviceAcl = aclCpConfig.cpConfig[ aclType ].serviceAcl
      for vrfName, serviceAclVrfConfig in serviceAcl.items():
         serviceAclConfig = serviceAclVrfConfig.service.get( 'ntp' )
         if serviceAclConfig and serviceAclConfig.aclName:
            if vrfName == DEFAULT_VRF:
               vrfArg = ""
            else:
               vrfArg = " vrf " + vrfName
            cmds.addCommand( 'ntp serve %s access-group %s%s in' %
                             ( aclType, serviceAclConfig.aclName, vrfArg ) )

   if options.saveAll:
      cfgIntfNames = set( allIpIntfNames( requireMounts ) )
   else:
      cfgIntfNames = set()
   cfgIntfNames.update( entity.serverModeEnabledIntf,
                        entity.serverModeDisabledIntf )

   clockConfig = requireMounts[ 'sys/time/clock/config' ]
   if clockConfig.source == 'ntp' and (
         clockConfig.source != clockConfig.defaultSource or options.saveAll ):
      cmds.addCommand( 'clock source ntp' )

   for intfName in Arnet.sortIntf( cfgIntfNames ):
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ "Ntp.intfConfig" ]
      if intfName in entity.serverModeDisabledIntf:
         cmds.addCommand( "no ntp serve" )
      elif intfName in entity.serverModeEnabledIntf:
         cmds.addCommand( "ntp serve" )
      elif options.saveAll:
         cmds.addCommand( "default ntp serve" )

   if entity.restartsForcedOnIntfChanges:
      cmds.addCommand( "ntp force-restarts" )
