# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import McastCommonCliLib
from CliSavePlugin.MrouteCliSave import ( getCliSaveVersion, LegacyConfig )
import socket
import Arnet
import Tac
import CliSave
from IpLibConsts import DEFAULT_VRF
from CliMode.Pim import RoutingPimMode, RoutingPimVrfMode, RoutingPimAfMode
from CliMode.Pim import RoutingPimSparseMode, RoutingPimSparseVrfMode
from CliMode.Pim import RoutingPimSparseAfMode
from CliMode.PimBidir import RoutingPimBidirBaseMode, RoutingPimBidirVrfMode
from CliMode.PimBidir import RoutingPimBidirAfMode
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "PimSave" )
t1 = Tracing.trace1
AddressFamily = Tac.Type( "Arnet::AddressFamily" )
PimLegacyConfig = Tac.Type( "McastCommon::LegacyConfig" )

class RouterPimBaseConfigMode( RoutingPimMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingPimMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterPimVrfConfigMode( RoutingPimVrfMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingPimVrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterPimAfConfigMode( RoutingPimAfMode, CliSave.Mode ):
   def __init__( self, vrfNameAndAf ):
      ( vrfName, af ) = vrfNameAndAf
      RoutingPimAfMode.__init__( self, vrfName, af )
      CliSave.Mode.__init__( self, ( vrfName, af ) )

class RouterPimSparseBaseConfigMode( RoutingPimSparseMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingPimSparseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class RouterPimSparseVrfConfigMode( RoutingPimSparseVrfMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingPimSparseVrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class RouterPimBidirBaseConfigMode( RoutingPimBidirBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingPimBidirBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class RouterPimBidirVrfConfigMode( RoutingPimBidirVrfMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingPimBidirVrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class RouterPimSparseAfConfigMode( RoutingPimSparseAfMode, CliSave.Mode ):
   def __init__( self, vrfNameAndAf ):
      ( vrfName, af ) = vrfNameAndAf
      RoutingPimSparseAfMode.__init__( self, vrfName, af )
      CliSave.Mode.__init__( self, ( vrfName, af ) )

   def skipIfEmpty( self ):
      return True

class RouterPimBidirAfConfigMode( RoutingPimBidirAfMode, CliSave.Mode ):
   def __init__( self, vrfNameAndAf ):
      ( vrfName, af ) = vrfNameAndAf
      RoutingPimBidirAfMode.__init__( self, vrfName, af )
      CliSave.Mode.__init__( self, ( vrfName, af ) )

   def skipIfEmpty( self ):
      return True

def _sortedByIp( coll ):
   return sorted( coll, key=lambda x: x.ipAddr )

def _sortedByPrefix( coll, af ):
   addrFamily = \
      socket.AF_INET if af == AddressFamily.ipv4 else socket.AF_INET6

   def keyFunc( p ):
      ( a, l ) = p.stringValue.split( '/' )
      return ( Arnet.IpAddress( a, addrFamily=addrFamily ), int( l ) )

   sPrefix = sorted( coll, key=keyFunc )
   return sPrefix

def getPimCmdRoot( root, vrfName, af, pimXLegacyVersion ):
   assert isinstance( pimXLegacyVersion, int )
   def getRouterMode():
      return root[ RouterPimBaseConfigMode ].getSingletonInstance()

   def getRouterVrfMode( vrfName ):
      routerMode = getRouterMode()
      return routerMode[ RouterPimVrfConfigMode ].getOrCreateModeInstance(
                                                         vrfName )
   def getIpMode( vrfName, af ):
      if vrfName == DEFAULT_VRF:
         parentMode = getRouterMode()
      else:
         parentMode = getRouterVrfMode( vrfName )
      return parentMode[ RouterPimAfConfigMode ].getOrCreateModeInstance(
                                                         ( vrfName, af ) )
   if pimXLegacyVersion == PimLegacyConfig.ipMode:
      cmds = getIpMode( vrfName, af )[ 'Pim.vrf.af.config' ]
   elif af == AddressFamily.ipv4:
      if vrfName == DEFAULT_VRF:
         cmds = getRouterMode()[ 'Pim.config' ]
      else:
         cmds = getRouterVrfMode( vrfName )[ 'Pim.vrf.config' ]

   return cmds

def getCmdRoot( root, vrfName, af, pimLegacyVersion ):
   #Remove, added to catch save plugin failure
   assert isinstance( pimLegacyVersion, int )
   def getRouterMode():
      return root[ RouterPimSparseBaseConfigMode ].getSingletonInstance()

   def getRouterVrfMode( vrfName ):
      routerMode = getRouterMode()
      return routerMode[ RouterPimSparseVrfConfigMode ].getOrCreateModeInstance(
                                                         vrfName )
   def getIpMode( vrfName, af ):
      if vrfName == DEFAULT_VRF:
         parentMode = getRouterMode()
      else:
         parentMode = getRouterVrfMode( vrfName )
      return parentMode[ RouterPimSparseAfConfigMode ].getOrCreateModeInstance(
                                                         ( vrfName, af ) )
   if pimLegacyVersion == PimLegacyConfig.ipMode:
      cmds = getIpMode( vrfName, af )[ 'Pim.vrf.af.config' ]
   elif af == AddressFamily.ipv4:
      if vrfName == DEFAULT_VRF:
         if pimLegacyVersion == PimLegacyConfig.globalMode:
            cmds = root[ 'Ip.Pim' ]
         else:
            cmds = getRouterMode()[ 'Pim.config' ]
      else:
         cmds = getRouterVrfMode( vrfName )[ 'Pim.vrf.config' ]

   return cmds

def getCmdRootBidir( root, vrfName, af, pimLegacyVersion ):
   def getRouterMode():
      return root[ RouterPimBidirBaseConfigMode ].getSingletonInstance()

   def getRouterVrfMode( vrfName ):
      routerMode = getRouterMode()
      return routerMode[ RouterPimBidirVrfConfigMode ].getOrCreateModeInstance(
                                                         vrfName )
   def getIpMode( vrfName, af ):
      if vrfName == DEFAULT_VRF:
         parentMode = getRouterMode()
      else:
         parentMode = getRouterVrfMode( vrfName )
      return parentMode[ RouterPimBidirAfConfigMode ].getOrCreateModeInstance(
                                                         ( vrfName, af ) )

   if pimLegacyVersion == PimLegacyConfig.ipMode:
      cmds = getIpMode( vrfName, af )[ 'PimBidir.vrf.af.config' ]
   elif af == AddressFamily.ipv4:
      if vrfName == DEFAULT_VRF:
         if pimLegacyVersion == PimLegacyConfig.globalMode:
            cmds = root[ 'Ip.PimBidir' ]
         else:
            cmds = getRouterMode()[ 'PimBidir.config' ]
      else:
         cmds = getRouterVrfMode( vrfName )[ 'PimBidir.vrf.config' ]

   return cmds

def saveConfig( pimConfigColl, root, requireMounts, saveAll, af ):
   pimLegacyConfig = requireMounts[ 'routing/pim/legacyconfig' ]
   cliVersion = getCliSaveVersion( pimLegacyConfig.version, saveAll )

   for vrfName in pimConfigColl.vrfConfig :
      config = pimConfigColl.vrfConfig[ vrfName ]

      if not config.rpTable and not saveAll:
         continue

      saveRoot = getCmdRoot( root, vrfName, af, cliVersion )
      saveConfigImpl( config, saveRoot, saveAll, cliVersion, af )

def savePimBidirConfig( pimConfigColl, root, requireMounts, options ):
   saveAll = options.saveAll
   # Save the default config only if the platform supports multicast routing

   if not McastCommonCliLib.mcastRoutingSupported(
         None,
         requireMounts[ 'routing/hardware/status' ] ):
      saveAll =  False

   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   if not routingHwStatus.pimBidirectionalSupported:
      return

   bidirLegacyConfig = requireMounts[ 'routing/pim/bidir/legacyconfig' ]
   cliVersion = getCliSaveVersion( bidirLegacyConfig.version, saveAll )

   for vrfName in pimConfigColl.vrfConfig :
      config = pimConfigColl.vrfConfig[ vrfName ]
      if not config.rpTable and not saveAll:
         continue
      saveRoot = getCmdRootBidir( root, vrfName, AddressFamily.ipv4, cliVersion )
      saveConfigImpl( config, saveRoot, saveAll, cliVersion, AddressFamily.ipv4 )

def saveConfigImpl( pimConfig, root, saveAll, version, af ):

   rpPriorityDefault = Tac.Type( "Routing::Pim::RpPriority" ).priorityStaticDefault
   hashMaskLenDefault = Tac.Type(
         "Routing::Pim::RpHashMaskLen" ).hashMaskLenDefault( af )
   if version == LegacyConfig.ipMode:
      prefix = "rp address"
   else:
      prefix = "ip pim rp-address"

   for rp in _sortedByIp( pimConfig.rpTable.values() ):
      for g in _sortedByPrefix( rp.group, af ):
         cmd = f"{prefix} {rp.ipAddr}"
         if g != McastCommonCliLib.defaultMcastPrefix( af ) or saveAll:
            # BUG25925
            cmd = f'{cmd} {g.stringValue}'
         if g in rp.priorityGrp:
            grpPriority = rp.priorityGrp[ g ]
            if  grpPriority != rpPriorityDefault:
               cmd = f'{cmd} priority {grpPriority}'
         if g in rp.hashMaskGrp:
            grpHashMask = rp.hashMaskGrp[ g ]
            if grpHashMask != hashMaskLenDefault:
               cmd = f'{cmd} hashmask {grpHashMask}'
         if g in rp.overrideDynGrp and rp.overrideDynGrp[ g ]:
            cmd = '%s override' % cmd # pylint: disable=consider-using-f-string
         root.addCommand( cmd )
      for acl in sorted( rp.acl ):
         cmd = f'{prefix} {rp.ipAddr} access-list {acl}'
         if acl in rp.priorityAcl:
            aclPriority = rp.priorityAcl[ acl ]
            if aclPriority !=  rpPriorityDefault:
               cmd = f'{cmd} priority {aclPriority}'
         if acl in rp.hashMaskAcl:
            aclHashMask = rp.hashMaskAcl[ acl ]
            if aclHashMask != hashMaskLenDefault:
               cmd = f'{cmd} hashmask {aclHashMask}'
         if acl in rp.overrideDynAcl and rp.overrideDynAcl[ acl ]:
            cmd = '%s override' % cmd # pylint: disable=consider-using-f-string
         root.addCommand( cmd )


def saveRpConfig( af, rpConfigColl, root, requireMounts, options, pimLegacyVersion ):
   saveAll = options.saveAll

   # Save the default config only if the platform supports multicast routing
   if not McastCommonCliLib.mcastRoutingSupported(
         None,
         requireMounts[ 'routing6/hardware/status' if af == AddressFamily.ipv6 \
                        else 'routing/hardware/status' ] ):
      saveAll = False

   for vrfName, config in rpConfigColl.vrfConfig.items():
      if pimLegacyVersion != PimLegacyConfig.ipMode:
         if vrfName == DEFAULT_VRF:
            mode = root[ RouterPimSparseBaseConfigMode ].getSingletonInstance()
            saveRoot = mode[ 'Pim.config' ]
         else:
            parentMode = root[ RouterPimSparseBaseConfigMode ].\
                         getSingletonInstance()
            mode = parentMode[ RouterPimSparseVrfConfigMode ].\
                   getOrCreateModeInstance( vrfName )
            saveRoot = mode[ 'Pim.vrf.config' ]
      else:
         saveRoot = getCmdRoot( root, vrfName, af, pimLegacyVersion )

      saveRpConfigImpl( config, saveRoot, saveAll, pimLegacyVersion )

def savePimBidirRpConfig( rpConfigColl, root, requireMounts, options,
                          pimLegacyVersion ):
   saveAll = options.saveAll
   # Save the default config only if the platform supports multicast routing

   if not McastCommonCliLib.mcastRoutingSupported(
         None,
         requireMounts[ 'routing/hardware/status' ] ):
      saveAll = False

   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   if not routingHwStatus.pimBidirectionalSupported:
      return

   for vrfName in rpConfigColl.vrfConfig :
      config = rpConfigColl.vrfConfig[ vrfName ]
      if pimLegacyVersion != PimLegacyConfig.ipMode:
         if vrfName == DEFAULT_VRF :
            mode = root[ RouterPimBidirBaseConfigMode ].getSingletonInstance()
            saveRoot = mode[ 'PimBidir.config' ]
         else:
            parentMode = root[ RouterPimBidirBaseConfigMode ].\
                         getSingletonInstance()
            mode = parentMode[ RouterPimBidirVrfConfigMode ].\
                   getOrCreateModeInstance( vrfName )
            saveRoot = mode[ 'PimBidir.vrf.config' ]
      else :
         saveRoot = getCmdRootBidir( root, vrfName, AddressFamily.ipv4,
                                     pimLegacyVersion )

      saveRpConfigImpl( config, saveRoot, saveAll, pimLegacyVersion )

# pylint: disable-next=useless-return
def saveRpConfigImpl( rpConfig, root, saveAll, pimLegacyVersion ):
   RpHashAlgorithm = Tac.Type( "Routing::Pim::RpHashAlgorithm" )
   hashAlgorithmDefault = RpHashAlgorithm.rpHashAlgorithmDefault

   if rpConfig.rpHashAlgorithm != hashAlgorithmDefault:
      if pimLegacyVersion != PimLegacyConfig.ipMode:
         cmd = 'ip pim rp-hash algorithm modulo'
      else:
         cmd = 'rp hash algorithm modulo'
      root.addCommand( cmd )
   else:
      if saveAll:
         if pimLegacyVersion != PimLegacyConfig.ipMode:
            cmd = 'no ip pim rp-hash algorithm modulo'
         else:
            cmd = 'no rp hash algorithm modulo'
         root.addCommand( cmd )
   return

def savePimServiceAclConfig( root, options, requireMounts ):
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ].cpConfig[ 'ip' ]
   for vrfName, serviceAclVrfConfig in aclCpConfig.serviceAcl.items():
      serviceAclConfig = serviceAclVrfConfig.service.get( 'pim' )
      aclName = serviceAclConfig.aclName if serviceAclConfig else None

      if not aclName and not options.saveAll:
         continue

      if vrfName == DEFAULT_VRF :
         mode = root[ RouterPimBaseConfigMode ].getSingletonInstance()
         saveRoot = mode[ 'Pim.config' ]
      else :
         parentMode = root[ RouterPimBaseConfigMode ].getSingletonInstance()
         mode = parentMode[ RouterPimVrfConfigMode ].getOrCreateModeInstance(
                        vrfName )
         saveRoot = mode[ 'Pim.vrf.config' ]

      if aclName:
         # pylint: disable-next=consider-using-f-string
         saveRoot.addCommand( "ip pim access-group %s" % ( aclName ) )
      else:
         saveRoot.addCommand( "no ip pim access-group" )
