#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliSave
from CliSavePlugin import IntfCliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.SamplePolicyCliSave import SamplePolicySaveMode, SamplePolicySaver
from CliSavePlugin.TrafficPolicyCliSave import ( MatchRuleSaveMode, ActionsSaveMode )
from CliMode.PostcardTelemetry import ( PostcardTelemetryModeBase,
                                        ProfileModeBase, FEATURE )

class PostcardTelemetryConfigMode( PostcardTelemetryModeBase, CliSave.Mode ):
   def __init__( self, param ):
      PostcardTelemetryModeBase.__init__( self )
      CliSave.Mode.__init__( self, self.longModeKey )

class ProfileConfigMode( ProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )

postcardDefaults = Tac.Type( 'PostcardTelemetry::Defaults' )

CliSave.GlobalConfigMode.addCommandSequence( 'PostcardTelemetry.Config' )
IntfCliSave.IntfConfigMode.addCommandSequence( 'PostcardTelemetry.Config' )

CliSave.GlobalConfigMode.addChildMode( PostcardTelemetryConfigMode,
                                       after=[ IntfConfigMode ] )

PostcardTelemetryConfigMode.addCommandSequence( 'PostcardTelemetry.Config' )

PostcardTelemetryConfigMode.addChildMode( SamplePolicySaveMode,
                                          after=[ 'PostcardTelemetry.Config' ] )
SamplePolicySaveMode.addCommandSequence( 'PostcardTelemetry.Policy' )
SamplePolicySaveMode.addChildMode( MatchRuleSaveMode )
MatchRuleSaveMode.addCommandSequence( 'PostcardSamplePolicy.MatchRule' )
ActionsSaveMode.addCommandSequence( 'PostcardSamplePolicy.Actions' )

PostcardTelemetryConfigMode.addChildMode( ProfileConfigMode,
                                          after=[ SamplePolicySaveMode ] )
ProfileConfigMode.addCommandSequence( 'PostcardTelemetry.profile' )

class PostcardSamplePolicySaver( SamplePolicySaver ):
   trafficPoliciesMode = PostcardTelemetryConfigMode
   defaultActionsMode = None

   def __init__( self, entity, root, requireMounts, options, feature,
                 commentKey ):
      SamplePolicySaver.__init__( self, entity, root, requireMounts, options,
                                  feature, commentKey )
      self.policyMapType = 'postcard-sample-policy'
      self.options = options

   def defaultActionsModeCmds( self, defaultActionsMode ):
      raise NotImplementedError

   def matchModeCmds( self, matchMode ):
      return matchMode[ 'PostcardSamplePolicy.MatchRule' ]

   def actionModeCmds( self, actionsMode ):
      return actionsMode[ 'PostcardSamplePolicy.Actions' ]

   def ipv6MatchSupported( self ):
      return True

   def saveTrafficPolicy( self, policyName, policiesMode ):
      if policyName == 'default' and not self.options.saveAll:
         return
      self.trafficPolicy = self.entity.pmapType.pmap[ policyName ].currCfg
      self.currPolicyName = policyName
      if not self.trafficPolicy:
         return
      param = ( self.feature, self.trafficPoliciesMode, policyName )
      trafficPolicyMode = \
         policiesMode[ self.trafficPolicyMode ].getOrCreateModeInstance( param )
      self.saveNamedCounters( trafficPolicyMode )
      self.saveMatchRules( trafficPolicyMode )

sampleRateValue = { 'sampleRate1in16k': '16384',
                    'sampleRate1in32k': '32768',
                    'sampleRate1in64k': '65536' }

@CliSave.saver( 'PostcardTelemetry::PolicyConfig', 'postcardtelemetry/policyconfig',
                requireMounts=( 'cli/config', 'postcardtelemetry/hwCapability',
                                'postcardtelemetry/samplePolicies/input/cli' ) )
def saveConfig( entity, root, requireMounts, options ):
   hwCapability = requireMounts[ 'postcardtelemetry/hwCapability' ]
   saveAll = options.saveAll
   if hwCapability.postcardTelemetrySupported:
      for intfName in entity.intfToPortProfile:
         profileName = entity.intfToPortProfile[ intfName ]
         intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
            intfName )
         intfCmds = intfMode[ 'PostcardTelemetry.Config' ]
         intfCmds.addCommand( 'telemetry postcard policy profile %s' % profileName )

   def addCommandProfile( profile ):
      profileName = profile.name
      cmdList = []
      isDefaultProfile = profileName == 'default'
      if profile:
         if not saveAll and isDefaultProfile:
            return
         profileMode = postConfigMode[ ProfileConfigMode ].\
                       getOrCreateModeInstance( profileName )
         if profile.samplePolicy != '':
            cmdList.append( 'ingress sample policy %s'
                            % profile.samplePolicy )
         if cmdList:
            cmdSeq = profileMode[ 'PostcardTelemetry.profile' ]
            for cmd in cmdList:
               cmdSeq.addCommand( cmd )

   def isDefaultSubTypeVersion( subTypeVersion ):
      return subTypeVersion == postcardDefaults.aristaSubTypeVersion

   def isDefaultDestination( dstConfig ):
      return dstConfig.srcIp.v4Addr == '0.0.0.0' and \
         dstConfig.srcIp.v6Addr.stringValue == '::' and \
         dstConfig.dstIp.v4Addr == '0.0.0.0' and \
         dstConfig.dstIp.v6Addr.stringValue == '::' and \
         isDefaultSubTypeVersion( dstConfig.subTypeVersion ) and \
         dstConfig.ttl == postcardDefaults.ttl and \
         dstConfig.dscp == postcardDefaults.dscp and \
         dstConfig.protocol == postcardDefaults.greProtocolType and \
         dstConfig.vrfName == postcardDefaults.vrf

   def isDefaultConfig( entity ):
      policiesCliConfig = requireMounts[
         'postcardtelemetry/samplePolicies/input/cli' ]
      return not entity.enable and len( entity.portProfile ) == 1 and \
         entity.sampleRate == 'sampleRateInvalid' and \
         entity.sampleDataAndMask is None and \
         entity.vxlanConfig is None and \
         isDefaultDestination( entity.destinationConfig ) and \
         list( policiesCliConfig.pmapType.pmapInput ) == [ 'default' ] and \
         not CliSave.hasComments( 'tele-postcard-policy', requireMounts )

   if hwCapability.postcardTelemetrySupported and (
         not isDefaultConfig( entity ) or saveAll ):
      postConfigMode = root[ PostcardTelemetryConfigMode ].getOrCreateModeInstance(
         None )
      postCmds = postConfigMode[ 'PostcardTelemetry.Config' ]
      if entity.enable:
         postCmds.addCommand( 'no disabled' )
      elif saveAll:
         postCmds.addCommand( 'disabled' )
      if entity.sampleRate != 'sampleRateInvalid':
         postCmds.addCommand( 'ingress sample rate %s' % sampleRateValue[
            entity.sampleRate ] )
      elif entity.sampleDataAndMask:
         checksumVal = hex( entity.sampleDataAndMask.data )
         checksumMask = hex( entity.sampleDataAndMask.mask )
         postCmds.addCommand(
            'ingress sample tcp-udp-checksum value {} mask {}'.format( checksumVal,
                                                                   checksumMask ) )
      elif saveAll:
         postCmds.addCommand( 'ingress sample rate %s' % sampleRateValue[
            'sampleRate1in32k' ] )
      if hwCapability.postcardTelemetryVxlanSupported:
         if entity.vxlanConfig:
            vxlanConfig = entity.vxlanConfig
            if vxlanConfig.vxlanMarkerBit and \
                  hwCapability.postcardTelemetryVxlanMarkerBitConfigurable:
               postCmds.addCommand( 'marker vxlan header word 0 bit %d' %
                  vxlanConfig.vxlanMarkerBit )
            else:
               postCmds.addCommand( 'marker vxlan' )

      if not isDefaultDestination( entity.destinationConfig ) or saveAll:
         dstConfig = entity.destinationConfig
         if dstConfig.srcIp.v6Addr.stringValue == '::' and \
            dstConfig.srcIp.v4Addr == '0.0.0.0':
            srcIp = '0.0.0.0'
         else:
            srcIp = dstConfig.srcIp
         if dstConfig.dstIp.v6Addr.stringValue == '::' and \
            dstConfig.dstIp.v4Addr == '0.0.0.0':
            dstIp = '0.0.0.0'
         else:
            dstIp = dstConfig.dstIp
         # ttl = dstConfig.ttl
         # dscp = dstConfig.dscp
         # protocol = dstConfig.protocol
         # vrfName = dstConfig.vrfName
         command = 'ingress collection gre source %s ' % srcIp
         command += 'destination %s ' % dstIp
         if ( saveAll or not
              isDefaultSubTypeVersion( dstConfig.subTypeVersion ) ):
            command += 'version %s ' % dstConfig.subTypeVersion
         # command += 'dscp %d ' % dscp
         # command += 'ttl %d ' % ttl
         # command += 'protocol %s ' % protocol
         # command += 'vrf %s' % vrfName
         postCmds.addCommand( command )
      cliDumper = PostcardSamplePolicySaver( requireMounts[
         'postcardtelemetry/samplePolicies/input/cli' ],
                                             root, requireMounts, options,
                                             FEATURE,
                                             'mon-telemetry-postcard' )
      cliDumper.save()
      for portProfile in sorted( entity.portProfile ):
         addCommandProfile( entity.portProfile[ portProfile ] )
