# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from operator import attrgetter

from ArPyUtils.Decorators import (
   traced,
   tracedFuncName,
)
from BgpLib import getVpwsName
from CliMode.Pseudowire import (
   FxcMode,
   LdpPseudowireMode,
   LdpPseudowireProfileMode,
   MplsLdpPseudowiresMode,
   MplsPseudowiresMode,
   MplsStaticPseudowiresMode,
   StaticPseudowireMode,
   PatchMode,
   PatchPanelMode,
   PatchQosBaseMode,
)
import CliSave
from CliSavePlugin.BgpMacVrfConfigCliSave import BgpMacVrfConfigSaveMode
from CliSavePlugin.LdpCliSave import LdpConfigMode
from CliSavePlugin.RoutingBgpCliSave import RouterBgpBaseConfigMode
# defaultMplsLdpPseudowiresConfig asserts that getResRibProfileTunnelRibs returns
# a non-empty resolution rib profile so there is a dependency on this
# DefaultConfigPlugin.
import DefaultConfigPlugin.PseudowireAgent # pylint: disable=unused-import
from PseudowireLib import (
   ConnectorType,
   FlowLabelMode,
   getBgpConnectorKey,
   getLdpConnectorKey,
   getNeighborResCmd,
   getResRibProfileTunnelRibs,
   QosMapType,
)
from RouteMapLib import isAsdotConfigured
import Tracing
from TypeFuture import TacLazyType
from natsort import (
   natsorted,
   natsort_key,
)
from Toggles.PseudowireToggleLib import (
   toggleVpwsStandbyErrDisableEnabled,
)

th = Tracing.Handle( "PseudowireCliSave" )
t0 = th.t0
t8 = th.t8

PseudowireType = TacLazyType( 'Pseudowire::PseudowireType' )
TunnelRibNameIdMap = TacLazyType( "Tunnel::TunnelTable::TunnelRibNameIdMap" )
VidNormalization = TacLazyType( "Pseudowire::Fxc::VidNormalization" )
ArnetMplsLabel = TacLazyType( "Arnet::MplsLabel" )

#----------------------------------------------------------
# Helper config mode classes
#----------------------------------------------------------

class PatchPanelConfigMode( PatchPanelMode, CliSave.Mode ):
   def __init__( self, param ):
      PatchPanelMode.__init__( self )
      CliSave.Mode.__init__( self, param )

class PatchConfigMode( PatchMode, CliSave.Mode ):
   def __init__( self, patchName ):
      PatchMode.__init__( self, patchName )
      CliSave.Mode.__init__( self, patchName )

class PatchQosMode( PatchQosBaseMode, CliSave.Mode ):
   def __init__( self, params ):
      patchName, connName, pwName, vpwsName, intfName, vlan, altPwName, \
            noErrDisable, staticMplsPw = params
      PatchQosBaseMode.__init__( self, patchName, connName, pwName=pwName,
                                 vpwsName=vpwsName, intfName=intfName, vlan=vlan,
                                 altPwName=altPwName, noErrDisable=noErrDisable,
                                 staticMplsPw=staticMplsPw )
      CliSave.Mode.__init__( self, self.longModeKey )

   def modeSeparator( self ):
      return False

   def instanceKey( self ):
      return ( natsort_key( self.connName ), self.pwName )

class FxcConfigMode( FxcMode, CliSave.Mode ):
   def __init__( self, patchName ):
      FxcMode.__init__( self, patchName )
      CliSave.Mode.__init__( self, patchName )

# CliSave mode for "mpls pseudowires" config mode - see AID10962.
class MplsPseudowiresConfigMode( MplsPseudowiresMode, CliSave.Mode ):
   def __init__( self, param ):
      MplsPseudowiresMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( MplsPseudowiresConfigMode )
MplsPseudowiresConfigMode.addCommandSequence( "Pseudowire.mplsPwsConfig" )

class MplsStaticPseudowiresConfigMode( MplsStaticPseudowiresMode, CliSave.Mode ):
   def __init__( self, param ):
      MplsStaticPseudowiresMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

MplsPseudowiresConfigMode.addChildMode( MplsStaticPseudowiresConfigMode )
MplsStaticPseudowiresConfigMode.addCommandSequence(
      "Pseudowire.mplsStaticPwsConfig" )

class StaticPseudowireConfigMode( StaticPseudowireMode, CliSave.Mode ):
   def __init__( self, pwName ):
      StaticPseudowireMode.__init__( self, pwName )
      CliSave.Mode.__init__( self, pwName )

MplsStaticPseudowiresConfigMode.addChildMode( StaticPseudowireConfigMode )
StaticPseudowireConfigMode.addCommandSequence( "Pseudowire.staticPwConfig" )

class MplsLdpPseudowiresConfigMode( MplsLdpPseudowiresMode, CliSave.Mode ):
   def __init__( self, param ):
      MplsLdpPseudowiresMode.__init__( self )
      CliSave.Mode.__init__( self, param )

class LdpPseudowireConfigMode( LdpPseudowireMode, CliSave.Mode ):
   def __init__( self, pwName ):
      LdpPseudowireMode.__init__( self, pwName )
      CliSave.Mode.__init__( self, pwName )

class LdpPseudowireProfileConfigMode( LdpPseudowireProfileMode, CliSave.Mode ):
   def __init__( self, profileName ):
      LdpPseudowireProfileMode.__init__( self, profileName )
      CliSave.Mode.__init__( self, profileName )

LdpConfigMode.addChildMode( MplsLdpPseudowiresConfigMode, after=[ 'Ldp.config' ] )
MplsLdpPseudowiresConfigMode.addCommandSequence( "Pseudowire.mplsLdpPwsConfig",
                                                 before=[ LdpPseudowireConfigMode ] )

MplsLdpPseudowiresConfigMode.addChildMode( LdpPseudowireConfigMode )
LdpPseudowireConfigMode.addCommandSequence( "Pseudowire.ldpPwConfig" )

MplsLdpPseudowiresConfigMode.addChildMode( LdpPseudowireProfileConfigMode,
                                           after=[ LdpPseudowireConfigMode ] )
LdpPseudowireProfileConfigMode.addCommandSequence( "Pseudowire.ldpPwProfileConfig" )

CliSave.GlobalConfigMode.addChildMode( PatchPanelConfigMode,
                                       after=[ LdpConfigMode ] )
PatchPanelConfigMode.addCommandSequence( "Pseudowire.patchPanelConfig",
                                         before=[ PatchConfigMode, FxcConfigMode ] )

PatchPanelConfigMode.addChildMode( FxcConfigMode )
FxcConfigMode.addCommandSequence( "Pseudowire.fxcConfig" )

PatchPanelConfigMode.addChildMode( PatchConfigMode, after=[ FxcConfigMode ] )
PatchConfigMode.addCommandSequence( "Pseudowire.patchConfig",
                                    before=[ PatchQosMode ] )

PatchConfigMode.addChildMode( PatchQosMode )
PatchQosMode.addCommandSequence( "Pseudowire.patchQosMode" )

#----------------------------------------------------------
# Patch panel CliSave
#----------------------------------------------------------

def fxcConnectors( patch ):
   """
   Natural sort connectors in patch
   """
   return natsorted( patch.connector.values(), key=attrgetter( 'name' ) )

def patchConnectors( patch ):
   return sorted( patch.connector.values(), key=attrgetter( 'name' ) )

def alternateConnector( patch ):
   """
   Try to get the alternate connector from patch.
   This function is for ldp pw redundancy. See aid/9570 for detail
   e.g. 'connector 1 pseudowire ldp pw1 alternate pw2'
   """
   for cliConnector in patchConnectors( patch ):
      if cliConnector.alternate:
         return cliConnector.connectorKey.name

   return None

def savePatchQosConfig( qosCmds, pwConfig, options, connKey, hwCapability ):
   connector = pwConfig.connector.get( connKey )
   if connector and connector.qosMap.mapType == QosMapType.pwDecapCosToTc:
      qosCmds.addCommand( 'qos map cos to traffic-class %s'
                           % connector.qosMap.mapName )
   elif connector and connector.qosMap.mapType == QosMapType.pwDecapDscpToTc:
      qosCmds.addCommand( 'qos map dscp to traffic-class %s'
                           % connector.qosMap.mapName )
   elif options.saveAll:
      if hwCapability.pwDecapCosQosMapSupported:
         qosCmds.addCommand( 'no qos map cos to traffic-class' )
      if hwCapability.pwDecapDscpQosMapSupported:
         qosCmds.addCommand( 'no qos map dscp to traffic-class' )

@tracedFuncName( trace=t8 )
def savePatchConfig( parentMode, pwConfig, options, patchName, hwCapability,
                     funcName=None ):
   patch = pwConfig.patch.get( patchName )
   if not patch:
      return

   if patch.fxc:
      t0( funcName, "save FXC", patchName )
      mode = parentMode[ FxcConfigMode ].getOrCreateModeInstance( patchName )
      cmds = mode[ "Pseudowire.fxcConfig" ]
   else:
      t0( funcName, "save patch", patchName )
      mode = parentMode[ PatchConfigMode ].getOrCreateModeInstance( patchName )
      cmds = mode[ "Pseudowire.patchConfig" ]

   if not patch.enabled:
      cmds.addCommand( "shutdown" )
   elif options.saveAll:
      cmds.addCommand( "no shutdown" )

   if patch.fxc:
      noNormalization = VidNormalization.noNormalization
      singleVid = VidNormalization.singleVid
      doubleVid = VidNormalization.doubleVid
      vidNormCmd = {
         noNormalization: 'no vlan tag normalization' if options.saveAll else None,
         singleVid: 'vlan tag normalization single',
         doubleVid: 'vlan tag normalization double',
      }[ patch.vidNormalization ]
      if vidNormCmd:
         t0( funcName, "vidNormCmd:", vidNormCmd )
         cmds.addCommand( vidNormCmd )

   connectors = fxcConnectors if patch.fxc else patchConnectors

   for cliConnector in connectors( patch ):
      t0( funcName, "save connector", cliConnector )
      if cliConnector.connectorKey.connectorType == ConnectorType.local:
         if not patch.fxc:
            if cliConnector.connectorKey.isLegacyTagged():
               qosMode = mode[ PatchQosMode ].getOrCreateModeInstance(
                ( patchName, cliConnector.name, None, None,
                  cliConnector.connectorKey.localIntfId(),
                  cliConnector.connectorKey.legacyDot1qTag(),
                  None, cliConnector.noErrDisable, False ) )
            else:
               qosMode = mode[ PatchQosMode ].getOrCreateModeInstance(
                ( patchName, cliConnector.name, None, None,
                  cliConnector.connectorKey.localIntfId(), None, None,
                  cliConnector.noErrDisable, False ) )
         else:
            if cliConnector.connectorKey.isLegacyTagged():
               cmds.addCommand( "connector %s interface %s dot1q vlan %d" %
                                ( cliConnector.name,
                                  cliConnector.connectorKey.localIntfId(),
                                  cliConnector.connectorKey.legacyDot1qTag() ) )
            else:
               cmds.addCommand( "connector %s interface %s" %
                                ( cliConnector.name,
                                  cliConnector.connectorKey.localIntfId() ) )

      elif cliConnector.connectorKey.connectorType == ConnectorType.ldp:
         if cliConnector.alternate: # pylint: disable=no-else-continue
            # Alternate connector will be added with primary one and both connector
            # will share the same qos map
            continue
         else:
            noErrDisable = False
            altPwName = alternateConnector( patch )
            qosMode = mode[ PatchQosMode ].getOrCreateModeInstance(
            ( patchName, cliConnector.name, cliConnector.connectorKey.name,
               None, None, None, altPwName, noErrDisable, False ) )
            qosCmds = qosMode[ "Pseudowire.patchQosMode" ]
            connKey = getLdpConnectorKey( cliConnector.connectorKey.name )
            savePatchQosConfig( qosCmds, pwConfig, options, connKey, hwCapability )

      elif cliConnector.connectorKey.connectorType == ConnectorType.bgp:
         if not patch.fxc:
            noErrDisable = False
            qosMode = mode[ PatchQosMode ].getOrCreateModeInstance(
             ( patchName, cliConnector.name, cliConnector.connectorKey.vpwsName,
               cliConnector.connectorKey.name, None, None, None, noErrDisable,
               False ) )
            qosCmds = qosMode[ "Pseudowire.patchQosMode" ]
            connKey = getBgpConnectorKey( cliConnector.connectorKey.name,
                                          cliConnector.connectorKey.vpwsName )

            savePatchQosConfig( qosCmds, pwConfig, options, connKey, hwCapability )
         else:
            cmds.addCommand( "connector %s pseudowire bgp vpws %s pseudowire %s" %
                             ( cliConnector.name,
                               cliConnector.connectorKey.name,
                               cliConnector.connectorKey.vpwsName ) )

      elif cliConnector.connectorKey.connectorType == ConnectorType.mplsStatic:
         # MPLS static connectors currently do not support Qos mapping. Although,
         # for proper ordering of commands in CliSave, a fake Qos mode
         # is created just like for interface connectors which do not support Qos
         # mapping either. See PatchQosBaseMode definition for more details.
         qosMode = mode[ PatchQosMode ].getOrCreateModeInstance(
          ( patchName, cliConnector.name, cliConnector.connectorKey.name,
            None, None, None, None,
            cliConnector.noErrDisable, True ) )

@traced( trace=t8 )
def defaultPatchConfig( pwConfig ):
   '''Returns true if all attributes related to patch panel have default values.
   This is used to determine whether the patch panel config should be CliSaved.'''
   if pwConfig.patch:
      return False

   if pwConfig.intfReviewDelayRange != pwConfig.intfReviewDelayRangeDefault:
      return False

   if pwConfig.patchBgpVpwsErrdisable != pwConfig.patchBgpVpwsErrdisableDefault:
      return False

   if toggleVpwsStandbyErrDisableEnabled():
      if pwConfig.bgpVpwsStandbyErrdisable != \
            pwConfig.bgpVpwsStandbyErrdisableDefault:
         return False

   return True

@tracedFuncName( trace=t8 )
def savePatchPanelConfig( pwConfig, root, options, hwCapability, funcName=None ):
   mode = root[ PatchPanelConfigMode ].getSingletonInstance()
   cmds = mode[ "Pseudowire.patchPanelConfig" ]

   if ( pwConfig.intfReviewDelayRange != pwConfig.intfReviewDelayRangeDefault or
         options.saveAll ):
      t0( funcName, "save recovery review delay", pwConfig.intfReviewDelayRange )
      cmds.addCommand( "connector interface recovery review delay %d %d" %
            ( pwConfig.intfReviewDelayRange.min,
              pwConfig.intfReviewDelayRange.max ) )

   if pwConfig.patchBgpVpwsErrdisable != pwConfig.patchBgpVpwsErrdisableDefault:
      t0( funcName, "save VPWS remote status indication" )
      cmds.addCommand( "connector interface patch bgp vpws "
                       "remote-failure errdisable" )
   elif options.saveAll:
      cmds.addCommand( "no connector interface patch bgp vpws "
                       "remote-failure errdisable" )

   if toggleVpwsStandbyErrDisableEnabled():
      if pwConfig.bgpVpwsStandbyErrdisable != \
            pwConfig.bgpVpwsStandbyErrdisableDefault:
         t0( funcName, "save VPWS standby mode indication" )
         cmds.addCommand( "connector interface patch bgp vpws "
                          "standby errdisable" )
      elif options.saveAll:
         cmds.addCommand( "no connector interface patch bgp vpws "
                          "standby errdisable" )

   for patchName in sorted( pwConfig.patch ):
      mode = root[ PatchPanelConfigMode ].getSingletonInstance()
      savePatchConfig( mode, pwConfig, options, patchName, hwCapability )

# ----------------------------------------------------------
# MPLS static pseudowires CliSave
# ----------------------------------------------------------

@traced( trace=t8 )
def saveStaticPseudowireConfig( parentMode, pwConfig, options, connKey,
                                hwCapability ):
   if pwConfig.connector[ connKey ].staticPwConfigMode is False:
      return

   saveAll = options.saveAll

   modeType = StaticPseudowireConfigMode
   mode = parentMode[ modeType ].getOrCreateModeInstance( connKey.name )
   cmds = mode[ "Pseudowire.staticPwConfig" ]

   connector = pwConfig.connector[ connKey ]

   if connector.transportIntf:
      cmds.addCommand( "transport " + str( connector.transportIntf ) )
   elif connector.nexthopGroupName:
      cmds.addCommand( "transport nexthop-group " +
                      str( connector.nexthopGroupName ) )
   elif saveAll:
      cmds.addCommand( "no transport" )

   if connector.localLabel != ArnetMplsLabel.null:
      cmds.addCommand( "local label " + str( connector.localLabel ) )
   elif saveAll:
      cmds.addCommand( "no local label" )

   if connector.neighborLabel != ArnetMplsLabel.null:
      cmds.addCommand( "neighbor label " + str( connector.neighborLabel ) )
   elif saveAll:
      cmds.addCommand( "no neighbor label" )

   if connector.controlWord:
      cmds.addCommand( "control-word" )
   elif saveAll:
      cmds.addCommand( "no control-word" )

@traced( trace=t8 )
def defaultMplsStaticPseudowiresConfig( pwConfig ):
   '''Returns if all attributes related to mpls static pseudowires have
   default values. This is used to determine whether the mpls static
   pseudowire config should be CliSaved.'''
   return not any( connectorKey.connectorType == ConnectorType.mplsStatic
                   for connectorKey in pwConfig.connector.keys() )

@traced( trace=t8 )
def saveMplsStaticPseudowiresConfig( mode, pwConfig, root, options,
                                     hwCapability ):

   staticPwConnKeys = []
   for connKey in pwConfig.connector:
      if connKey.connectorType == ConnectorType.mplsStatic:
         staticPwConnKeys.append( connKey )

   staticPwConnKeys = sorted( staticPwConnKeys )

   for connKey in staticPwConnKeys:
      saveStaticPseudowireConfig( mode, pwConfig, options, connKey,
                                  hwCapability )

#----------------------------------------------------------
# MPLS LDP pseudowires CliSave
#----------------------------------------------------------

@traced( trace=t8 )
def saveLdpPseudowireConfig( parentMode, pwConfig, options, connKey,
                             hwCapability ):
   if pwConfig.connector[ connKey ].ldpPwConfigMode is False:
      return

   saveAll = options.saveAll

   isPwProfile = ( connKey.connectorType == ConnectorType.ldpAutoDiscoveryProfile )
   modeType = ( LdpPseudowireProfileConfigMode if isPwProfile else
                LdpPseudowireConfigMode )
   mode = parentMode[ modeType ].getOrCreateModeInstance( connKey.name )
   cmds = mode[ "Pseudowire.ldpPwProfileConfig" if isPwProfile else
                "Pseudowire.ldpPwConfig" ]

   connector = pwConfig.connector[ connKey ]

   tunnelRibs = getResRibProfileTunnelRibs( connector.resolutionRibProfileConfig )

   if tunnelRibs:
      cmds.addCommand( getNeighborResCmd( tunnelRibs ) )

   if not isPwProfile:
      if connector.neighborAddrPresent:
         cmds.addCommand( "neighbor " + str( connector.neighborAddr ) )
      elif saveAll:
         cmds.addCommand( "no neighbor" )

      if connector.pwIdPresent:
         cmds.addCommand( "pseudowire-id %d" % connector.pwId )
      elif saveAll:
         cmds.addCommand( "no pseudowire-id" )

   if connector.mtuPresent:
      cmds.addCommand( 'mtu %d' % connector.mtu )
   elif saveAll:
      cmds.addCommand( 'no mtu' )

   if not isPwProfile:
      if connector.colorPresent:
         cmds.addCommand( 'color %d' % connector.color )
      elif saveAll:
         cmds.addCommand( 'no color' )

   if connector.controlWord:
      cmds.addCommand( 'control-word' )
   elif saveAll:
      cmds.addCommand( 'no control-word' )

   if hwCapability.flowLabelSupported:
      if connector.flowLabelMode == FlowLabelMode.transmit:
         cmds.addCommand( 'label flow transmit' )
      elif connector.flowLabelMode == FlowLabelMode.receive:
         cmds.addCommand( 'label flow receive' )
      elif connector.flowLabelMode == FlowLabelMode.both:
         cmds.addCommand( 'label flow' )
      elif saveAll:
         cmds.addCommand( 'no label flow' )

   if connector.pwType != PseudowireType.pwTypeNone:
      cmds.addCommand( 'pseudowire-type {}'.format(
         'raw' if connector.pwType == PseudowireType.pwType5 else 'tagged' ) )
   elif saveAll:
      cmds.addCommand( 'no pseudowire-type' )

@traced( trace=t8 )
def defaultMplsLdpPseudowiresConfig( pwConfig ):
   '''Returns if all attributes related to mpls ldp pseudowires have
   default values. This is used to determine whether the mpls ldp
   pseudowire config should be CliSaved.'''
   if pwConfig.mtu != pwConfig.getMtuDefault():
      return False

   if pwConfig.mtuIgnore != pwConfig.getMtuIgnoreDefault():
      return False

   if any( connectorKey.connectorType in ( ConnectorType.ldp,
                                           ConnectorType.ldpAutoDiscoveryProfile )
           for connectorKey in pwConfig.connector.keys() ):
      return False

   tunnelRibs = getResRibProfileTunnelRibs( pwConfig.resolutionRibProfileConfig )
   assert tunnelRibs

   defaultTunnelRibs = [ TunnelRibNameIdMap.systemColoredTunnelRibName,
                         TunnelRibNameIdMap.systemTunnelRibName ]

   if tunnelRibs != defaultTunnelRibs:
      return False

   return True

@traced( trace=t8 )
def saveMplsLdpPseudowiresConfig( mode, pwConfig, root, options,
                                  hwCapability ):
   saveAll = options.saveAll

   cmds = mode[ "Pseudowire.mplsLdpPwsConfig" ]

   tunnelRibs = getResRibProfileTunnelRibs( pwConfig.resolutionRibProfileConfig )
   assert tunnelRibs

   defaultTunnelRibs = [ TunnelRibNameIdMap.systemColoredTunnelRibName,
                         TunnelRibNameIdMap.systemTunnelRibName ]

   if saveAll or tunnelRibs != defaultTunnelRibs:
      cmds.addCommand( getNeighborResCmd( tunnelRibs ) )

   if saveAll or pwConfig.mtu != pwConfig.getMtuDefault():
      if pwConfig.mtu == pwConfig.getMtuDefault():
         cmds.addCommand( "no mtu" )
      else:
         cmds.addCommand( "mtu %d" % pwConfig.mtu )

   # TODO: For BUG146565, "no mtu ignore" is not allowed, and "mtu ignore" is
   # the only permissible value
   if saveAll:
      cmds.addCommand( "mtu ignore" )

   ldpPwConnKeys = []
   ldpPwProfileConnKeys = []
   for connKey in pwConfig.connector:
      if connKey.connectorType == ConnectorType.ldp:
         ldpPwConnKeys.append( connKey )
      elif connKey.connectorType == ConnectorType.ldpAutoDiscoveryProfile:
         ldpPwProfileConnKeys.append( connKey )

   ldpPwConnKeys = sorted( ldpPwConnKeys )
   ldpPwProfileConnKeys = sorted( ldpPwProfileConnKeys )

   for connKey in ldpPwConnKeys:
      saveLdpPseudowireConfig( mode, pwConfig, options, connKey,
                               hwCapability )

   for connKey in ldpPwProfileConnKeys:
      saveLdpPseudowireConfig( mode, pwConfig, options, connKey,
                               hwCapability )

@traced( trace=t8 )
@CliSave.saver( "Pseudowire::Config", "pseudowire/config",
                requireMounts=(
                   "routing/hardware/pseudowire/capability",
                ) )
def saveConfig( pwConfig, root, requireMounts, options ):
   saveAll = options.saveAll

   hwCapability = requireMounts[ 'routing/hardware/pseudowire/capability' ]
   if not defaultMplsStaticPseudowiresConfig( pwConfig ) or saveAll:
      pwMode = \
               root[ MplsPseudowiresConfigMode ].getSingletonInstance()
      staticPwMode = \
         pwMode[ MplsStaticPseudowiresConfigMode ].getSingletonInstance()
      saveMplsStaticPseudowiresConfig( staticPwMode, pwConfig, root, options,
                                       hwCapability )

   if not defaultMplsLdpPseudowiresConfig( pwConfig ) or saveAll:
      mode = root[ LdpConfigMode ].getSingletonInstance()
      mode = mode[ MplsLdpPseudowiresConfigMode ].getSingletonInstance()
      saveMplsLdpPseudowiresConfig( mode, pwConfig, root, options,
                                    hwCapability )

   if not defaultPatchConfig( pwConfig ) or saveAll:
      savePatchPanelConfig( pwConfig, root, options, hwCapability )

@traced( trace=t8 )
def getMockLdpResponseConfigCmd( pwaMockLdpStatus, respKey ):
   resp = pwaMockLdpStatus.response.get( respKey )
   if not resp:
      return ""

   pwId, neighbor = respKey.split( '/' )
   cmd = "debug pseudowire-id %s" % pwId
   cmd += " neighbor %s" % neighbor
   cmd += " peer-label %d" % resp.peerLabel
   cmd += " type %d" % ( 4 if resp.remotePwType == "pwType4" else 5 )
   if resp.mtu:
      cmd += " mtu %d" % resp.mtu
   if resp.controlWord:
      cmd += " control-word"
   if resp.peerFlowLabelMode != FlowLabelMode.disabled:
      cmd += " flow-label %s" % resp.peerFlowLabelMode
   if resp.requestedDot1qTag:
      cmd += " dot1q vlan %d" % resp.requestedDot1qTag
   if resp.groupId:
      cmd += " group-id %d" % resp.groupId

   return cmd

@traced( trace=t8 )
@CliSave.saver( "Pseudowire::PseudowireLdpStatus", "pseudowire/ldp/mockStatus" )
def saveDebugLdpConfig( pwaMockLdpStatus, root, requireMounts, options ):
   mockLdpRespKeys = sorted( pwaMockLdpStatus.response.keys() )

   if mockLdpRespKeys:
      mode = root[ LdpConfigMode ].getSingletonInstance()
      mode = mode[ MplsLdpPseudowiresConfigMode ].getSingletonInstance()

      cmds = mode[ "Pseudowire.mplsLdpPwsConfig" ]

      for respKey in mockLdpRespKeys:
         cmds.addCommand( getMockLdpResponseConfigCmd( pwaMockLdpStatus,
                                                       respKey ) )

@traced( trace=t8 )
def getMockBgpResponseConfigCmd( pwaMockBgpStatus, respKey ):
   resp = pwaMockBgpStatus.bgpResponse.get( respKey )
   if not resp:
      return ""

   pwName = respKey.pwName
   cmd = "debug pseudowire %s" % pwName

   for peerAddr, peerInfo in resp.peerInfo.items():
      cmd += " neighbor %s" % peerAddr
      cmd += " peer-label %d" % peerInfo.label
      # peerInfo currently has only one element
      break

   if resp.mtu or resp.controlWord:
      cmd += " local"
      if resp.mtu:
         cmd += " mtu %d" % resp.mtu
      if resp.controlWord:
         cmd += " control-word"

   for peerInfo in resp.peerInfo.values():
      if peerInfo.mtu or peerInfo.controlWord:
         cmd += " peer"
         if peerInfo.mtu:
            cmd += " mtu %d" % peerInfo.mtu
         if peerInfo.controlWord:
            cmd += " control-word"
      # peerInfo currently has only one element
      break

   return cmd

@traced( trace=t8 )
@CliSave.saver( "Pseudowire::PseudowireBgpStatus", "pseudowire/bgp/mockStatus",
                requireMounts=( "routing/bgp/config",
                                "routing/bgp/asn/config" ) )
def saveDebugBgpConfig( pwaMockBgpStatus, root, requireMounts, options ):
   # This must be imported here so it does not interfere with
   # Eos/test/HwEpochVersionCliSanity.py
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   if bgpConfig.asNumber == 0:
      return

   asdotConfigured = isAsdotConfigured( asnConfig )

   parentMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )

   mockBgpRespKeys = sorted( pwaMockBgpStatus.bgpResponse.keys() )

   for respKey in mockBgpRespKeys:
      vpwsName = getVpwsName( respKey.vpwsEviName )
      mode = parentMode[ BgpMacVrfConfigSaveMode ].getOrCreateModeInstance(
         ( vpwsName, False, False, True ) )
      cmds = mode[ 'Bgp.macvrf.config' ]
      cmds.addCommand( getMockBgpResponseConfigCmd( pwaMockBgpStatus, respKey ) )
