# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliSave implementation for common pseudowire profiles (see AID10962), for example:
#    mpls pseudowires
#       profiles
#          profile PROFILE_1
#             control-word
#             mtu 9000
#             pseudowire-type raw
#             label flow transmit

from ArPyUtils.Decorators import traced
from CliMode.PseudowireProfile import (
   MplsPseudowireProfileMode,
   MplsPseudowireProfilesMode,
)
import CliSave
from CliSavePlugin.PseudowireCliSave import MplsPseudowiresConfigMode
from PseudowireLib import FlowLabelMode
from Toggles.PseudowireToggleLib import toggleVplsBgpSignalingEnabled
import Tracing
from TypeFuture import TacLazyType

PseudowireType = TacLazyType( 'Pseudowire::PseudowireType' )

th = Tracing.Handle( "PseudowireProfileCliSave" )
t0 = th.t0
t8 = th.t8

# -------------------------------------------------------------------------------
# CliSave mode for "profiles" config mode under "mpls pseudowires" config mode.
# -------------------------------------------------------------------------------

class MplsPseudowireProfilesConfigMode( MplsPseudowireProfilesMode, CliSave.Mode ):
   def __init__( self, param ):
      MplsPseudowireProfilesMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

if toggleVplsBgpSignalingEnabled():
   MplsPseudowiresConfigMode.addChildMode( MplsPseudowireProfilesConfigMode )
   MplsPseudowireProfilesConfigMode.addCommandSequence(
      "Pseudowire.mplsPwProfilesConfig" )

@traced( trace=t8 )
@CliSave.saver( "Pseudowire::PseudowireProfileDir", "pseudowire/profiles",
                requireMounts=( "routing/hardware/pseudowire/capability", ) )
def saveMplsPseudowireProfiles( pwProfileDir, root, requireMounts, options ):
   if not toggleVplsBgpSignalingEnabled():
      return
   hwCapability = requireMounts[ 'routing/hardware/pseudowire/capability' ]

   # Create the config mode hierarchy under which we will save the profiles.
   mplsPwsMode = root[ MplsPseudowiresConfigMode ].getSingletonInstance()
   mplsPwsProfilesMode = \
      mplsPwsMode[ MplsPseudowireProfilesConfigMode ].getSingletonInstance()

   # Save each profile, sorted by name.
   for profileName in sorted( pwProfileDir.profile ):
      saveMplsPseudowireProfileConfig( mplsPwsProfilesMode, pwProfileDir, options,
                                       profileName, hwCapability )

# -------------------------------------------------------------------------------
# CliSave mode for "profile NAME" config mode under "profiles" config mode.
# -------------------------------------------------------------------------------

class MplsPseudowireProfileConfigMode( MplsPseudowireProfileMode, CliSave.Mode ):
   def __init__( self, profileName ):
      MplsPseudowireProfileMode.__init__( self, profileName )
      CliSave.Mode.__init__( self, profileName )

if toggleVplsBgpSignalingEnabled():
   MplsPseudowireProfilesConfigMode.addChildMode( MplsPseudowireProfileConfigMode )
   MplsPseudowireProfileConfigMode.addCommandSequence(
      "Pseudowire.mplsPwProfileConfig" )

@traced( trace=t8 )
def saveMplsPseudowireProfileConfig( parentMode, pwProfileDir, options, profileName,
                                     hwCapability ):
   # Create the config mode for the specified profile.
   mode = parentMode[ MplsPseudowireProfileConfigMode ].getOrCreateModeInstance(
      profileName )
   cmds = mode[ "Pseudowire.mplsPwProfileConfig" ]

   saveAll = options.saveAll
   profile = pwProfileDir.profile[ profileName ]

   if profile.mtu != profile.mtuDefault:
      cmds.addCommand( f"mtu {profile.mtu}" )
   elif saveAll:
      cmds.addCommand( "no mtu" )

   if profile.controlWord != profile.controlWordDefault:
      cmds.addCommand( "control-word" )
   elif saveAll:
      cmds.addCommand( "no control-word" )

   # TODO: BUG792276, BUG980715
   # Currently Pseudowire profiles under mpls pseudowires only work with VPLS-BGP
   # groups, which do not use the pseudowire type config in a pseudowire profile,
   # therefore the pseudowire type config has been disabled. Once BUG 792276 is
   # implemented and VPLS-LDP groups can also use 'mpls pseudowire' profiles, then
   # we should re-enable the pseudowire type config for 'mpls pseudowire' profiles.
   #
   # if profile.pwType != profile.pwTypeDefault:
   #    if profile.pwType == PseudowireType.pwType4:
   #       cmds.addCommand( "pseudowire-type tagged" )
   #    elif profile.pwType == PseudowireType.pwType5:
   #       cmds.addCommand( "pseudowire-type raw" )
   # elif saveAll:
   #    cmds.addCommand( "no pseudowire-type" )

   if hwCapability.flowLabelSupported:
      if profile.flowLabelMode != profile.flowLabelModeDefault:
         if profile.flowLabelMode == FlowLabelMode.transmit:
            cmds.addCommand( "label flow transmit" )
         elif profile.flowLabelMode == FlowLabelMode.receive:
            cmds.addCommand( "label flow receive" )
         elif profile.flowLabelMode == FlowLabelMode.both:
            cmds.addCommand( "label flow" )
      elif saveAll:
         cmds.addCommand( "no label flow" )
