# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import sys
import CliSave
import EthIntfUtil
# pylint: disable-next=consider-using-from-import
import CliSavePlugin.IntfCliSave as IntfCliSave
from CliSavePlugin.AclCliSave import IpAclConfigMode, CpConfigMode
from CliMode import ConfigMode
from CliMode.Intf import IntfMode
from CliMode.Qos import IntfTxQueueMode, ClassMapModeBase, PolicyMapModeBase, \
   QosProfileModeBase, PolicyMapClassModeBase, DscpRewriteModeBase, \
   CosToTcProfileModeBase,ExpToTcProfileModeBase, TcToCosModeBase, \
   TcToExpModeBase, QosSchedulingModeBase, \
   QosSchedulingIntfModeBase, QosSchedulingIntfGroupModeBase, \
   QosSchedulingPolicyModeBase, PolicingModeBase, DscpToTcModeBase
from CliMode.Fabric import FabricModeBase
import QosLib
from QosLib import mapTypeFromEnum, directionFromEnum, actionFromEnum, \
    pmapQosActionTypes, qosMapType, matchFromEnum, fabricIntfName, \
    pdpMapType, burstUnitFromEnum, isTxQueue7Configurable
import AclCliLib
from QosTypes import tacTrafficClass, tacCos, tacDscp, tacExp, tacTrustMode, \
    tacTxQueueId, tacPriorityGroup, tacPercent, tacTxQueuePriority, tacBwWeight, \
    tacQueueType, tacActionType, tacCMapNm, tacDropPrecedence, \
    tacMatchOption, tacRewriteEnableMode, tacShapeRateUnit, tacShapeRateVal, \
    tacGuaranteedBwVal, tacGuaranteedBwUnit, tacEcnDelayThreshold, \
    tacDscpRewriteMapName, tacCosToTcProfileName, tacTcDpToExpMapName, \
       tacExpToTcProfileName, tacTcToCosMapName, tacBurstVal, \
    tacPolicerType, tacSchedulerCompensation, tacPolicerPktSizeAdjProfileName, \
    tacDscpToTcMapName, tacDei, tacQueueWeight, tacPercentAllowedZero, \
    tacTxQueueSchedulerProfileName

from PfcTypes import tacPfcWatchdogAction
import MultiRangeRule
import Cell

tacIntfConfig = Tac.newInstance( 'Qos::IntfConfig', '' )
invalidShapeRate = Tac.Value( 'Qos::ShapeRate' )

CliSave.GlobalConfigMode.addCommandSequence( 'Qos.config',
                                             after=[ IpAclConfigMode ] )
IntfCliSave.IntfConfigMode.addCommandSequence( 'Qos.config' )
CpConfigMode.addCommandSequence( 'Qos.cp' )

# -------------------------------------------------------------------------------
# Object used for saving commands in "config-if-txq" mode.
# -------------------------------------------------------------------------------
class IntfTxQueueConfigMode( IntfTxQueueMode, CliSave.Mode ):
   def __init__( self, param ):
      IntfTxQueueMode.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )


IntfCliSave.IntfConfigMode.addChildMode( IntfTxQueueConfigMode,
                                         after=[ 'Qos.config' ] )
IntfTxQueueConfigMode.addCommandSequence( 'Intf.txQueue' )

# -------------------------------------------------------------------------------
# Object used for saving commands in "config-cmap" mode.
# -------------------------------------------------------------------------------
class ClassMapConfigMode( ClassMapModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ClassMapModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )


CliSave.GlobalConfigMode.addChildMode( ClassMapConfigMode,
                                       after=[ 'Qos.config' ] )
ClassMapConfigMode.addCommandSequence( 'Qos.cmap' )

# -------------------------------------------------------------------------------
# Object used for saving commands in "config-pmap" mode.
# -------------------------------------------------------------------------------
class PolicyMapConfigMode( PolicyMapModeBase, CliSave.Mode ):
   def __init__( self, param ):
      PolicyMapModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )


CliSave.GlobalConfigMode.addChildMode( PolicyMapConfigMode,
                                       after=[ ClassMapConfigMode ] )
PolicyMapConfigMode.addCommandSequence( 'Qos.pmap' )

# -------------------------------------------------------------------------------
# Object used for saving commands in "config-pmap-c" mode.
# -------------------------------------------------------------------------------
class PolicyMapClassConfigMode( PolicyMapClassModeBase, CliSave.Mode ):
   def __init__( self, param ):
      self.mapType, self.pmapName, self.cmapName, self.classPrio = param
      PolicyMapClassModeBase.__init__( self, param[ : 3 ] )
      CliSave.Mode.__init__( self, self.longModeKey )

   def instanceKey( self ):
      return self.classPrio

   @classmethod
   def useInsertionOrder( cls ):
      # because `instanceKey` is overridden with self.classPrio
      return True


PolicyMapConfigMode.addChildMode( PolicyMapClassConfigMode )
PolicyMapClassConfigMode.addCommandSequence( 'Qos.pmapc' )

class DscpRewriteConfigMode( DscpRewriteModeBase, CliSave.Mode ):
   def __init__( self, param ):
      DscpRewriteModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )


CliSave.GlobalConfigMode.addChildMode( DscpRewriteConfigMode,
                                       after=[ 'Qos.config' ] )
DscpRewriteConfigMode.addCommandSequence( 'Qos.dscpRewriteMap' )

class TcToExpMapConfigMode( TcToExpModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TcToExpModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )

CliSave.GlobalConfigMode.addChildMode( TcToExpMapConfigMode,
                                       after=[ 'Qos.config' ] )
TcToExpMapConfigMode.addCommandSequence( 'Qos.tcDpToExpNamedMap' )

class CosToTcProfileConfigMode( CosToTcProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      CosToTcProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )


CliSave.GlobalConfigMode.addChildMode( CosToTcProfileConfigMode,
                                       after=[ 'Qos.config' ] )
CosToTcProfileConfigMode.addCommandSequence( 'Qos.cosToTcProfile' )

class ExpToTcProfileConfigMode( ExpToTcProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ExpToTcProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )


CliSave.GlobalConfigMode.addChildMode( ExpToTcProfileConfigMode,
                                       after=[ 'Qos.config' ] )
ExpToTcProfileConfigMode.addCommandSequence( 'Qos.expToTcProfile' )

class TcToCosMapConfigMode( TcToCosModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TcToCosModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )


CliSave.GlobalConfigMode.addChildMode( TcToCosMapConfigMode,
                                       after=[ 'Qos.config' ] )
TcToCosMapConfigMode.addCommandSequence( 'Qos.tcToCosMap' )

class DscpToTcMapConfigMode( DscpToTcModeBase, CliSave.Mode ):
   def __init__( self, param ):
      DscpToTcModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )


CliSave.GlobalConfigMode.addChildMode( DscpToTcMapConfigMode,
                                       after=[ 'Qos.config' ] )
DscpToTcMapConfigMode.addCommandSequence( 'Qos.dscpToTcNamedMap' )

class QosProfileConfigMode( QosProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      QosProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )


CliSave.GlobalConfigMode.addChildMode( QosProfileConfigMode,
                                       before=[ 'Qos.config' ] )
QosProfileConfigMode.addCommandSequence( 'Qos.profile' )
QosProfileConfigMode.addChildMode( IntfTxQueueConfigMode,
                                       after=[ 'Qos.profile' ] )
IntfTxQueueConfigMode.addCommandSequence( 'QosProfile.txQueue' )

# -------------------------------------------------------------------------------
# Object used for saving commands in "config-if-fabric" mode.
# -------------------------------------------------------------------------------
class FabricConfigMode( FabricModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FabricModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )


CliSave.GlobalConfigMode.addChildMode( FabricConfigMode,
                                       after=[ 'Qos.config' ] )
FabricConfigMode.addCommandSequence( 'Qos.fabricIntf' )
_servicePolicyNoDirectionStr = {}

# -------------------------------------------------------------------------------
# Object used for saving commands in "config-policing" mode.
# -------------------------------------------------------------------------------
class PolicingConfigMode( PolicingModeBase, CliSave.Mode ):
   def __init__( self, param ):
      PolicingModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True


CliSave.GlobalConfigMode.addChildMode( PolicingConfigMode,
                                       after=[ 'Qos.config' ] )
PolicingConfigMode.addCommandSequence( 'Qos.policing' )

IntfCliSave.IntfConfigMode.addCommandSequence( 'Qos.policing',
                                               after=[ 'Qos.config' ] )

@CliSave.saver( 'Qos::Input::Config', 'qos/input/config/cli',
                requireMounts=( 'qos/hardware/status/global',
                                  'qos/hardware/acl/status/global', 'qos/config',
                                  'hwEpoch/status', 'qos/profile',
                                  'qos/acl/input/defaultPdpPmapCfg',
                                  'dcb/pfc/status',
                                  'interface/config/eth/phy/slice',
                                  'interface/config/all', 'interface/status/all',
                                  'interface/hardware/capability',
                                  'interface/config/subintf',
                                  'qos/global/config',
                                  'qos/input/config/cli' ) )
def saveConfig( entity, root, requireMounts, options ):
   # Display qos config/defaults

   # First save the global configuration
   cmds = root[ 'Qos.config' ]
   hwStatus = requireMounts[ 'qos/hardware/status/global' ]
   subIntfCapability = requireMounts[ 'interface/hardware/capability' ]

   saveAll = options.saveAll

   if not entity.hwConfigVerificationEnabled:
      cmds.addCommand( 'no service config verification qos' )
   elif saveAll:
      cmds.addCommand( 'service config verification qos' )

   if entity.cosRewriteEnabled == tacRewriteEnableMode.rewriteDisabled:
      cmds.addCommand( 'no qos rewrite cos' )
   elif entity.cosRewriteEnabled == tacRewriteEnableMode.rewriteEnabled:
      cmds.addCommand( 'qos rewrite cos' )
   elif saveAll:
      cmds.addCommand( 'default qos rewrite cos' )

   if entity.dscpRewriteEnabled:
      cmds.addCommand( 'qos rewrite dscp' )
   elif saveAll:
      cmds.addCommand( 'no qos rewrite dscp' )

   ecnDelayThreshold = entity.ecnDelayThreshold
   if ecnDelayThreshold != tacEcnDelayThreshold:
      unit = ''
      if ecnDelayThreshold.unit == 'ecnDelayThresholdNs':
         # Currently we store as ns internally while displaying
         # microseconds in CLI.
         unit = 'microseconds'
      cmds.addCommand( 'qos ecn delay threshold %d %s' % \
                          ( ecnDelayThreshold.threshold / 1000, unit ) )
   elif saveAll:
      cmds.addCommand( 'no qos ecn delay threshold' )

   if entity.globalEcnConfig:
      minThd = entity.globalEcnConfig.minThd
      maxThd = entity.globalEcnConfig.maxThd
      unit = entity.globalEcnConfig.unit
      weight = entity.globalEcnConfig.weight
      commandStr = ( 'qos random-detect ecn global-buffer '
                     'minimum-threshold %s %s '
                     'maximum-threshold %s %s' ) % ( minThd, unit, maxThd, unit )
      if weight != tacQueueWeight.defaultWeight:
         commandStr += ' weight %d' % ( weight )
      cmds.addCommand( commandStr )
   elif saveAll:
      cmds.addCommand( 'no qos random-detect ecn global-buffer' )

   if entity.ecnAllowNonEct:
      cmds.addCommand( 'qos random-detect ecn allow non-ect' )
   elif saveAll:
      cmds.addCommand( 'no qos random-detect ecn allow non-ect' )

   if entity.wredAllowEct:
      cmds.addCommand( 'qos random-detect drop allow ect' )
   elif saveAll:
      cmds.addCommand( 'no qos random-detect drop allow ect' )

   if entity.ecnAllowNonEctWithDropPrecedence:
      cmds.addCommand( 'qos random-detect ecn allow non-ect chip-based' )
   elif saveAll:
      cmds.addCommand( 'no qos random-detect ecn allow non-ect chip-based' )

   if hwStatus.hwInitialized:
      tcRange = range( tacTrafficClass.min, hwStatus.numTcSupported )
      for tcName, tc in \
            entity.forwardingGroupNameToTcMap.forwardingGroupNameToTc.items():
         if tc in tcRange:
            cmds.addCommand( 'qos traffic-class %i name %s' % ( tc, tcName ) )

   # pylint: disable-next=singleton-comparison
   if entity.txQueueShapeRateAdaptive == True or saveAll:
      cmdFormat = 'qos tx-queue shape rate percent adaptive'
      if not entity.txQueueShapeRateAdaptive:
         cmdFormat = "no " + cmdFormat
      cmds.addCommand( cmdFormat )

   if hwStatus.hwInitialized:
      txQueueRange = range( tacTxQueueId.min, hwStatus.numTxQueueSupported )
      for txQName, txQ in \
            entity.queueNameToTxQueueIdMap.queueNameToTxQueueId.items():
         if txQ in txQueueRange:
            cmds.addCommand( 'qos tx-queue %i name %s' % ( txQ, txQName ) )

      if hwStatus.txQueueSchedulerProfileFixedSupported or \
            hwStatus.txQueueSchedulerProfileCustomSupported:
         for txQ, schProfile in entity.globalTxQueueToSchedulerProfile.items():
            if not txQ in txQueueRange:
               continue
            if schProfile != tacTxQueueSchedulerProfileName.defaultProfile:
               cmds.addCommand( 'qos tx-queue %i scheduler profile %s' %
                                ( txQ, schProfile ) )
            elif saveAll:
               cmds.addCommand( 'default qos tx-queue %i scheduler profile' %
                                ( txQ ) )

   valKeyMap = {}
   valKeyMapNumEntries = max( tacCos.max, tacDscp.max, tacExp.max,
                              tacTrafficClass.max, tacTxQueueId.max )

   def initValKeyMap():
      for v in range( 0, valKeyMapNumEntries + 1 ):
         valKeyMap[ v ] = []
         for u in range( tacDropPrecedence.min, tacDropPrecedence.max + 1 ):
            valKeyMap[ ( v, u ) ] = []

   def saveMapConfigHelper( mapFromStr, mapToStrList,
                            prefixStr='qos map' ):
      for mapToVal, mapFromList in valKeyMap.items():
         if len( mapFromList ) > 0:
            mapFromListStr = " ".join( [ str( v ) for v in mapFromList ] )
            if not isinstance( mapToVal, tuple ):
               mapToStr = mapToStrList[ 0 ] if isinstance( mapToStrList, tuple ) \
                          else mapToStrList
               cmds.addCommand( '%s %s %s to %s %i' %
                                ( prefixStr, mapFromStr, mapFromListStr,
                                  mapToStr, mapToVal ) )
            else:
               cmds.addCommand( '%s %s %s to %s %i %s %i' %
                                ( prefixStr, mapFromStr, mapFromListStr,
                                  mapToStrList[ 0 ], mapToVal[ 0 ],
                                  mapToStrList[ 1 ], mapToVal[ 1 ] ) )
      initValKeyMap()

   initValKeyMap()

   for cos in range( tacCos.min, tacCos.max + 1 ):
      if ( entity.cosToTcMap[ cos ] != tacTrafficClass.invalid and
           entity.cosToTcMap[ cos ] != hwStatus.defaultCosToTcMap[ cos ] ):
         valKeyMap[ entity.cosToTcMap[ cos ] ].append( cos )
      elif saveAll and hwStatus.hwInitialized:
         assert hwStatus.defaultCosToTcMap[ cos ] != tacTrafficClass.invalid
         valKeyMap[ hwStatus.defaultCosToTcMap[ cos ] ].append( cos )
   saveMapConfigHelper( 'cos', 'traffic-class' )

   if hwStatus.deiPreserveSupported:
      for dei in range( tacDei.min, tacDei.max + 1 ):
         if ( entity.deiToDpMap[ dei ] != tacDropPrecedence.invalid and
              entity.deiToDpMap[ dei ] != hwStatus.defaultDeiToDpMap[ dei ] ):
            valKeyMap[ entity.deiToDpMap[ dei ] ].append( dei )
         elif saveAll and hwStatus.hwInitialized:
            assert hwStatus.defaultDeiToDpMap[ dei ] != tacDropPrecedence.invalid
            valKeyMap[ hwStatus.defaultDeiToDpMap[ dei ] ].append( dei )
         saveMapConfigHelper( 'dei', 'drop-precedence' )

   for dscp in range( tacDscp.min, tacDscp.max + 1 ):
      if entity.dscpToDpMap[ dscp ] != tacDropPrecedence.invalid:
         if entity.dscpToTcMap[ dscp ] != tacTrafficClass.invalid:
            valKeyMap[ ( entity.dscpToTcMap[ dscp ],
                         entity.dscpToDpMap[ dscp ] ) ].append( dscp )
         else:
            valKeyMap[ ( hwStatus.defaultDscpToTcMap[ dscp ],
                         entity.dscpToDpMap[ dscp ] ) ].append( dscp )
      elif entity.dscpToTcMap[ dscp ] != tacTrafficClass.invalid:
         valKeyMap[ ( entity.dscpToTcMap[ dscp ] ) ].append( dscp )
      elif saveAll and hwStatus.hwInitialized:
         assert hwStatus.defaultDscpToTcMap[ dscp ] != tacTrafficClass.invalid
         assert hwStatus.defaultDropPrecedence != tacDropPrecedence.invalid
         if hwStatus.defaultDropPrecedence in QosLib.validDpConfigValues(
                                                     hwStatus ):
            valKeyMap[ ( hwStatus.defaultDscpToTcMap[ dscp ],
                         hwStatus.defaultDropPrecedence ) ].append( dscp )
         else:
            valKeyMap[ hwStatus.defaultDscpToTcMap[ dscp ] ].append( dscp )
   saveMapConfigHelper( 'dscp', ( 'traffic-class', 'drop-precedence' ) )

   if hwStatus.dynamicExpMappingSupported:
      for exp in range( tacExp.min, tacExp.max + 1 ):
         if ( entity.expToTcMap[ exp ] != tacTrafficClass.invalid and
            entity.expToTcMap[ exp ] != hwStatus.defaultExpToTcMap[ exp ] ):
            valKeyMap[ entity.expToTcMap[ exp ] ].append( exp )
         elif saveAll and hwStatus.hwInitialized:
            assert hwStatus.defaultExpToTcMap[ exp ] != tacTrafficClass.invalid
            valKeyMap[ hwStatus.defaultExpToTcMap[ exp ] ].append( exp )
      saveMapConfigHelper( 'exp', 'traffic-class' )

   for tc in range( tacTrafficClass.min, hwStatus.numTcSupported ):
      if ( entity.tcToCosMap[ tc ] != tacCos.invalid and
           entity.tcToCosMap[ tc ] != hwStatus.defaultTcToCosMap[ tc ] ):
         valKeyMap[ entity.tcToCosMap[ tc ] ].append( tc )
      elif saveAll and hwStatus.hwInitialized:
         assert hwStatus.defaultTcToCosMap[ tc ] != tacCos.invalid
         valKeyMap[ hwStatus.defaultTcToCosMap[ tc ] ].append( tc )
   saveMapConfigHelper( 'traffic-class', 'cos' )

   if hwStatus.hwInitialized and hwStatus.tcToDscpMapSupported:
      for tc in range( tacTrafficClass.min, hwStatus.numTcSupported ):
         if ( entity.tcToDscpMap[ tc ] != tacDscp.invalid and
              entity.tcToDscpMap[ tc ] != hwStatus.defaultTcToDscpMap[ tc ] ):
            valKeyMap[ entity.tcToDscpMap[ tc ] ].append( tc )
         elif saveAll and hwStatus.hwInitialized:
            assert hwStatus.defaultTcToDscpMap[ tc ] != tacDscp.invalid
            valKeyMap[ hwStatus.defaultTcToDscpMap[ tc ] ].append( tc )
      saveMapConfigHelper( 'traffic-class', 'dscp' )

   if hwStatus.dynamicExpMappingSupported:
      for tc in range( tacTrafficClass.min, hwStatus.numTcSupported ):
         if ( entity.tcToExpMap[ tc ] != tacExp.invalid and
              entity.tcToExpMap[ tc ] != hwStatus.defaultTcToExpMap[ tc ] ):
            valKeyMap[ entity.tcToExpMap[ tc ] ].append( tc )
         elif saveAll and hwStatus.hwInitialized:
            assert hwStatus.defaultTcToExpMap[ tc ] != tacExp.invalid
            valKeyMap[ hwStatus.defaultTcToExpMap[ tc ] ].append( tc )
      saveMapConfigHelper( 'traffic-class', 'exp' )

   if hwStatus.hwInitialized and hwStatus.numTxQueueSupported != 0:
      if hwStatus.numMulticastQueueSupported == 0:
         txqToken = 'tx-queue'
      else:
         txqToken = 'uc-tx-queue'
      for tc in range( tacTrafficClass.min, hwStatus.numTcSupported ):
         if ( entity.tcToTxQueueMap[ tc ] != tacTxQueueId.invalid and
              entity.tcToTxQueueMap[ tc ] != hwStatus.defaultTcToTxQueueMap[ tc ] ):
            valKeyMap[ entity.tcToTxQueueMap[ tc ] ].append( tc )
         elif saveAll:
            if tc == 7 and hwStatus.tc7ToAnyTxQueueRestricted:
               continue
            assert hwStatus.defaultTcToTxQueueMap[ tc ] != tacTxQueueId.invalid
            valKeyMap[ hwStatus.defaultTcToTxQueueMap[ tc ] ].append( tc )
      saveMapConfigHelper( 'traffic-class', txqToken )

      if hwStatus.numMulticastQueueSupported != 0:
         for tc in range( tacTrafficClass.min, hwStatus.numTcSupported ):
            if ( entity.tcToMcTxQueueMap[ tc ] != tacTxQueueId.invalid and
                 ( entity.tcToMcTxQueueMap[ tc ] !=
                   hwStatus.defaultTcToMcTxQueueMap[ tc ] ) ):
               valKeyMap[ entity.tcToMcTxQueueMap[ tc ] ].append( tc )
            elif saveAll:
               assert hwStatus.defaultTcToMcTxQueueMap[ tc ] != tacTxQueueId.invalid
               valKeyMap[ hwStatus.defaultTcToMcTxQueueMap[ tc ] ].append( tc )
         saveMapConfigHelper( 'traffic-class', 'mc-tx-queue' )

   if hwStatus.hwInitialized and hwStatus.tcToPriorityGroupMapSupported:
      for tc in range( tacTrafficClass.min, hwStatus.numTcSupported ):
         if ( entity.tcToPriorityGroupMap[ tc ] != tacPriorityGroup.invalid and
              entity.tcToPriorityGroupMap[ tc ] !=
                                    hwStatus.defaultTcToPriorityGroupMap[ tc ] ):
            valKeyMap[ entity.tcToPriorityGroupMap[ tc ] ].append( tc )
         elif saveAll and hwStatus.hwInitialized:
            assert ( hwStatus.defaultTcToPriorityGroupMap[ tc ] !=
                                                         tacPriorityGroup.invalid )
            valKeyMap[ hwStatus.defaultTcToPriorityGroupMap[ tc ] ].append( tc )
      saveMapConfigHelper( 'traffic-class', 'priority-group' )

   def addDscpRewriteMap( mapName ):
      dscpRewriteMode = root[ DscpRewriteConfigMode ].getOrCreateModeInstance(
         mapName )
      cmds = dscpRewriteMode[ 'Qos.dscpRewriteMap' ]
      rewriteMap = entity.dscpRewriteMap[ mapName ]
      for tc in range( tacTrafficClass.min,
                       tacTrafficClass.min + hwStatus.numTcSupported ):
         for dp in QosLib.tcDpToDscpSupportedDpValues( hwStatus ):
            tcDpPair = Tac.Value( "Qos::TcDpPair", tc, dp )
            dscpVal = rewriteMap.tcToDscpMap.get( tcDpPair, tacDscp.invalid )
            if dscpVal == tacDscp.invalid:  # pylint: disable=no-else-continue
               continue
            elif hwStatus.dscpRewriteDropPrecedenceSupported:
               cmds.addCommand( 'traffic-class %d drop-precedence %d to dscp %d' % (
                                tcDpPair.tc, tcDpPair.dp, dscpVal ) )
            elif tcDpPair.dp == tacDropPrecedence.min:
               cmds.addCommand( 'traffic-class %d to dscp %d' % (
                                tcDpPair.tc, dscpVal ) )

   for mapName in entity.dscpRewriteMap:
      addDscpRewriteMap( mapName )

   def addTcDpToExpNamedMap( mapName ):
      tcDpToExpNamedMapMode = root[ TcToExpMapConfigMode ].getOrCreateModeInstance(
         mapName )
      cmds = tcDpToExpNamedMapMode[ 'Qos.tcDpToExpNamedMap' ]
      namedExpMap = entity.tcDpToExpNamedMap[ mapName ]
      for tcDpPair, expVal in sorted( namedExpMap.tcDpToExpMap.items() ):
         cmds.addCommand( 'traffic-class %d drop-precedence %d to exp %d' % (
                              tcDpPair.tc, tcDpPair.dp, expVal ) )

   for mapName in sorted( entity.tcDpToExpNamedMap ):
      addTcDpToExpNamedMap( mapName )

   def addCosToTcProfile( profileName ):
      cosToTcProfileMode = root[ CosToTcProfileConfigMode ].getOrCreateModeInstance(
         profileName )
      cmds = cosToTcProfileMode[ 'Qos.cosToTcProfile' ]
      cosToTcProfile = entity.cosToTcProfile[ profileName ]
      for cos in cosToTcProfile.cosToTcMap:
         if cosToTcProfile.cosToTcMap[ cos ] != tacTrafficClass.invalid:
            cmds.addCommand( 'cos %d to traffic-class %d' % (
               cos, cosToTcProfile.cosToTcMap[ cos ] ) )

   for profileName in entity.cosToTcProfile:
      addCosToTcProfile( profileName )

   def addExpToTcProfile( profileName ):
      expToTcProfileMode = root[ ExpToTcProfileConfigMode ].getOrCreateModeInstance(
         profileName )
      cmds = expToTcProfileMode[ 'Qos.expToTcProfile' ]
      expToTcProfile = entity.expToTcProfile[ profileName ]
      for exp in expToTcProfile.expToTcMap:
         if expToTcProfile.expToTcMap[ exp ] != tacTrafficClass.invalid:
            cmds.addCommand( 'exp %d to traffic-class %d' % (
               exp, expToTcProfile.expToTcMap[ exp ] ) )

   for profileName in entity.expToTcProfile:
      addExpToTcProfile( profileName )

   def addDscpToTcMap( mapName ):
      dscpToTcMapMode = root[ DscpToTcMapConfigMode ].getOrCreateModeInstance(
            mapName )
      cmds = dscpToTcMapMode[ 'Qos.dscpToTcNamedMap' ]
      dscpToTcNamedMap = entity.dscpToTcNamedMap[ mapName ]
      for dscp in range( tacDscp.min, tacDscp.max + 1 ):
         if dscpToTcNamedMap.dscpToTcNamedMap[ dscp ] != tacTrafficClass.invalid:
            cmds.addCommand( 'dscp %d to traffic-class %d' % ( dscp,
               dscpToTcNamedMap.dscpToTcNamedMap[ dscp ] ) )

   for mapName in entity.dscpToTcNamedMap:
      addDscpToTcMap( mapName )

   def addTcToCosMap( mapName ):
      # BUG403294: Need to make use of tcToCosPerInterfaceSupported,
      # please see bug description for more information
      tcToCosMapMode = root[ TcToCosMapConfigMode ].getOrCreateModeInstance(
            mapName )
      cmds = tcToCosMapMode[ 'Qos.tcToCosMap' ]
      tcToCosNamedMap = entity.tcToCosNamedMap[ mapName ]
      for tc in range( tacTrafficClass.min,
                       tacTrafficClass.min + hwStatus.numTcSupported ):
         for dp in QosLib.tcDpToCosSupportedDpValues():
            tcDpPair = Tac.Value( "Qos::TcDpPair", tc, dp )
            if tcToCosNamedMap.tcToCosNamedMap[ tcDpPair ] != tacCos.invalid:
               cmds.addCommand( 'traffic-class %d drop-precedence %d to cos %d' %
                     ( tc, dp, tcToCosNamedMap.tcToCosNamedMap[ tcDpPair ] ) )

   for mapName in entity.tcToCosNamedMap:
      addTcToCosMap( mapName )

   def intfPolicerPktSizeAdjCmd( profileName, saveAll ):
      cmd = ''
      if profileName != tacPolicerPktSizeAdjProfileName.defaultProfileName:
         cmd = 'policer packet size adjustment profile %s input' % profileName
      elif saveAll and hwStatus.perPortPolicerPacketSizeAdjSupported:
         cmd = 'default policer packet size adjustment profile'
      return cmd

   cmds = root[ 'Qos.config' ]

   def addCommandServicePolicy():
      def _addCommandServicePolicy( key, intfName ):
         intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
                    intfName )
         cmds = intfMode[ 'Qos.config' ]
         if key.type in _servicePolicyNoDirectionStr:
            directionStr = ""
         else:
            directionStr = directionFromEnum( key.direction ) + " "
         cmds.addCommand( 'service-policy type %s %s%s' %
                          ( mapTypeFromEnum( key.type ), directionStr,
                          key.pmapName ) )

      for key in entity.servicePolicyConfig:
         spCfg = entity.servicePolicyConfig[ key ]
         for intfName in spCfg.intfIds:
            if key.type != pdpMapType:
               _addCommandServicePolicy( key, intfName )

   addCommandServicePolicy()

   def schedulerCompensationCmd( schedulerCompensation, saveAll ):
      cmd = ''
      if schedulerCompensation != tacSchedulerCompensation.invalid:
         compSign = "plus"
         if schedulerCompensation < 0:
            compSign = "minus"
            schedulerCompensation = -schedulerCompensation
         cmd = 'tx-scheduler packet size adjustment %s %s bytes' % (
            compSign, schedulerCompensation )
      elif saveAll and hwStatus.perPortSchedulerCompensationSupported:
         cmd = 'default tx-scheduler packet size adjustment'
      return cmd

   def shapeRateCmd( shapeRate, saveAll ):
      cmd = ''
      if shapeRate.rate != tacShapeRateVal.invalid:
         if shapeRate.unit == tacShapeRateUnit.shapeRatePps:
            cmd = 'shape rate %d pps' % shapeRate.rate
         else:
            if shapeRate.percent != tacPercent.invalid:
               cmd = 'shape rate %d percent' % shapeRate.percent
            else:
               cmd = 'shape rate %d' % shapeRate.rate
            if shapeRate.burstSize.value != tacBurstVal.invalid:
               cmd += ' burst-size %s %s' % ( shapeRate.burstSize.value,
                                              burstUnitFromEnum (
                                                 shapeRate.burstSize.unit ) )
            if shapeRate.shared:
               cmd += ' shared'
      elif saveAll:
         cmd = 'no shape rate'
      return cmd

   def guaranteedBwCmd( bandwidth, saveAll ):
      cmd = ''
      if bandwidth.bw != tacGuaranteedBwVal.invalid:
         if bandwidth.unit == tacGuaranteedBwUnit.guaranteedBwPps:
            cmd = 'bandwidth guaranteed %d pps' % bandwidth.bw
         elif bandwidth.unit == tacGuaranteedBwUnit.guaranteedBwKbps:
            cmd = 'bandwidth guaranteed %d' % bandwidth.bw
         else:
            cmd = 'bandwidth guaranteed percent %d' % bandwidth.bw
      elif saveAll:
         cmd = 'no bandwidth guaranteed'
      return cmd

   def addCommandTxQueue( clitxq, txQueueConfig, ecnTxQueueCounterConfig=None,
                          isSubIntf=False ):
      cmdList = []
      if txQueueConfig:
         if isSubIntf:
            if tacTxQueuePriority.priorityRoundRobin == txQueueConfig.priority:
               cmdList.append( 'no priority' )
            elif tacTxQueuePriority.priorityStrict == txQueueConfig.priority:
               cmdList.append( 'priority strict' )
            elif saveAll:
               cmdList.append( 'default priority' )
         else:
            if tacTxQueuePriority.priorityRoundRobin == txQueueConfig.priority:
               cmdList.append( 'no priority' )
            elif saveAll:
               cmdList.append( 'priority strict' )

         if tacPercent.invalid != txQueueConfig.bandwidth:
            cmdList.append( 'bandwidth percent %d' % txQueueConfig.bandwidth )
         elif tacBwWeight.invalid != txQueueConfig.bandwidthWeight:
            cmdList.append( 'bandwidth weight %d' % txQueueConfig.bandwidthWeight )
         elif saveAll:
            cmdList.append( 'no bandwidth' )

         cmd = guaranteedBwCmd( txQueueConfig.guaranteedBw, saveAll )
         if cmd:
            cmdList.append( cmd )

         cmd = shapeRateCmd( txQueueConfig.shapeRate, saveAll )
         if cmd:
            cmdList.append( cmd )
         if hwStatus.dropPrecedenceThresholdSupported and \
            clitxq.type != tacQueueType.mcq:
            for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
               if dp in QosLib.validDpConfigValues( hwStatus ):
                  if dp in txQueueConfig.dropThresholds:
                     cmdList.append( 'drop-precedence %d drop-threshold percent %d' %
                                  ( dp, txQueueConfig.dropThresholds[ dp ] ) )
                  elif saveAll:
                     cmdList.append( 'default drop-precedence %d drop-threshold '
                                  'percent' % dp )

         if clitxq.type != tacQueueType.mcq:
            if txQueueConfig.ecnConfig:
               minThd = txQueueConfig.ecnConfig.minThd
               maxThd = txQueueConfig.ecnConfig.maxThd
               unit = txQueueConfig.ecnConfig.unit
               maxDroprate = txQueueConfig.ecnConfig.maxDroprate
               weight = txQueueConfig.ecnConfig.weight
               commandStr = ( 'random-detect ecn minimum-threshold %s %s'
                              ' maximum-threshold %s %s' ) % ( minThd,
                                                               unit, maxThd, unit )
               if maxDroprate in range( tacPercentAllowedZero.min,
                     tacPercentAllowedZero.max + 1 ) or saveAll:
                  commandStr += ' max-mark-probability %s' % ( maxDroprate )
               if weight != tacQueueWeight.defaultWeight:
                  commandStr += ' weight %d' % ( weight )

               cmdList.append( commandStr )
            elif saveAll:
               cmdList.append( 'no random-detect ecn minimum-threshold' )

            if txQueueConfig.ecnDelayConfig:
               thd = txQueueConfig.ecnDelayConfig.maxThd
               unit = txQueueConfig.ecnDelayConfig.unit
               commandStr = ( 'random-detect ecn delay threshold %s %s'
                               % ( thd, unit ) )
               cmdList.append( commandStr )
            elif saveAll:
               cmdList.append( 'no random-detect ecn delay threshold' )

            decnConfig = txQueueConfig.bufferBasedDecnConfig
            if hwStatus.bufferBasedDecnSupported and decnConfig:
               cmdList.append( f'random-detect ecn dynamic floor {decnConfig.floor} '
                               f'{decnConfig.unit} offset {decnConfig.offset} '
                               f'{decnConfig.unit}' )
            elif hwStatus.bufferBasedDecnSupported and saveAll:
               cmdList.append( 'no random-detect ecn dynamic' )

         if hwStatus.ecnCountersSupported and ecnTxQueueCounterConfig:
            cmdList.append( 'random-detect ecn count' )

         if clitxq.type != tacQueueType.mcq:
            if txQueueConfig.nonEctConfig:
               minThd = txQueueConfig.nonEctConfig.minThd
               maxThd = txQueueConfig.nonEctConfig.maxThd
               unit = txQueueConfig.nonEctConfig.unit
               weight = txQueueConfig.nonEctConfig.weight
               commandStr = ( 'random-detect non-ect ' +
                    'minimum-threshold %s %s maximum-threshold %s %s'
                    % ( minThd, unit, maxThd, unit ) )
               if weight != tacQueueWeight.defaultWeight:
                  commandStr += ' weight %d' % ( weight )
               cmdList.append( commandStr )
            elif saveAll:
               cmdList.append( 'no random-detect non-ect' )

         if clitxq.type != tacQueueType.mcq:
            if txQueueConfig.delayEcnEnabled:
               ecnDelayThreshold = txQueueConfig.ecnDelayThreshold
               if ecnDelayThreshold != tacEcnDelayThreshold:
                  unit = ''
                  if ecnDelayThreshold.unit == 'ecnDelayThresholdNs':
                     # Currently we store as ns internally while displaying
                     # microseconds in CLI.
                     unit = 'microseconds'
                  cmdList.append( 'ecn delay threshold %d %s' %
                                  ( ecnDelayThreshold.threshold / 1000, unit ) )
               else:
                  cmdList.append( 'ecn delay' )
            elif saveAll:
               cmdList.append( 'no ecn delay' )

         if hwStatus.txQueueSchedulerProfileFixedSupported or \
               hwStatus.txQueueSchedulerProfileCustomSupported:
            schProfile = txQueueConfig.schedulerProfile
            if schProfile != tacTxQueueSchedulerProfileName.defaultProfile:
               cmdList.append( 'scheduler profile %s' % schProfile )
            elif saveAll:
               cmdList.append( 'default scheduler profile' )

         if clitxq.type != tacQueueType.mcq:
            if txQueueConfig.weightConfig != tacQueueWeight.defaultWeight:
               weight = txQueueConfig.weightConfig
               commandStr = 'queue length weight %d' % ( weight )
               cmdList.append( commandStr )
            elif saveAll:
               cmdList.append( 'no queue length weight' )

         if clitxq.type != tacQueueType.mcq:
            if txQueueConfig.wredConfig:
               minThd = txQueueConfig.wredConfig.minThd
               maxThd = txQueueConfig.wredConfig.maxThd
               unit = txQueueConfig.wredConfig.unit
               weight = txQueueConfig.wredConfig.weight
               maxDroprate = txQueueConfig.wredConfig.maxDroprate
               commandStr = 'random-detect drop ' + \
                     'minimum-threshold %s %s maximum-threshold %s %s ' \
                     % ( minThd, unit, maxThd, unit ) + \
                     'drop-probability %s' \
                     % ( maxDroprate )
               if weight != tacQueueWeight.defaultWeight:
                  commandStr += ' weight %d' % ( weight )
               cmdList.append( commandStr )
            elif saveAll:
               cmdList.append( 'no random-detect drop' )

         if ( clitxq.type != tacQueueType.mcq and
              txQueueConfig.dpConfig ):
            for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
               if dp in txQueueConfig.dpConfig.dpWredConfig:
                  dpWredParam = txQueueConfig.dpConfig.dpWredConfig[ dp ]
                  minThd = dpWredParam.minThd
                  maxThd = dpWredParam.maxThd
                  unit = dpWredParam.unit
                  weight = dpWredParam.weight
                  maxDroprate = dpWredParam.maxDroprate
                  commandStr = 'random-detect drop ' + \
                        'drop-precedence %s ' % ( dp ) + \
                        'minimum-threshold %s %s maximum-threshold %s %s ' \
                        % ( minThd, unit, maxThd, unit ) + \
                        'drop-probability %s' \
                        % ( maxDroprate )
                  if weight != tacQueueWeight.defaultWeight:
                     commandStr += ' weight %d' % ( weight )
                  cmdList.append( commandStr )
               elif saveAll:
                  cmdList.append( 'no random-detect drop drop-precedence %s' % dp )

         if clitxq.type != tacQueueType.mcq:
            tacLatencyThreshold = Tac.Type( 'Qos::LatencyThreshold' )
            if txQueueConfig.latencyThreshold != tacLatencyThreshold():
               cmdList.append( 'latency maximum %d %s' %
                               ( txQueueConfig.latencyThreshold.configThreshold(),
                                 txQueueConfig.latencyThreshold.configUnit ) )
            elif saveAll:
               cmdList.append( 'no latency maximum' )

      elif hwStatus.ecnCountersSupported and ecnTxQueueCounterConfig:
         cmdList.append( 'random-detect ecn count' )
      elif saveAll:
         if isSubIntf:
            cmdList.append( 'default priority' )
         else:
            cmdList.append( 'priority strict' )
         cmdList.append( 'no bandwidth' )
         cmdList.append( 'no shape rate' )
         cmdList.append( 'no bandwidth guaranteed' )
         if hwStatus.ecnSupported and clitxq.type != tacQueueType.mcq:
            cmdList.append( 'no random-detect ecn minimum-threshold' )
         if hwStatus.bufferBasedDecnSupported and clitxq.type != tacQueueType.mcq:
            cmdList.append( 'no random-detect ecn dynamic' )
         if hwStatus.ecnCountersSupported:
            cmdList.append( 'random-detect ecn count' )
         if hwStatus.ecnDelaySupported and clitxq.type != tacQueueType.mcq:
            cmdList.append( 'no ecn delay' )
         if hwStatus.wredSupported and clitxq.type != tacQueueType.mcq:
            cmdList.append( 'no random-detect drop' )
         if hwStatus.nonEctThdSupported and clitxq.type != tacQueueType.mcq:
            cmdList.append( 'no random-detect non-ect' )
         if hwStatus.latencyThresholdSupported and clitxq.type != tacQueueType.mcq:
            cmdList.append( 'no latency maximum' )
         if hwStatus.txQueueSchedulerProfileFixedSupported or \
               hwStatus.txQueueSchedulerProfileCustomSupported:
            cmdList.append( 'no scheduler profile' )
      return cmdList

   def getAppliedQosProfileName( intfName ):
      # Only one qosProfile will be active at a time
      # Get that from intfToProfileMap
      intToProfileMap = requireMounts[ 'qos/profile' ].intfToProfileMap
      if intfName in intToProfileMap:
         return intToProfileMap[ intfName ]
      return None

   def addCommandQosProfile( profileName ):
      qosProfileMode = root[ QosProfileConfigMode ].getOrCreateModeInstance(
            profileName )
      qosProfileConfig = requireMounts[ 'qos/profile' ].config
      qosProfile = qosProfileConfig[ profileName ].qosProfile
      cmds = qosProfileMode[ 'Qos.profile' ]

      if qosProfile.trustMode != tacTrustMode.invalid:
         if tacTrustMode.untrusted == qosProfile.trustMode:
            cmds.addCommand( 'no qos trust' )
         else:
            cmds.addCommand( 'qos trust %s' % qosProfile.trustMode )
      elif saveAll:
         cmds.addCommand( 'default qos trust' )

      if qosProfile.defaultCos != tacCos.invalid:
         cmds.addCommand( 'qos cos %s' % qosProfile.defaultCos )
      elif saveAll and hwStatus.hwInitialized and hwStatus.defaultCosSupported:
         cmds.addCommand( 'qos cos %s' % hwStatus.defaultCos )

      if qosProfile.defaultDscp != tacDscp.invalid:
         cmds.addCommand( 'qos dscp %s' % qosProfile.defaultDscp )
      elif saveAll and hwStatus.hwInitialized and hwStatus.defaultDscpSupported:
         cmds.addCommand( 'qos dscp %s' % hwStatus.defaultDscp )

      cmd = schedulerCompensationCmd( qosProfile.schedulerCompensation, saveAll )
      if cmd:
         cmds.addCommand( cmd )

      dscpPreserveIpMplsEncapCmd = 'qos rewrite dscp mpls encapsulation disabled'
      if qosProfile.dscpPreserveIpMplsEncapMode:
         cmds.addCommand( dscpPreserveIpMplsEncapCmd )
      elif saveAll:
         cmds.addCommand( "no " + dscpPreserveIpMplsEncapCmd )

      cmd = shapeRateCmd( qosProfile.shapeRate, saveAll )
      if cmd:
         cmds.addCommand( cmd )

      if subIntfCapability.subIntfBandwidthSupported:
         cmd = guaranteedBwCmd( qosProfile.guaranteedBw, saveAll )
         if cmd:
            cmds.addCommand( cmd )

      portConfig = qosProfile.pfcPortConfig
      if pfcSaveAll:
         if not portConfig:
            portConfig = Tac.newInstance( 'Pfc::PortConfig', profileName, False )

      if portConfig:
         if portConfig.enabled:
            cmds.addCommand( 'priority-flow-control on' )
         elif pfcSaveAll:
            cmds.addCommand( 'no priority-flow-control' )

         if not portConfig.watchdogEnabled:
            cmds.addCommand( 'no priority-flow-control pause watchdog' )
         elif pfcSaveAll:
            cmds.addCommand( 'priority-flow-control pause watchdog' )

         if portConfig.priorities or pfcSaveAll:
            priorities = portConfig.priorities
            dlbPriorities = qosProfile.fabricPfcDlb
            for priority in range( 8 ):
               bit = 1 << priority
               if bit & dlbPriorities != 0:
                  cmds.addCommand( 'priority-flow-control priority %d no-drop dlb'
                                      % ( priority ) )
               elif bit & priorities != 0:
                  cmds.addCommand( 'priority-flow-control priority %d no-drop'
                                      % ( priority ) )
               elif pfcSaveAll and ( pfcStatus.prioritiesSupported & bit ):
                  cmds.addCommand( 'no priority-flow-control priority %d' %
                                   priority )

         if portConfig.portTimerConfig:
            portTimerConfig = portConfig.portTimerConfig
            if portTimerConfig.usePerPortTimerValues:
               cmdString = 'priority-flow-control pause watchdog port timer'
               timeoutString = ' timeout %2.2f' % (
                  portTimerConfig.portWatchdogTimeout )
               cmdString = cmdString + timeoutString
               pollingIntervalValue = "auto" if not \
                        portTimerConfig.portWatchdogPollingInterval else \
                        "%.3f" % ( portTimerConfig.portWatchdogPollingInterval )
               pollingIntervalString = " polling-interval %s" % (
                  pollingIntervalValue )
               cmdString = cmdString + pollingIntervalString
               recoveryCfg = portTimerConfig.portWatchdogRecoveryCfg
               recoveryTimeValue = "0" if not recoveryCfg.recoveryTime else \
                                   "%2.2f" % ( recoveryCfg.recoveryTime )
               if recoveryCfg.forcedRecovery:
                  recoveryTimeValue = recoveryTimeValue + " forced"
               recoveryTimeString = " recovery-time %s" % ( recoveryTimeValue )
               cmdString = cmdString + recoveryTimeString
               cmds.addCommand( cmdString )
            elif pfcSaveAll:
               cmds.addCommand(
                  'no priority-flow-control pause watchdog port timer' )
         elif pfcSaveAll:
            cmds.addCommand( 'no priority-flow-control pause watchdog port timer' )

         if portConfig.watchdogPortAction == tacPfcWatchdogAction.drop:
            cmds.addCommand( 'priority-flow-control pause watchdog '
                             'port action drop' )
         elif portConfig.watchdogPortAction == tacPfcWatchdogAction.errdisable:
            cmds.addCommand( 'priority-flow-control pause watchdog '
                             'port action errdisable' )
         elif portConfig.watchdogPortAction == tacPfcWatchdogAction.notifyOnly:
            cmds.addCommand( 'priority-flow-control pause watchdog '
                             'port action notify-only' )
         elif pfcSaveAll:
            cmds.addCommand( 'no priority-flow-control pause watchdog '
                             'port action' )

      if qosProfile.servicePolicyConfig:
         key = qosProfile.servicePolicyConfig.key
         cmds.addCommand( 'service-policy type qos %s %s' %
                          ( directionFromEnum( key.direction ),
                            key.pmapName ) )

      if hwStatus.hwInitialized and hwStatus.numTxQueueSupported != 0:
         for hwtxqid in range( hwStatus.numTxQueueSupported ):
            if hwStatus.hwTxQueue.get( hwtxqid ) is not None:
               clitxq = hwStatus.hwTxQueue[ hwtxqid ].txQueue
               if clitxq.type == tacQueueType.unknown:
                  queueType = 'txq'
               elif clitxq.type == tacQueueType.ucq:
                  queueType = 'uc-txq'
               elif clitxq.type == tacQueueType.mcq:
                  queueType = 'mc-txq'
               txQueueConfig = qosProfile.txQueueConfig.get( clitxq )
               ecnTxQueueCounterCfg = qosProfile.ecnTxQueueCounterConfig.get(
                  clitxq.id )

               cmdList = addCommandTxQueue( clitxq, txQueueConfig,
                  ecnTxQueueCounterConfig=ecnTxQueueCounterCfg )

               if len( cmdList ) > 0:
                  mode = qosProfileMode[ IntfTxQueueConfigMode ].\
                         getOrCreateModeInstance( ( 'qos-profile-%s' % profileName,
                                                    queueType,
                                                    clitxq.id, ConfigMode ) )
                  cmdSeq = mode[ 'QosProfile.txQueue' ]
                  for cmd in cmdList:
                     cmdSeq.addCommand( cmd )

   pfcStatus = requireMounts[ 'dcb/pfc/status' ]
   pfcSaveAll = options.saveAll

   if not pfcStatus.prioritiesSupported:
      pfcSaveAll = False

   qosProfileConfig = requireMounts[ 'qos/profile' ].config
   for profileConfig in sorted( qosProfileConfig ):
      addCommandQosProfile( qosProfileConfig[ profileConfig ].name )

   cmds = root[ 'Qos.config' ]
   # Now save interface specific settings
   phyIntfConfigDir = requireMounts[ 'interface/config/eth/phy/slice' ]
   allIntfNames = set( EthIntfUtil.allPhyEthernetInterfaceNames( phyIntfConfigDir ) )
   allIntfNames.update( ( i for i in
                             EthIntfUtil.allSwitchportNames(
                                requireMounts,
                                includeEligible=True,
                                includeSubIntf=True )
                          if i.startswith( 'Port-Channel' ) ),
                        entity.intfConfig )
   if saveAll:
      cfgIntfNames = allIntfNames
   else:
      cfgIntfNames = entity.intfConfig

   intfToProfileMap = requireMounts[ 'qos/profile' ].intfToProfileMap
   for intfName in intfToProfileMap:
      # skip filtered interfaces
      if options.intfFilter and intfName not in options.intfFilter:
         continue
      qosProfileName = getAppliedQosProfileName( intfName )
      if intfName == fabricIntfName:
         fabricIntfMode = root[ FabricConfigMode ].getSingletonInstance()
         cmds = fabricIntfMode[ 'Qos.fabricIntf' ]
      else:
         intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
            intfName )
         cmds = intfMode[ 'Qos.config' ]
      if qosProfileName:
         cmds.addCommand( 'service-profile %s' % ( qosProfileName ) )

   cmds = root[ 'Qos.config' ]
   globalPfcDisabledStr = 'priority-flow-control all off'
   if not entity.pfcGlobalEnabled:
      cmds.addCommand( globalPfcDisabledStr )
   elif pfcSaveAll:
      cmds.addCommand( 'no %s' % globalPfcDisabledStr )

   # If selective honoring of PFC priorities isn't supported,
   # set the list of honored priorities to the default value.
   # This is needed to fix BUG479718 and BUG482581.
   if ( not pfcStatus.pfcRxSelectivePriorities and
        # BUG881130: this needs to be removed; but work around this
        # to unblock myself.
        "Sysdb/session/cleanConfig" not in entity.fullName ):
      requireMounts[ 'qos/input/config/cli' ].rxPfcPrioritiesHonored = 0xFF

   rxPfcPrioritiesHonoredStr = \
         'priority-flow-control priority default'
   rxPfcPrioritiesHonored = entity.rxPfcPrioritiesHonored
   # By default, 8 PFC priorities (0-7) would be honored.
   if rxPfcPrioritiesHonored != 0xFF:
      prioList = [
         str( p ) for p in range( 8 )
            if ( 1 << p ) & rxPfcPrioritiesHonored ]
      prioListStr = ' '.join( prioList )
      cmds.addCommand( '%s %s' % ( rxPfcPrioritiesHonoredStr, prioListStr ) )
   elif pfcSaveAll:
      cmds.addCommand( 'no %s' % rxPfcPrioritiesHonoredStr )

   timeoutStr = 'priority-flow-control pause watchdog default timeout'
   if entity.watchdogTimeout:
      cmds.addCommand( '%s %2.2f' % ( timeoutStr, entity.watchdogTimeout ) )
   elif pfcSaveAll:
      cmds.addCommand( 'no %s' % timeoutStr )

   recoveryTimeStr = 'priority-flow-control pause watchdog default recovery-time'
   watchdogRecoveryCfg = entity.watchdogRecoveryCfg
   if watchdogRecoveryCfg.recoveryTime:
      if watchdogRecoveryCfg.forcedRecovery:
         cmds.addCommand( '%s %2.2f forced' %
                          ( recoveryTimeStr, watchdogRecoveryCfg.recoveryTime ) )
      else:
         cmds.addCommand( '%s %2.2f' %
                          ( recoveryTimeStr, watchdogRecoveryCfg.recoveryTime ) )
   elif pfcSaveAll:
      cmds.addCommand( 'no %s' % recoveryTimeStr )

   pollingIntervalStr = \
         'priority-flow-control pause watchdog default polling-interval'
   if entity.watchdogPollingInterval:
      cmds.addCommand( '%s %.3f' %
                       ( pollingIntervalStr, entity.watchdogPollingInterval ) )
   elif pfcSaveAll:
      cmds.addCommand( 'no %s' % pollingIntervalStr )

   if entity.watchdogAction == tacPfcWatchdogAction.drop:
      cmds.addCommand( 'priority-flow-control pause watchdog action drop' )
   elif entity.watchdogAction == tacPfcWatchdogAction.notifyOnly:
      cmds.addCommand( 'priority-flow-control pause watchdog action notify-only' )
   elif pfcSaveAll:
      cmds.addCommand( 'priority-flow-control pause watchdog action errdisable' )

   if entity.overrideAction:
      cmds.addCommand( 'priority-flow-control pause watchdog override action\
 drop' )
   elif pfcSaveAll:
      cmds.addCommand( 'no priority-flow-control pause watchdog override action\
 drop' )

   nonDisruptivePrioStr = \
         'priority-flow-control pause watchdog hardware non-disruptive priority'
   if not pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported == 8:
      nonDisruptivePrio = entity.watchdogNonDisruptivePriorities
      if nonDisruptivePrio:
         prioList = [ str( p ) for p in range( 8 )
                      if ( 1 << p ) & nonDisruptivePrio ]
         prioListStr = ' '.join( prioList )
         cmds.addCommand( '%s %s' % ( nonDisruptivePrioStr, prioListStr ) )
      elif pfcSaveAll:
         cmds.addCommand( 'no %s' % nonDisruptivePrioStr )

   if entity.watchdogNonDisruptiveAction == tacPfcWatchdogAction.ignorePfc:
      cmds.addCommand( 'priority-flow-control pause watchdog '
                       'hardware non-disruptive action forward' )
   elif pfcSaveAll:
      cmds.addCommand( 'no priority-flow-control pause watchdog '
                       'hardware non-disruptive action forward' )

   if entity.forcedNonDisruptiveBehaviour:
      cmds.addCommand( 'priority-flow-control pause watchdog hardware non-disruptive\
 port non-disruptive-only' )
   elif pfcSaveAll:
      cmds.addCommand( 'no priority-flow-control pause watchdog hardware\
 non-disruptive port non-disruptive-only' )

   hwMonitorStr = 'priority-flow-control pause watchdog hardware monitor priority '
   if entity.hwMonitoredPri:
      for i in entity.hwMonitoredPri:
         cmds.addCommand( "%s %d" % ( hwMonitorStr, i ) )
   elif saveAll:
      for i in range( 8 ):
         cmds.addCommand( "no %s %d" % ( hwMonitorStr, i ) )

   # Save settings for regular interfaces
   subIntfConfig = requireMounts[ 'interface/config/subintf' ]
   for intfName in cfgIntfNames:
      # skip filtered interfaces
      if options.intfFilter and intfName not in options.intfFilter:
         continue
      intfConfig = entity.intfConfig.get( intfName )
      if not intfConfig:
         if saveAll:
            intfConfig = tacIntfConfig
         else:
            continue

      intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
         intfName )
      cmds = intfMode[ 'Qos.config' ]

      isSubIntf = Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intfName )
      isIntfVlan = Tac.Type( "Arnet::VlanIntfId" ).isVlanIntfId( intfName )
      if intfConfig.trustMode != tacTrustMode.invalid:
         if tacTrustMode.untrusted == intfConfig.trustMode:
            cmds.addCommand( 'no qos trust' )
         else:
            cmds.addCommand( 'qos trust %s' % intfConfig.trustMode )
      elif saveAll:
         if not isSubIntf and not isIntfVlan:
            cmds.addCommand( 'default qos trust' )

      if intfConfig.defaultCos != tacCos.invalid:
         cmds.addCommand( 'qos cos %s' % intfConfig.defaultCos )
      elif saveAll and hwStatus.hwInitialized and hwStatus.defaultCosSupported:
         if not isSubIntf and not isIntfVlan:
            cmds.addCommand( 'qos cos %s' % hwStatus.defaultCos )

      if intfConfig.defaultDscp != tacDscp.invalid:
         cmds.addCommand( 'qos dscp %s' % intfConfig.defaultDscp )
      elif saveAll and hwStatus.hwInitialized and hwStatus.defaultDscpSupported:
         if not isSubIntf and not isIntfVlan:
            cmds.addCommand( 'qos dscp %s' % hwStatus.defaultDscp )

      cmd = shapeRateCmd( intfConfig.shapeRate, saveAll )
      if cmd:
         if not isIntfVlan:
            cmds.addCommand( cmd )

      if ( ( intfName in subIntfConfig.intfConfig and
             subIntfCapability.subIntfBandwidthSupported ) or
           ( not isSubIntf and not isIntfVlan and
             hwStatus.parentGuaranteedBwSupported )
      ):
         cmd = guaranteedBwCmd( intfConfig.guaranteedBw, saveAll )
         if cmd:
            cmds.addCommand( cmd )

      if intfConfig.dscpRewriteMapName != tacDscpRewriteMapName.defaultMapName:
         cmds.addCommand( 'qos rewrite traffic-class to dscp %s'
                          % intfConfig.dscpRewriteMapName )
      elif saveAll and hwStatus.dscpRewritePerInterfaceSupported:
         if ( ( intfName not in subIntfConfig.intfConfig ) or
               hwStatus.dscpRewritePerSubInterfaceSupported ):
            cmds.addCommand( 'no qos rewrite traffic-class to dscp' )

      if intfConfig.tcDpToExpMapName != tacTcDpToExpMapName.defaultMapName:
         cmds.addCommand( 'qos rewrite traffic-class to exp %s'
                          % intfConfig.tcDpToExpMapName )
      elif saveAll and hwStatus.expRewriteSupported:
         cmds.addCommand( 'no qos rewrite traffic-class to exp' )

      if intfConfig.cosToTcProfileName != tacCosToTcProfileName.defaultProfileName:
         cmds.addCommand( 'qos map cos to traffic-class %s'
                          % intfConfig.cosToTcProfileName )
      elif ( saveAll and hwStatus.cosToTcPerInterfaceSupported ):
         cmds.addCommand( 'no qos map cos to traffic-class' )

      if intfConfig.expToTcProfileName != tacExpToTcProfileName.defaultProfileName:
         cmds.addCommand( 'qos map exp to traffic-class %s'
                          % intfConfig.expToTcProfileName )
      elif ( saveAll and hwStatus.expToTcPerInterfaceSupported ):
         cmds.addCommand( 'no qos map exp to traffic-class' )

      if intfConfig.dscpToTcMapName != tacDscpToTcMapName.defaultMapName:
         cmds.addCommand( 'qos map dscp to traffic-class %s'
                          % intfConfig.dscpToTcMapName )
      elif ( saveAll and hwStatus.dscpToTcPerSubInterfaceSupported ):
         cmds.addCommand( 'no qos map dscp to traffic-class' )

      if intfConfig.tcToCosMapName != tacTcToCosMapName.defaultMapName:
         cmds.addCommand( 'qos map traffic-class to cos %s'
                          % intfConfig.tcToCosMapName )
      elif ( saveAll and hwStatus.tcToCosPerInterfaceSupported ):
         cmds.addCommand( 'no qos map traffic-class to cos' )

      if intfConfig.cpuTcToCosMapName != tacTcToCosMapName.defaultMapName:
         cmds.addCommand( 'qos map cpu traffic-class to cos %s'
                          % intfConfig.cpuTcToCosMapName )
      elif( saveAll and (
            ( not isSubIntf and not intfName.startswith( "Port-Channel" ) and
               hwStatus.cpuTcToCosPerInterfaceSupported ) or
            ( isSubIntf and not intfName.startswith( "Port-Channel" ) and
               hwStatus.cpuTcToCosPerSubInterfaceSupported ) or
            ( intfName.startswith( "Port-Channel" ) and
               hwStatus.cpuTcToCosPerLagSupported ) ) ):
         cmds.addCommand( 'no qos map cpu traffic-class to cos' )

      if intfConfig.dscpPreserveIpMplsEncapMode:
         cmds.addCommand( 'qos rewrite dscp mpls encapsulation disabled' )
      elif saveAll:
         cmds.addCommand( 'no qos rewrite dscp mpls encapsulation disabled' )

      cmd = schedulerCompensationCmd( intfConfig.schedulerCompensation, saveAll )
      if cmd:
         cmds.addCommand( cmd )

      cmd = intfPolicerPktSizeAdjCmd( intfConfig.policerPktSizeAdjProfileName,
                                      saveAll )
      if not isSubIntf and not isIntfVlan and cmd:
         cmds.addCommand( cmd )

      portConfig = intfConfig.pfcPortConfig

      if not portConfig:
         if pfcSaveAll:
            portConfig = Tac.newInstance( 'Pfc::PortConfig', intfName, False )

      if portConfig:
         if portConfig.enabled:
            cmds.addCommand( 'priority-flow-control on' )
         elif pfcSaveAll:
            cmds.addCommand( 'no priority-flow-control' )

         if not portConfig.watchdogEnabled:
            cmds.addCommand( 'no priority-flow-control pause watchdog' )
         elif pfcSaveAll:
            cmds.addCommand( 'priority-flow-control pause watchdog' )

         for priority in range( 8 ):
            if portConfig.priorities & ( 1 << priority ):
               cmds.addCommand( 'priority-flow-control priority %d no-drop' %
                                priority )
            elif pfcSaveAll and ( pfcStatus.prioritiesSupported &
                                     ( 1 << priority ) ):
               cmds.addCommand( 'no priority-flow-control priority %d' % priority )

         if portConfig.watchdogPortAction == tacPfcWatchdogAction.drop:
            cmds.addCommand( 'priority-flow-control pause watchdog '
                             'port action drop' )
         elif portConfig.watchdogPortAction == tacPfcWatchdogAction.errdisable:
            cmds.addCommand( 'priority-flow-control pause watchdog '
                             'port action errdisable' )
         elif portConfig.watchdogPortAction == tacPfcWatchdogAction.notifyOnly:
            cmds.addCommand( 'priority-flow-control pause watchdog '
                             'port action notify-only' )
         elif pfcSaveAll:
            cmds.addCommand( 'no priority-flow-control pause watchdog port action' )

         if portConfig.portTimerConfig:
            portTimerConfig = portConfig.portTimerConfig
            if portTimerConfig.usePerPortTimerValues:
               cmdString = 'priority-flow-control pause watchdog port timer'
               timeoutString = ' timeout %2.2f' % (
                  portTimerConfig.portWatchdogTimeout )
               cmdString = cmdString + timeoutString
               pollingIntervalValue = "auto" if not \
                        portTimerConfig.portWatchdogPollingInterval else \
                        "%.3f" % ( portTimerConfig.portWatchdogPollingInterval )
               pollingIntervalString = " polling-interval %s" % (
                  pollingIntervalValue )
               cmdString = cmdString + pollingIntervalString
               recoveryCfg = portTimerConfig.portWatchdogRecoveryCfg
               recoveryTimeValue = "0" if not recoveryCfg.recoveryTime else \
                                   "%2.2f" % ( recoveryCfg.recoveryTime )
               if recoveryCfg.forcedRecovery:
                  recoveryTimeValue = recoveryTimeValue + " forced"
               recoveryTimeString = " recovery-time %s" % ( recoveryTimeValue )
               cmdString = cmdString + recoveryTimeString
               cmds.addCommand( cmdString )
            elif pfcSaveAll:
               cmds.addCommand(
                  'no priority-flow-control pause watchdog port timer' )
         elif pfcSaveAll:
            cmds.addCommand( 'no priority-flow-control pause watchdog port timer' )

   if hwStatus.hwInitialized and hwStatus.numTxQueueSupported != 0:
      for intfName in allIntfNames:
         intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
            intfName )
         intfConfig = entity.intfConfig.get( intfName, tacIntfConfig )
         numTxQueues = hwStatus.numTxQueueSupported
         isSubIntf = Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intfName ) \
                  if intfName is not None else False
         if isSubIntf:
            numTxQueues = requireMounts[ 'qos/global/config' ].numTxqsPerSubIntf
         for hwtxqid in range( numTxQueues ):
            if not isTxQueue7Configurable( hwStatus ) and hwtxqid == 7:
               continue
            # XXX_adityavd SCH_FIXES
            if hwStatus.hwTxQueue.get( hwtxqid ) is None:
               continue
            clitxq = hwStatus.hwTxQueue[ hwtxqid ].txQueue
            if clitxq.type == tacQueueType.unknown:
               queueType = 'txq'
            elif clitxq.type == tacQueueType.ucq:
               queueType = 'uc-txq'
            elif clitxq.type == tacQueueType.mcq:
               queueType = 'mc-txq'
            txQueueConfig = intfConfig.txQueueConfig.get( clitxq )

            cmdList = addCommandTxQueue( clitxq, txQueueConfig, isSubIntf=isSubIntf )

            intfShortname = IntfMode.getShortname( intfName )
            intfShortname = 'if-%s' % ( intfShortname )
            mode = intfMode[ IntfTxQueueConfigMode ].getOrCreateModeInstance(
            ( intfShortname, queueType, clitxq.id, ConfigMode ) )
            cmdSeq = mode[ 'Intf.txQueue' ]
            for cmd in cmdList:
               cmdSeq.addCommand( cmd )

   if hwStatus.ecnQueueCounterSupported:
      cliCounterConfig = entity
      if saveAll:
         phyIntfConfigDir = requireMounts[ 'interface/config/eth/phy/slice' ]
         phyIntfNames = EthIntfUtil.allPhyEthernetInterfaceNames( phyIntfConfigDir )
         cfgIntfNames = set( cliCounterConfig.ecnIntfCounterConfig )
         cfgIntfNames.update( phyIntfNames )
      else:
         cfgIntfNames = cliCounterConfig.ecnIntfCounterConfig

      for intfName in cfgIntfNames:
         # skip filtered interfaces
         if options.intfFilter and intfName not in options.intfFilter:
            continue
         intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
            intfName )
         intfCounterCfg = cliCounterConfig.ecnIntfCounterConfig.get( intfName )
         numTxQIds = hwStatus.numTxQueueSupported
         txQStr = 'txq'
         if hwStatus.numMulticastQueueSupported > 0:
            numTxQIds = hwStatus.numUnicastQueueSupported
            txQStr = 'uc-txq'
         for txQId in range( numTxQIds ):
            cmd = None
            if ( not intfCounterCfg ) and saveAll:
               cmd = 'no random-detect ecn count'
            elif intfCounterCfg:
               queueCounterCfg = intfCounterCfg.ecnTxQueueCounterConfig.get( txQId )
               if ( not queueCounterCfg ) and saveAll:
                  cmd = 'no random-detect ecn count'
               elif queueCounterCfg and queueCounterCfg.counterEnable:
                  cmd = 'random-detect ecn count'
            if cmd:
               intfShortname = IntfMode.getShortname( intfName )
               intfShortname = 'if-%s' % ( intfShortname )
               mode = intfMode[ IntfTxQueueConfigMode ].getOrCreateModeInstance(
                                    ( intfShortname, txQStr, txQId, ConfigMode ) )
               cmdSeq = mode[ 'Intf.txQueue' ]
               cmdSeq.addCommand( cmd )


_addCommandMapTypeCallback = {}

def addCommandMapTypeQos( hwStatus, sliceHwStatus, aclHwStatus, hwEpochStatus ):
   return aclHwStatus.policyQosSupported


_addCommandMapTypeCallback[ qosMapType ] = addCommandMapTypeQos

_addCommandCMapCallback = {}

def addCommandQosCMap( cmap ):
   return True


_addCommandCMapCallback[ qosMapType ] = addCommandQosCMap

def addCommandQosPMap( root, entity, hwStatus, sliceHwStatus,
                       defaultPdpPmapCfg, mapType, pmapName, saveAll ):
   if ( mapType == pdpMapType and defaultPdpPmapCfg and
        pmapName == defaultPdpPmapCfg.name ):
      # PDP support is removed
      return
   else:
      pmap = entity.pmapType[ mapType ].pmap[ pmapName ]
   shared = pmap.shared

   def addCommandQosPMapClass( cmapName, classPrio ):
      actions = None
      policer = None
      count = None
      dscpConfiguredAsName = False
      dscpConfiguredAsNameYellow = False

      if QosLib.isDefaultClass( mapType, cmapName ) is True:
         if pmap.classActionDefault != None:  # pylint: disable=singleton-comparison
            actions = pmap.classActionDefault.policyAction
            policer = pmap.classActionDefault.policer
            count = pmap.classActionDefault.count
            dscpConfiguredAsName = pmap.classActionDefault.dscpConfiguredAsName
            dscpConfiguredAsNameYellow = pmap.classActionDefault.\
                                         dscpConfiguredAsNameYellow
      else:
         actions = pmap.classAction[ cmapName ].policyAction
         policer = pmap.classAction[ cmapName ].policer
         count = pmap.classAction[ cmapName ].count
         dscpConfiguredAsName = pmap.classAction[ cmapName ].dscpConfiguredAsName
         dscpConfiguredAsNameYellow = pmap.classAction[ cmapName ].\
                                      dscpConfiguredAsNameYellow

      pmapMode = root[ PolicyMapConfigMode ].getOrCreateModeInstance(
         ( mapType, pmapName, shared ) )
      configPmapClassMode = pmapMode[ PolicyMapClassConfigMode ].\
                            getOrCreateModeInstance( (
                               mapType,
                               pmapName,
                               cmapName,
                               ( 'dynamic',
                                 classPrio ) ) )
      cmds = configPmapClassMode[ 'Qos.pmapc' ]

      def addCommandQosPMapClassAction():
         def addCommandQosPMapAction( actions, actionType ):
            def addCommandActionU32( action, actStr ):
               cmds.addCommand( 'set %s %d' %
                                ( actStr, action.value ) )

            if actionType in [ tacActionType.actionSetDscp,
                               tacActionType.actionSetCos,
                               tacActionType.actionSetTc,
                               tacActionType.actionSetDropPrecedence ]:
               actStr = actionFromEnum( actionType )

               # For DSCP, the name-string should be printed, if configured.
               if ( actionType == tacActionType.actionSetDscp ) and \
                  dscpConfiguredAsName:
                  action = actions[ actionType ]
                  name = AclCliLib.dscpNameFromValue( action.value )
                  cmds.addCommand( 'set %s %s' % ( actStr, name ) )
               else:
                  addCommandActionU32( actions[ actionType ], actStr )

            if actionType in [ tacActionType.actionSetDrop ]:
               cmds.addCommand( 'drop' )

         # we want to maintain the order how we write the config
         if actions:
            for actionType in pmapQosActionTypes:
               if actionType in actions:
                  addCommandQosPMapAction( actions, actionType )

         if policer:
            cirUnitStr = QosLib.rateUnitFromEnum( policer.cirUnit )
            bcUnitStr = QosLib.burstUnitFromEnum( policer.bcUnit )
            if policer.pir:
               pirUnitStr = QosLib.rateUnitFromEnum( policer.pirUnit )
               beUnitStr = QosLib.burstUnitFromEnum( policer.beUnit )
               yellowActions = policer.yellowActions
               cmd = 'police rate %d %s burst-size %d %s ' % ( policer.cir,
                                                cirUnitStr, policer.bc, bcUnitStr )
               for actionType in yellowActions:
                  if actionType in [ tacActionType.actionSetDscp,
                                     tacActionType.actionSetCos,
                                     tacActionType.actionSetTc ]:
                     action = actionFromEnum( actionType )
                     if dscpConfiguredAsNameYellow:
                        name = AclCliLib.dscpNameFromValue(
                           yellowActions[ actionType ].value )
                        cmd += 'action set %s %s ' % ( action, name )
                     else:
                        cmd += 'action set %s %d ' % ( action,
                               yellowActions[ actionType ].value )
                  else:
                     cmd += 'action set drop-precedence '
               cmd += 'rate %d %s burst-size %d %s' % ( policer.pir, pirUnitStr,
                                                        policer.be, beUnitStr )
               cmds.addCommand( cmd )
            else:
               # BUG197856: Currently a named policer can only be a
               # shared policer. When we support policer profiles for
               # unshared policers, we need to update this check to
               # look at shared and named attributes.
               if policer.named:
                  cmds.addCommand( 'police shared %s' % (
                                     policer.name ) )
               elif policer.cmdVersion == 1:
                  cmds.addCommand( 'police cir %d %s bc %d %s' % (
                           policer.cir, cirUnitStr, policer.bc,
                           bcUnitStr ) )
               else:
                  cmds.addCommand( 'police rate %d %s burst-size '
                                     '%d %s' % ( policer.cir, cirUnitStr,
                                                 policer.bc,
                                                 bcUnitStr ) )
         if count:
            cmds.addCommand( 'count' )

      addCommandQosPMapClassAction()

   # walk in the prio order to display the class-maps
   prio = pmap.classPrio
   indices = list( prio )
   indices.sort()
   for index in indices:
      cmapName = prio[ index ].cmapName
      addCommandQosPMapClass( cmapName, ( 'dynamic', index ) )
   cmapName = tacCMapNm.classDefault
   addCommandQosPMapClass( cmapName, ( 'dynamic', sys.maxsize ) )


_addCommandPMapCallback = {}
_addCommandPMapCallback[ qosMapType ] = addCommandQosPMap
_addCommandPMapCallback[ pdpMapType ] = addCommandQosPMap

@CliSave.saver( 'Qos::Input::AclConfig', 'qos/acl/input/cli',
                requireMounts=( 'qos/hardware/status/global',
                                   Cell.path( 'qos/hardware/status/slice' ),
                                  'qos/hardware/acl/status/global',
                                  'qos/acl/input/defaultPdpPmapCfg',
                                  'hwEpoch/status',
                                  'interface/config/eth/phy/slice',
                                  'interface/config/all', 'interface/status/all' ) )
def saveQosAclConfig( entity, root, requireMounts, options ):
   # servicePolicyConfig, pmapType, cmapType are in qos/acl/input/cli
   hwStatus = requireMounts[ 'qos/hardware/status/global' ]
   sliceHwStatus = requireMounts[ Cell.path( 'qos/hardware/status/slice' ) ]
   aclHwStatus = requireMounts[ 'qos/hardware/acl/status/global' ]
   hwEpochStatus = requireMounts[ 'hwEpoch/status' ]

   saveAll = options.saveAll

   def addCommandCMapType( mapType ):
      def addCommandCMap( cmapName ):
         cmap = entity.cmapType[ mapType ].cmap[ cmapName ]

         cb = _addCommandCMapCallback.get( cmap.type )
         if not( cb and cb( cmap ) ):
            return

         configCMapMode = root[ ClassMapConfigMode ].getOrCreateModeInstance(
            ( mapType, cmapName ) )
         cmds = configCMapMode[ 'Qos.cmap' ]

         def addCommandCMapMatch( cmap ):
            def addCommandMatch( m ):
               def addCommandAcl( m ):
                  cmds.addCommand( 'match %s %s' % ( matchFromEnum( m.option ),
                                                     m.strValue ) )

               def addCommandL2Params( m ):
                  assert m.vlanValue or m.cosValue or m.innerVlanValue or \
                         m.innerCosValue or m.deiValue
                  cliStr = 'match'
                  if m.vlanValue:
                     vlanId = m.vlanValue.vlan
                     vlanMask = m.vlanValue.vlanMask
                     maskValid = m.vlanValue.maskValid
                     if maskValid:
                        cliStr += ' vlan %d %#5.3x' % ( vlanId, vlanMask )
                     else:
                        ids = set()
                        for key in m.vlanValue.vlanColl:
                           ids.update( list( range( key.min, key.max + 1 ) ) )
                        vlanString = \
                           MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
                        cliStr += ' vlan %s' % ( vlanString )
                  if m.innerVlanValue:
                     innerVlanId = m.innerVlanValue.innerVlan
                     innerVlanMask = m.innerVlanValue.innerVlanMask
                     maskValid = m.innerVlanValue.maskValid
                     if maskValid:
                        cliStr += ' vlan inner %d %#5.3x' % ( innerVlanId,
                                                              innerVlanMask )
                     else:
                        ids = set()
                        for key in m.innerVlanValue.innerVlanColl:
                           ids.update( list( range( key.min, key.max + 1 ) ) )
                        innerVlanString = \
                           MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
                        cliStr += ' vlan inner %s' % ( innerVlanString )
                  if m.cosValue:
                     ids = set()
                     for key in m.cosValue.cosColl:
                        ids.update( list( range( key.min, key.max + 1 ) ) )
                     cosString = \
                        MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
                     cliStr += ' cos %s' % cosString
                  if m.innerCosValue:
                     ids = set()
                     for key in m.innerCosValue.innerCosColl:
                        ids.update( list( range( key.min, key.max + 1 ) ) )
                     innerCosString = \
                        MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
                     cliStr += ' cos inner %s' % innerCosString
                  if m.deiValue:
                     cliStr += ' dei ' + str( m.deiValue.deiVal )
                  cmds.addCommand( cliStr )

               def addCommandDscpEcn( m ):
                  assert m.dscpEcnValue
                  dscp = m.dscpEcnValue.dscp
                  ecn = m.dscpEcnValue.ecn
                  dscpNameValid = m.dscpEcnValue.dscpNameValid
                  ecnName = ""
                  if ecn != Tac.Type( "Acl::Ecn" ).dontCare:
                     ecnName = " ecn %s" % ( AclCliLib.ecnNameFromValue( ecn ) )
                  if dscp == Tac.Value( "Qos::DscpVal" ).invalid:
                     # pylint: disable-next=use-implicit-booleaness-not-len
                     if not len( m.dscpEcnValue.dscpColl ):
                        cmds.addCommand( 'match%s' % ( ecnName ) )
                     else:
                        ids = set()
                        for key in m.dscpEcnValue.dscpColl:
                           ids.update( list( range( key.min, key.max + 1 ) ) )
                        dscpStr = \
                           MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
                        cmds.addCommand( 'match dscp %s%s' % ( dscpStr, ecnName ) )
                  else:
                     dscpName = str( dscp )
                     if dscpNameValid:
                        dscpName = AclCliLib.dscpNameFromValue( dscp )
                     cmds.addCommand( 'match dscp %s%s' % ( dscpName, ecnName ) )

               def addCommandMplsTrafficClass( m ):
                  assert m.mplsTrafficClassVal
                  ids = set()
                  if len( m.mplsTrafficClassVal.mplsTrafficClassColl ) == 1:
                     for key in m.mplsTrafficClassVal.mplsTrafficClassColl:
                        if key.min == key.max:
                           cmds.addCommand( 'match mpls traffic-class %d' % (
                              key.min ) )
                        else:
                           ids.update( list( range( key.min, key.max + 1 ) ) )
                           mplsTrafficClassString = \
                               MultiRangeRule.multiRangeToCanonicalString(
                                     list( ids ) )
                           cmds.addCommand(
                              'match mpls traffic-class %s' %
                              mplsTrafficClassString )
                  else:
                     for key in m.mplsTrafficClassVal.mplsTrafficClassColl:
                        ids.update( list( range( key.min, key.max + 1 ) ) )
                     mplsTrafficClassString = \
                         MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
                     cmds.addCommand(
                           'match mpls traffic-class %s' % mplsTrafficClassString )
               if m.option in [ tacMatchOption.matchIpAccessGroup,
                     tacMatchOption.matchIpv6AccessGroup,
                     tacMatchOption.matchMacAccessGroup ]:
                  addCommandAcl( m )
               if m.option == tacMatchOption.matchL2Params:
                  addCommandL2Params( m )
               if m.option == tacMatchOption.matchDscpEcn:
                  addCommandDscpEcn( m )
               if m.option == tacMatchOption.matchMplsTrafficClass:
                  addCommandMplsTrafficClass( m )

            # Add all the matches
            for mname in cmap.match:
               m = cmap.match[ mname ]
               addCommandMatch( m )

         addCommandCMapMatch( cmap )

      for cmapName in entity.cmapType[ mapType ].cmap:
         addCommandCMap( cmapName )

   def addCommandPMapType( mapType ):
      # display all pmaps in sorted order of names
      pmapNames = sorted( entity.pmapType[ mapType ].pmap.keys() )
      for pmapName in pmapNames:
         cb = _addCommandPMapCallback.get( mapType )
         if cb:
            cb( root, entity, hwStatus, sliceHwStatus,
                None, mapType, pmapName, saveAll )

   def addCommandAllCMapType():
      for t in QosLib.mapTypes:
         cb = _addCommandMapTypeCallback.get( t )
         if ( not cb ) or cb( hwStatus, sliceHwStatus, aclHwStatus, hwEpochStatus ):
            addCommandCMapType( t )

   def addCommandAllPMapType():
      for t in QosLib.mapTypes:
         cb = _addCommandMapTypeCallback.get( t )
         if ( not cb ) or cb( hwStatus, sliceHwStatus, aclHwStatus, hwEpochStatus ):
            addCommandPMapType( t )

   def addCommandPMapSharingKnob():
      if entity.sviPolicyQosSharing:
         root[ 'Qos.config' ].addCommand(
            'hardware access-list qos resource sharing vlan in' )

   def addCommandSubIntfRoutedPortAclSharingKnob():
      if entity.routedPortSubIntfQosAclSharing:
         if entity.usingQosRoutedPortSubIntfAclSharingCli:
            root[ 'Qos.config' ].addCommand(
               'hardware access-list qos resource sharing routedport subinterface' )
         else:
            root[ 'Qos.config' ].addCommand(
               'hardware access-list resource sharing routedport subinterface' )

   def addCommandAllNamedPolicers():
      def addCommandNamedPolicer( policerName ):
         namedPolicer = entity.namedPolicer[ policerName ]
         cirUnitStr = QosLib.rateUnitFromEnum( namedPolicer.cirUnit )
         bcUnitStr = QosLib.burstUnitFromEnum( namedPolicer.bcUnit )
         root[ 'Qos.config' ].addCommand(
             'qos policer %s rate %d %s burst-size %d %s' % (
              namedPolicer.name, namedPolicer.cir, cirUnitStr,
              namedPolicer.bc, bcUnitStr ) )

      if aclHwStatus.namedSharedPolicerSupported:
         for policerName in entity.namedPolicer:
            addCommandNamedPolicer( policerName )

   def addCommandEnablePolicerDropCounter():
      if entity.enablePolicerDropCounter:
         root[ 'Qos.config' ].addCommand(
               'policy-map type quality-of-service policer drop counter' )

   def addCommandEnablePerInterfaceCounter():
      if entity.perIntfCounter:
         root[ 'Qos.config' ].addCommand(
               'policy-map type quality-of-service counter per-interface' )

   def addCommandPMapCMapServicePolicy():
      addCommandAllCMapType()
      addCommandAllPMapType()
      addCommandPMapSharingKnob()
      addCommandSubIntfRoutedPortAclSharingKnob()

   addCommandEnablePolicerDropCounter()
   addCommandEnablePerInterfaceCounter()
   addCommandAllNamedPolicers()
   addCommandPMapCMapServicePolicy()

class QosSchedulingConfigMode( QosSchedulingModeBase, CliSave.Mode ):
   def __init__( self, param ):
      QosSchedulingModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True


CliSave.GlobalConfigMode.addChildMode( QosSchedulingConfigMode,
                                       after=[ 'Qos.config' ] )
QosSchedulingConfigMode.addCommandSequence( 'Qos.schedulingConfig' )

class QosSchedulingIntfConfigMode( QosSchedulingIntfModeBase, CliSave.Mode ):
   def __init__( self, param ):
      QosSchedulingIntfModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True


QosSchedulingConfigMode.addChildMode( QosSchedulingIntfConfigMode,
                                      after=[ 'Qos.schedulingConfig' ] )
QosSchedulingIntfConfigMode.addCommandSequence( 'Qos.schedulingIntfConfig' )

class QosSchedulingIntfGroupConfigMode( QosSchedulingIntfGroupModeBase,
                                        CliSave.Mode ):
   def __init__( self, param ):
      QosSchedulingIntfGroupModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )


QosSchedulingIntfConfigMode.addChildMode( QosSchedulingIntfGroupConfigMode,
                                          after=[ 'Qos.schedulingIntfConfig' ] )
QosSchedulingIntfGroupConfigMode.addCommandSequence( 'Qos.schedulingGroupConfig' )

class QosSchedulingPolicyConfigMode( QosSchedulingPolicyModeBase, CliSave.Mode ):
   def __init__( self, param ):
      QosSchedulingPolicyModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )


QosSchedulingConfigMode.addChildMode( QosSchedulingPolicyConfigMode,
                                      after=[ 'Qos.schedulingConfig' ] )
QosSchedulingPolicyConfigMode.addCommandSequence( 'Qos.schedulingPolicyConfig' )

@CliSave.saver( 'Qos::QosSchedulerConfig', 'qos/scheduler/config',
                requireMounts=( 'qos/hardware/status/global',
                                'interface/hardware/capability' ) )
def saveSchedulerConfig( entity, root, requireMounts, options ):
   topMode = root[ QosSchedulingConfigMode ].getOrCreateModeInstance( entity.name )

   def shapeRateCmd( shapeRate, saveAll ):
      cmd = ''
      if shapeRate.rate != tacShapeRateVal.invalid:
         if shapeRate.unit == tacShapeRateUnit.shapeRatePps:
            cmd = 'shape rate %d pps' % shapeRate.rate
         else:
            if shapeRate.percent != tacPercent.invalid:
               cmd = 'shape rate %d percent' % shapeRate.percent
            else:
               cmd = 'shape rate %d' % shapeRate.rate
            if shapeRate.burstSize.value != tacBurstVal.invalid:
               cmd += ' burst-size %s %s' % ( shapeRate.burstSize.value,
                                              burstUnitFromEnum (
                                                 shapeRate.burstSize.unit ) )
            if shapeRate.shared:
               cmd += ' shared'
      elif saveAll:
         cmd = 'no shape rate'
      return cmd

   def guaranteedBwCmd( bandwidth, saveAll ):
      cmd = ''
      if bandwidth.bw != tacGuaranteedBwVal.invalid:
         if bandwidth.unit == tacGuaranteedBwUnit.guaranteedBwPps:
            cmd = 'bandwidth guaranteed %d pps' % bandwidth.bw
         elif bandwidth.unit == tacGuaranteedBwUnit.guaranteedBwKbps:
            cmd = 'bandwidth guaranteed %d' % bandwidth.bw
         else:
            cmd = 'bandwidth guaranteed percent %d' % bandwidth.bw
      elif saveAll:
         cmd = 'no bandwidth guaranteed'
      return cmd

   def addCommandSchedulingPolicy( policyName ):
      policyMode = topMode[ QosSchedulingPolicyConfigMode ].getOrCreateModeInstance(
         policyName )
      policyConfig = entity.policy
      policy = policyConfig[ policyName ]
      childCmds = policyMode[ 'Qos.schedulingPolicyConfig' ]
      nextCmd = shapeRateCmd( policy.shapeRate, options.saveAll )
      if nextCmd:
         childCmds.addCommand( nextCmd )
      nextCmd = guaranteedBwCmd( policy.guaranteedBw, options.saveAll )
      if nextCmd:
         childCmds.addCommand( nextCmd )

   def addCommandSchedulingGroup( intfMode, intf, groupName ):
      groupMode = intfMode[
         QosSchedulingIntfGroupConfigMode ].getOrCreateModeInstance(
            ( intf.intfId, groupName ) )
      groupConfig = intf.group
      group = groupConfig[ groupName ]
      childCmds = groupMode[ 'Qos.schedulingGroupConfig' ]
      if group.policyName:
         childCmds.addCommand( 'policy %s' % group.policyName )
      elif options.saveAll:
         childCmds.addCommand( 'no policy' )
      if not group.members:
         if options.saveAll:
            childCmds.addCommand( 'no members' )
      else:
         for m in group.lines:
            cmd = 'members'
            memberLine = group.lines[ m ]
            for m2 in memberLine.members:
               intfId = memberLine.members[ m2 ]
               if intfId:
                  assert group.members[ intfId ] == m
                  cmd += ' ' + intfId
            childCmds.addCommand( cmd )

   def addCommandSchedulingIntf( intfId ):
      intfMode = topMode[ QosSchedulingIntfConfigMode ].getOrCreateModeInstance(
         intfId )
      intfConfig = entity.qosSchedulerIntfConfig
      intf = intfConfig[ intfId ]
      for groupName in sorted( intf.group ):
         addCommandSchedulingGroup( intfMode, intf, groupName )

   for policyName in sorted( entity.policy ):
      addCommandSchedulingPolicy( policyName )
   for intfId in sorted( entity.qosSchedulerIntfConfig ):
      addCommandSchedulingIntf( intfId )


CliSave.GlobalConfigMode.addCommandSequence( 'Qos.global',
                                             before=[ IntfCliSave.IntfConfigMode ] )

@CliSave.saver( 'Qos::GlobalConfig', 'qos/global/config',
                requireMounts=( 'interface/hardware/capability', ) )
def saveGlobalConfig( entity, root, requireMounts, options ):
   subIntfCapability = requireMounts[ 'interface/hardware/capability' ]
   if not subIntfCapability.subIntfNumTxqsConfigSupported:
      return
   cmds = root[ 'Qos.global' ]
   defaultNumTxqs = subIntfCapability.subintfNumTxQueuesPerPort
   if entity.numTxqsPerSubIntf != defaultNumTxqs or options.saveAll:
      cmdFormat = 'qos subinterface tx-queue count %d'
      cmds.addCommand( cmdFormat % entity.numTxqsPerSubIntf )

   defaultLagRebalanceDisabled = subIntfCapability.subintfLagRebalanceDisabled
   if entity.lagRebalanceDisabled != defaultLagRebalanceDisabled or \
         options.saveAll:
      if entity.lagRebalanceDisabled:
         cmdFormat = 'no qos subinterface port-channel rebalancing'
      else:
         cmdFormat = 'qos subinterface port-channel rebalancing'
      cmds.addCommand( cmdFormat )

   # pylint: disable-next=singleton-comparison
   if entity.lagSubIntfLoadbalance == True or options.saveAll:
      if entity.lagSubIntfLoadbalance:
         cmdFormat = 'qos subinterface port-channel load-balancing'
      else:
         cmdFormat = 'no qos subinterface port-channel load-balancing'
      cmds.addCommand( cmdFormat )

   if entity.parentFairSched or options.saveAll:
      if entity.parentFairSched:
         cmdFormat = 'qos subinterface scheduling parent round-robin'
      else:
         cmdFormat = 'no qos subinterface scheduling parent round-robin'
      cmds.addCommand( cmdFormat )

@CliSave.saver( 'Qos::PolicingConfig', 'qos/input/config/cli' )
def savePolicingConfig( entity, root, requireMounts, options ):
   policingMode = root[ PolicingConfigMode ].getSingletonInstance()
   cmds = policingMode[ 'Qos.policing' ]

   # Display profiles
   def addCommandAllProfiles():
      for profile in sorted( entity.profile ):
         profileConfig = entity.profile[ profile ].profileConfig
         cirUnitStr = QosLib.rateUnitFromEnum( profileConfig.cirUnit )
         bcUnitStr = QosLib.burstUnitFromEnum( profileConfig.bcUnit )
         cmd = 'profile %s rate %d %s burst-size %d %s' % ( profile,
                                                            profileConfig.cir,
                                                            cirUnitStr,
                                                            profileConfig.bc,
                                                            bcUnitStr )
         if profileConfig.pir:
            pirUnitStr = QosLib.rateUnitFromEnum( profileConfig.pirUnit )
            beUnitStr = QosLib.burstUnitFromEnum( profileConfig.beUnit )
            cmd += ' rate %d %s burst-size %d %s' % ( profileConfig.pir, pirUnitStr,
                                                      profileConfig.be, beUnitStr )
         cmds.addCommand( cmd )

   # Display policerInstances
   def addCommandAllPolicers():
      for policer in sorted( entity.policerInstance ):
         profileName = entity.policerInstance[ policer ].profileName
         cmd = 'policer %s profile %s' % ( policer, profileName )
         cmds.addCommand( cmd )

   # Display interface-policers
   def addCommandInterfacePolicers():
      for intfPair in entity.intfPolicer:
         intfName = intfPair.lagIntf if intfPair.lagIntf else intfPair.intfId
         intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
            intfName )
         cmds = intfMode[ 'Qos.policing' ]
         profileName = ""
         policerName = ""
         if tacPolicerType.dedicatedPolicer in entity.intfPolicer[
               intfPair ].policerName:
            profileName = entity.intfPolicer[ intfPair ].policerName[
               tacPolicerType.dedicatedPolicer ]
            profileName = ' profile %s ' % profileName
         if tacPolicerType.policerInstance in entity.intfPolicer[
               intfPair ].policerName:
            policerName = entity.intfPolicer[ intfPair ].policerName[
               tacPolicerType.policerInstance ]
            if profileName:
               policerName = 'group %s ' % policerName
            else:
               policerName = ' group %s ' % policerName
         if profileName or policerName:
            cmd = 'policer%s%sinput' % ( profileName, policerName )
            cmds.addCommand( cmd )

      # output
      for intfPair in entity.intfPolicerOutput:
         intfName = intfPair.lagIntf if intfPair.lagIntf else intfPair.intfId
         intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
            intfName )
         cmds = intfMode[ 'Qos.policing' ]
         if tacPolicerType.dedicatedPolicer in entity.intfPolicerOutput[
               intfPair ].policerName:
            profileName = entity.intfPolicerOutput[ intfPair ].policerName[
               tacPolicerType.dedicatedPolicer ]
            profileName = ' profile %s ' % profileName
            cmd = 'policer%soutput' % ( profileName )
            cmds.addCommand( cmd )

      # control-plane
      for intfPair in entity.intfPolicerCpu:
         intfName = intfPair.lagIntf if intfPair.lagIntf else intfPair.intfId
         intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
            intfName )
         cmds = intfMode[ 'Qos.policing' ]
         if tacPolicerType.dedicatedPolicer in entity.intfPolicerCpu[
               intfPair ].policerName:
            profileName = entity.intfPolicerCpu[ intfPair ].policerName[
               tacPolicerType.dedicatedPolicer ]
            profileName = ' profile %s ' % profileName
            cmd = 'policer%scpu' % ( profileName )
            cmds.addCommand( cmd )

   def addCommandPolicerPktSizeAdjProfile():
      for profileName in sorted( entity.policerPktSizeAdjProfile ):
         profile = entity.policerPktSizeAdjProfile[ profileName ]
         value = profile.policerPktSizeAdj
         sign = "plus"
         if value < 0:
            sign = "minus"
            value = -value
         cmds.addCommand( 'profile %s packet size adjustment %s %d bytes' % (
            profileName, sign, value ) )

   addCommandAllProfiles()
   addCommandAllPolicers()
   addCommandInterfacePolicers()
   addCommandPolicerPktSizeAdjProfile()
