#!/usr/bin/env python3
# Copyright (c) 2008-2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
import RadiusConstants
import Tac
from CliMode.Radius import RadiusServerGroupMode
from IpLibConsts import DEFAULT_VRF
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.NetworkCliSave import networkConfigCmdSeq
from CliSavePlugin.Security import mgmtSecurityConfigPath
from CliSavePlugin.Security import SecurityConfigMode
from Toggles import RadiusToggleLib
import ReversibleSecretCli
from AaaPluginLib import hostProtocol

#-------------------------------------------------------------------------------
# Object used for saving commands in "config-sg-radius" mode.
#-------------------------------------------------------------------------------
class AaaRadiusServerGroupConfigMode( RadiusServerGroupMode, CliSave.Mode ):
   def __init__( self, param ):
      RadiusServerGroupMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( AaaRadiusServerGroupConfigMode,
                                       before=[ 'Aaa.global' ] )
AaaRadiusServerGroupConfigMode.addCommandSequence( 'Radius.serverGroup' )

# I want the Radius commands to go after Network.config because the
# "radius-server host" command takes hostname-or-ip-addrs and attempt to
# resolve hostnames, so it's better if the nameserver config is done first.
CliSave.GlobalConfigMode.addCommandSequence( 'AaaRadius.global',
                                             before=[ 
                                                'Aaa.global',
                                                AaaRadiusServerGroupConfigMode ],
                                             after=[ networkConfigCmdSeq,
                                                     SecurityConfigMode ] )

CliSave.GlobalConfigMode.addCommandSequence( 'Radius.srcIntf',
   after=[ IntfConfigMode ] )

# When someone types "show running-config", the Cli code walks the sysdb object
# tree, calling all the functions which registered with CliSave.saver to be
# called for a particular object. Below, I register saveRadiusConfig to be
# called for 'security/aaa/radius/config'. When I'm called, I walk our entire
# Radius config object tree, generating all non-default config.

@CliSave.saver( 'Aaa::HostGroup', 'security/aaa/config' )
def saveHostGroup( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   if entity.groupType != 'radius':
      return
   mode = root[ AaaRadiusServerGroupConfigMode ].getOrCreateModeInstance( 
                                                            entity.name )
   cmds = mode[ 'Radius.serverGroup' ]
   for m in entity.member.values():
      cmd = "server %s" % ( m.spec.hostname )
      if m.spec.vrf != DEFAULT_VRF:
         cmd += " vrf %s" % m.spec.vrf
      if m.spec.protocol != hostProtocol.protoRadsec:
         if m.spec.port != RadiusConstants.defaultPort or saveAll:
            cmd += " auth-port %d" % ( m.spec.port )
         if m.spec.acctPort != RadiusConstants.defaultAcctPort or saveAll:
            cmd += " acct-port %d" % ( m.spec.acctPort )
      else:
         cmd += " tls"
         if m.spec.port != RadiusConstants.defaultTlsPort or saveAll:
            cmd += " port %d" % ( m.spec.port )
      cmds.addCommand( cmd )

@CliSave.saver( 'Radius::Config', 'security/aaa/radius/config',
                requireMounts=( mgmtSecurityConfigPath, ) )
def saveRadiusConfig( cfg, root, requireMounts, options ):

   cmds = root[ 'AaaRadius.global' ]
   saveAll = options.saveAll
   securityConfig = requireMounts[ mgmtSecurityConfigPath ]

   if cfg.key:
      cmd = ReversibleSecretCli.getCliSaveCommand( 'radius-server key {}',
                                                   securityConfig, cfg.key )
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( 'no radius-server key' )

   if cfg.secretProfileName:
      cmds.addCommand( 'radius-server shared-secret profile %s' %
                        cfg.secretProfileName )
   elif saveAll:
      cmds.addCommand( 'no radius-server shared-secret profile' )

   if cfg.timeout != cfg.defaultTimeout or saveAll:
      cmd ='radius-server timeout %d' % ( int( cfg.timeout ) )
      cmds.addCommand( cmd )

   if cfg.retries != cfg.defaultRetries or saveAll:
      cmd ='radius-server retransmit %d' % ( int( cfg.retries ) )
      cmds.addCommand( cmd )

   if cfg.deadtime != 0:
      cmd = 'radius-server deadtime %d' % ( int( cfg.deadtime / 60 ) )
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no radius-server deadtime'
      cmds.addCommand( cmd )

   if cfg.nasId != "":
      cmd = 'radius-server attribute 32 include-in-access-req format %s' % cfg.nasId
      cmds.addCommand( cmd )
   elif cfg.nasIdType != "disabled":
      cmd = 'radius-server attribute 32 include-in-access-req %s' % cfg.nasIdType
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( 'no radius-server attribute 32 include-in-access-req' )

   if cfg.dynAuthPort != cfg.dynAuthPortDefault:
      cmd = 'radius-server dynamic-authorization port %d' % ( cfg.dynAuthPort )
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no radius-server dynamic-authorization port'
      cmds.addCommand( cmd )

   if cfg.sslProfile != "":
      cmd = 'radius-server tls ssl-profile %s' % ( cfg.sslProfile )
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no radius-server tls ssl-profile'
      cmds.addCommand( cmd )

   if RadiusToggleLib.toggleRadsecCoaToggleEnabled():
      if cfg.dynAuthSslProfile != "":
         cmd = 'radius-server dynamic-authorization tls ssl-profile %s' % \
               cfg.dynAuthSslProfile
         cmds.addCommand( cmd )
      elif saveAll:
         cmd = 'no radius-server dynamic-authorization tls ssl-profile'
         cmds.addCommand( cmd )

   hosts = cfg.host
   for h in sorted( hosts.values(), key=lambda host: host.index ):
      cmd = "radius-server host " + h.hostname
      if h.vrf != DEFAULT_VRF:
         cmd += ' vrf %s' % h.vrf
      if h.protocol != hostProtocol.protoRadsec:
         if h.port != RadiusConstants.defaultPort or saveAll:
            cmd += " auth-port %d" % ( h.port )
         if h.acctPort != RadiusConstants.defaultAcctPort or saveAll:
            cmd += " acct-port %d" % ( h.acctPort )
         if h.useTimeout:
            cmd += " timeout %d" % ( h.timeout )
         if h.useRetries:
            cmd += " retransmit %d" % ( h.retries )
         if h.useKey:
            if h.secretProfileName:
               cmd += " shared-secret profile %s" % h.secretProfileName
            else:
               # cmd could contain curly braces, maybe? escape to make sure
               cmd = ReversibleSecretCli.getCliSaveCommand(
                  CliSave.escapeFormatString( cmd ) + " key {}",
                  securityConfig,
                  h.key )
      else:
         cmd += " tls"
         if h.sslProfile:
            cmd += " ssl-profile %s" % ( h.sslProfile )
         if h.port != RadiusConstants.defaultTlsPort or saveAll:
            cmd += " port %d" % ( h.port )
         if h.useTimeout:
            cmd += " timeout %d" % ( h.timeout )
         if h.useRetries:
            cmd += " retransmit %d" % ( h.retries )
      cmds.addCommand( cmd )  

   if cfg.srcIntfName:
      for vrf, srcIntf in cfg.srcIntfName.items():
         vrfString = ' vrf %s' % vrf if vrf != DEFAULT_VRF else '' 
         root[ 'Radius.srcIntf' ].addCommand( 'ip radius%s source-interface %s' % \
                                                 ( vrfString, srcIntf ) )
   elif saveAll:
      root[ 'Radius.srcIntf' ].addCommand( 'no ip radius source-interface' )

   # radius-server qos dscp <dscpValue>
   if cfg.dscpValue != cfg.dscpValueDefault:
      cmd ='radius-server qos dscp %s' % cfg.dscpValue
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'radius-server qos dscp %s' % cfg.dscpValueDefault
      cmds.addCommand( cmd )
