#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliSave import escapeFormatString
import ReversibleSecretCli
import Tac
from CliMode.RadiusProxyMode import RadiusProxyClientGroupMode
from CliMode.RadiusProxyMode import RadiusProxyMode
from CliSavePlugin.Security import mgmtSecurityConfigPath
from Toggles import RadiusProxyToggleLib
from TypeFuture import TacLazyType

TimeoutConstants = TacLazyType( 'RadiusProxy::TimeoutConstants' )

class RadiusProxySaveConfigMode( RadiusProxyMode, CliSave.Mode ):
   def __init__( self, param ):
      RadiusProxyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   # Mode is displayed only if sub-commands are configured otherwise
   # mode is not displayed at all.
   # Here, we want the mode to be displayed if configured by default
   # as "radius proxy" configuration is responsible for the
   # runnability of the agent RadiusProxy.
   def skipIfEmpty( self ):
      return False

CliSave.GlobalConfigMode.addChildMode( RadiusProxySaveConfigMode )
RadiusProxySaveConfigMode.addCommandSequence( 'RadiusProxy.config' )
CliSave.GlobalConfigMode.addCommandSequence( 'RadiusProxy.global' )

class RadiusProxySaveClientGroupMode( RadiusProxyClientGroupMode,
                                      CliSave.Mode ):
   def __init__( self, param ):
      RadiusProxyClientGroupMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RadiusProxySaveConfigMode.addChildMode( RadiusProxySaveClientGroupMode )
RadiusProxySaveClientGroupMode.addCommandSequence( 'RadiusProxy.clientGroupConfig' )

@CliSave.saver( 'RadiusProxy::Config', 'radiusproxy/config',
                requireMounts=( mgmtSecurityConfigPath, ) )
def saveRadiusProxyConfig( cfg, root, requireMounts, options ):
   if not RadiusProxyToggleLib.toggleRadiusProxyForAPEnabled():
      return

   saveAll = options.saveAll
   if not cfg.radiusProxyEnabled:
      if saveAll:
         root[ 'RadiusProxy.global' ].addCommand( 'no radius proxy' )
      return

   configMode = root[ RadiusProxySaveConfigMode ].getOrCreateModeInstance(
      'radius proxy' )
   cmds = configMode[ 'RadiusProxy.config' ]
   securityConfig = requireMounts[ mgmtSecurityConfigPath ]

   if cfg.dynAuthEnabled:
      cmds.addCommand( 'dynamic-authorization' )
   elif saveAll:
      cmds.addCommand( 'no dynamic-authorization' )

   if cfg.key:
      cmd = ReversibleSecretCli.getCliSaveCommand( 'client key {}',
                                                   securityConfig,
                                                   cfg.key )
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( 'no client key' )

   if cfg.clientMappingTimeout != TimeoutConstants.defaultTimeout or saveAll:
      cmds.addCommand(
         f'client session idle-timeout { int( cfg.clientMappingTimeout ) } seconds' )

   groups = cfg.clientGroup
   for g in groups.values():
      groupMode = configMode[
         RadiusProxySaveClientGroupMode ].getOrCreateModeInstance( g.name )
      cmds = groupMode[ 'RadiusProxy.clientGroupConfig' ]
      clients = g.client

      # Add client hosts config
      for c in clients.values():
         try:
            hostType = Tac.Value( 'Arnet::IpGenAddrWithMask', c.clientName ).af
         except IndexError:
            # Tac::RangeException
            hostType = 'host'
         cmd = f'client { hostType } { c.clientName } vrf { c.vrf }'
         if c.key:
            formatStr = escapeFormatString( cmd ) + ' key {}'
            cmd = ReversibleSecretCli.getCliSaveCommand( formatStr,
                                                         securityConfig,
                                                         c.key )
         cmds.addCommand( cmd )

      # Add server groups configs
      l = len( g.serverGroup )
      if l == 0:
         continue
      cmd = 'server group'
      for i in range( l ):
         cmd += f' { g.serverGroup[ i ] }'
      cmds.addCommand( cmd )
