# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools

import CliSave
from CliMode.Rcf import ControlFunctionsBaseMode
from CliSavePlugin.RouterGeneralCliSave import RouterGeneralCliSaveMode
from Toggles import RcfLibToggleLib

class ControlFunctionsMode( ControlFunctionsBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      ControlFunctionsBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

RouterGeneralCliSaveMode.addChildMode( ControlFunctionsMode )
ControlFunctionsMode.addCommandSequence( 'Rcf.config' )

#---------------------------------------------------------------------------------
#                        C L I      S A V E R S
#---------------------------------------------------------------------------------

@CliSave.saver( 'Rcf::Config', 'routing/rcf/config',
      requireMounts=( 'configTag/configTagIdState', ),
      commandTagSupported=True )
def saveRcfConfig( rcfConfig, root, requireMounts, options ):
   routerGeneralMode = root[ RouterGeneralCliSaveMode ].getSingletonInstance()
   controlFunctionsMode = routerGeneralMode[ ControlFunctionsMode ].\
         getSingletonInstance()
   configTagIdState = requireMounts[ 'configTag/configTagIdState' ]
   cmds = controlFunctionsMode[ 'Rcf.config' ]

   def saveRcfCodeUnitUrlInfo( unitName, unitUrlInfo ):

      if unitUrlInfo.lastPulledUrl:
         cmds.addCommand( 'code %ssource pulled-from %s%s' %
               ( 'unit %s ' % unitName if unitName else '',
                 unitUrlInfo.lastPulledUrl,
                 ' edited' if unitUrlInfo.editSincePull else '' ) )

   def saveMultiLineCmd( baseCmd, text ):
      trailer = "EOF"
      if text and not text.endswith( "\n" ):
         # EOF must be on its own line, see 978673.
         trailer = "\nEOF"
      cmds.addCommand( f"{baseCmd}\n{text}{trailer}" )

   def saveRcfCodeUnitText( unitName, unitText ):
      if unitName:
         saveMultiLineCmd( f"code unit {unitName}", unitText )
      else:
         saveMultiLineCmd( "code", unitText )

   def saveRcfCodeUnitConfigTag( unitName, tagId ):
      tagStr = configTagIdState.tagIdToTagStr.get( tagId, None )
      if tagStr is not None:
         if unitName:
            codeCmd = ( "code unit " + unitName + " command-tag " + tagStr )
         else:
            codeCmd = ( "code command-tag " + tagStr )
         cmds.addCommand( codeCmd )

   def saveRcfOpenConfigFunction( functionName, openConfigFunction ):
      saveMultiLineCmd( f"function openconfig {functionName}",
                         openConfigFunction )

   # Code Units are always rendered in alphabetical order with interlined URLs
   # and command tag associations
   allUnitNames = set( itertools.chain( rcfConfig.rcfCodeUnitUrlInfo,
                                        rcfConfig.rcfCodeUnitText,
                                        rcfConfig.codeUnitToConfigTag ) )
   for unitName in sorted( allUnitNames ):
      unitUrlInfo = rcfConfig.rcfCodeUnitUrlInfo.get( unitName )
      if unitUrlInfo is not None:
         saveRcfCodeUnitUrlInfo( unitName, unitUrlInfo )

      unitConfigTagId = rcfConfig.codeUnitToConfigTag.get( unitName )
      if unitConfigTagId is not None:
         saveRcfCodeUnitConfigTag( unitName, unitConfigTagId )

      unitText = rcfConfig.rcfCodeUnitText.get( unitName )
      if unitText is not None:
         saveRcfCodeUnitText( unitName, unitText )

   if RcfLibToggleLib.toggleRcfPolicyDefinitionsEnabled():
      for functionName, openConfigFunction in rcfConfig.openConfigFunction.items():
         saveRcfOpenConfigFunction( functionName, openConfigFunction )
