# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac, CliSave, Tracing
from CliMode.RouteMap import RouteMapCliMode, Ipv6PrefixListCliMode
from CliMode.RouteMap import IpPrefixListCliMode, PeerFilterCliMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliToken.RouteMapCliTokens import statementConfigToken
from CliMode.RouteMap import generateCommentKey # pylint: disable=ungrouped-imports
from CliSave import GlobalConfigMode
from RouteMapLib import bgpOriginEnum, matchPermitEnum, regexModeEnum
from RouteMapLib import printRouteMapEntryAttributes, Output, commValueToPrint, \
    printPeerFilter
from RouteMapLib import duplicateOperationToCliStr, isAsdotConfigured, CommunityType
from RouteMapLib import extCommType

__defaultTraceHandle__ = Tracing.Handle( 'RouteMap' )
t0 = Tracing.trace0

class RouteMapConfigMode( RouteMapCliMode, CliSave.Mode ):
   def __init__( self, param ):
      RouteMapCliMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def instanceKey( self ):
      # Note: The param # value is split and tested.
      # `param_` in this case
      # is the route map sequence line of the config,
      # without the 'route-map' token.
      #
      # It is a string, which takes the form of
      # <route map name> [statement <statement>] <permit|deny> <seqno>
      #
      # CLI:   route-map FOO permit 10
      # param: FOO permit 10
      # CLI:   route-map FOO statement BAR deny 20
      # param: FOO statement BAR deny 20
      name, *_, seqno = self.param_.split()
      return ( name, int( seqno ) )

   # Because the long-mode-key for RouteMapMode is
   # "route-map-mapName", not unique to each of individual mode, we
   # have to override commentKey() to give the
   # complete unique string for saving the comment.
   def commentKey( self ):
      ''' return a unique route-map key based on name, seqno and permit|deny '''
      # route-map cli command currently has two forms
      #         route-map FOO statement BAR permit|deny SEQNO
      #         route-map FOO permit|deny SEQNO
      paramTokens = self.param_.split()
      assert isinstance( paramTokens, list )

      if statementConfigToken in paramTokens:
         # assume 'BAR' always follows 'statement', discard both
         index = paramTokens.index( statementConfigToken )
         del paramTokens[ index ] # remove 'statement'
         assert paramTokens
         del paramTokens[ index ] # remove 'BAR'

      # assume remaining tokens have:
      # name first, seqno last and permitDeny preceding seqno
      assert len( paramTokens ) > 2
      routeMapName = paramTokens[ 0 ]
      seqno = paramTokens[ -1 ]
      assert seqno.isdigit()
      permitDeny = paramTokens[ -2 ]
      # pylint: disable-next=consider-using-in
      assert permitDeny == 'permit' or permitDeny == 'deny'
      return generateCommentKey( routeMapName, permitDeny, seqno )

class IpPrefixListConfigMode( IpPrefixListCliMode, CliSave.Mode ):

   def __init__( self, param ):
      IpPrefixListCliMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class Ipv6PrefixListConfigMode( Ipv6PrefixListCliMode, CliSave.Mode ):

   def __init__( self, param ):
      Ipv6PrefixListCliMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class PeerFilterConfigMode( PeerFilterCliMode, CliSave.Mode ):
   def __init__( self, param ):
      PeerFilterCliMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouteMapOpModeTypeEnum = Tac.Type( "Routing::RouteMap::RouteMapOpModeType" )

CliSave.GlobalConfigMode.addCommandSequence( 'RouteMap.config' )
@CliSave.saver( 'Routing::RouteMap::Config', 'routing/routemap/config' )
def saveRouteMapOpMode( config, root, requireMounts, options ):
   cmds = root[ 'RouteMap.config' ]

   command = "service routing configuration route-map set-operations "

   if config.routeMapOpMode == RouteMapOpModeTypeEnum.routeMapSetOpModeMerged:
      command += "merged"
   else:
      command += "sequential"

   if config.routeMapOpMode != config.routeMapOpModeDefault or options.saveAll:
      addToOutput( command, cmds )

RouteMapConfigMode.addCommandSequence( 'RouteMap.config' )   
GlobalConfigMode.addChildMode( RouteMapConfigMode, after=[ IntfConfigMode ] )

#--------------------------------------------------------------------------------
CliSave.GlobalConfigMode.addCommandSequence( 'RouteMap.config' )

@CliSave.saver( 'Routing::RouteMap::Config', 'routing/routemap/config' )
def saveRouteMapSetCommExplicit( config, root, requireMounts, options ):
   cmds = root[ 'RouteMap.config' ]

   command = "service routing configuration route-map no-set-community " + \
             "explicit-members"

   # There are two different valid configurations for service routing configuration
   # route-map, they are operations, and set-operations. The currently set enum
   #  value allows us to differentiate between which variant was configured.

   # The system can either be configured to ignore or read trailing communities in
   # "no set community ..." commands ( ignores by default )
   if config.noSetCommExplicitMembers:
      addToOutput( command, cmds )

RouteMapConfigMode.addCommandSequence( 'RouteMap.config' )
GlobalConfigMode.addChildMode( RouteMapConfigMode, after=[ IntfConfigMode ] )
#--------------------------------------------------------------------------------
@CliSave.saver( 'Routing::RouteMap::Config', 'routing/routemap/config',
                 requireMounts = ( 'routing/bgp/asn/config', ) )
def saveRouteMapConfig( config, root, requireMounts, options ):

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdot = isAsdotConfigured( asnConfig )
   for mapName in config.routeMap:
      rtMap = config.routeMap[ mapName ]
      for seqno in rtMap.mapEntry:
         entry = rtMap.mapEntry[ seqno ]
         statementName = entry.statementName
         if not entry.statementNameFromCli:
            param = "%s %s %s" % (
               mapName, matchPermitEnum[ entry.permit ], seqno )
         else:
            param = "%s %s %s %s %s" % (
               mapName, 'statement', statementName,
               matchPermitEnum[ entry.permit ], seqno )
         mode = root[ RouteMapConfigMode ].getOrCreateModeInstance( param )
         cmds = mode[ 'RouteMap.config' ]
         printRouteMapEntryAttributes( entry, output=cmds, asdotConfigured=asdot,
                                       showHidden=True )

GlobalConfigMode.addCommandSequence( 'Ira.routeFilter',
                                     after=[ 'Ira.routing' ] )
@CliSave.saver( 'Acl::AclListConfig', 'routing/acl/config',
      requireMounts = ( 'routing/bgp/asn/config', ) )
def saveRoutingAcl( entity, root, requireMounts, options ):
   cmds = root[ 'Ira.routeFilter' ]
   saveAll = options.saveAll

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdot = isAsdotConfigured( asnConfig )
   output = Output( cmds )
   printAsPathList( entity, output=output, saveAll=saveAll, showRunning=True )
   printImportAsPathList( entity, output=output, saveAll=saveAll, showRunning=True )
   printImportPrefixList( entity.prefixList, ipCmd="ip", output=output )
   printImportPrefixList( entity.ipv6PrefixList, ipCmd="ipv6", output=output )
   if entity.prefixListModeCount == 0:
      printPrefixList( entity, output=output )
   printCommunityList( entity, output=output )
   printCommunityList( entity, output=output,
                       commType=CommunityType.communityTypeExtended,
                       asdotConfigured=asdot )
   printCommunityList( entity, output=output,
                       commType=CommunityType.communityTypeLarge,
                       asdotConfigured=asdot )
   printTagSet( entity.tagSet, output=output )

def printAsPathListRegexMode( config, output=None, saveAll=False ):
   regexMode = config.regexMode
   msg = "ip as-path regex-mode %s" % regexModeEnum[ regexMode ]
   if regexMode == config.regexModeDefault and not saveAll:
      return
   if output:
      output.append( msg )
   else:
      print( msg )

def isImportAsPathList( asPathList ):
   return asPathList.source != ""

def getActiveInactiveAsPathList( config, listName=None ):
   pathNames = []
   activeList = []
   inactiveList = []

   if listName:
      if listName in config.pathList:
         pathNames = [ listName ]
   else:
      # pylint: disable-next=unnecessary-comprehension
      pathNames = [ name for name in config.pathList ]

   posixRegexMode = config.regexMode == 'regexModePosix'

   for name in pathNames:
      if isImportAsPathList( config.pathList[name] ): 
         continue
      entries = list( config.pathList[ name ].pathEntry.values() )
      entries.sort( key=lambda entry: entry.seqno )

      acl = []
      aclActive = True
      for entry in entries:
         permit = matchPermitEnum[ entry.permit ]
         regex = entry.userRegex
         aclActive = aclActive and ( entry.validPosixRegex if posixRegexMode 
                                     else entry.validDfaRegex )
         msg = "ip as-path access-list %s %s %s %s" % (
            name, permit, regex, bgpOriginEnum[ entry.origin ] )
         acl.append( msg )

      if aclActive:
         activeList.extend( acl )
      else:
         inactiveList.extend( acl )

   return ( activeList, inactiveList )

def printAsPathList( config, listName=None, output=None, saveAll=False,
                     showRegexMode=True, showRunning=False ):
   if showRegexMode or saveAll:
      printAsPathListRegexMode( config, output=output, saveAll=saveAll )

   if listName and listName not in config.pathList:
      return

   activeList, inactiveList = getActiveInactiveAsPathList( config, listName )

   def usePrint( s ):
      print( s )

   # if we are collecting the result in output, 
   # use output.append, else print the result
   # on screen
   dump = output.append if output else usePrint

   for cmd in activeList:
      dump( cmd )

   for cmd in inactiveList:
      dump( cmd )

def printImportAsPathList( config, output, listName=None, 
                           saveAll=False, showRunning=True  ):
   pathNames = []
   if listName is not None:
      if listName in config.pathList:
         pathNames = [ listName ]
   else:
      # pylint: disable-next=unnecessary-comprehension
      pathNames = [ name for name in config.pathList ]

   for name in pathNames: 
      if isImportAsPathList( config.pathList[ name ] ):
         cmd = "ip as-path access-list %s source %s" % (
                config.pathList[ name ].name, config.pathList[ name ].source )
         dupStr = duplicateOperationToCliStr( \
               config.pathList[ name ].duplicateHandling )
         if dupStr is not None:
            cmd += " " + dupStr
         addToOutput( cmd, output )

def addToOutput( cmd, output ):
   if output is not None:
      if isinstance( output, Output ):
         output.append( cmd ) 
      elif hasattr( output, "addCommand" ):
         output.addCommand( cmd )
      elif isinstance( output, list ):
         output.append( cmd )
   else:
      print( cmd )

def prefixEntryToStr( entry ):
   '''Get seq ... part of the prefix-list entry'''
   if hasattr( entry, "prefix" ):
      cmd = "seq %s %s %s" % (
         entry.seqno, matchPermitEnum[ entry.permit ], entry.prefix )
   else:
      cmd = "seq %s %s %s" % (
         entry.seqno, matchPermitEnum[ entry.permit ], entry.ip6Prefix )
   if entry.eq == 0:
      if entry.ge > 0:
         cmd += " ge %s" % entry.ge
      if entry.le > 0:
         cmd += " le %s" % entry.le
   else:
      cmd += " eq %s" % entry.eq
   return cmd

def isImportPrefixList( pfxList ):
   return pfxList.source != ""

def printImportPrefixList( pfxLists, ipCmd, output, listName=None ):
   '''Print import prefix-lists in global config mode'''
   if listName is None:
      for listName in pfxLists: # pylint: disable=redefined-argument-from-local
         pfxList = pfxLists[ listName ]
         if isImportPrefixList( pfxList ):
            cmd = "%s prefix-list %s source %s" % (
               ipCmd, pfxList.name, pfxList.source )
            dupStr = duplicateOperationToCliStr( pfxList.duplicateHandling )
            if pfxList.vrfName != "default":
               cmd += " vrf " + pfxList.vrfName
            if dupStr is not None:
               cmd += " " + dupStr
            addToOutput( cmd, output )
   else:
      assert listName in pfxLists and isImportPrefixList( pfxLists[ listName ] )
      pfxList = pfxLists[ listName ]
      cmd = "%s prefix-list %s source %s" % (
         ipCmd, pfxList.name, pfxList.source )
      dupStr = duplicateOperationToCliStr( pfxList.duplicateHandling )
      if pfxList.vrfName != "default":
         cmd += " vrf " + pfxList.vrfName
      if dupStr is not None:
         cmd += " " + dupStr
      addToOutput( cmd, output )      

def printPrefixList( config, listName=None, output=None, ipv6=False,
                     inConfigMode=False ):
   if listName is None:
      prefixList = config.ipv6PrefixList if ipv6 else config.prefixList
      for name in prefixList:
         if not isImportPrefixList( prefixList[ name ] ):
            printPrefixList( config, listName=name, output=output, ipv6=ipv6,
                             inConfigMode=inConfigMode )
      return

   if ipv6 and ( listName in config.ipv6PrefixList or
                 listName in config.ipv6RemarkCollection ):
      assert not isImportPrefixList( config.ipv6PrefixList[ listName ] )

      entries, remarks = [], []
      if listName in config.ipv6PrefixList:
         entries = config.ipv6PrefixList[ listName ].ipv6PrefixEntry
      if listName in config.ipv6RemarkCollection:
         remarks = config.ipv6RemarkCollection[ listName ].remarkEntries
      printPrefixesAndRemarks( entries, remarks, output )

   if not ipv6 and ( listName in config.prefixList or
                     listName in config.remarkCollection ):
      assert not isImportPrefixList( config.prefixList[ listName ] )
      inPrefixListMode = config.prefixListModeCount > 0
      if not inPrefixListMode and inConfigMode:
         # If all (v4) prefix-list are one-liners and caller is prefix-list
         # config mode we don't print because it is done in saveRoutingAcl()
         return

      entries, remarks = [], []
      if listName in config.prefixList:
         entries = config.prefixList[ listName ].prefixEntry
      if not inPrefixListMode:
         printGlobalPrefixes( entries, listName, output )
         return
      if listName in config.remarkCollection:
         remarks = config.remarkCollection[ listName ].remarkEntries
      printPrefixesAndRemarks( entries, remarks, output )

def printGlobalPrefixes( entries, listName, output ):
   # entries is already presorted since its an ordered data structure
   # casting entries to a list will extract the keys from the dictionary
   eSeqs = list( entries )
   for e in eSeqs:
      seq = "ip prefix-list %s %s" % ( listName, prefixEntryToStr( entries[ e ] ) )
      addToOutput( seq, output )

def printPrefixesAndRemarks( entries, remarks, output ):
   # get sequence numbers (the keys inside entries and remarks)
   eSeqs = list( entries )
   rSeqs = list( remarks )

   # iterate across both lists, adding to output the smaller of the seq nums
   e, r = 0, 0
   while e < len( eSeqs ) or r < len( rSeqs ):
      # if reached the end of a list, keep iterating on the other list
      if e == len( eSeqs ):
         addToOutput( f"seq { rSeqs[ r ]} remark { remarks[ rSeqs[ r ] ] }",
                      output )
         r += 1
      elif r == len( rSeqs ):
         addToOutput( prefixEntryToStr( entries[ eSeqs[ e ] ] ),
                      output )
         e += 1
      # if both lists have entries not accounted for, pick the smallest seqno
      elif eSeqs[ e ] < rSeqs[ r ]:
         addToOutput( prefixEntryToStr( entries[ eSeqs[ e ] ] ),
                      output )
         e += 1
      else:
         addToOutput( f"seq { rSeqs[ r ]} remark { remarks[ rSeqs[ r ] ] }",
                      output )
         r += 1


def printCommunityList( config, listName=None, output=None,
                        commType=CommunityType.communityTypeStandard,
                        asdotConfigured=False ):
   assert type( commType ) == CommunityType # pylint: disable=unidiomatic-typecheck

   if listName is None:
      for name in sorted( config.communityList ):
         printCommunityList( config, listName=name, output=output,
                             commType=commType,
                             asdotConfigured=asdotConfigured )
      return
   if not listName in config.communityList:
      return
   communityList = config.communityList[ listName ]
   for s in communityList.communityListEntry.values():
      if s.version == 0:
         continue
      if s.listType == 'communityStandard' and \
         commType == CommunityType.communityTypeStandard:
         msg = "ip community-list %s %s " % (
            listName, matchPermitEnum[ s.permit ] )
         for value in s.community:
            msg += commValueToPrint( value ) + " "
      elif s.listType == 'communityExpanded' and \
           commType == CommunityType.communityTypeStandard:
         msg = "ip community-list regexp %s %s %s" % (
            listName, matchPermitEnum[ s.permit ],
            s.commRegexDisplay )
      elif s.listType == 'extCommunityStandard' and \
           commType == CommunityType.communityTypeExtended:
         msg = "ip extcommunity-list %s %s " % (
            listName, matchPermitEnum[ s.permit ] )
         lbwRange = False
         for value in s.community:
            extType = extCommType( value )
            if extType == 36:
               msg += commValueToPrint( value,
                                        commType=CommunityType.communityTypeExtended,
                                        lbwDisplay=s.lbwDisplay,
                                        lbwRange=lbwRange ) + " "
               lbwRange = True
            else:
               msg += commValueToPrint( value,
                                        commType=CommunityType.communityTypeExtended,
                                        lbwDisplay=s.lbwDisplay,
                                        asdotConfigured=asdotConfigured ) + " "

      elif s.listType == 'extCommunityExpanded' and \
           commType == CommunityType.communityTypeExtended:
         msg = "ip extcommunity-list regexp %s %s %s" % (
            listName, matchPermitEnum[ s.permit ],
            s.commRegexDisplay )
      elif s.listType == 'largeCommunityStandard' and \
           commType == CommunityType.communityTypeLarge:
         msg = "ip large-community-list %s %s " % (
            listName, matchPermitEnum[ s.permit ] )
         for value in s.community:
            msg += commValueToPrint( value,
                                     commType=CommunityType.communityTypeLarge,
                                     asdotConfigured=asdotConfigured ) + " "
      elif s.listType == 'largeCommunityExpanded' and \
           commType == CommunityType.communityTypeLarge:
         msg = "ip large-community-list regexp %s %s %s" % (
            listName, matchPermitEnum[ s.permit ],
            s.commRegexDisplay )
      else:
         continue
      msg = msg.rstrip()
      if output:
         output.append( msg )
      else:
         print( msg )

def printTagSet( tagSets, output=None ):
   for name, tagSet in tagSets.items():
      if len( tagSet.tagValue ):
         tags = ' '.join( map( str, tagSet.tagValue.values() ) )
         cmd = f'tag set {name} {tags}'
         addToOutput( cmd, output )
   
IpPrefixListConfigMode.addCommandSequence( 'IpPrefixList.config' )
GlobalConfigMode.addChildMode( IpPrefixListConfigMode,
                               after=[ 'Ira.routeFilter' ] )
@CliSave.saver( 'Acl::AclListConfig', 'routing/acl/config' )
def saveIpPrefixListConfig( config, root, requireMounts, options ):

   if config.prefixListModeCount == 0:
      return # All one-liners
   for listName in config.prefixList:
      if isImportPrefixList( config.prefixList[ listName ] ):
         continue
      mode = root[ IpPrefixListConfigMode ].getOrCreateModeInstance( listName )
      cmds = mode[ 'IpPrefixList.config' ]
      printPrefixList( config, listName, output=cmds,
                       inConfigMode=True )

Ipv6PrefixListConfigMode.addCommandSequence( 'Ipv6PrefixList.config' )   
GlobalConfigMode.addChildMode( Ipv6PrefixListConfigMode,
                               after=[ 'Ira.routeFilter' ] )
@CliSave.saver( 'Acl::AclListConfig', 'routing/acl/config' )
def saveIpv6PrefixListConfig( config, root, requireMounts, options ):

   for listName in config.ipv6PrefixList:
      if isImportPrefixList( config.ipv6PrefixList[ listName ] ):
         continue
      mode = root[ Ipv6PrefixListConfigMode ].getOrCreateModeInstance( listName )
      cmds = mode[ 'Ipv6PrefixList.config' ]
      printPrefixList( config, listName, output=cmds, ipv6=True,
                       inConfigMode=True )

PeerFilterConfigMode.addCommandSequence( 'PeerFilter.config' )
GlobalConfigMode.addChildMode( PeerFilterConfigMode, after=[ RouteMapConfigMode ] )
@CliSave.saver( 'Routing::RouteMap::PeerFilterConfig', 'routing/peerfilter/config' )
def savePeerFilterConfig( config, root, requireMounts, options ):
   for filterName in config.peerFilter:
      pFilter = config.peerFilter[ filterName ]
      mode = root[ PeerFilterConfigMode ].getOrCreateModeInstance( filterName )
      cmds = mode[ 'PeerFilter.config' ]
      printPeerFilter( pFilter, output=cmds )
