# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in
# pylint: disable=superfluous-parens
# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements the CliSave code for routing bgp commands
#-------------------------------------------------------------------------------
from __future__ import absolute_import, division, print_function
from Toggles import (
   BgpCommonToggleLib,
   RoutingLibToggleLib,
   RcfLibToggleLib,
   ConfigTagToggleLib,
   MvpnLibToggleLib
)
import CliSave
from CliSave import escapeFormatString
from CliSavePlugin.IntfCliSave import IntfConfigMode
from BgpLib import (
   createKey,
   Ip6AddrType,
   routeTargetToPrint,
   tcpAoAlgorithmMap,
   addressFamilies,
   vpnAfTypeCliKeywords,
   vpnAfTypeMap,
   vpnTypeAttrMap,
   YangSourceEnum,
)
import BgpLib
from CliMode.BgpCommon import ( RoutingBgpBaseMode, RoutingBgpVrfMode,
                                BgpFieldSetMappingsVrfMode,
                                FieldSetIpv4BaseMode, FieldSetIpv6BaseMode,
                                RoutingBgpBaseAfMode, RoutingBgpVrfAfMode,
                                BgpOrrPositionMode, BgpRfdPolicyMode,
                                BgpRouteDistinguisherMode )
from CliToken.RoutingBgpConstants import evpnNexthopSelfSupportNlris
# pylint: disable-next=ungrouped-imports
from CliSavePlugin.Security import SecurityConfigMode
from CliSavePlugin.Security import mgmtSecurityConfigPath
# Need to import this to get the ConfigTag.config config sequence
import CliSavePlugin.ConfigTagCliSave # pylint: disable=unused-import
import MultiRangeRule
from ArnetLib import bgpFormatAsn, formatRd
from RouteMapLib import isAsdotConfigured, commValueToPrint
import CliExtensions
import ReversibleSecretCli
from TypeFuture import TacLazyType
from ProtoEnums import protoDict
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
from IpLibConsts import DEFAULT_VRF
import Tac
import Tracing
import six

traceHandle = Tracing.Handle( 'Bgp' )
#pylint: disable-msg=C0321
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

toggleRcfFlowspecVpnPoaEnabled = RcfLibToggleLib.toggleRcfFlowspecVpnPoaEnabled()
toggleFeatureConfigTagBgpEnabled = \
      ConfigTagToggleLib.toggleFeatureConfigTagBgpEnabled()

def vpnNlriTypeKeyword( vpnNlriType ):
   return { 'evpnType5Ipv4': 'evpn ipv4',
            'evpnType5Ipv6': 'evpn ipv6',
            'mplsVpnIpv4Unicast': 'vpn-ipv4',
            'mplsVpnIpv6Unicast': 'vpn-ipv6' }[ vpnNlriType.toStrep() ]

GrHelperRestartTimeClassEnum = TacLazyType(
   "Routing::Bgp::GrHelperRestartTimeClass" )

def getTimeFromSeconds( time, timeClass ):
   if timeClass is GrHelperRestartTimeClassEnum.minutes:
      return time / 60
   elif timeClass is GrHelperRestartTimeClassEnum.hours:
      return time / 60 / 60
   elif timeClass is GrHelperRestartTimeClassEnum.days:
      return time / 60 / 60 / 24
   else:
      return time

# The CLI hook for BGP is used to allow external features dependant on BGP mode
# to clear their configurations. This function is called as
#
# hook( bgpConfig, defaultVrfBgpConfig, saveAll )
#
# bgpConfig is the current VRF Bgp::Config
# defaultVrfBgpConfig is the default VRF Bgp::Config
# saveAll is an boolean that must print the config
#
# Expects to return a cli command as a string or None
# if the config doesn't exist
bgpMonitoringHook = CliExtensions.CliHook()

# The CLI hook for BGP is used to allow external features dependant on BGP mode
# to clear their configurations. This function is called as
#
# hook( peer, peerConfig, saveAll )
#
# peer is the BGP peer address or peer group name
# peerConfig is the configuration of the peer in the current VRF
# saveAll is an boolean that must print the config
#
# Expects to return a cli command as a string or None
# if the config doesn't exist
neighborMonitoringHook = CliExtensions.CliHook()
neighborRpkiHook = CliExtensions.CliHook()

# defining new extensions for route-reflector commands
neighborRrHook = CliExtensions.CliHook()

reflectedRouteAttrStateEnum = Tac.Type( "Routing::Bgp::ReflectedRouteAttrState" )
redistInternalStateEnum = Tac.Type( "Routing::Bgp::RedistInternalState" )
SoftReconfigInboundStateEnum = Tac.Type( "Routing::Bgp::SoftReconfigInboundState" )
MetricOutStateEnum = Tac.Type( "Routing::Bgp::MetricOutState" )
triStateBoolEnum = Tac.Type( "Ip::TristateBool" )
SendCommunityLinkBandwidthStateEnum = \
    Tac.Type( "Routing::Bgp::SendCommunityLinkBandwidthState" )
LinkBandwidthGenerationStateEnum = \
    Tac.Type( "Routing::Bgp::LinkBandwidthGenerationState" )
ExtendedNextHopCapabilityEnum = Tac.Type( "Routing::Bgp::ExtendedNextHopCapability" )
policyActionEnum = Tac.Type( "Routing::Bgp::PolicyActionType" )
outDelayApplyEnum = Tac.Type( "Routing::Bgp::OutDelayApplyType" )
NlriType = Tac.Type( "Routing::Bgp::NlriType" )
PeerType = Tac.Type( "Routing::Bgp::PeerType" )
PeerFieldSetFilter = Tac.Type( "Routing::Bgp::PeerFieldSetFilter" )
LuLabelLocalTerminationModeEnum = \
      Tac.Type( "Routing::Bgp::LuLabelLocalTerminationMode" )

# Below address families always follow no-equals-default semantics for the
# configuration commands irrespective of the explicit no-equals-default configuration
alwaysNedModeAfs = [ 'mvpn-ipv4' ]

policyActions = dict(
   policyActionUnspecified='unspecified',
   policyActionPermit='permit',
   policyActionDeny='deny',
   policyActionDenyBoth='deny-in-out' )

class RouterBgpBaseConfigMode( RoutingBgpBaseMode, CliSave.Mode ):

   def __init__( self, param ):
      assert isinstance( param, tuple )
      asn, asdotConfigured = param
      asnParam = bgpFormatAsn( asn, asdotConfigured )
      RoutingBgpBaseMode.__init__( self, asnParam )
      CliSave.Mode.__init__( self, asnParam )

class RouterBgpVrfConfigMode( RoutingBgpVrfMode, CliSave.Mode ):

   def __init__( self, param ):
      RoutingBgpVrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

# Traffic-policy field-set config mode(s)
class TrafficPolicyFieldSetMappingsVrfConfigMode(
      BgpFieldSetMappingsVrfMode, CliSave.Mode ):
   def __init__( self, param ):
      mapCtx = 'tp'
      BgpFieldSetMappingsVrfMode.__init__( self, mapCtx, param )
      CliSave.Mode.__init__( self, param )

class TpFieldSetIpv4ConfigMode( FieldSetIpv4BaseMode, CliSave.Mode ):
   def __init__( self, param ):
      fsName, vrfName = param
      mapCtx = 'tp'
      FieldSetIpv4BaseMode.__init__( self, fsName, mapCtx, vrfName )
      CliSave.Mode.__init__( self, param )

class TpFieldSetIpv6ConfigMode( FieldSetIpv6BaseMode, CliSave.Mode ):
   def __init__( self, param ):
      fsName, vrfName = param
      mapCtx = 'tp'
      FieldSetIpv6BaseMode.__init__( self, fsName, mapCtx, vrfName )
      CliSave.Mode.__init__( self, param )

# VRF selection policy field-set config mode(s)
class VrfSelectionPolicyFieldSetMappingsVrfConfigMode(
      BgpFieldSetMappingsVrfMode, CliSave.Mode ):
   def __init__( self, param ):
      assert param == DEFAULT_VRF
      mapCtx = 'vsp'
      BgpFieldSetMappingsVrfMode.__init__( self, mapCtx, param )
      CliSave.Mode.__init__( self, param )

class VspFieldSetIpv4ConfigMode( FieldSetIpv4BaseMode, CliSave.Mode ):
   def __init__( self, param ):
      fsName, vrfName = param
      mapCtx = 'vsp'
      FieldSetIpv4BaseMode.__init__( self, fsName, mapCtx, vrfName )
      CliSave.Mode.__init__( self, param )

class VspFieldSetIpv6ConfigMode( FieldSetIpv6BaseMode, CliSave.Mode ):
   def __init__( self, param ):
      fsName, vrfName = param
      mapCtx = 'vsp'
      FieldSetIpv6BaseMode.__init__( self, fsName, mapCtx, vrfName )
      CliSave.Mode.__init__( self, param )

class BgpOrrPositionConfigMode( BgpOrrPositionMode, CliSave.Mode ):
   def __init__( self, param ):
      BgpOrrPositionMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class BgpRfdPolicyConfigMode( BgpRfdPolicyMode, CliSave.Mode ):
   def __init__( self, param ):
      BgpRfdPolicyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class BgpRouteDistinguisherConfigMode( BgpRouteDistinguisherMode, CliSave.Mode ):
   def __init__( self, param ):
      BgpRouteDistinguisherMode.__init__( self )
      CliSave.Mode.__init__( self, param )

routeTypeDict = { 'protoOspf' : 'internal', 'protoOspfAse' : 'external',
      'protoOspfNssa': 'nssa-external', 'protoOspf3': 'internal',
      'protoOspf3Ase' : 'external', 'protoOspf3Nssa' : 'nssa-external' }

isLevelDict = { 1 : 'level-1', 2 : 'level-2', 3 : 'level-1-2' }

def redistributeCommand( protocol, saveAll=False, addrFamily=None ):
   # internally we redistribute ospf-ase along with ospf depending on whether
   # it is match external or internal.
   if type( protocol ) is str: # pylint: disable=unidiomatic-typecheck
      proto = protocol
      routeType = ''
      metricType = 0
      isLevel = 0
      leaked = False
      redistEnabled = True
   else:
      proto = protocol.proto
      routeType = protocol.routeType
      metricType = protocol.metricType
      isLevel = protocol.isisLevel
      leaked = protocol.includeLeaked
      redistEnabled = protocol.redistEnabled

   command = ''
   if proto in ( 'protoOspf', 'protoOspfAse', 'protoOspfNssa',
         'protoOspf3', 'protoOspf3Ase', 'protoOspf3Nssa' ):
      if proto in ( 'protoOspf', 'protoOspfAse', 'protoOspfNssa' ):
         command += 'redistribute ospf'
      else:
         command += 'redistribute ospfv3'
      if routeType != '':
         command += ' match %s' % routeType
      elif saveAll:
         command += ' match %s' % routeTypeDict[ proto ]
      if metricType != 0:
         command += ' %d' % metricType
   elif proto == 'protoIsis':
      if isLevel:
         command += 'redistribute isis %s' % isLevelDict[ isLevel ]
      else:
         command += 'redistribute isis'
   elif proto == 'protoLdp':
      command += 'tunnel source-protocol ldp'
   elif proto == 'protoIsisSr':
      command += 'tunnel source-protocol isis segment-routing'
   elif proto == 'protoBgp' and addrFamily and not leaked:
      # This is redistribution within bgp address-family
      # If it is not leaked routes the command is "route input ..."
      # We currently only support unicast to multicast redistribution
      afStr = 'ipv4' if 'ipv4' in addrFamily else 'ipv6'
      command += 'route input address-family %s unicast' % afStr
   else:
      command += 'redistribute %s' % protoDict[ proto ]
   if leaked:
      if proto == 'protoBgp':
         command += ' leaked'
      else:
         command += ' include leaked'
   if not redistEnabled:
      command += ' disabled'

   return command

def addPathSendCommon( app, appConfig=None, saveAll=False ):
   command = 'additional-paths send'
   if app == 'appAny':
      command += ' any'
   elif app == 'appLimit':
      command += ' limit %d' % appConfig.pathLimit
   elif app == 'appBackup':
      command += ' backup'
   elif app == 'appEcmp':
      limit = ''
      if appConfig.pathLimit:
         limit = ' limit %d' % appConfig.pathLimit
      command += ' ecmp%s' % limit
   elif app == 'appNone':
      pass
   else:
      # It should never hit this condition
      pass
   if appConfig and appConfig.prefixList:
      command += ' prefix-list %s' % appConfig.prefixList
   return command

peerAfStateEnum = Tac.Type( "Routing::Bgp::PeerAFState" )
def hasAfConfig( bgpConfig ):
   for nconf in bgpConfig.neighborConfig.values():
      peerAfStateAttrs = BgpLib.peerConfigAttrsAfMap[ 'af' ]
      for attrName in peerAfStateAttrs.values():
         if getattr( nconf, attrName ) != peerAfStateEnum.afDefault:
            return True
   return False

# The 'service ...' command must appear before interface configuration,
# see BUG163576
# Additionally, the 'service ...' command must appear before BGP configuration.
# This is ensured by rendering 'Bgp.service' before IntfConfigMode while
# RouterBgpBaseConfigMode comes after IntfConfigMode.
CliSave.GlobalConfigMode.addCommandSequence( 'Bgp.service',
                                             before=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addChildMode( RouterBgpBaseConfigMode,
                                       after=[ 'ConfigTag.config',
                                          SecurityConfigMode, IntfConfigMode ] )
RouterBgpBaseConfigMode.addCommandSequence( 'Bgp.config' )

RouterBgpBaseConfigMode.addChildMode( RouterBgpVrfConfigMode )
RouterBgpVrfConfigMode.addCommandSequence( 'Bgp.vrf.config' )

RouterBgpVrfConfigMode.addChildMode( TrafficPolicyFieldSetMappingsVrfConfigMode )
TrafficPolicyFieldSetMappingsVrfConfigMode.addCommandSequence(
   'Bgp.vrf.traffic-policy.config' )
TrafficPolicyFieldSetMappingsVrfConfigMode.addChildMode( TpFieldSetIpv4ConfigMode )
TpFieldSetIpv4ConfigMode.addCommandSequence( 'Bgp.vrf.traffic-policy.ipv4.config' )
TrafficPolicyFieldSetMappingsVrfConfigMode.addChildMode( TpFieldSetIpv6ConfigMode )
TpFieldSetIpv6ConfigMode.addCommandSequence( 'Bgp.vrf.traffic-policy.ipv6.config' )

RouterBgpVrfConfigMode.addChildMode(
   VrfSelectionPolicyFieldSetMappingsVrfConfigMode )
VrfSelectionPolicyFieldSetMappingsVrfConfigMode.addCommandSequence(
   'Bgp.vrf.vrf-selection-policy.config' )
VrfSelectionPolicyFieldSetMappingsVrfConfigMode.addChildMode(
   VspFieldSetIpv4ConfigMode )
VspFieldSetIpv4ConfigMode.addCommandSequence(
   'Bgp.vrf.vrf-selection-policy.ipv4.config' )
VrfSelectionPolicyFieldSetMappingsVrfConfigMode.addChildMode(
   VspFieldSetIpv6ConfigMode )
VspFieldSetIpv6ConfigMode.addCommandSequence(
   'Bgp.vrf.vrf-selection-policy.ipv6.config' )

# Returns a tuple: isSet, value
#  isSet identifies if the attribute 'attr' is set in the vrfBgpConfig.
#  If the value inside vrfBgpConfig is "invalid", we try inheriting from
#  the baseBgpConfig.
def evaluateValue( vrfBgpConfig, baseBgpConfig, attr, defVal=None, invalVal=None ):
   if defVal == None: # pylint: disable=singleton-comparison
      defVal = getattr( vrfBgpConfig, attr + 'Default' )
   if invalVal == None: # pylint: disable=singleton-comparison
      invalVal = getattr( vrfBgpConfig, attr + 'Invalid' )
   val = getattr( vrfBgpConfig, attr )
   if val == invalVal:
      val = getattr( baseBgpConfig, attr )
      if val == invalVal:
         return False, defVal
      else:
         return False, val
   return True, val

# Returns a tuple: isSet, bool-value
def evaluateTristate( vrfBgpConfig, baseBgpConfig, attr ):
   meaningOfInvalid = getattr( vrfBgpConfig, attr + 'Default' )
   val = getattr( vrfBgpConfig, attr )
   if val == 'isInvalid':
      isSet = False
      val = getattr( baseBgpConfig, attr )
      if val == 'isInvalid':
         return isSet, meaningOfInvalid
      else:
         return ( isSet, True ) if val == 'isTrue' else ( isSet, False )
   else:
      isSet = True
      return ( isSet, True ) if val == 'isTrue' else ( isSet, False )

# Returns a tuple: isSet, pref-value
def evaluateRoutePref( vrfBgpConfig, baseBgpConfig, attr ):
   rp = getattr( vrfBgpConfig, attr )
   rpDef = getattr( baseBgpConfig, attr )
   isSet = rp.isSet
   val = rp.pref if isSet else rpDef.pref
   return isSet, val

def isGrConfigured( vrfBgpConfig, baseBgpConfig ):
   def _isGrConfigured( bgpOrPeerConfig, configuredValue ):
      grAttrs = BgpLib.bgpConfigAttrsAfMap[ 'gracefulRestart' ]
      for attrName in grAttrs.values():
         if getattr( bgpOrPeerConfig, attrName ) == configuredValue:
            return True
      return False

   # Check if GR is in the BGP config
   if _isGrConfigured( vrfBgpConfig, triStateBoolEnum.isTrue ):
      return True
   # Or check if a static nbr, dynPeerGroup, static Peergroup is configured for GR
   for neighborConfig in vrfBgpConfig.neighborConfig.values():
      if not neighborConfig.isPeerGroup and _isGrConfigured( neighborConfig, True ):
         return True
      if neighborConfig.isPeerGroupPeer:
         peerGroupConfig = \
               baseBgpConfig.neighborConfig.get( neighborConfig.peerGroupKey )
         if peerGroupConfig and _isGrConfigured( peerGroupConfig, True ):
            return True
   for pgName in vrfBgpConfig.listenRange:
      peerGroupConfig = baseBgpConfig.neighborConfig.get( createKey( pgName ) )
      if peerGroupConfig and _isGrConfigured( peerGroupConfig, True ):
         return True
   return False

def getExplicitNoCmd( cmd, nedMode, defaultValue=None ):
   if nedMode:
      if defaultValue is not None:
         # explicit-no default value for remote-port is 179, not 0
         if 'transport remote-port' in cmd:
            defaultValue = 179
         return cmd + ' ' + str( defaultValue )
      else:
         # If no defaultValue is provided, this command has a 'disabled' form
         return cmd + ' disabled'
   else:
      return 'no ' + cmd

def weightConfiguredInAf( config ):
   attrs = BgpLib.peerConfigAttrsAfMap[ 'weight' ]
   weightAfAttrs = [ ( af, c ) for af, c in six.iteritems( attrs ) if af != 'all' ]
   for _, c in weightAfAttrs:
      if getattr( config, c + 'Present' ):
         return True
   return False

def outDelayConfiguredInAf( config ):
   attrs = BgpLib.peerConfigAttrsAfMap[ 'outDelay' ]
   outDelayAfAttrs = [ ( af, c ) for af, c in six.iteritems( attrs ) if af != 'all' ]
   for _, c in outDelayAfAttrs:
      if getattr( config, c + 'Present' ):
         return True
   return False

def multiRangeToBgpAsdotString( inputString ):
   result = []
   tmpList = inputString.split()
   for substr in tmpList:
      if '-' in substr:
         low, high = substr.split( '-' )
         lowInAsdot = bgpFormatAsn( int( low ), True )
         highInAsdot = bgpFormatAsn( int(high), True )
         result.append( lowInAsdot + '-' + highInAsdot )
      else:
         result.append( bgpFormatAsn( int(substr), True ) )
   return ' '.join( result )

def saveNetworkList( cmds, bgpConfig, addrFamily='all', filterNetworkAf=None ):
   attrName = BgpLib.bgpConfigAttrsAfMap[ 'networkList' ].get( addrFamily )
   networkList = getattr( bgpConfig, attrName )
   for network in networkList.values():
      if filterNetworkAf and network.network.af != filterNetworkAf:
         continue
      cmd =  'network %s%s' % ( network.network, '' )
      # Once we support 'restrict', then:
      #' restrict' if network.restricted else '' ) )
      routeMap = network.routeMap
      if RcfLibToggleLib.toggleRcfNetworkCommandEnabled() and network.rcf:
         cmd += " rcf %s()" % network.rcf
      elif routeMap:
         cmd += " route-map %s" % routeMap
      cmds.append( cmd )

def saveEpeNexthopOriginateConfig( cmds, bgpConfig, addrFamily ):
   if addrFamily == 'ipv4 labeled-unicast':
      attrName = 'epeNexthopAfV4LabeledUni'
   elif addrFamily == 'ipv6 labeled-unicast':
      attrName = 'epeNexthopAfV6LabeledUni'
   else:
      return
   nexthopSet = getattr( bgpConfig, attrName )
   # need to sort manually until TACC supports ordered sets
   for nexthop in sorted( nexthopSet ):
      cmd = 'next-hop ' + str( nexthop ) + ' originate'
      cmds.append( cmd )

def saveEpeNexthopOriginateLfibBackupConfig( cmds, bgpConfig, addrFamily ):
   if addrFamily == 'ipv4 labeled-unicast':
      attrName = 'epeNexthopLfibBackupIpFwdAfV4LabeledUni'
   elif addrFamily == 'ipv6 labeled-unicast':
      attrName = 'epeNexthopLfibBackupIpFwdAfV6LabeledUni'
   else:
      return
   nexthopList = getattr( bgpConfig, attrName )
   for nexthop in nexthopList:
      cmd = 'next-hop ' + str( nexthop ) + ' originate lfib-backup ip-forwarding'
      if not nexthopList[ nexthop ]:
         cmd = cmd + ' disabled'
      cmds.append( cmd )

def saveRouteTargetConfig( bgpConfig, asdotConfigured, addrFamily=None ):
   cmds = []
   rtList = ( ( 'import', 'routeTargetImport' ),
              ( 'export', 'routeTargetExport' ) )
   for importExport, rtAttrName in rtList:
      attrName = rtAttrName
      allAttrName = rtAttrName + 'All'
      remoteAttrName = rtAttrName + 'RemoteDomain'
      if addrFamily is not None:
         attrName = BgpLib.bgpConfigAttrsAfMap[ attrName ].get( addrFamily, '' )
         allAttrName = BgpLib.bgpConfigAttrsAfMap.get( allAttrName, {} ).get(
               addrFamily, '' )
         remoteAttrName = BgpLib.bgpConfigAttrsAfMap.get( remoteAttrName, {}
               ).get( addrFamily, '' )
      for rt, isRtSet in getattr( bgpConfig, attrName, {} ).items():
         assert isRtSet
         cmds.append( "route-target %s %s" % ( importExport,
            routeTargetToPrint( rt, asdotConfigured ) ) )
      # Display these sorted on the cli keyword for the vpnAf.
      for vpnType, vpnTypeStr in sorted(
            vpnTypeAttrMap.items(),
            key=( lambda vpnType_vpnTypeStr1: vpnType_vpnTypeStr1[ 1 ] ) ):
         vpnTypeAttrName = attrName + vpnType
         vpnTypeAllAttrName = allAttrName + vpnType
         rtAll = getattr( bgpConfig, vpnTypeAllAttrName, {} )
         for rt, isRtSet in getattr( bgpConfig, vpnTypeAttrName, {} ).items():
            assert isRtSet
            if rt not in rtAll:
               cmds.append( 'route-target %s %s %s' % ( importExport,
                  vpnTypeStr, routeTargetToPrint( rt, asdotConfigured ) ) )

      for vpnType, vpnTypeStr in sorted(
            vpnTypeAttrMap.items(),
            key=( lambda vpnType_vpnTypeStr1: vpnType_vpnTypeStr1[ 1 ] ) ):
         vpnTypeAttrName = allAttrName + vpnType
         for rt, isRtSet in getattr( bgpConfig, vpnTypeAttrName, {} ).items():
            assert isRtSet
            cmds.append( 'route-target %s %s domain all %s' % ( importExport,
               vpnTypeStr, routeTargetToPrint( rt, asdotConfigured ) ) )

         vpnTypeAttrName = remoteAttrName + vpnType
         vpnTypeAllAttrName = allAttrName + vpnType
         rtAll = getattr( bgpConfig, vpnTypeAllAttrName, {} )
         for rt, isRtSet in getattr( bgpConfig, vpnTypeAttrName, {} ).items():
            assert isRtSet
            if rt not in rtAll:
               cmds.append( 'route-target %s %s domain remote %s' % (
                  importExport, vpnTypeStr,
                  routeTargetToPrint( rt, asdotConfigured ) ) )

   return cmds

def saveBgpVpnConfig( vrfName, bgpConfig, defaultVrfBgpConfig, asdotConfigured,
                      saveAll ):
   """This method returns the VPN config for the given VRF"""
   cmds = saveRouteTargetConfig( bgpConfig, asdotConfigured )

   for importExport, attrName in [ ( 'import', 'rcfImport' ),
                                   ( 'export', 'rcfExport' ) ]:
      # An explicit key is required for sort method here because vpnAf is of
      # type Routing::Bgp::AfiSafi but we need to sort them based on their cli
      # keyword here.
      afiSafiKwList = sorted( getattr( bgpConfig, attrName ).items(),
                              key=lambda item: vpnAfTypeMap.get( item[ 0 ] ) )
      for vpnAf, rcfMainAndFilter in afiSafiKwList:
         if vpnAfTypeMap.get( vpnAf ):
            # Check if the optional vpn/vrf rcf function exists for the given vpnAf
            # and create the optional argument for the command
            direction = "vpn" if importExport == "import" else "vrf"
            filterArg = "" if not rcfMainAndFilter.filterFunction else (
               " %s-route filter-rcf %s()" % ( direction,
                                               rcfMainAndFilter.filterFunction ) )
            cmds.append( "route-target %s %s rcf %s()%s"
               % ( importExport, vpnAfTypeMap[ vpnAf ],
                   rcfMainAndFilter.mainFunction, filterArg ) )

   for importExport, attrName in [ ( 'import', 'routeMapImport' ),
                                   ( 'export', 'routeMapExport' ) ]:
      # An explicit key is required for sort method here because vpnAf is of type
      # Routing::Bgp::AfiSafi but we need to sort them based on thier cli keyword
      # here.
      for vpnAf, rmName in sorted(
            getattr( bgpConfig, attrName ).items(),
                     key=lambda item: vpnAfTypeMap.get( item[ 0 ] ) ):
         if vpnAfTypeMap.get( vpnAf ):
            cmds.append( "route-target %s %s route-map %s" % ( importExport,
               vpnAfTypeMap[ vpnAf ], rmName ) )

   for importExport, attrName in [ ( 'export', 'allowImportedRouteToExport' ) ]:
      for vpnAf, isImpRoSet in sorted(
            getattr( bgpConfig, attrName ).items(),
                     key=lambda item: vpnAfTypeMap.get( item[ 0 ] ) ):
         assert isImpRoSet
         if vpnAfTypeMap.get( vpnAf ):
            cmds.append( "route-target %s %s %s" % ( importExport,
                         vpnAfTypeMap[ vpnAf ], 'imported-route' ) )

   if bgpConfig.vrfLocalLabelAfVpnV4 != bgpConfig.vrfLocalLabelAfVpnV4Default:
      cmds.append( 'vrf-label vpn-ipv4 %s' % bgpConfig.vrfLocalLabelAfVpnV4.label )
   if bgpConfig.vrfLocalLabelAfVpnV6 != bgpConfig.vrfLocalLabelAfVpnV6Default:
      cmds.append( 'vrf-label vpn-ipv6 %s' % bgpConfig.vrfLocalLabelAfVpnV6.label )

   for peerKey in bgpConfig.neighborConfig:
      addrOrPgName = peerKey.stringValue
      peer = bgpConfig.neighborConfig.get( peerKey )
      savePeerRouteTargetExportFilterDisabled( peer, cmds, addrOrPgName, 'all',
                                               saveAll )

   return cmds

def saveAggregates( lst ):
   cmds = []
   for agg in lst:
      cmd = 'aggregate-address {}'.format( agg.address )
      if agg.asSet:
         cmd += ' as-set'
      if agg.summaryOnly:
         cmd += ' summary-only'
      if agg.attrMap and agg.attrMap != agg.attrMapDefault:
         cmd += ' attribute-map {}'.format( agg.attrMap )
      if agg.attrRcf and agg.attrRcf != agg.attrRcfDefault:
         cmd += ' attribute rcf {}()'.format( agg.attrRcf )
      if agg.matchMap and agg.matchMap != agg.matchMapDefault:
         cmd += ' match-map {}'.format( agg.matchMap )
      if agg.advertiseOnly:
         cmd += ' advertise-only'
      cmds.append( cmd )
   return cmds

def saveOrrPositions( orrPosition ):
   cmds = []
   for igpPrefix in six.itervalues( orrPosition.prefixByPriority ):
      cmds.append( 'igp-prefix %s' % ( igpPrefix ) )
   return cmds

def saveRfdPolicy( rfdPolicy, saveAll ):
   rfdPolicyParams = rfdPolicy.rfdPolicyParams
   cmds = []

   def shouldSaveConfig( attr ):
      if saveAll:
         # For saveAll case, always save the config
         return True
      if getattr( rfdPolicyParams, attr ) != \
            getattr( rfdPolicyParams, attr + "Default" ):
         # If config is set to non default value, save it
         return True
      return False

   if shouldSaveConfig( "penaltyAttrUpdate" ):
      cmds.append( "penalty change {0}".format( rfdPolicyParams.penaltyAttrUpdate ) )

   if shouldSaveConfig( "penaltyUnreach" ):
      cmds.append( "penalty withdrawal {0}".format(
         rfdPolicyParams.penaltyUnreach ) )

   if shouldSaveConfig( "penaltyReach" ):
      cmds.append( "penalty advertisement withdrawal-tracking {0}".format(
         rfdPolicyParams.penaltyReach ) )

   if ( shouldSaveConfig( "reuseThreshold" ) or
        shouldSaveConfig( "suppressThreshold" ) or
        shouldSaveConfig( "ceilingThreshold" ) ):
      cmds.append( "penalty threshold reuse {0} suppression {1} maximum {2}".format(
         rfdPolicyParams.reuseThreshold,
         rfdPolicyParams.suppressThreshold,
         rfdPolicyParams.ceilingThreshold ) )

   if shouldSaveConfig( "halfLife" ):
      halfLifeCmd = "penalty decay half-life {0} {1}"
      if rfdPolicyParams.halfLifeUnitIsMinute:
         cmds.append( halfLifeCmd.format(
            int( rfdPolicyParams.halfLife / 60 ), "minutes" ) )
      else:
         cmds.append( halfLifeCmd.format( rfdPolicyParams.halfLife, "seconds" ) )

   v4UniPfxLst = rfdPolicy.suppressionIgnoredPrefixListV4Uni
   v4UniPfxLstDef = rfdPolicy.suppressionIgnoredPrefixListV4UniDefault
   v6UniPfxLst = rfdPolicy.suppressionIgnoredPrefixListV6Uni
   v6UniPfxLstDef = rfdPolicy.suppressionIgnoredPrefixListV6UniDefault
   if v4UniPfxLst != v4UniPfxLstDef or \
         v6UniPfxLst != v6UniPfxLstDef:
      suppIgnrCmd = "suppression ignored"
      if v4UniPfxLst != v4UniPfxLstDef:
         suppIgnrCmd += " prefix-list ipv4 " + v4UniPfxLst
      if v6UniPfxLst != v6UniPfxLstDef:
         suppIgnrCmd += " prefix-list ipv6 " + v6UniPfxLst
      cmds.append( suppIgnrCmd )
   elif saveAll:
      cmds.append( "no suppression ignored" )

   return cmds

def saveEpePeerSetConfig( cmds, bgpConfig, addrFamily ):
   if addrFamily != 'ipv4' and addrFamily != 'ipv6':
      return

   attrName = BgpLib.bgpConfigAttrsAfMap[ 'epePeerSetConfig' ].get( addrFamily )
   peerSetColl = getattr( bgpConfig, attrName )
   # need to sort manually until TACC supports ordered sets
   for peerSet in six.itervalues( peerSetColl ):
      if peerSet.communityPresent:
         cmd = ( 'peer-set %s community %s' %
            ( peerSet.name, commValueToPrint( peerSet.community ) ) )
      else:
         cmd = ( 'peer-set %s' % peerSet.name )
      cmds.append( cmd )
   return

def saveRouteDistinguisherConfig( rdAutoUnifiedConfig,
      rdAutoUnifiedConfigDefault, saveAll ):
   cmds = []
   if ( saveAll or
         rdAutoUnifiedConfig.assignedNumberStart !=
               rdAutoUnifiedConfigDefault.assignedNumberStart or
         rdAutoUnifiedConfig.assignedNumberEnd !=
               rdAutoUnifiedConfigDefault.assignedNumberEnd ):
      cmds.append(
            f"assignment auto range {rdAutoUnifiedConfig.assignedNumberStart} "
            f"{rdAutoUnifiedConfig.assignedNumberEnd}" )
   rdAutoAfi = rdAutoUnifiedConfig.rdAutoAfi
   addressFamilyConfiguration = "assignment auto address-family {0}"
   if rdAutoAfi.l2evpn:
      cmds.append( addressFamilyConfiguration.format( "l2-evpn" ) )
   if rdAutoAfi.vpls:
      cmds.append( addressFamilyConfiguration.format( "vpls" ) )
   if rdAutoAfi.l3vpn:
      cmds.append( addressFamilyConfiguration.format( "mpls-vpn l3-evpn" ) )
   return cmds

def saveBgpConfig( bgpConfig, defaultVrfBgpConfig,
                   routingHwStatus, vrfName, securityConfig, aclCpConfig,
                   commandTagIdState, asdotConfigured=False, saveAll=False ):
   cmds = []
   afCfgPresent = hasAfConfig( bgpConfig )

   # "command clear all disabled" is never written to for a non-default VRF
   if not vrfName:
      if ( defaultVrfBgpConfig.clearAllDisabled !=
           defaultVrfBgpConfig.clearAllDisabledDefault ):
         cmds.append( 'command clear all disabled' )
      elif saveAll:
         cmds.append( 'no command clear all disabled' )

   cmd = 'bgp labeled-unicast rib'
   if bgpConfig.bgpLuRib.rib == 'ipAndTunnel':
      cmd += ' ip'
      if bgpConfig.bgpLuRib.ipRmName != \
         bgpConfig.bgpLuRib.rmNameDefault:
         cmd += ' route-map %s' % bgpConfig.bgpLuRib.ipRmName
      cmd += ' tunnel'
      if bgpConfig.bgpLuRib.tunnelRmName != \
         bgpConfig.bgpLuRib.rmNameDefault:
         cmd += ' route-map %s' % bgpConfig.bgpLuRib.tunnelRmName
   elif bgpConfig.bgpLuRib.rib == 'tunnel':
      cmd += ' tunnel'
      if bgpConfig.bgpLuRib.tunnelRmName != \
         bgpConfig.bgpLuRib.rmNameDefault:
         cmd += ' route-map %s' % bgpConfig.bgpLuRib.tunnelRmName
   elif bgpConfig.bgpLuRib.rib == 'ip':
      cmd += ' ip'
      if bgpConfig.bgpLuRib.ipRmName != \
         bgpConfig.bgpLuRib.rmNameDefault:
         cmd += ' route-map %s' % bgpConfig.bgpLuRib.ipRmName
   else:
      assert False, "Unexpected BGP LU RIB"

   if cmd != 'bgp labeled-unicast rib tunnel' or saveAll:
      cmds.append( cmd )

   # The per VPN route-target and route-maps are printed in sorted order based on
   # their display vpnAf string. The display vpnAf string (cli keyword for the vpn
   # Af) is storeed in vpnAfTypeCliKeywords in a sorted manner.

   if vrfName:
      if bgpConfig.asNumber != 0:
         cmds.append( 'local-as %d' % bgpConfig.asNumber )
      elif saveAll:
         cmds.append( 'local-as %d' % defaultVrfBgpConfig.asNumber )
      cmds.extend( saveBgpVpnConfig( vrfName, bgpConfig, defaultVrfBgpConfig,
                                     asdotConfigured, saveAll ) )

   if bgpConfig.shutdown != bgpConfig.shutdownDefault:
      if bgpConfig.shutdownMsg != bgpConfig.shutdownMsgDefault:
         reasonStr = " reason %s" % bgpConfig.shutdownMsg
      else:
         reasonStr = ""
      cmds.append( 'shutdown%s' % reasonStr )
   elif saveAll:
      cmds.append( 'no shutdown' )

   if bgpConfig.cpFilterDefaultAllow:
      cmds.append( 'bgp control-plane-filter default-allow' )
   elif saveAll:
      cmds.append( 'no bgp control-plane-filter default-allow' )

   if bgpConfig.routerId != bgpConfig.routerIdDefault:
      cmds.append( 'router-id %s' % bgpConfig.routerId )
   elif saveAll:
      cmds.append( 'no router-id' )

   isStSet, stVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                   'convergenceTime' )
   if isStSet or saveAll:
      cmds.append( 'bgp convergence time %d' % stVal )

   isStSet, stVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                   'convergenceIdlePeerTime' )
   if isStSet or saveAll:
      cmds.append( 'bgp convergence slow-peer time %d' % stVal )

   if bgpConfig.confedId != 0:
      cmds.append( 'bgp confederation identifier %s' %
            bgpFormatAsn( bgpConfig.confedId, asdotConfigured ) )
   elif saveAll:
      cmds.append( 'no bgp confederation identifier' )

   for substr in MultiRangeRule.multiRangeToCanonicalStringGen(
            bgpConfig.confedPeer, 50, ' ' ):
      if asdotConfigured:
         substr = multiRangeToBgpAsdotString( substr )
      cmds.append( 'bgp confederation peers %s' % substr )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'convergenceNoSync' )
   if isSet or saveAll:
      cmds.append( ( 'no ' if val else '' ) +
                         'update wait-for-convergence' )

   isSet = bgpConfig.fibSync != bgpConfig.fibSyncDefault
   if isSet or saveAll:
      if bgpConfig.fibSync.batchSize:
         batchStr = ' batch-size %s' % (bgpConfig.fibSync.batchSize)
      else:
         batchStr = ''
      cmds.append( '%supdate wait-install%s' \
            %('' if isSet else 'no ', batchStr ) )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'logNeighborChanges' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp log-neighbor-changes' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'defaultV4Uni' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp default ipv4-unicast' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'defaultV4UniOverV6' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) +
                       'bgp default ipv4-unicast transport ipv6' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'defaultV6Uni' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp default ipv6-unicast' )

   for direction in [ "In", "Out" ]:
      tok = [ 'bgp', 'missing-policy', 'direction', direction.lower(), 'action' ]
      actionSet, val = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                  'missingPolicyAction' + direction )
      addrFamilyAllSet, addrFamilyAllVal = evaluateTristate(
                                                 bgpConfig, defaultVrfBgpConfig,
                                                 'missingPolicyAddressFamilyAll' +
                                                 direction )
      subRmSet, subRmVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                             'missingPolicyIncludeSubRouteMap' +
                                             direction )
      pfxListSet, pfxListVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                                 'missingPolicyIncludePrefixList' +
                                                 direction )
      commListSet, commListVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                                   'missingPolicyIncludeCommList' +
                                                   direction )
      isSet = actionSet or addrFamilyAllSet
      if isSet:
         tok.append( policyActions[ val ] )
         insertPos = 2
         if addrFamilyAllSet and addrFamilyAllVal:
            tok[ insertPos : insertPos ] = [ 'address-family', 'all' ]
            insertPos += 2
         if ( ( subRmSet and subRmVal ) or ( pfxListSet and pfxListVal ) or
              ( commListSet and commListVal ) ):
            # Insert include keywords into the command token list alphabetically
            includeTokens = [ 'include' ]
            if commListSet and commListVal:
               includeTokens.append( 'community-list' )
            if pfxListSet and pfxListVal:
               includeTokens.append( 'prefix-list' )
            if subRmSet and subRmVal:
               includeTokens.append( 'sub-route-map' )
            tok[ insertPos : insertPos ] = includeTokens
         missingPolicyCommand = ' '.join( tok )
         cmds.append( missingPolicyCommand )
      elif saveAll:
         missingPolicyCommand = 'no ' + ' '.join( tok )
         cmds.append( missingPolicyCommand )

   isKtSet, ktVal = evaluateValue( bgpConfig, defaultVrfBgpConfig, 'keepaliveTime' )
   isHtSet, htVal = evaluateValue( bgpConfig, defaultVrfBgpConfig, 'holdTime' )
   isMhtSet, mhtVal = evaluateValue( bgpConfig, defaultVrfBgpConfig, 'minHoldTime' )
   isShtSet, shtVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                    'sendFailureHoldTime' )
   timersCmd = 'timers bgp'
   if ( isKtSet or isHtSet ) or saveAll:
      timersCmd += ' %d %d' % ( ktVal, htVal )
   if isMhtSet or saveAll:
      timersCmd += ' min-hold-time %d' % ( mhtVal )
   if isShtSet or saveAll:
      timersCmd += ' send-failure hold-time %d' % ( shtVal )
   if timersCmd != 'timers bgp':
      cmds.append( timersCmd )

   idSet, idVal = evaluateRoutePref( bgpConfig, defaultVrfBgpConfig,
                                     'internalDistance' )
   edSet, edVal = evaluateRoutePref( bgpConfig, defaultVrfBgpConfig,
                                     'externalDistance' )
   ldSet, ldVal = evaluateRoutePref( bgpConfig, defaultVrfBgpConfig,
                                     'localDistance' )
   if edSet or idSet or ldSet or saveAll:
      cmds.append( 'distance bgp %d %d %d' % ( edVal, idVal, ldVal ) )

   isRtSet, rtVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                   'grRestartTime' )
   if isRtSet or saveAll or isGrConfigured( bgpConfig, defaultVrfBgpConfig ):
      cmds.append( 'graceful-restart restart-time %d' % rtVal )

   isStSet, stVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                   'grStalepathTime' )
   if isStSet or saveAll:
      cmds.append( 'graceful-restart stalepath-time %d' % stVal )

   # Only CliSave this command for the default VRF.
   if not vrfName:
      if bgpConfig.bgpHitlessRestartDisabled:
         cmds.append( 'bgp restart hitless disabled' )
      elif saveAll:
         cmds.append( 'bgp restart hitless' )

   aaVal = bgpConfig.allowAs
   if aaVal != bgpConfig.allowAsDefault:
      if aaVal == bgpConfig.allowAsEnabledDefault:
         cmds.append( 'bgp allowas-in' )
      else:
         cmds.append( 'bgp allowas-in %s' % ( aaVal ) )
   elif saveAll:
      if aaVal:
         cmds.append( 'bgp allowas-in %s' % ( aaVal ) )
      else:
         cmds.append( 'no bgp allowas-in' )

   if bgpConfig.bgpClusterId != bgpConfig.bgpClusterIdDefault:
      cmds.append( 'bgp cluster-id %s' % ( bgpConfig.bgpClusterId ) )
   elif saveAll:
      cmds.append( 'no bgp cluster-id' )

   if not vrfName and bgpConfig.autoRd:
      cmds.append( 'rd auto' )
   elif saveAll and not vrfName:
      cmds.append( 'no rd auto' )

   if not vrfName:
      for vpnNlriType in [ NlriType( n ) for n in
                           [ 'evpnType5Ipv4', 'evpnType5Ipv6',
                             'mplsVpnIpv4Unicast', 'mplsVpnIpv6Unicast' ] ]:
         present = vpnNlriType in bgpConfig.routeTargetExportFilterDisabled
         if present or saveAll:
            cmds.append( '%sroute-target export %s filter%s' %
                         ( '' if present else 'no ',
                           vpnNlriTypeKeyword( vpnNlriType ),
                           ' disabled' if present else '' ) )

   if bgpConfig.rrDisabled:
      cmds.append( 'no bgp client-to-client reflection' )
   elif saveAll:
      cmds.append( 'bgp client-to-client reflection' )

   isGrSet, grVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                      'gracefulRestart' )
   if isGrSet:
      cmds.append( ( '' if grVal else 'no ' ) + 'graceful-restart' )
   elif saveAll:
      cmds.append( 'default graceful-restart' )

   isGrHelperSet, grHelperVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                      'grHelper' )
   isGrHelperRestartTimeSet, grHelperRestartTimeVal = evaluateValue( bgpConfig,
            defaultVrfBgpConfig, 'grHelperRestartTime' )
   isLlgrHelperSet, llgrHelperVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                                      'llgrHelper' )
   isGrHelperRmSet, grHelperRmVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
         'grHelperRouteMap' )
   isGrHelperTimeClassSet, grHelperTimeClassVal = evaluateValue( bgpConfig,
         defaultVrfBgpConfig, 'grHelperRestartTimeClass' )
   isGrHelperRmHoldTimeSet, grHelperRmHoldTime = evaluateTristate( bgpConfig,
         defaultVrfBgpConfig, 'grHelperRmGrOptional' )

   grHelperCmd = 'graceful-restart-helper'
   if isGrHelperSet and not grHelperVal:
      cmds.append( 'no graceful-restart-helper' )
   elif isGrHelperRestartTimeSet or isLlgrHelperSet:
      grHelperCmdStr = grHelperCmd
      if isGrHelperRestartTimeSet and grHelperRestartTimeVal:
         grHelperRestartTimeVal = getTimeFromSeconds( grHelperRestartTimeVal,
                                                      grHelperTimeClassVal )
         grHelperCmdStr += ' restart-time %d' % grHelperRestartTimeVal
      if isGrHelperTimeClassSet and grHelperTimeClassVal:
         grHelperCmdStr += ' %s' % grHelperTimeClassVal
      if isLlgrHelperSet and llgrHelperVal:
         grHelperCmdStr += ' long-lived'
      elif isGrHelperRmSet and grHelperRmVal.value:
         grHelperCmdStr += ' stale-route route-map %s' % grHelperRmVal.value
         if isGrHelperRmHoldTimeSet and grHelperRmHoldTime:
            grHelperCmdStr += ' session-failure graceful-restart-negotiation ' \
                              'optional'
      cmds.append( grHelperCmdStr )
   elif saveAll:
      cmds.append( grHelperCmd )

   isPmrtSet, pmrtVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                       'peerMacResolutionTimeout' )
   if isPmrtSet or saveAll:
      cmds.append( 'bgp peer-mac-resolution-timeout %d' % pmrtVal )

   isEnforceFirstAsSet, enforceFirstAsVal = evaluateTristate( bgpConfig,
                                                              defaultVrfBgpConfig,
                                                              'enforceFirstAs' )
   if isEnforceFirstAsSet and not enforceFirstAsVal:
      cmds.append( 'no bgp enforce-first-as' )
   elif saveAll:
      cmds.append( 'bgp enforce-first-as' )

   isfecSkipIarDivDetSet, fecSkipIarDivDetVal = evaluateTristate( bgpConfig,
      defaultVrfBgpConfig, 'fecSkipIarDivergenceDetection' )
   isfecSkipIarSet, fecSkipIarVal = evaluateTristate( bgpConfig,
      defaultVrfBgpConfig, 'fecSkipIarTracking' )
   if isfecSkipIarDivDetSet:
      if fecSkipIarDivDetVal:
         cmds.append( 'bgp fec skip in-place update event route-changes' )
      else:
         cmds.append( 'no bgp fec skip in-place update' )
   elif isfecSkipIarSet:
      if fecSkipIarVal:
         cmds.append( 'bgp fec skip in-place update event all' )
      else:
         cmds.append( 'no bgp fec skip in-place update' )
   elif saveAll:
      cmds.append( 'no bgp fec skip in-place update' )

   isfecSkipIarDrUndrSet, fecSkipIarDrUndrVal = evaluateTristate( bgpConfig,
      defaultVrfBgpConfig, 'fecSkipIarDrainUndrain' )
   if isfecSkipIarDrUndrSet:
      if fecSkipIarDrUndrVal:
         cmds.append( 'no bgp fec in-place update event drain-undrain' )
      else:
         isfecDelayNewRouteDuringIarSet, fecDelayNewRouteDuringIarVal = \
               evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                 'fecDelayNewRouteDuringIar' )
         if isfecDelayNewRouteDuringIarSet and fecDelayNewRouteDuringIarVal:
            cmds.append( 'bgp fec in-place update event drain-undrain new-route' )
         else:
            cmds.append( 'bgp fec in-place update event drain-undrain' )
   elif saveAll:
      cmds.append( 'no bgp fec in-place update event drain-undrain' )

   isfecSkipIarPeerInitSet, fecSkipIarPeerInitVal = evaluateTristate( bgpConfig,
      defaultVrfBgpConfig, 'fecSkipIarPeerInit' )
   if isfecSkipIarPeerInitSet:
      if fecSkipIarPeerInitVal:
         cmds.append( 'no bgp fec in-place update event peer-init' )
      else:
         cmds.append( 'bgp fec in-place update event peer-init' )
   elif saveAll:
      cmds.append( 'no bgp fec in-place update event peer-init' )

   disabledVal = bgpConfig.fecIarTimeoutDisabled
   defVal = bgpConfig.fecIarTimeoutDefault
   iarTimeout = bgpConfig.fecIarTimeout
   isSet = ( bgpConfig.fecIarTimeout != bgpConfig.fecIarTimeoutDefault )
   if iarTimeout == defVal:
      # IAR timeout is not explicitly configured at BGP VRF level. Inherit
      # the value from default VRF
      iarTimeout = defaultVrfBgpConfig.fecIarTimeout
   if iarTimeout == bgpConfig.fecIarTimeoutDefault:
      if saveAll:
         cmds.append( 'no bgp fec in-place update timeout' )
   elif isSet or saveAll:
      if iarTimeout == disabledVal:
         cmds.append( 'bgp fec in-place update timeout disabled' )
      else:
         cmds.append( 'bgp fec in-place update timeout %d' % iarTimeout )

   if bgpConfig.routeInstallMap != bgpConfig.routeInstallMapDefault:
      cmds.append( 'bgp route install-map %s' %
                       bgpConfig.routeInstallMap )
   elif saveAll:
      cmds.append( 'no bgp route install-map' )

   aclVrfName = vrfName
   if vrfName == '':
      aclVrfName = 'default'
   for aclType in [ 'ip', 'ipv6' ]:
      serviceAclConfig = aclCpConfig.cpConfig[ aclType ].serviceAcl
      if ( aclVrfName in serviceAclConfig and
           'bgpsacl' in serviceAclConfig[ aclVrfName ].service and
           serviceAclConfig[ aclVrfName ].service[ 'bgpsacl' ].aclName ):
         cmds.append( '%s access-group %s' % ( aclType,
                      serviceAclConfig[ aclVrfName ].service[ 'bgpsacl' ].aclName ) )
      elif saveAll:
         cmds.append( 'no %s access-group' % aclType )

   if bgpConfig.listenPort != bgpConfig.listenPortInvalid:
      cmds.append( 'bgp transport listen-port %d' % bgpConfig.listenPort )
   elif saveAll:
      cmds.append( 'no bgp transport listen-port' )

   if bgpConfig.sockMaxSegSizeIpv4 != bgpConfig.sockMaxSegSizeInvalid:
      cmds.append( 'bgp transport ipv4 mss %u' % bgpConfig.sockMaxSegSizeIpv4 )
   elif saveAll:
      cmds.append( 'no bgp transport ipv4 mss' )

   if bgpConfig.sockMaxSegSizeIpv6 != bgpConfig.sockMaxSegSizeInvalid:
      cmds.append( 'bgp transport ipv6 mss %u' % bgpConfig.sockMaxSegSizeIpv6 )
   elif saveAll:
      cmds.append( 'no bgp transport ipv6 mss' )

   if BgpCommonToggleLib.toggleBgpSockBufferSizeEnabled():
      if bgpConfig.sockBufferSize:
         size = bgpConfig.sockBufferSize.txBufferSize
         cmds.append( 'bgp transport buffer %u kbytes' % size )
      elif saveAll:
         cmds.append( 'no bgp transport buffer' )

   if bgpConfig.pathMtuDiscoverySet:
      if bgpConfig.pathMtuDiscovery == 'isTrue':
         cmds.append( 'bgp transport pmtud' )
      else:
         cmds.append( 'bgp transport pmtud disabled' )
   elif saveAll:
      cmds.append( 'no bgp transport pmtud' )

   if not vrfName and bgpConfig.dscp != bgpConfig.dscpDefault:
      cmds.append( 'bgp transport qos dscp %d' % bgpConfig.dscp )
   elif saveAll and not vrfName:
      cmds.append( 'bgp transport qos dscp %d' % bgpConfig.dscp )

   if bgpConfig.defaultMetricPresent != False: # pylint: disable=singleton-comparison
      cmds.append( 'default-metric %d' % bgpConfig.defaultMetric )
   elif saveAll:
      cmds.append( 'no default-metric' )

   if not vrfName:
      maintCmd = "maintenance receiver route-map default "
      if bgpConfig.maintRecvLocalPref:
         cmds.append(
               maintCmd + 'local-preference %d' % bgpConfig.maintRecvLocalPref )
      elif bgpConfig.maintRecvRouteMapDisabled:
         cmds.append( maintCmd + "disabled" )
      elif saveAll:
         cmds.append( maintCmd + 'local-preference 0' )

   isCmpMedSet, cmpMedVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                              'alwaysCompareMed' )
   if isCmpMedSet or saveAll:
      cmds.append( ( '' if cmpMedVal else 'no ' ) + 'bgp always-compare-med' )

   isMedWrstSet, medWrstVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                                'medMissingAsWorst' )
   if isMedWrstSet or saveAll:
      cmds.append( ( '' if medWrstVal else 'no ' ) + \
                          'bgp bestpath med missing-as-worst' )

   isMedCfdSet, medCfdVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                              'medConfed' )
   isMedCfdMissWrstSet, medCfdMissWrstVal = evaluateTristate( bgpConfig,
                                                         defaultVrfBgpConfig,
                                                         'medConfedMissingAsWorst' )
   if isMedCfdSet or isMedCfdMissWrstSet or saveAll:
      cmds.append( ( '' if medCfdVal or medCfdMissWrstVal else 'no ') + \
                          'bgp bestpath med confed' + \
                          ( ' missing-as-worst' if isMedCfdMissWrstSet else '') )

   if bgpConfig.reflectedRouteAttr != bgpConfig.reflectedRouteAttrDefault:
      cmd = 'bgp route-reflector preserve-attributes'
      if bgpConfig.reflectedRouteAttr == reflectedRouteAttrStateEnum.alwaysPreserved:
         cmd += ' always'
      cmds.append( cmd )
   elif saveAll:
      cmds.append( 'no bgp route-reflector preserve-attributes' )

   isMpSet, mpVal = evaluateValue( bgpConfig, defaultVrfBgpConfig, 'maxPaths' )
   isEcmpSet, ecmpVal = evaluateValue( bgpConfig, defaultVrfBgpConfig, 'maxEcmp',
                                       routingHwStatus.maxLogicalProtocolEcmp )
   if isMpSet or isEcmpSet or saveAll:
      cmd = 'maximum-paths %d' % mpVal
      if len( bgpConfig.maxPathsYangSource ) > 0:
         # YangSource configuration does not support max-ecmp
         assert not isEcmpSet
         cmd += ' source openconfig'
         for enumVal, yangSrcStr in \
               [ ( YangSourceEnum.yangSourceGlobal, 'global' ),
                 ( YangSourceEnum.yangSourceIpv4Unicast, 'ipv4-unicast' ),
                 ( YangSourceEnum.yangSourceIpv6Unicast, 'ipv6-unicast' ) ]:
            if enumVal in bgpConfig.maxPathsYangSource:
               cmd += f' {yangSrcStr}'
      elif isEcmpSet or saveAll:
         if isEcmpSet or ( not isMpSet ):
            ecmpVal = ecmpVal if ecmpVal != getattr( defaultVrfBgpConfig,
               'maxEcmpInvalid' ) else routingHwStatus.maxLogicalProtocolEcmp
         else:
            # Default VRF -> ecmp not configured, set it to hwMax.
            # Non-Default VRF -> ecmp and also max-path not configured, set ecmp to
            # hwMax.
            ecmpVal = routingHwStatus.maxLogicalProtocolEcmp

         if ecmpVal:
            cmd += ' ecmp %s' % ecmpVal
      cmds.append( cmd )

   # Print additional-paths command in RouterBgpMode only if there's
   # no AF mode under bgp; this command is for IPv4 Unicast AF.
   if not hasAfConfig( bgpConfig ):
      if bgpConfig.picIPv4Uni:
         if not bgpConfig.picEcmpPrimaryIPv4Uni:
            cmds.append( "bgp additional-paths install" )
         else:
            cmds.append( "bgp additional-paths install ecmp-primary" )
      elif saveAll:
         cmds.append( "no bgp additional-paths install" )

   # Add-path receive command
   isAPRecvSet, apRecvVal = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                              'apRecv' )
   if isAPRecvSet:
      if apRecvVal:
         cmds.append( 'bgp additional-paths receive' )
      else:
         cmds.append( 'no bgp additional-paths receive' )
   elif saveAll:
      # Add-path receive is enabled by default
      cmds.append( 'bgp additional-paths receive' )

   # Add-path send command
   cmd = ''
   for app in bgpConfig.apSend:
      appConfig = bgpConfig.apSend.get( app )
      if appConfig.enable:
         # app shouldn't be 'appNone'
         cmd = 'bgp '
      else:
         # app must be 'appNone' or 'appAny'
         cmd = 'no bgp '
      cmd += addPathSendCommon( app, appConfig=appConfig )
      # There can be only one app present
      break
   if saveAll and cmd == '':
      cmd = 'default bgp '
      app = 'appNone'
      cmd += addPathSendCommon( app, saveAll=saveAll )
   if cmd != '':
      cmds.append( cmd )

   isLlSet, val = evaluateValue( bgpConfig, defaultVrfBgpConfig, 'listenLimit' )
   if isLlSet or saveAll:
      cmd = 'dynamic peer max %d' % val
      cmds.append( cmd )

   listenRange = set( bgpConfig.listenRange )
   for peergroupName in sorted( list( listenRange ) ):
      peergroup = bgpConfig.listenRange.get( peergroupName )
      # TODO .stringValue should not be necessary below, BUG25925
      for prefix in peergroup.prefixList:
         peerSpec = peergroup.prefixList[prefix]
         if peerSpec.hasAsn:
            spec = 'remote-as %s' % bgpFormatAsn( peerSpec.asn, asdotConfigured )
         else:
            spec = 'peer-filter %s' % peerSpec.peerFilter
         peerId = ''
         if peerSpec.includeRouterId:
            peerId = ' peer-id include router-id'
         cmd = 'bgp listen range %s%s peer-group %s %s' % \
               ( prefix.stringValue,
                 peerId,
                 peergroup.peergroupName,
                 spec )
         cmds.append( cmd )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'asPathMultiPathRelax' )
   _, matchVal = evaluateValue( bgpConfig, defaultVrfBgpConfig,
                                         'asPathMultiPathRelaxMatch' )
   if isSet or matchVal != bgpConfig.asPathMultiPathRelaxMatchDefault or saveAll:
      if matchVal != bgpConfig.asPathMultiPathRelaxMatchInvalid and \
         matchVal != bgpConfig.asPathMultiPathRelaxMatchDefault:
         cmd = ( 'bgp bestpath as-path multipath-relax match %d' %
                 bgpConfig.asPathMultiPathRelaxMatch )
      else:
         cmd = ( ( '' if val else 'no ' ) +
                 'bgp bestpath as-path multipath-relax' )
      cmds.append( cmd )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'ignoreAsPathLen' )

   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp bestpath as-path ignore' )

   # bestpath d-path configuration applies only to default-vrf.
   if not vrfName:
      val = defaultVrfBgpConfig.includeDomainPathLen == 'isTrue'
      if val or saveAll:
         cmds.append( ( '' if val else 'no ' ) + 'bgp bestpath d-path' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
         'nexthopResolutionNhgTunnelEcmpDisabled' )

   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) +
         'bgp bestpath next-hop resolution tunnel nexthop-group ecmp disabled' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
         'skipNextHopIgpCost' )

   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) +
                   'bgp bestpath skip next-hop igp-cost' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
         'skipAigpMetric' )

   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) +
                   'bgp bestpath skip aigp-metric' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'compatibleEBgpIBgp' )

   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) +
                   'bgp bestpath skip peer type ebgp ibgp' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'asPathCmpIncNh' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp aspath-cmp-include-nexthop' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'ecmpFast' )
   if isSet and not val:
      cmds.append( 'no bgp bestpath ecmp-fast' )
   elif saveAll:
      cmds.append( 'bgp bestpath ecmp-fast' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig, 'ecmpTbOnAge' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp bestpath tie-break age' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'ecmpTbOnRouterId' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + \
                          'bgp bestpath tie-break router-id' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'ecmpTbOnOrigId' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + \
                          'bgp bestpath tie-break originator-id' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'ecmpTbOnClusterListLen' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + \
                          'bgp bestpath tie-break cluster-list-length' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'advertiseInactive' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp advertise-inactive' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'autoLocalAddr' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp auto-local-addr' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'nexthopUnchanged' )
   if isSet or saveAll:
      cmds.append( ( '' if val else 'no ' ) + 'bgp next-hop-unchanged' )

   val = bgpConfig.aggregateCommInheritLoose
   if val or saveAll:
      cmds.append( ( '' if val else 'no ' ) +
                   'bgp aggregate-route community inheritance loose' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'includeOriginAsValidity' )
   if isSet:
      cmds.append( 'bgp bestpath origin-as validity' +
                   ( '' if val else ' disabled' ) )
   elif saveAll:
      cmds.append( 'no bgp bestpath origin-as validity' )

   if not vrfName:
      val = defaultVrfBgpConfig.lfibUninstallDelay
   else:
      val = bgpConfig.lfibUninstallDelay

   if val == 0:
      cmds.append( 'next-hop labeled-unicast originate '
                   'lfib-backup uninstall-delay disabled' )
   elif val != defaultVrfBgpConfig.lfibUninstallDelayDefault:
      cmds.append( 'next-hop labeled-unicast originate '
                   'lfib-backup uninstall-delay %d seconds' % val )
   elif saveAll:
      cmds.append( 'no next-hop labeled-unicast originate '
                   'lfib-backup uninstall-delay' )

   isSet, val = evaluateTristate( bgpConfig, defaultVrfBgpConfig,
                                  'nhLuOrigLfibBackupIpFwd' )
   if isSet:
      cmds.append( 'next-hop labeled-unicast originate lfib-backup '
                   'ip-forwarding' + ( '' if val else ' disabled' ) )
   elif saveAll:
      cmds.append( 'no next-hop labeled-unicast originate lfib-backup '
                   'ip-forwarding' )

   for hook in bgpMonitoringHook.extensions():
      cmd = hook( bgpConfig, saveAll )
      if cmd:
         cmds.append( cmd )

   nbrDefaultRcvdBgpAttrDiscardCmd = 'neighbor default received attribute discard'
   bgpAttrsString = MultiRangeRule.multiRangeToCanonicalString(
         bgpConfig.rcvdBgpAttrDiscard )
   if bgpAttrsString:
      cmds.append( nbrDefaultRcvdBgpAttrDiscardCmd + ' %s' % bgpAttrsString )
   elif bgpConfig.rcvdBgpAttrDiscardDisabled:
      # Hardcode nedMode=True until BUG433700 is resolved
      cmds.append( getExplicitNoCmd( nbrDefaultRcvdBgpAttrDiscardCmd, True ) )
   elif not defaultVrfBgpConfig.rcvdBgpAttrDiscard and saveAll:
      cmds.append( getExplicitNoCmd( nbrDefaultRcvdBgpAttrDiscardCmd,
                                     getattr( defaultVrfBgpConfig,
                                              'noEqualsDefaultMode' ) ) )

   neighborDefaultSendCommunityCmd = 'neighbor default send-community'
   if bgpConfig.sendCommunityPresent:
      if ( bgpConfig.sendCommunity or
           bgpConfig.sendStandardCommunity or
           bgpConfig.sendExtendedCommunity or
           bgpConfig.sendLargeCommunity ):
         if bgpConfig.sendStandardCommunity:
            neighborDefaultSendCommunityCmd += ' standard'
         if bgpConfig.sendExtendedCommunity:
            neighborDefaultSendCommunityCmd += ' extended'
         if bgpConfig.sendLargeCommunity:
            neighborDefaultSendCommunityCmd += ' large'
         cmds.append( neighborDefaultSendCommunityCmd )
      else:
         cmds.append( getExplicitNoCmd( neighborDefaultSendCommunityCmd,
                                        getattr( defaultVrfBgpConfig,
                                                 'noEqualsDefaultMode' ) ) )
   if bgpConfig.indexRoutesPeerDisabled:
      cmds.append( 'neighbor default index routes peer disabled' )

   def peerConfigSave( peerKey ):
      nedMode = getattr( defaultVrfBgpConfig, 'noEqualsDefaultMode' )
      v6Addr = peerKey.type in { PeerType.peerIpv6, PeerType.peerIpv6Ll }
      v4Addr = peerKey.type == PeerType.peerIpv4

      addrOrPgName = peerKey.stringValue

      peer = bgpConfig.neighborConfig.get( peerKey )
      if not peer and saveAll:
         # Display default peer config for saveAll.
         # Using defaultPeerConfig to get the default values for the peer.
         # Assigning the peer with default config changes the entity
         # name (addrOrPgName).
         # Therefore, 'addrOrPgName' should be used to access neighbor ip.
         # (XXX.Shall we create a dummy peer instance instead?)
         peer = bgpConfig.defaultPeerConfig
      peerGroup = peer.isPeerGroup
      peerGroupPeer = peer.isPeerGroupPeer
      if peerGroup:
         cmd = 'neighbor %s peer group' % addrOrPgName
         if peer.isPeerGroupWithAncestors:
            pgKeys = peer.ancestorPeerGroupConfig.ancestorPeerGroupKeys.values()
            for ancestorPgKey in pgKeys:
               cmd += ' %s' % ancestorPgKey.group
         cmds.append( cmd )

      def addSimpleSaveCommand( attrStr, cmd, disabledKeywordInNedMode,
                                hasGlobalAttr=False, valuePrint=None, trail=None ):
         attr = getattr( peer, attrStr )
         attrDefault = getattr( peer, attrStr + 'Default' )
         attrPresent = getattr( peer, attrStr + 'Present' )

         saveCmd = 'neighbor %s %s' % ( addrOrPgName, cmd )
         if attrPresent:
            if ( attr == attrDefault ) and ( peerGroupPeer or hasGlobalAttr ):
               if disabledKeywordInNedMode:
                  cmds.append( getExplicitNoCmd( saveCmd, nedMode ) )
               else:
                  cmds.append( getExplicitNoCmd( saveCmd, nedMode, attrDefault ) )
               return
            if attrStr == 'removePrivateAs':
               if attr.always:
                  saveCmd += ' all'
               if attr.replaceAs:
                  saveCmd += ' replace-as'
            elif attrStr == 'removePrivateAsIngress':
               if attr.replaceAs:
                  saveCmd += ' replace-as'
            elif attrStr == 'neighborShutdown':
               shutdownMsgAttr = peer.neighborShutdownMsg
               shutdownMsgAttrDefault = peer.neighborShutdownMsgDefault
               if shutdownMsgAttr != shutdownMsgAttrDefault:
                  saveCmd += ' reason %s' % shutdownMsgAttr
            elif attrStr in [ 'idleHoldTime', 'bgpTtlSecMaxHop' ]:
               saveCmd = '%s %d' % ( saveCmd, attr )
            elif valuePrint:
               saveCmd += valuePrint( peer, attr )
            elif not isinstance( attr, bool ):
               if cmd == 'remote-as':
                  saveCmd = saveCmd + ' ' + bgpFormatAsn( attr, asdotConfigured )
               else:
                  saveCmd = saveCmd + ' ' + str( attr )
            if trail:
               saveCmd = saveCmd + ' ' + trail
            cmds.append( saveCmd )
         elif saveAll and not peerGroupPeer:
            if disabledKeywordInNedMode:
               cmd = getExplicitNoCmd( saveCmd, nedMode )
            else:
               cmd = getExplicitNoCmd( saveCmd, nedMode, attrDefault )
            cmds.append( cmd )

      def addSimpleBoolSaveCommand( attrStr, baseCmd, positiveCmdDefault=False ):
         attr = getattr( peer, attrStr )
         attrPresent = getattr( peer, attrStr + 'Present' )

         saveCmd = 'neighbor %s %s' % ( addrOrPgName, baseCmd )
         if attrPresent:
            if attr:
               if attrStr == 'defaultOriginate':
                  if peer.defaultOriginateRouteMap:
                     saveCmd += " route-map %s" % peer.defaultOriginateRouteMap
                  if peer.defaultOriginateAlways:
                     saveCmd += " always"

            else:
               saveCmd = getExplicitNoCmd( saveCmd, nedMode )
            cmds.append( saveCmd )
         elif saveAll and not peerGroupPeer:
            if positiveCmdDefault:
               cmds.append( saveCmd )
            else:
               cmds.append( getExplicitNoCmd( saveCmd, nedMode ) )

      # Used for route-map and prefix-list in|out commands
      def addSimpleInOutSaveCommand( attrStr, baseCmd, inOrOut ):
         assert inOrOut in [ 'in', 'out' ]
         attr = getattr( peer, attrStr )
         attrPresent = getattr( peer, attrStr + 'Present' )

         saveCmd = 'neighbor %s %s' % ( addrOrPgName, baseCmd )
         if attrPresent:
            if attr:
               cmds.append( '%s %s %s' % ( saveCmd, attr, inOrOut ) )
            elif peer.isPeerGroupPeer:
               cmds.append( getExplicitNoCmd( '%s %s' % ( saveCmd, inOrOut ),
                                              nedMode ) )
         elif saveAll and not peerGroupPeer:
            cmds.append( getExplicitNoCmd( '%s %s' % ( saveCmd, inOrOut ),
                                           nedMode ) )


      # COMPATIBILITY WARNING
      # Please don't reorder the cmds.append, unless you have a *really* good
      # reason and are ready to break backward-compatibility.
      #
      # Add new attributes at the very end.
      #
      # See BUG198992.

      if peerGroupPeer:
         pgName = peer.peerGroupKey.group
         cmds.append( 'neighbor %s peer group %s' % ( addrOrPgName,
                                                      pgName ) )
      elif saveAll and not peerGroup and not v6Addr:
         cmds.append( 'no neighbor %s peer group' % addrOrPgName )

      addSimpleSaveCommand( 'asNumber', 'remote-as', True )
      addSimpleSaveCommand( 'importLocalPref', 'import-localpref', False )
      addSimpleSaveCommand( 'exportLocalPref', 'export-localpref', False )
      addSimpleSaveCommand( 'nextHopSelf', 'next-hop-self', True,
                            hasGlobalAttr=True )
      addSimpleSaveCommand( 'nextHopPeer', 'next-hop-peer', True )
      addSimpleSaveCommand( 'nexthopUnchanged', 'next-hop-unchanged', True,
                            hasGlobalAttr=True )
      addSimpleSaveCommand( 'neighborShutdown', 'shutdown', True )
      addSimpleSaveCommand( 'removePrivateAs', 'remove-private-as', True )
      addSimpleSaveCommand( 'linkbwDelay', 'link-bandwidth update-delay', False )
      addSimpleSaveCommand( 'prependOwnDisabled', 'as-path prepend-own disabled',
                            True )
      addSimpleSaveCommand( 'replaceRemoteAs',
                            'as-path remote-as replace out', True )

      if toggleFeatureConfigTagBgpEnabled and not peerGroup:
         commandTagCmd = 'neighbor %s command-tag' % addrOrPgName
         commandTagId = peer.commandTagId
         if commandTagId != peer.commandTagIdDefault:
            commandTagStr = commandTagIdState.tagIdToTagStr[ commandTagId ]
            assert commandTagStr
            commandTagCmd += ' %s' % commandTagStr
            cmds.append( commandTagCmd )
         elif saveAll:
            cmds.append( 'no %s' % commandTagCmd )

      if peer.pathMtuDiscoveryPresent:
         if peer.pathMtuDiscovery:
            addSimpleSaveCommand( 'pathMtuDiscovery', 'transport pmtud', True )
         else:
            addSimpleSaveCommand( 'pathMtuDiscovery', 'transport pmtud disabled',
                                  True )

      def localAsValuePrint( peer, localAs ):
         cmd = ' %s no-prepend replace-as' % \
            bgpFormatAsn( localAs.asn, asdotConfigured )
         if localAs.fallback:
            cmd += " fallback"
         return cmd

      addSimpleSaveCommand( 'localAs', 'local-as', True,
                            valuePrint=localAsValuePrint )

      weightCmd = "neighbor %s weight" % addrOrPgName
      if peer.weightPresent:
         if peer.weight == peer.weightDefault and peerGroupPeer:
            cmds.append( getExplicitNoCmd( weightCmd, nedMode,
                                           defaultValue=peer.weightDefault ) )
         else:
            cmds.append( '%s %s' % ( weightCmd, peer.weight ) )
      elif saveAll and not peerGroupPeer:
         # Per-neighbor weight cannot be configured both globally and per address
         # family. If it is not configured in any address family, we should render
         # the default neighbor weight cmd for save-all at the global level.
         if not weightConfiguredInAf( peer ):
            cmds.append( getExplicitNoCmd( weightCmd, nedMode,
                                           defaultValue=peer.weightDefault ) )

      outDelayCmd = "neighbor %s out-delay" % addrOrPgName
      if peer.outDelayPresent:
         if peer.outDelay == peer.outDelayDefault and peerGroupPeer:
            odCmd = getExplicitNoCmd( outDelayCmd, nedMode,
                                      defaultValue=peer.outDelayDefault )
         else:
            odCmd = '%s %s' % ( outDelayCmd, peer.outDelay )
         if peer.outDelayApply == outDelayApplyEnum.outDelayApplyChanges:
            odCmd += ' changes'
         elif peer.outDelayApply == \
              outDelayApplyEnum.outDelayApplyChangesWithdrawals:
            odCmd += ' changes withdrawals'
         cmds.append( odCmd )

      elif saveAll and not peerGroupPeer:
         # Per-neighbor out-delay cannot be configured both globally and per address
         # family. If it is not configured in any address family, we should render
         # the default neighbor out-delay cmd for save-all at the global level.
         if not outDelayConfiguredInAf( peer ):
            cmds.append( getExplicitNoCmd( outDelayCmd, nedMode,
                                           defaultValue=peer.outDelayDefault ) )

      addSimpleSaveCommand( 'passive', 'passive', True )
      addSimpleSaveCommand( 'remotePort', 'transport remote-port', False )
      updateSrcCmd = 'neighbor %s update-source' % addrOrPgName
      if peer.updateSrcIntfPresent or peer.updateSrcAddrPresent:
         if peer.updateSrcIntf != peer.updateSrcIntfDefault:
            cmds.append( '%s %s' % ( updateSrcCmd, peer.updateSrcIntf ) )
         elif peer.updateSrcAddr != peer.updateSrcAddrDefault:
            cmds.append( '%s %s' % ( updateSrcCmd, peer.updateSrcAddr ) )
         elif peerGroupPeer:
            cmds.append( getExplicitNoCmd( updateSrcCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( updateSrcCmd, nedMode ) )
      addSimpleSaveCommand( 'dontCapabilityNegotiate', 'dont-capability-negotiate',
                            True )
      # Only use the deprecated syntax of the neighbor bfd command if the c-bit
      # option of the command is not being used, and the deprecated syntax for this
      # command was used.
      bfdCmd = "bfd"
      if ( getattr( peer, 'bfdCBitEnabled' ) ):
         bfdCmd += " c-bit"
      if ( getattr( peer, 'bfdDampingEnabled' ) ):
         bfdCmd += " damping"
      if ( getattr( peer, 'bfdDampingTimePresent' ) ):
         bfdCmd += " seconds %d" % getattr( peer, 'bfdDampingTime' )
      if ( getattr( peer, 'bfdDampingMultiplierPresent' ) ):
         bfdCmd += " max-multiplier %s" % getattr( peer, 'bfdDampingMultiplier' )

      addSimpleSaveCommand( 'bfdEnabledState',
                            bfdCmd,
                            True )

      bfdIntervalCmd = 'neighbor %s bfd interval' % addrOrPgName
      if peer.bfdIntervalConfigPresent:
         cfg = peer.bfdIntervalConfig
         cmds.append( '%s %d min-rx %d multiplier %d' %
               ( bfdIntervalCmd, cfg.minTx, cfg.minRx, cfg.mult ) )
      elif saveAll:
         cmds.append( 'no %s' % bfdIntervalCmd )

      if not v4Addr:
         addSimpleSaveCommand( 'localIp4Addr', 'local-v4-addr', True )
      if not v6Addr:
         addSimpleSaveCommand( 'localIp6Addr', 'local-v6-addr', True )
      addSimpleSaveCommand( 'autoLocalAddr', 'auto-local-addr', True )

      for hook in neighborMonitoringHook.extensions():
         cmd = hook( addrOrPgName, peer, saveAll )
         if cmd:
            cmds.append( cmd )

      if not v6Addr:
         addSimpleSaveCommand( 'remoteIp6Addr', 'next-hop-v6-addr', True,
                               trail='in' )

      descriptionCmd = 'neighbor %s description' % addrOrPgName
      if peer.descriptionPresent:
         if peer.description != peer.descriptionDefault:
            cmds.append( '%s %s' % ( descriptionCmd, peer.description ) )
         elif peerGroupPeer:
            cmds.append( 'no %s' % descriptionCmd )
      elif saveAll and not peerGroupPeer:
         cmds.append( 'no %s' % descriptionCmd )

      addSimpleSaveCommand( 'allowAs', 'allowas-in', True, True )

      # softreconfig inbound is default true
      sciCmd = 'neighbor %s rib-in pre-policy retain' % addrOrPgName
      if peer.softReconfigInboundPresent:
         if peer.softReconfigInbound != peer.softReconfigInboundDefault:
            if peer.softReconfigInbound == SoftReconfigInboundStateEnum.sciAll:
               cmds.append( '%s all' % sciCmd )
            else:
               cmds.append( getExplicitNoCmd( sciCmd, nedMode ) )
         elif peerGroupPeer:
            cmds.append( sciCmd )
      elif saveAll and not peerGroupPeer:
         cmds.append( sciCmd )

      # ebgpMultiHop may or may not print out a ttl
      ebgpMultiHopCmd = 'neighbor %s ebgp-multihop' % addrOrPgName
      if peer.ebgpMultiHopPresent:
         if ( peer.ebgpMultiHop == peer.ebgpMultiHopDefault ) and peerGroupPeer:
            ebgpMultiHopCmd = getExplicitNoCmd( ebgpMultiHopCmd, nedMode )
         elif peer.ebgpMultiHop != peer.ebgpMultiHopEnabledTtlDefault:
            ebgpMultiHopCmd = '%s %d' % ( ebgpMultiHopCmd, peer.ebgpMultiHop )
         cmds.append( ebgpMultiHopCmd )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( ebgpMultiHopCmd, nedMode ) )

      bgpTtlSecMaxHopCmdTrail = None
      if peer.bgpTtlSecMaxHopLog != peer.bgpTtlSecMaxHopLogDefault:
         bgpTtlSecMaxHopCmdTrail = 'log'
      addSimpleSaveCommand( 'bgpTtlSecMaxHop', 'ttl maximum-hops', False,
                            trail=bgpTtlSecMaxHopCmdTrail )

      # route reflector has another option meshed
      rrClientCmd = 'neighbor %s route-reflector-client' % addrOrPgName
      rrClientEnum = Tac.Type( "Routing::Bgp::ReflectorClient" )
      if peer.rrClientPresent:
         if peer.rrClient == rrClientEnum.reflectorMeshed:
            rrClientCmd = '{} reflection intra-cluster disabled'.format(
               rrClientCmd )
         elif ( peer.rrClient == peer.rrClientDefault ) and peerGroupPeer:
            rrClientCmd = getExplicitNoCmd( rrClientCmd, nedMode )
         cmds.append( rrClientCmd )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( rrClientCmd, nedMode ) )
      for hook in neighborRrHook.extensions():
         cmd = hook( addrOrPgName, peer, saveAll )
         if cmd:
            cmds.append( cmd )

      timersCmdPrefix = 'neighbor %s timers' % addrOrPgName
      timersCmd = ''
      if peer.timersPresent:
         timersCmd += ' %d %d' % ( peer.keepaliveTime, peer.holdTime )
      if peer.minHoldTimePresent:
         timersCmd += ' min-hold-time %d' % ( peer.minHoldTime )
      if peer.sendFailureHoldTimePresent:
         timersCmd += ' send-failure hold-time %d' % ( peer.sendFailureHoldTime )
      if timersCmd:
         cmds.append( timersCmdPrefix + timersCmd )
      elif saveAll and not peerGroupPeer:
         timersDefaultValue = '%d %d' % (
            peer.keepaliveTimeDefault, peer.holdTimeDefault
         )
         timersDefaultValue += ' min-hold-time %d' % ( peer.minHoldTimeDefault )
         timersDefaultValue += ' send-failure hold-time %d' % (
            peer.sendFailureHoldTimeDefault
         )
         cmds.append( getExplicitNoCmd( timersCmdPrefix, nedMode,
                         defaultValue=timersDefaultValue ) )

      addSimpleInOutSaveCommand( 'routeMapIn', 'route-map', 'in' )

      gracefulRestartCmd = "neighbor %s graceful-restart" % addrOrPgName
      if peer.gracefulRestartPresent:
         if peer.gracefulRestart:
            cmds.append( gracefulRestartCmd )
         else:
            cmds.append( getExplicitNoCmd( gracefulRestartCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( f"default {gracefulRestartCmd}" )

      grHelperCmd = 'neighbor %s graceful-restart-helper' % addrOrPgName
      # grHelperPresent is False in case of default cmd
      if peer.grHelperPresent:
         # peer.grHelper is False in case of 'no' cmd
         if peer.grHelper:
            if peer.grHelperRestartTime:
               grHelperRestartTimeVal = getTimeFromSeconds(
                     peer.grHelperRestartTime,
                     peer.grHelperRestartTimeClass )
               grHelperCmd += ( ' restart-time %d' % grHelperRestartTimeVal )
               if ( peer.grHelperRestartTimeClass !=
                     peer.grHelperRestartTimeClassDefault ):
                  grHelperCmd += ' %s' % peer.grHelperRestartTimeClass
            if peer.llgrHelper:
               grHelperCmd += ' long-lived'
            elif peer.grHelperRouteMapPresent:
               grHelperCmd += ' stale-route route-map %s' % peer.grHelperRouteMap
               if peer.grHelperRmGrOptionalPresent:
                  grHelperCmd += ' session-failure graceful-restart-negotiation ' \
                                 'optional'
            cmds.append( grHelperCmd )
         else:
            cmds.append( getExplicitNoCmd( grHelperCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         # Add default if no explicit configuration exists
         cmds.append( 'default ' + grHelperCmd )
      addSimpleBoolSaveCommand( 'apRecv', 'additional-paths receive', True )

      # Add-path send command
      cmd = ''
      for app in peer.apSend:
         appConfig = peer.apSend.get( app )
         if appConfig.enable:
            cmd = 'neighbor %s %s' % ( addrOrPgName,
                     addPathSendCommon( app, appConfig=appConfig ) )
         else:
            # app must be 'appNone' or 'appAny'
            cmd = getExplicitNoCmd( 'neighbor %s %s' % ( addrOrPgName,
                     addPathSendCommon( app, appConfig=appConfig ) ), nedMode )
         # There can be only one app present
         break
      if saveAll and not peerGroupPeer and cmd == '':
         app = 'appNone'
         cmd = getExplicitNoCmd( 'neighbor %s %s' % ( addrOrPgName,
                  addPathSendCommon( app, saveAll=saveAll ) ), nedMode )
      if cmd != '':
         cmds.append( cmd )

      addSimpleBoolSaveCommand( 'routeToPeer', 'route-to-peer', True )

      addSimpleInOutSaveCommand( 'routeMapOut', 'route-map', 'out' )

      passwordCmd = 'neighbor %s password' % addrOrPgName
      if peer.passwordPresent:
         defaultSecret = Tac.Value( 'SecretCliLib::Secret' )
         if peer.password.md5Password != defaultSecret:
            formatStr = escapeFormatString( passwordCmd ) + ' {}'
            cmd = ReversibleSecretCli.getCliSaveCommand( formatStr,
                                                         securityConfig,
                                                         peer.password.md5Password,
                                                         uniqueKey=( addrOrPgName +
                                                                     '_passwd' ),
                                                         algorithm='DES' )
            cmds.append( cmd )
         elif ( peer.password.sharedSecretProfile ):
            cmds.append(
               'neighbor %s password shared-secret profile %s algorithm %s' %
               ( addrOrPgName, peer.password.sharedSecretProfile,
                 tcpAoAlgorithmMap.reverse()[ peer.password.algorithm ] ) )
         elif peer.isPeerGroupPeer:
            cmds.append( 'no ' + passwordCmd )
      elif saveAll and not peerGroupPeer:
         cmds.append( 'no ' + passwordCmd )

      addSimpleBoolSaveCommand( 'defaultOriginate', 'default-originate' )
      addSimpleBoolSaveCommand( 'enforceFirstAs', 'enforce-first-as', True )

      metricOutCmd = 'neighbor %s metric-out' % addrOrPgName
      if ( peer.metricOutState != MetricOutStateEnum.metricNotConfigured ):
         if ( peer.metricOutState == MetricOutStateEnum.metricOutSet ):
            cmds.append( '%s %d' % ( metricOutCmd, peer.metricOut ) )
         else:
            cmds.append( getExplicitNoCmd( metricOutCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( metricOutCmd, nedMode ) )

      addSimpleSaveCommand( 'idleHoldTime', 'idle-restart-timer', True )

      if BgpCommonToggleLib.toggleArBgpNbrRcvdBgpAttributeFilteringEnabled():
         nbrRcvdBgpAttrDiscardCmd = (
               f'neighbor {addrOrPgName} received attribute discard' )
         bgpAttrsString = MultiRangeRule.multiRangeToCanonicalString(
               peer.rcvdBgpAttrDiscard )
         if bgpAttrsString:
            cmds.append( f'{nbrRcvdBgpAttrDiscardCmd} {bgpAttrsString}' )
         elif peer.rcvdBgpAttrDiscardDisabled:
            # Hardcode nedMode=True until BUG433700 is resolved
            cmds.append( getExplicitNoCmd( nbrRcvdBgpAttrDiscardCmd, True ) )
         elif not peerGroupPeer and saveAll:
            cmds.append( getExplicitNoCmd( nbrRcvdBgpAttrDiscardCmd, nedMode ) )

      sendCommunityCmd = 'neighbor %s send-community' % addrOrPgName
      if peer.sendCommunityPresent:
         lbwPresenceValid = \
               peer.sendCommunityLinkBwPresent and \
               ( peer.sendExtendedCommunity or \
                not ( peer.sendLargeCommunity or peer.sendStandardCommunity ) )
         if peer.sendCommunity or \
               peer.sendStandardCommunity or \
               peer.sendExtendedCommunity or \
               peer.sendLargeCommunity:
            if peer.sendStandardCommunity:
               sendCommunityCmd += ' standard'
            if peer.sendExtendedCommunity:
               sendCommunityCmd += ' extended'
            if peer.sendLargeCommunity:
               sendCommunityCmd += ' large'
            if lbwPresenceValid:
               sendCommunityCmd += ' link-bandwidth'
               if peer.sendCommunityLinkBw == \
                      SendCommunityLinkBandwidthStateEnum.\
                      sendCommunityLinkBandwidthAdditive:
                  sendCommunityCmd += ' aggregate'
               elif peer.sendCommunityLinkBw == \
                      SendCommunityLinkBandwidthStateEnum.\
                      sendCommunityLinkBandwidthAdditiveRef:
                  sendCommunityCmd += ' aggregate %s' \
                      % peer.sendCommunityLinkBwAdditiveRefDisplay
               elif peer.sendCommunityLinkBw == \
                      SendCommunityLinkBandwidthStateEnum.\
                      sendCommunityLinkBandwidthDivideEqual:
                  sendCommunityCmd += ' divide equal'
               elif peer.sendCommunityLinkBw == \
                      SendCommunityLinkBandwidthStateEnum.\
                      sendCommunityLinkBandwidthDivideRatio:
                  sendCommunityCmd += ' divide ratio'
               else:
                  raise ValueError()
            cmds.append( sendCommunityCmd )
         else:
            cmds.append( getExplicitNoCmd( sendCommunityCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( sendCommunityCmd, nedMode ) )

      savePeerMaxRoutes( peer, cmds, addrOrPgName, saveAll, nedMode )
      savePeerMaxAdvRoutes( peer, cmds, addrOrPgName, saveAll, nedMode )
      savePeerMissingPolicy( peer, cmds, addrOrPgName, saveAll, nedMode )

      # linkBw may have auto or default generation configured or not
      linkBwCmd = 'neighbor %s link-bandwidth' % addrOrPgName
      if peer.linkBwPresent:
         if peer.linkBw != \
                LinkBandwidthGenerationStateEnum.linkBandwidthNotConfigured:
            if peer.linkBw == \
                  LinkBandwidthGenerationStateEnum.linkBandwidthAutoGeneration:
               linkBwCmd += " auto"
               if peer.linkBwAutoPercentPresent:
                  linkBwCmd += " percent %s" % peer.linkBwAutoPercent
            elif peer.linkBw == \
                   LinkBandwidthGenerationStateEnum.linkBandwidthDefaultGeneration:
               linkBwCmd += " default %s" \
                   % peer.linkBwDefDisplay
            cmds.append( linkBwCmd )
         else:
            cmds.append( getExplicitNoCmd( linkBwCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( linkBwCmd, nedMode ) )

      linkBwAdjustCmd = 'neighbor %s link-bandwidth adjust auto' % addrOrPgName
      if peer.linkBwAdjustPresent:
         if peer.linkBwAdjust:
            if peer.linkBwAdjustPercentPresent:
               linkBwAdjustCmd += ' percent %s' % peer.linkBwAdjustPercent
            cmds.append( linkBwAdjustCmd )
         else:
            cmds.append( getExplicitNoCmd( linkBwAdjustCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( linkBwAdjustCmd, nedMode ) )

      # rtRefreshCmd is disabled by default
      rtRefreshCmd = \
         'neighbor %s route refresh demarcated stale-path removal' % addrOrPgName
      if peer.enhRtRefreshInPresent:
         if peer.enhRtRefreshIn:
            if peer.enhRtRefreshStalePathTimePresent:
               rtRefreshCmd += " timeout %d" % peer.enhRtRefreshStalePathTime
            cmds.append( rtRefreshCmd )
         else:
            if peerGroupPeer:
               cmds.append( getExplicitNoCmd( rtRefreshCmd, nedMode ) )
      elif saveAll and not peerGroupPeer:
         cmds.append( getExplicitNoCmd( rtRefreshCmd, nedMode ) )

      for hook in neighborRpkiHook.extensions():
         cmd = hook( addrOrPgName, peer, saveAll )
         if cmd:
            cmds.append( cmd )

      addSimpleSaveCommand( 'removePrivateAsIngress',
                            'remove-private-as ingress', True )

      # Save Link State peering segment config (see AID11677).
      savePeerSegmentNode( vrfName, peer, cmds, addrOrPgName, saveAll )

      # Save rib-in delay config.
      saveRibInDelay( vrfName, peer, cmds, addrOrPgName, saveAll )

      # END of peerConfigSave()

   # Iterate through the ordered collection bgpConfig.neighborConfig
   # and call peerConfigSave on every element.
   for peerKey in bgpConfig.neighborConfig:
      peerConfigSave( peerKey )

   # Only print network statements in RouterBgpMode if
   # no address-family configuration exists.
   if not afCfgPresent:
      saveNetworkList( cmds, bgpConfig )

   # Print [no] bgp redistribute-internal in router bgp mode only
   # if it wasn't configured in address-family mode.
   if bgpConfig.bgpRedistInternal != redistInternalStateEnum.notConfigured:
      if bgpConfig.bgpRedistInternal != \
         bgpConfig.bgpRedistInternalDefault:
         cmds.append( 'no bgp redistribute-internal' )
      elif saveAll:
         cmds.append( 'bgp redistribute-internal' )

   def saveAutoAggDomains( lst ):
      for domain in lst:
         cmd = 'auto-aggregation-domain %s' % domain.prefix
         if domain.asSet:
            cmd += ' as-set'
         if domain.attrMap and domain.attrMap != domain.attrMapDefault:
            cmd += ' aggregate-attributes %s' % domain.attrMap
         if domain.matchMap and domain.matchMap != domain.matchMapDefault:
            cmd += ' eligible-routes %s' % domain.matchMap
         if any( [ domain.groupByAsPath, domain.groupByLocPref, domain.groupByMed,
                   domain.groupByCommunity, domain.groupByExtCommunity ] ):
            cmd += ' group-by'
            if domain.groupByAsPath:
               cmd += ' as-path'
            if domain.groupByLocPref:
               cmd += ' local-preference'
            if domain.groupByMed:
               cmd += ' med'
            if domain.groupByCommunity:
               cmd += ' community'
            if domain.groupByExtCommunity:
               cmd += ' ext-community'
         cmds.append( cmd )

   for af in ( 'ipv4', 'ipv6' ):
      afAggregates = [ agg for agg in bgpConfig.aggregateList.values()
                       if agg.address.af == af ]
      cmds.extend( saveAggregates( afAggregates ) )

   saveAutoAggDomains( list( bgpConfig.autoAggDomainList.values() ) )

   protos = [ 'protoDirect', 'protoIsis', 'protoOspf', 'protoOspfAse',
              'protoOspfNssa', 'protoOspf3', 'protoOspf3Ase', 'protoOspf3Nssa',
              'protoStatic', 'protoRip', 'protoBgpAggregate',
              'protoAttachedHost', 'protoDynamic', 'protoBgp', 'protoEosSdk' ]

   for proto in protos:
      protocol = bgpConfig.redistributeConfig.get( proto )
      if protocol:
         command = redistributeCommand( protocol )
         if protocol.routeMap:
            command += ' route-map %s' % protocol.routeMap
         if protocol.rcf:
            command += ' rcf %s()' % protocol.rcf
      elif saveAll:
         command = 'no '
         command += redistributeCommand( proto, saveAll=saveAll )
      else:
         continue
      cmds.append( command )

   # BGP 'neighbor interface' config commands
   neighIntfConfig = {}
   for intf in bgpConfig.neighIntfConfig:
      peerConfig = bgpConfig.neighIntfConfig[ intf ]
      key = ( peerConfig.asnOrPeerFilter, peerConfig.peerGroup.stringValue )
      neighIntfConfig.setdefault( key, [ ] ).append( peerConfig.intfId )

   intfRangeCmds = []
   for asnOrPeerFilter, pg in neighIntfConfig:
      intfRanges = IntfRange.intfListToCanonical(
                             neighIntfConfig[ ( asnOrPeerFilter, pg ) ] )
      for intfRange in intfRanges:
         if asnOrPeerFilter.hasAsn:
            peerSpec = ' remote-as %s' % \
                       bgpFormatAsn( asnOrPeerFilter.asn, asdotConfigured )
         elif asnOrPeerFilter.peerFilter != "":
            peerSpec = ' peer-filter %s' % asnOrPeerFilter.peerFilter
         else:
            peerSpec = ''
         cmd = 'neighbor interface %s peer-group %s%s' % \
               ( intfRange, pg, peerSpec )
         intfRangeCmds.append( cmd )
   # Since we have to sort based on interface names and different interface
   # types (ethernet, vlans etc) can be grouped together as they have the
   # same asn and peer-group, we have to first construct the individual config
   # and then sort the final list of commands.
   intfRangeCmds.sort()
   cmds += intfRangeCmds
   return cmds

def saveUcmpConfig( routingHwStatus, ucmpConfig, defaultVrfUcmpConfig,
                    saveAll=False ):
   cmds = []
   isUcmpEnabledSet, ucmpEnabledVal = evaluateTristate( ucmpConfig,
                                                        defaultVrfUcmpConfig,
                                                        'ucmpEnabled' )
   isMaxUcmpSet, maxUcmpVal = evaluateValue( ucmpConfig, defaultVrfUcmpConfig,
                                             'maxUcmp',
                                             routingHwStatus.maxUcmp )
   isUcmpDeviationSet, ucmpDeviation = evaluateValue(
      ucmpConfig, defaultVrfUcmpConfig, 'ucmpDeviation',
      ucmpConfig.ucmpDeviationDefault )

   if isUcmpEnabledSet or saveAll:
      if ucmpEnabledVal:
         cmd = 'ucmp mode 1'
         if isMaxUcmpSet or saveAll:
            cmd += ' %s' % ( maxUcmpVal )
            if isUcmpDeviationSet or saveAll:
               cmd += ' %s' % ( ucmpDeviation )
      else:
         cmd = 'no ucmp mode'
      cmds.append( cmd )

   isUcmpLinkbwDelaySet, ucmpLinkbwDelayVal = evaluateValue( ucmpConfig,
                                                             defaultVrfUcmpConfig,
                                                             'ucmpLinkbwDelay' )
   if isUcmpLinkbwDelaySet or saveAll:
      cmds.append( 'ucmp link-bandwidth update-delay %s' % ucmpLinkbwDelayVal )

   isUcmpWeightedSet, ucmpWeightedVal = evaluateTristate( ucmpConfig,
                                                          defaultVrfUcmpConfig,
                                                          'ucmpLinkbwWeighted' )
   if isUcmpWeightedSet or saveAll:
      cmds.append( ( '' if ucmpWeightedVal else 'no ' ) +
                          'ucmp link-bandwidth encoding-weighted' )

   isUcmpTriggerFecThresholdSet, ucmpTriggerFecThresholdVal = evaluateValue(
                                   ucmpConfig, defaultVrfUcmpConfig,
                                   'ucmpTriggerFecThreshold' )
   isUcmpClearFecThresholdSet, ucmpClearFecThresholdVal = evaluateValue(
                                   ucmpConfig, defaultVrfUcmpConfig,
                                   'ucmpClearFecThreshold' )
   isUcmpRecursiveSet, ucmpRecursiveVal = \
      evaluateTristate( ucmpConfig, defaultVrfUcmpConfig, 'ucmpLinkbwRecursive' )
   isUcmpRecursiveUserWeightSet, ucmpRecursiveUserWeightVal = \
      evaluateTristate( ucmpConfig, defaultVrfUcmpConfig,
                        'ucmpLinkbwRecursiveUserWeight' )

   # When future protocols are added (see cli-review thread at
   # https://groups.google.com/a/arista.com/g/cli-review/c/LBuCGdczi1c), refactor
   # this section to add all enabled tokens to a base string and save only one
   # command if multiple protocols are enabled
   if isUcmpRecursiveSet:
      cmds.append( ( '' if ucmpRecursiveVal else 'no ' ) +
                   'ucmp link-bandwidth recursive' )
   elif isUcmpRecursiveUserWeightSet:
      cmds.append( ( '' if ucmpRecursiveUserWeightVal else 'no ' ) +
                   'ucmp link-bandwidth recursive user weight' )
   elif saveAll:
      # In this case, both recursive tristates are 'isInvalid'
      cmds.append( 'no ucmp link-bandwidth recursive' )

   if saveAll and not isUcmpTriggerFecThresholdSet and not\
          isUcmpClearFecThresholdSet:
      cmds.append( 'no ucmp fec threshold' )
   elif saveAll or ( ucmpTriggerFecThresholdVal !=
                        ucmpConfig.ucmpTriggerFecThresholdDefault or\
                        ucmpClearFecThresholdVal != \
                        ucmpConfig.ucmpClearFecThresholdDefault ):
      cmds.append( 'ucmp fec threshold trigger %s clear %s warning-only' %
                       ( ucmpTriggerFecThresholdVal, ucmpClearFecThresholdVal ) )
   return cmds

# pkgdeps : library MgmtSecurity

@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config',
      requireMounts=( 'routing/hardware/status',
                      'routing/bgp/asn/config', 'acl/cpconfig/cli',
                      'ip/vrf/routeDistinguisherInputDir/config',
                      'configTag/configTagIdState',
                      mgmtSecurityConfigPath ),
      commandTagSupported=True )
def saveConfig( bgpConfig, root, requireMounts, options ):
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   saveAll = options.saveAll
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   rdInputDir = requireMounts[ 'ip/vrf/routeDistinguisherInputDir/config' ]
   commandTagIdState = requireMounts[ 'configTag/configTagIdState' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )
   cmds = saveBgpConfig( bgpConfig, bgpConfig,
                         routingHwStatus, "",
                         securityConfig, aclCpConfig, commandTagIdState,
                         asdotConfigured, saveAll=saveAll )
   for cmd in cmds:
      bgpMode[ 'Bgp.config' ].addCommand( cmd )

   # Add ORR positions
   for orrPosition in six.itervalues( bgpConfig.orrNamedPosition ):
      orrMode = bgpMode[ BgpOrrPositionConfigMode ].\
         getOrCreateModeInstance( orrPosition.name )
      for cmd in saveOrrPositions( orrPosition ):
         orrMode[ 'Bgp.orrPosition.config' ].addCommand( cmd )

   # Save Rfd Policy
   for rfdPolicyName in sorted( bgpConfig.rfdPolicy ):
      rfdMode = bgpMode[ BgpRfdPolicyConfigMode ].\
            getOrCreateModeInstance( rfdPolicyName )
      rfdPolicy = bgpConfig.rfdPolicy[ rfdPolicyName ]
      for cmd in saveRfdPolicy( rfdPolicy, saveAll ):
         rfdMode[ 'Bgp.rfdPolicy.config' ].addCommand( cmd )

   # Save route-distinguisher config
   if bgpConfig.rdAutoUnifiedConfig != bgpConfig.rdAutoUnifiedConfigDefault:
      routeDistinguisherMode = bgpMode[ BgpRouteDistinguisherConfigMode ].\
            getOrCreateModeInstance( None )
      for cmd in saveRouteDistinguisherConfig( bgpConfig.rdAutoUnifiedConfig,
            bgpConfig.rdAutoUnifiedConfigDefault, saveAll ):
         routeDistinguisherMode[ 'Bgp.routeDistinguisher.config' ].addCommand( cmd )

   # Add router-bgp-vrf-default mode commands now. This mode only has VPN,
   # traffic-policy and vrf selection policy commands for now.
   vpnCmds = []
   rd = bgpConfig.routeDistinguisher
   rdRemoteDomainConfig = rdInputDir.get( 'bgpRemoteDomain' )
   if bgpConfig.rdAll:
      remoteRd = rdRemoteDomainConfig.routeDistinguisher[ DEFAULT_VRF ]
      assert rd != 'INVALID'
      assert rd == remoteRd
      vpnCmds.append( 'rd evpn domain all %s' % formatRd( rd, asdotConfigured ) )
   else:
      if rd != 'INVALID':
         vpnCmds.append( 'rd %s' % formatRd( rd, asdotConfigured ) )
      remoteRd = rdRemoteDomainConfig.routeDistinguisher.get( DEFAULT_VRF, None )
      if remoteRd is not None:
         vpnCmds.append( 'rd evpn domain remote %s' %
               formatRd( remoteRd, asdotConfigured ) )
   vpnCmds.extend( saveBgpVpnConfig( DEFAULT_VRF, bgpConfig, bgpConfig,
                                     asdotConfigured, saveAll ) )
   # Add this mode only if there is some config under it.
   if vpnCmds:
      bgpDefaultVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( DEFAULT_VRF )
      for cmd in vpnCmds:
         bgpDefaultVrfMode[ 'Bgp.vrf.config' ].addCommand( cmd )

   # Add config commands for router-bgp-vrf-default-af modes
   for addrFamily in BgpLib.bgpConfigAttrAfList( DEFAULT_VRF ):
      vpnCmds = saveBgpVpnAfConfig( bgpConfig, bgpConfig, addrFamily,
                                    asdotConfigured, saveAll )
      # Add AF VPN config only if there is some config under it
      if vpnCmds:
         # The default VRF mode may get created for the first time here if there was
         # no config outside address family mode
         bgpDefaultVrfMode = bgpMode[ RouterBgpVrfConfigMode
               ].getOrCreateModeInstance( DEFAULT_VRF )
         defaultVrfAfMode = bgpDefaultVrfMode[ RouterBgpVrfAfConfigMode
               ].getOrCreateModeInstance( ( addrFamily, DEFAULT_VRF ) )
         for cmd in vpnCmds:
            defaultVrfAfMode[ 'Bgp.vrf.afConfig.%s' % addrFamily ].addCommand( cmd )

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts = ( 'routing/bgp/config', 'routing/hardware/status', \
                                  'routing/bgp/asn/config',
                                  'acl/cpconfig/cli',
                                  'configTag/configTagIdState',
                                  mgmtSecurityConfigPath,
                                  'routing/ucmp/bgp/vrf/config',
                                  'routing/ucmp/bgp/config' ),
                commandTagSupported=True )
def saveVrfConfig( vrfConfigDir, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   commandTagIdState = requireMounts[ 'configTag/configTagIdState' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )
   for vrfName in vrfConfigDir.vrfConfig:
      vrfBgpConfig = vrfConfigDir.vrfConfig[ vrfName ]
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( vrfName )
      cmds = saveBgpConfig( vrfBgpConfig, bgpConfig,
                            routingHwStatus,
                            vrfName, securityConfig, aclCpConfig,
                            commandTagIdState, asdotConfigured,
                            saveAll=options.saveAll )
      for cmd in cmds:
         bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( cmd )

@CliSave.saver( 'Routing::Ucmp::UcmpConfig', 'routing/ucmp/bgp/config',
                requireMounts = ( 'routing/hardware/status',
                                  'routing/bgp/config',
                                  'routing/bgp/asn/config',
                                  mgmtSecurityConfigPath, ) )
def saveUcmpBgpConfig( ucmpConfig, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )
   cmds = saveUcmpConfig( routingHwStatus, ucmpConfig,
                          ucmpConfig, saveAll=options.saveAll )
   for cmd in cmds:
      bgpMode[ 'Bgp.config' ].addCommand( cmd )

@CliSave.saver( 'Routing::Ucmp::VrfUcmpConfigDir',
                 'routing/ucmp/bgp/vrf/config',
                requireMounts = ( 'routing/hardware/status',
                                  'routing/bgp/config',
                                  'routing/bgp/vrf/config',
                                  'routing/bgp/asn/config',
                                  'routing/ucmp/bgp/config' ) )
def saveVrfUcmpConfig( vrfConfigDir, root, requireMounts, options ):
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   ucmpConfig = requireMounts[ 'routing/ucmp/bgp/config' ]

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )
   for vrfName in vrfConfigDir.vrfConfig:
      vrfUcmpConfig = vrfConfigDir.vrfConfig[ vrfName ]
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( vrfName )
      cmds = saveUcmpConfig( routingHwStatus, vrfUcmpConfig,
                             ucmpConfig, saveAll=options.saveAll )
      for cmd in cmds:
         bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( cmd )

@CliSave.saver( 'Tac::Dir', 'ip/vrf/routeDistinguisherInputDir/config',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config',
                   'routing/bgp/vrf/config' ) )
def saveVrfRdConfig( entity, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   vrfConfigDir = requireMounts[ 'routing/bgp/vrf/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )
   bgpRouteDistinguisherInput = entity.get( 'bgp' )
   if not bgpRouteDistinguisherInput:
      return
   rdRemoteDomainConfig = entity.get( 'bgpRemoteDomain' )
   for vrfName in sorted( bgpRouteDistinguisherInput.routeDistinguisher ):
      if vrfName == DEFAULT_VRF:
         # For default VRF, we populate RD (and other VPN) config when Bgp config
         # ("routing/bgp/config") is loaded
         continue
      rd = bgpRouteDistinguisherInput.routeDistinguisher[ vrfName ]
      assert rd != 'INVALID'
      bgpVrfConfig = vrfConfigDir.vrfConfig.get( vrfName, None )
      if bgpVrfConfig is None:
         # BUG842636: the entry in bgpRouteDistinguisherInput.routeDistinguisher
         # doesn't have a matching entry in vrfConfigDir.vrfConfig.
         # This shouldn't happen, here we skip the processing to
         # prevent the Cli from failing.
         continue
      cmds = []
      if bgpVrfConfig.rdAll:
         remoteRd = rdRemoteDomainConfig.routeDistinguisher[ vrfName ]
         assert rd == remoteRd
         cmds.append( 'rd evpn domain all %s' % formatRd( rd, asdotConfigured ) )
      else:
         cmds.append( 'rd %s' % formatRd( rd, asdotConfigured ) )
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( vrfName )[ 'Bgp.vrf.config' ]
      for cmd in cmds:
         bgpVrfMode.addCommand( cmd )
   for vrfName in sorted( rdRemoteDomainConfig.routeDistinguisher ):
      if vrfName == DEFAULT_VRF:
         # For default VRF, we populate RD (and other VPN) config when Bgp config
         # ("routing/bgp/config") is loaded
         continue
      bgpVrfConfig = vrfConfigDir.vrfConfig.get( vrfName, None )
      if bgpVrfConfig is None:
         # BUG842636: the entry in bgpRouteDistinguisherInput.routeDistinguisher
         # doesn't have a matching entry in vrfConfigDir.vrfConfig.
         # This shouldn't happen, here we skip the processing to
         # prevent the Cli from failing.
         continue
      cmds = []
      if not bgpVrfConfig.rdAll:
         # BUG966763: 'show active' should display remote domain RD after 'no RD'
         remoteRd = rdRemoteDomainConfig.routeDistinguisher.get( vrfName, None )
         if remoteRd is not None:
            cmds.append( 'rd evpn domain remote %s' %
               formatRd( remoteRd, asdotConfigured ) )
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( vrfName )[ 'Bgp.vrf.config' ]
      for cmd in cmds:
         bgpVrfMode.addCommand( cmd )

def getLsImportCmds( lsImportCfg ):
   cmds = []
   for protoConfig in sorted( lsImportCfg.protoConfig.values() ):
      proto = protoConfig.proto
      if proto == 'protoIsis':
         proto = 'isis'
      if proto == 'protoDps':
         proto = 'path-selection'
      for instName in sorted( protoConfig.instanceCfg ):
         if proto == 'protoBgp':
            if not RoutingLibToggleLib.toggleBgpLsProducerBgpEpeEnabled():
               continue
            # BGP supports importing from a single default instance only.
            cmd = "bgp peer segments"
         elif proto == 'path-selection':
            # DPS supports importing from a single default instance only.
            cmd = "%s" % proto
         else:
            cmd = "%s instance %s" % ( proto, instName )
         instanceCfg = protoConfig.instanceCfg[ instName ]
         identifierPresent = instanceCfg.identifierPresent
         if identifierPresent:
            identifier = instanceCfg.identifier
            cmd += " identifier %lu" % identifier
         if proto == 'isis' and instanceCfg.twoWayCheckDisabled:
            cmd += " link connectivity-check disabled"
         cmds.append( cmd )
   return cmds

@CliSave.saver( 'Routing::Bgp::LinkStateImportConfig', 'routing/bgp/lsImportConfig',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config',
                                mgmtSecurityConfigPath, ) )
def saveLsImportConfig( lsImportCfg, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if not lsImportCfg.protoConfig:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode ].getOrCreateModeInstance(
         'link-state' )
   cmds = getLsImportCmds( lsImportCfg )
   for cmd in cmds:
      bgpAfMode[ 'Bgp.afConfig.link-state' ].addCommand( cmd )

def getLsRoleCmds( lsRoleCfg ):
   bgpLsProtoEnumToStringMap = {
      'protoDps': 'path-selection',
      'protoIsis': 'isis',
   }
   cmds = []
   for proto in sorted( lsRoleCfg.protoConfig.keys() ):
      protoRoleCfg = lsRoleCfg.protoConfig.get( proto )
      if not protoRoleCfg.explicitRoleEnabled:
         continue
      # only path-selection and isis support explicit role configuration
      if proto != 'protoDps' and proto != 'protoIsis':
         continue
      cmd = ' '.join( role
                for role in ( 'consumer', 'propagator' )
                if getattr( protoRoleCfg, role + 'Enabled' ) )
      if cmd:
         protoCliName = bgpLsProtoEnumToStringMap[ proto ]
         cmds.append( protoCliName + ' role ' + cmd )
   return cmds

@CliSave.saver( 'Routing::Bgp::LinkStateRoleConfig', 'routing/bgp/lsRoleConfig',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config',
                                mgmtSecurityConfigPath, ) )
def saveLsRoleConfig( lsRoleCfg, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if not bgpConfig.asNumber:
      return
   if not lsRoleCfg.protoConfig:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode ].getOrCreateModeInstance(
         'link-state' )
   cmds = getLsRoleCmds( lsRoleCfg )
   for cmd in cmds:
      bgpAfMode[ 'Bgp.afConfig.link-state' ].addCommand( cmd )

#------------------------------------------------------------------------------------
# address-family mode
#------------------------------------------------------------------------------------
class RouterBgpBaseAfConfigMode( RoutingBgpBaseAfMode, CliSave.Mode ):
   def __init__( self, param, label=False, srTe=False ):
      RoutingBgpBaseAfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

# Used to generate the address-family config level commands
RouterBgpBaseConfigMode.addChildMode( RouterBgpBaseAfConfigMode )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv4' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv6' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv4 labeled-unicast' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv6 labeled-unicast' )
# XXX_FIX_BGP_CORE_CLI_EVPN_CONFIG: Move this to Evpn package eventually ?
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.evpn' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv4 multicast' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv6 multicast' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv4 sr-te' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.ipv6 sr-te' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.link-state' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.path-selection' )

class RouterBgpVrfAfConfigMode( RoutingBgpVrfAfMode, CliSave.Mode ):
   def __init__( self, afAndVrfName ):
      af = afAndVrfName[ 0 ]
      vrfName = afAndVrfName[ 1 ]
      RoutingBgpVrfAfMode.__init__( self, af, vrfName )
      CliSave.Mode.__init__( self, af )

RouterBgpVrfConfigMode.addChildMode( RouterBgpVrfAfConfigMode )
RouterBgpVrfAfConfigMode.addCommandSequence( 'Bgp.vrf.afConfig.ipv4' )
RouterBgpVrfAfConfigMode.addCommandSequence( 'Bgp.vrf.afConfig.ipv6' )
RouterBgpVrfAfConfigMode.addCommandSequence( 'Bgp.vrf.afConfig.ipv4 multicast' )
RouterBgpVrfAfConfigMode.addCommandSequence( 'Bgp.vrf.afConfig.ipv6 multicast' )

RouterBgpBaseConfigMode.addChildMode( BgpOrrPositionConfigMode,
                                      before=[ RouterBgpVrfConfigMode,
                                               RouterBgpBaseAfConfigMode ] )
BgpOrrPositionConfigMode.addCommandSequence( 'Bgp.orrPosition.config' )

RouterBgpBaseConfigMode.addChildMode( BgpRfdPolicyConfigMode )
BgpRfdPolicyConfigMode.addCommandSequence( 'Bgp.rfdPolicy.config' )

RouterBgpBaseConfigMode.addChildMode( BgpRouteDistinguisherConfigMode )
BgpRouteDistinguisherConfigMode.addCommandSequence( 'Bgp.routeDistinguisher.config' )

def savePeerAfActivation( peer, cmds, key, afConfig, addrFamily, saveAll, nedMode ):
   if afConfig == peerAfStateEnum.afActive:
      cmds.append( 'neighbor %s activate' % key )
   elif afConfig == peerAfStateEnum.afInactive:
      if nedMode:
         cmds.append( 'neighbor %s deactivate' % key )
      else:
         cmds.append( 'no neighbor %s activate' % key )
   elif saveAll:
      cmds.append( 'default neighbor %s activate' % key )

def savePeerPerAfRouteMap( peer, cmds, key, addrFamily, bgpConfig, saveAll,
                           nedMode ):

   for direction in [ "In", "Out" ]:
      attr = "routeMap%s" % direction
      attrName = BgpLib.peerConfigAttrsAfMap[ attr ].get( addrFamily )
      if not attrName:
         return

      attrNamePresent = attrName + "Present"

      if getattr( peer, attrNamePresent ):
         if getattr( peer, attrName ):
            cmd = 'neighbor %s route-map %s %s' \
                     % ( key, getattr( peer, attrName ), direction.lower() )
         else:
            cmd = getExplicitNoCmd( 'neighbor %s route-map %s' \
                                       % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )
      elif saveAll and not peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( 'neighbor %s route-map %s' \
                                    % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )

def saveLfibEntryInstallationSkippedCmd( cmds, bgpConfig, addrFamily, saveAll ):
   if addrFamily == 'ipv4 labeled-unicast':
      attrName = 'lfibSkipIPv4LabeledUni'
   elif addrFamily == 'ipv6 labeled-unicast':
      attrName = 'lfibSkipIPv6LabeledUni'
   else:
      return
   skipped = getattr( bgpConfig, attrName )
   if skipped:
      cmds.append( 'lfib entry installation skipped' )
   elif saveAll:
      cmds.append( 'no lfib entry installation skipped' )

def saveLabelLocalTerminationExplicitNullCmd( cmds, bgpConfig, addrFamily, saveAll ):
   if addrFamily == 'ipv4 labeled-unicast':
      attrName = 'labelLocalTerminationModeAfV4LabeledUni'
   elif addrFamily == 'ipv6 labeled-unicast':
      attrName = 'labelLocalTerminationModeAfV6LabeledUni'
   else:
      return
   nullLabel = getattr( bgpConfig, attrName )
   if nullLabel == LuLabelLocalTerminationModeEnum.explicitNull:
      cmds.append( 'label local-termination explicit-null' )
   if saveAll and nullLabel == LuLabelLocalTerminationModeEnum.implicitNull:
      cmds.append( 'label local-termination implicit-null' )

def savePeerPerAfRcf( peer, cmds, key, addrFamily, bgpConfig, saveAll,
                     nedMode ):
   directions = [ "In", "Out" ]
   for direction in directions:
      attr = "rcf%s" % direction
      attrName = BgpLib.peerConfigAttrsAfMap[ attr ].get( addrFamily )
      if not attrName:
         return
      attrNamePresent = attrName + "Present"

      if getattr( peer, attrNamePresent ):
         funcName = getattr( peer, attrName )
         if funcName:
            funcName += "()"
            cmd = 'neighbor %s rcf %s %s' \
                     % ( key, direction.lower(), funcName )
         else:
            cmd = getExplicitNoCmd( 'neighbor %s rcf %s'
                     % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )
      elif saveAll and not peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( 'neighbor %s rcf %s'
                                    % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )

def savePeerPerAfPrefixList( peer, cmds, key, addrFamily, bgpConfig, saveAll,
                             afCfgPresent, nedMode ):

   for direction in [ "In", "Out" ]:
      attr = "prefixList%s" % direction
      attrName = BgpLib.peerConfigAttrsAfMap[ attr ].get( addrFamily )
      attrNamePresent = attrName + "Present"

      if getattr( peer, attrNamePresent ):
         if getattr( peer, attrName ):
            cmd = 'neighbor %s prefix-list %s %s' \
                     % ( key, getattr( peer, attrName ), direction.lower() )
         else:
            cmd = getExplicitNoCmd( 'neighbor %s prefix-list %s' \
                                       % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )
      elif saveAll and not peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( 'neighbor %s prefix-list %s' \
                                    % ( key, direction.lower() ), nedMode )
         cmds.append( cmd )

def savePeerAfDefOrigin( peer, cmds, key, addrFamily, saveAll, nedMode ):

   attrBase = \
         BgpLib.peerConfigAttrsAfMap[ 'defaultOriginate' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   attrRouteMap = attrBase + 'RouteMap'
   attrRcf = attrBase + 'Rcf'
   attrAlways = attrBase + 'Always'
   if getattr( peer, attrPresent ):
      if addrFamily in vpnAfTypeCliKeywords:
         cmd = "neighbor %s default-route" % key
      else:
         cmd = "neighbor %s default-originate" % key
      if getattr( peer, attrBase ):
         rm = getattr( peer, attrRouteMap )
         rcf = getattr( peer, attrRcf )
         if rm:
            cmd += ' route-map %s' % rm
         if rcf:
            cmd += ' rcf %s()' % rcf
         if getattr( peer, attrAlways ):
            cmd += ' always'
      else:
         if addrFamily in vpnAfTypeCliKeywords:
            cmd += ' disabled'
         else:
            cmd = getExplicitNoCmd( cmd, nedMode )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      if addrFamily in vpnAfTypeCliKeywords:
         cmd = 'no neighbor %s default-route' % key
         cmds.append( cmd )
      else:
         cmds.append( getExplicitNoCmd( "neighbor %s default-originate" % key,
                                        nedMode ) )

def savePeerAfGracefulRestart( peer, cmds, key, addrFamily, saveAll, nedMode ):

   attrBase = BgpLib.peerConfigAttrsAfMap[ 'gracefulRestart' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      cmd = "neighbor %s graceful-restart" % key
      if not getattr( peer, attrBase ):
         cmd = getExplicitNoCmd( cmd, nedMode )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      if addrFamily in ( 'ipv4 multicast', 'ipv6 multicast', 'link-state' ):
         # Graceful restart is currently not supported for IPv4/IPv6 multicast,
         # or Link State modes.
         # Once the check for ipv4/ipv6 multicast address-family is removed from
         # RoutingBgpCli.configureNeighborGracefulRestart(), this if block can
         # be removed.
         return
      if addrFamily == 'path-selection' and \
            not BgpCommonToggleLib.toggleHandlePfLossEnabled():
         # support for 'path-selection' address family is under feature toggle
         # once feature toggle is removed we can removed this check.
         return
      cmds.append( f"default neighbor {key} graceful-restart" )

def savePeerAfGrHelperRouteMap( peer, cmds, key, addrFamily, saveAll, nedMode ):

   attrBase = BgpLib.peerConfigAttrsAfMap[ 'grHelperRouteMap' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   cmdStr = "neighbor %s graceful-restart-helper stale-route route-map %s"
   if getattr( peer, attrPresent ):
      cmd = cmdStr % ( key, getattr( peer, attrBase ) )
      attrBase = BgpLib.peerConfigAttrsAfMap[ 'grHelperRmGrOptional' ].get(
            addrFamily )
      attrPresent = attrBase + 'Present'
      if getattr( peer, attrPresent ):
         cmd += " session-failure graceful-restart-negotiation optional"
      cmds.append( cmd )


def savePeerAfOrr( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'orr' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      cmd = 'neighbor %s' % key
      cmd += ' route-reflector-client optimal-route-reflection'
      attr = getattr( peer, attrBase )
      assert attr is not None
      if attr:
         if attr.peerAddress:
            cmd += ' position peer-address'
         else:
            cmd += ' position ' + attr.positionName
         if addrFamily == 'evpn':
            cmd += ' route-type ip-prefix'
      else:
         cmd = getExplicitNoCmd( cmd, nedMode )
      cmds.append( cmd )

def savePeerAfRfd( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'rfdPolicyAttachment' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      cmd = 'neighbor {0} route-flap-damping '.format( key )
      rfdPolicy = getattr( peer, attrBase )
      rfdPolicyDefault = getattr( peer, attrBase + "Default" )
      if rfdPolicy == rfdPolicyDefault:
         cmd += "disabled"
      else:
         cmd += rfdPolicy.rfdPolicyName
      if rfdPolicy.rfdMonitorOnly:
         cmd += " monitor-only"
      if rfdPolicy.rfdLogPathEvents or rfdPolicy.rfdLogConfigEvents:
         cmd += " log"
         if rfdPolicy.rfdLogPathEvents:
            cmd += " path-events"
         if rfdPolicy.rfdLogConfigEvents:
            cmd += " config-events"
      cmds.append( cmd )
   elif saveAll:
      cmd = getExplicitNoCmd( 'neighbor {0} route-flap-damping '.format( key ),
                              nedMode )
      cmds.append( cmd )

def savePeerAfAddPathRecv( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'apRecv' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmd = "neighbor %s additional-paths receive" % key
      else:
         cmd = getExplicitNoCmd( "neighbor %s additional-paths receive" \
                                    % key, nedMode )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      # Add-path receive is enabled by default
      cmds.append( "neighbor %s additional-paths receive" % key )

# Add-path send command
def savePeerAfAddPathSend( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'apSend' ].get( addrFamily )
   apSendConfig = getattr( peer, attrBase )
   cmd = ''
   for app in apSendConfig:
      appConfig = apSendConfig.get( app )
      if appConfig.enable:
         cmd = 'neighbor %s %s' \
                  % ( key, addPathSendCommon( app, appConfig=appConfig ) )
      else:
         #  app must be 'appNone' or 'appAny'
         cmd = getExplicitNoCmd( 'neighbor %s %s'
                  % ( key, addPathSendCommon( app, appConfig=appConfig ) ),
                  nedMode )
      # There can be only one app present
      break
   if saveAll and not peer.isPeerGroupPeer and cmd == '':
      app = 'appNone'
      cmd = getExplicitNoCmd( 'neighbor %s %s' \
               % ( key, addPathSendCommon( app, saveAll=saveAll ) ), nedMode )
   if cmd != '':
      cmds.append( cmd )

def savePeerAfNexthopUnchanged( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = \
         BgpLib.peerConfigAttrsAfMap[ 'nexthopUnchanged' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmd = "neighbor %s next-hop-unchanged" % key
      else:
         cmd = getExplicitNoCmd( "neighbor %s next-hop-unchanged" % key, nedMode )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      # disabled by default
      cmds.append( getExplicitNoCmd( "neighbor %s next-hop-unchanged" % key,
                                     nedMode ) )

def savePeerAfNexthopSelf( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = \
          BgpLib.peerConfigAttrsAfMap[ 'nextHopSelf' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmd = "neighbor %s next-hop-self" % key
      else:
         cmd = getExplicitNoCmd( "neighbor %s next-hop-self" % key, nedMode )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      # disabled by default
      cmds.append( getExplicitNoCmd( "neighbor %s next-hop-self" % key,
                                     nedMode ) )

def savePeerAfNhSelfSrcIntf( peer, cmds, key, addrFamily, saveAll, nedMode ):
   if type( key ) is Ip6AddrType: # pylint: disable=unidiomatic-typecheck
      key = key.stringValue
   attr = BgpLib.peerConfigAttrsAfMap[ 'nhSelfSrcIntf' ].get( addrFamily )
   v4MappedV6Str = ""
   if addrFamily == 'ipv6 labeled-unicast':
      v4MappedV6Attr = BgpLib.peerConfigAttrsAfMap[ 'nhSelfV4MappedV6' ].get(
            addrFamily )
      if getattr( peer, v4MappedV6Attr ):
         v4MappedV6Str = 'v4-mapped-v6 '

   if getattr( peer, '%sPresent' % attr ):
      val = getattr( peer, attr )
      if val != getattr( peer, 'nhSelfSrcIntfDefault' ):
         cmd = "neighbor %s next-hop-self %ssource-interface %s" % ( key,
               v4MappedV6Str, val )
      elif peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( "neighbor %s next-hop-self %ssource-interface"
                  % ( key, v4MappedV6Str ), nedMode )
      else:
         return
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      # disabled by default
      cmds.append( getExplicitNoCmd( "neighbor %s next-hop-self %ssource-interface"
            % ( key, v4MappedV6Str ), nedMode ) )

def savePeerAfWeight( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'weight' ].get( addrFamily )
   attrDefault = attrBase + 'Default'
   attrPresent = attrBase + 'Present'
   defaultWeight = getattr( peer, attrDefault )
   if getattr( peer, attrPresent ):
      if ( getattr( peer, attrBase ) == defaultWeight ) and peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( "neighbor %s weight" % key, nedMode,
                                 defaultValue=defaultWeight )
      else:
         cmd = "neighbor %s weight %s" % ( key, getattr( peer, attrBase ) )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      # Per-neighbor weight cannot be configured both globally and per address
      # family. If it is not configured at the global level but it is configured at
      # some address-family level, we should render the default neighbor weight cmd
      # for save-all for all other address-families.
      if weightConfiguredInAf( peer ):
         cmds.append( getExplicitNoCmd( "neighbor %s weight" % key, nedMode,
                                        defaultValue=defaultWeight ) )

def savePeerAfOutDelay( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'outDelay' ].get( addrFamily )
   attrDefault = attrBase + 'Default'
   attrPresent = attrBase + 'Present'
   defaultOutDelay = getattr( peer, attrDefault )
   if getattr( peer, attrPresent ):
      if ( getattr( peer, attrBase ) == defaultOutDelay ) and peer.isPeerGroupPeer:
         cmd = getExplicitNoCmd( "neighbor %s out-delay" % key, nedMode,
                                 defaultValue=defaultOutDelay )
      else:
         cmd = "neighbor %s out-delay %s" % ( key, getattr( peer, attrBase ) )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      # Per-neighbor out-delay cannot be configured both globally and per address
      # family. If it is not configured at the global level but it is configured at
      # some address-family level, we should render the default neighbor outdelay cmd
      # for save-all for all other address-families.
      if outDelayConfiguredInAf( peer ):
         cmds.append( getExplicitNoCmd( "neighbor %s out-delay" % key, nedMode,
                                        defaultValue=defaultOutDelay ) )

def savePeerRouteTargetExportFilterDisabled( peer, cmds, key, addrFamily, saveAll ):
   if addrFamily not in [ 'ipv4', 'ipv6', 'all' ]:
      return
   if addrFamily == 'all':
      coll = peer.routeTargetExportFilterDisabled
   else:
      coll = peer.routeTargetExportFilterDisabledAf
   vpnNlriTypeList = [ NlriType( n ) for n in
                       [ 'evpnType5Ipv4', 'evpnType5Ipv6',
                        'mplsVpnIpv4Unicast', 'mplsVpnIpv6Unicast' ] ]
   for vpnNlriType in vpnNlriTypeList:
      tristate = coll.get( vpnNlriType, 'isInvalid' )
      if addrFamily != 'all':
         # AF collection is used for both v4 and v6. Skip entries not
         # belonging to this AF.
         ipVersion = int( addrFamily[ -1 ] )
         if vpnNlriType.ipVersion != ipVersion:
            continue
      if tristate == 'isInvalid' and ( not saveAll or peer.isPeerGroupPeer ):
         continue
      cmds.append( 'neighbor %s route-target export %s filter%s' %
                   ( key, vpnNlriTypeKeyword( vpnNlriType ),
                     ' disabled' if tristate == 'isTrue' else '' ) )

def savePeerAfNextHopAfV6( peer, cmds, key, saveAll, nedMode ):
   attrBase = 'v6NextHopAfV4Uni'
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ) == \
                              ExtendedNextHopCapabilityEnum.isEnabledWithOriginate:
         cmds.append( 'neighbor %s next-hop address-family ipv6 originate' % key )
      elif getattr( peer, attrBase ) == ExtendedNextHopCapabilityEnum.isEnabled:
         cmds.append( 'neighbor %s next-hop address-family ipv6' % key )
      else:
         cmds.append( getExplicitNoCmd( 'neighbor %s next-hop address-family ipv6' \
                                           % key, nedMode ) )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( getExplicitNoCmd( 'neighbor %s next-hop address-family ipv6' \
                                        % key, nedMode ) )

def savePeerSixPeActivation( peer, cmds, addrOrPgName, afConfig, addrFamily,
                             saveAll, nedMode ):
   if peer.sixPe:
      if peer.afV6Uni == 'afActive':
         cmds.append(
            "neighbor %s activate 6pe ipv6-unicast receive" % addrOrPgName )
      else:
         cmds.append( "neighbor %s activate 6pe" % addrOrPgName )
   else:
      savePeerAfActivation( peer, cmds, addrOrPgName, afConfig, addrFamily,
                            saveAll, nedMode )

def savePeerMaxRoutes( peer, cmds, addrOrPgName, saveAll, nedMode,
                       addrFamily='all' ):
   attr = BgpLib.peerConfigAttrsAfMap[ 'maxRoutesConfig' ].get( addrFamily )
   if attr is not None:
      maxRoutesConfig = getattr( peer, attr )
      maxRoutesConfigDefault = getattr( peer, attr + 'Default' )
      maxRoutesConfigPresent = getattr( peer, attr + 'Present' )
   else:
      # Additional address families are added via the maxRoutesAfiSafiConfig
      afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                           *BgpLib.addressFamilies.get( addrFamily ) )
      maxRoutesConfig = peer.maxRoutesAfiSafiConfig.get( afiSafi )
      maxRoutesConfigDefault = peer.afiSafiMaxRoutesConfigDefault
      maxRoutesConfigPresent = ( maxRoutesConfig is not None )

   if ( maxRoutesConfigPresent or
         ( saveAll and not peer.isPeerGroupPeer and addrFamily == 'all' ) ):
      maxRoutesCmd = 'neighbor %s maximum-routes %d' % ( addrOrPgName,
                                                         maxRoutesConfig.maxRoutes )
      if maxRoutesConfig.maxRoutesThresholdPercentagePresent:
         valueType = ' percent'
         threshold = str( maxRoutesConfig.maxRoutesThresholdPercentage )
      else:
         valueType = ''
         threshold = str( maxRoutesConfig.maxRoutesThreshold )
      if threshold != '0':
         maxRoutesCmd += ' warning-limit %s%s' % ( threshold, valueType )
      if ( maxRoutesConfig.maxRoutesWarningOnly !=
               maxRoutesConfigDefault.maxRoutesWarningOnly ):
         maxRoutesCmd += ' warning-only'

      cmds.append( maxRoutesCmd )

   attr = BgpLib.peerConfigAttrsAfMap[ 'maxAcceptedRoutesConfig' ].get(
      addrFamily )
   if attr is not None:
      maxAccRoutesConfigPresent = getattr( peer, attr + 'Present' )
      maxAccRoutesConfig = getattr( peer, attr )
      maxAccRoutesConfigDefault = getattr( peer, attr + 'Default' )
   else:
      # Additional address families are added via the maxAcceptedRoutesAfiSafiConfig
      afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                           *BgpLib.addressFamilies.get( addrFamily ) )
      maxAccRoutesConfig = peer.maxAcceptedRoutesAfiSafiConfig.get( afiSafi )
      maxAccRoutesConfigDefault = peer.maxAcceptedRoutesConfigDefault
      maxAccRoutesConfigPresent = ( maxAccRoutesConfig is not None )
   saveAll = saveAll and addrFamily == 'all'
   if maxAccRoutesConfigPresent or ( saveAll and not peer.isPeerGroupPeer ):
      numRoutes = maxAccRoutesConfig.maxAcceptedRoutes
      threshold = maxAccRoutesConfig.maxAcceptedRoutesThreshold
      cmd = 'neighbor %s maximum-accepted-routes' % addrOrPgName
      if ( not maxAccRoutesConfigPresent or
           ( numRoutes == maxAccRoutesConfigDefault.maxAcceptedRoutes and
             threshold == maxAccRoutesConfigDefault.maxAcceptedRoutesThreshold and
             peer.isPeerGroupPeer ) ):
         cmd = getExplicitNoCmd( cmd, nedMode,
                                 maxAccRoutesConfigDefault.maxAcceptedRoutes )
      else:
         threshold = ' warning-limit %d' % threshold if \
            threshold != maxAccRoutesConfigDefault.maxAcceptedRoutesThreshold else ''
         cmd += ' %s%s' % ( numRoutes, threshold )

      cmds.append( cmd )

def savePeerMaxAdvRoutes( peer, cmds, addrOrPgName, saveAll, nedMode,
                       addrFamily='all' ):
   attr = 'maxAdvRoutesConfig'
   if addrFamily == 'all':
      present = getattr( peer, attr + 'Present' )
   else:
      afiSafi = afiSafiCache.getAfiSafi( addrFamily )
      present = afiSafi in peer.maxAdvRoutesAfiSafiConfig
   saveAll = saveAll and addrFamily == 'all'
   if present or saveAll:
      config = getattr( peer, attr ) if addrFamily == 'all' else \
               peer.maxAdvRoutesAfiSafiConfig.get( afiSafi )
      configDefault = getattr( peer, attr + 'Default' )
      numRoutes = config.maxAdvRoutes
      threshold = config.maxAdvRoutesThreshold
      cmd = 'neighbor %s maximum-advertised-routes' % addrOrPgName
      if not present:
         # Hardcode nedMode=True until BUG433700 is resolved
         cmd = getExplicitNoCmd( cmd, True, configDefault.maxAdvRoutes )
      else:
         cmd += ' %s' % numRoutes
         if config.maxAdvRoutesThresholdPercentagePresent:
            valueType = ' percent'
            threshold = str( config.maxAdvRoutesThresholdPercentage )
         else:
            valueType = ''
            threshold = str( config.maxAdvRoutesThreshold )
         if threshold != '0':
            cmd += ' warning-limit %s%s' % ( threshold, valueType )

      cmds.append( cmd )

class AfiSafisCache( object ): # pylint: disable=useless-object-inheritance
   slots = ( 'afiSafis', )

   def __init__( self ):
      self.afiSafis = {}

   def getAfiSafi( self, addrFamily ):
      afiSafi = self.afiSafis.get( addrFamily )
      if afiSafi is None:
         afiSafi = Tac.Value( "Routing::Bgp::AfiSafi",
               *addressFamilies[ addrFamily ] )
         self.afiSafis[ addrFamily ] = afiSafi

      return afiSafi

afiSafiCache = AfiSafisCache()

def savePeerMissingPolicy( peer, cmds, addrOrPgName, saveAll, nedMode,
                           addrFamily='all' ):
   # missing-policy at instance level
   if addrFamily == 'all':
      baseConfig = peer # No base config to dervie and set to itself
      for direction in [ "In", "Out" ]:
         tok = [ 'missing-policy', 'address-family', 'all',
                 'direction', direction.lower(), 'action' ]
         isSet, val = evaluateValue(
            peer, baseConfig, 'missingPolicyAction' + direction )
         subRmSet, subRmVal = evaluateTristate(
            peer, baseConfig, 'missingPolicyIncludeSubRouteMap' + direction )
         pfxListSet, pfxListVal = evaluateTristate(
            peer, baseConfig, 'missingPolicyIncludePrefixList' + direction )
         commListSet, commListVal = evaluateTristate(
            peer, baseConfig, 'missingPolicyIncludeCommList' + direction )
         if isSet:
            tok.append( policyActions[ val ] )
            if ( ( subRmSet and subRmVal ) or ( pfxListSet and pfxListVal ) or
                 ( commListSet and commListVal ) ):
               # Insert include keywords into the command token list alphabetically
               includeTokens = [ 'include' ]
               if commListSet and commListVal:
                  includeTokens.append( 'community-list' )
               if pfxListSet and pfxListVal:
                  includeTokens.append( 'prefix-list' )
               if subRmSet and subRmVal:
                  includeTokens.append( 'sub-route-map' )
               tok[ 3 : 3 ] = includeTokens
            missingPolicyCommand = 'neighbor %s ' % addrOrPgName + ' '.join( tok )
            cmds.append( missingPolicyCommand )
         elif saveAll and not peer.isPeerGroupPeer:
            missingPolicyCommand = 'no neighbor %s ' % addrOrPgName + ' '.join( tok )
            cmds.append( missingPolicyCommand )
      return

   # missing-policy at AF level
   # Some address families like rt-membership will not influence
   # missing policy, so do not CliSave the missing policy command
   skipMissingPolicyAf = [ 'rt-membership', 'path-selection', 'mvpn-ipv4',
                           'flow-spec vpn-ipv4', 'flow-spec vpn-ipv6' ]
   for direction in [ "In", "Out" ]:
      tok = [ 'missing-policy', 'direction', direction.lower(), 'action' ]
      actionAfAttr = 'missingPolicyActionAf' + direction
      actionIncludeSubAfAttr = 'missingPolicyIncludeSubRouteMapAf' + direction
      actionIncludePfxAfAttr = 'missingPolicyIncludePrefixListAf' + direction
      actionIncludeCommAfAttr = 'missingPolicyIncludeCommListAf' + direction
      actionAfConfig = getattr( peer, actionAfAttr )
      actionIncludeSubAfConfig = getattr( peer, actionIncludeSubAfAttr )
      actionIncludePfxAfConfig = getattr( peer, actionIncludePfxAfAttr )
      actionIncludeCommAfConfig = getattr( peer, actionIncludeCommAfAttr )
      af = afiSafiCache.getAfiSafi( addrFamily )
      if af in actionAfConfig:
         tok.append( policyActions[ actionAfConfig[ af ] ] )
         insertPos = 1
         if 'isTrue' in ( actionIncludeSubAfConfig.get( af ),
                          actionIncludePfxAfConfig.get( af ),
                          actionIncludeCommAfConfig.get( af ) ):
            # insert include keywords into the command token list alphabetically
            includeTokens = [ 'include' ]
            if actionIncludeCommAfConfig.get( af ) == 'isTrue':
               includeTokens.append( 'community-list' )
            if actionIncludePfxAfConfig.get( af ) == 'isTrue':
               includeTokens.append( 'prefix-list' )
            if actionIncludeSubAfConfig.get( af ) == 'isTrue':
               includeTokens.append( 'sub-route-map' )
            tok[ insertPos : insertPos ] = includeTokens
         missingPolicyCommand = 'neighbor %s ' % addrOrPgName + ' '.join( tok )
         cmds.append( missingPolicyCommand )
      elif ( saveAll and addrFamily not in skipMissingPolicyAf ):
         missingPolicyCommand = 'no neighbor %s ' % addrOrPgName + ' '.join( tok )

def savePeerAfAigpSession( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'aigpSession' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmd = "neighbor %s aigp-session" % key
      else:
         cmd = getExplicitNoCmd( "neighbor %s aigp-session" % key, nedMode )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( "default neighbor %s aigp-session" % key )

def savePeerAfNexthopLuOriginate( peer, cmds, addrOrPgName, saveAll, nedMode,
                                  addrFamily ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuOriginate' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   dynAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuOriginateDyn' ].get(
         addrFamily )
   cmd = "neighbor %s next-hop labeled-unicast originate" % addrOrPgName
   if getattr( peer, attrPresent ):
      if getattr( peer, dynAttr ):
         cmd += " third-party"
      elif not getattr( peer, attrBase ):
         cmd = "%s disabled" % cmd
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( "default %s" % cmd )

def savePeerAfNexthopLuPeerSetOriginate( peer, cmds, addrOrPgName, saveAll, nedMode,
                                  addrFamily ):
   peerSetAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuPeerSet' ].\
      get( addrFamily )
   peerSetOrigAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuPeerSetOriginate' ].\
      get( addrFamily )
   peerSet = getattr( peer, peerSetAttr )
   cmd = "neighbor %s next-hop labeled-unicast peer-set %s originate"\
         % ( addrOrPgName, peerSet )
   if getattr( peer, peerSetAttr + 'Present' ):
      # pylint: disable-next=singleton-comparison
      if getattr( peer, peerSetOrigAttr ) == False:
         cmd = "%s disabled" % cmd
      cmds.append( cmd )

def savePeerAfNhLuOrigLfibBackupIpFwd( peer, cmds, addrOrPgName, saveAll,
                                       addrFamily ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'nhLuOrigLfibBackupIpFwd' ].get(
                                                                       addrFamily )
   attrPresent = attrBase + 'Present'
   cmd = "neighbor %s next-hop labeled-unicast originate lfib-backup ip-forwarding"\
         % addrOrPgName
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmds.append( cmd )
      else:
         cmds.append( "%s disabled" % cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( "default %s" % cmd )

def savePeerFwdFailoverTriggerSession( peer, cmds, addrOrPgName, saveAll,
                                         nedMode, addrFamily ):
   attr = 'fwdFailoverTriggerSession'
   attrBase = BgpLib.peerConfigAttrsAfMap[ attr ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   cmd = "neighbor %s forwarding failover trigger session" % addrOrPgName
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmds.append( cmd )
      else:
         cmds.append( "%s disabled" % cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( "default %s" % cmd )

def savePeerAfV4MappedV6NextHopTranslation( peer, cmds, addrOrPgName, addrFamily,
                                            saveAll ):
   attr = 'v4MappedV6NextHopTranslation'
   attrBase = BgpLib.peerConfigAttrsAfMap[ attr ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   cmd = "neighbor %s next-hop resolution v4-mapped-v6 translation" % addrOrPgName
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmds.append( cmd )
      else:
         cmds.append( "%s disabled" % cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( "default %s" % cmd )

def savePeerAfRejectAsSets( peer, cmds, key, addrFamily, saveAll, nedMode ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'rejectAsSets' ].get( addrFamily )
   attrPresent = attrBase + 'Present'

   cmd = f"neighbor {key} path-attribute as-path as-sets treat-as-withdraw"
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmds.append( cmd )
      else:
         cmds.append( getExplicitNoCmd( cmd, nedMode ) )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( getExplicitNoCmd( cmd, nedMode ) )

def savePeerMultipath( peer, cmds, addrOrPgName, addrFamily, saveAll ):
   attrBase = BgpLib.peerConfigAttrsAfMap[ 'multipath' ].get( addrFamily )
   attrPresent = attrBase + 'Present'
   if getattr( peer, attrPresent ):
      if getattr( peer, attrBase ):
         cmd = 'neighbor {} multi-path'.format( addrOrPgName )
      else:
         cmd = 'neighbor {} multi-path disabled'.format( addrOrPgName )
      cmds.append( cmd )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( 'default neighbor {} multi-path'.format( addrOrPgName ) )

def savePeerSegmentNode( vrfName, peer, cmds, addrOrPgName, saveAll ):
   """ Save the (BGP-LS) EPE PeerNode segment config (see AID11677).
   """
   if not RoutingLibToggleLib.toggleBgpLsProducerBgpEpeEnabled():
      return

   # This command only applies to the default vrf.
   if vrfName and vrfName != DEFAULT_VRF:
      return

   # This command does not apply to IPv6 link-local peers.
   if peer.key.type == 'peerIpv6Ll':
      return

   attr = 'linkStateEpePeerSegmentNode'
   attrPresent = f'{attr}Present'

   cmd = f'neighbor {addrOrPgName} peer segment node'
   if getattr( peer, attrPresent ):
      if getattr( peer, attr ):
         cmds.append( cmd )
      else:
         cmds.append( f'{cmd} disabled' )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( f'default {cmd}' )

def saveRibInDelay( vrfName, peer, cmds, addrOrPgName, saveAll ):
   """ Save the Rib-in delay.
   """
   if not BgpCommonToggleLib.toggleBgpInDelayEnabled():
      return

   cmd = f'neighbor {addrOrPgName} rib-in delay'
   attr = 'ribInDelay'
   ribInDelayPresent = getattr( peer, f'{attr}Present' )
   if ribInDelayPresent:
      ribInDelay = getattr( peer, attr )
      cmds.append( f'{cmd} {ribInDelay} event peer-init' )
   elif saveAll and not peer.isPeerGroupPeer:
      cmds.append( f'default {cmd}' )

# saveAfPeerCallbackDict is a dictionary keyed by address family.
# Entries are lists of callbacks.
# Each callback has the signature fn( cmds, peer, saveAll, addrOrPgName )
saveAfPeerConfigCallbackDict = {}

def registerAfPeerConfigCallback( af, callback ):
   saveAfPeerConfigCallbackDict[ af ] = callback

def savePeerAfConfig( cmds, bgpConfig, defaultVrfBgpConfig, peerKey, saveAll,
                      addrFamily, afCfgPresent ):
   nedMode = ( getattr( defaultVrfBgpConfig, 'noEqualsDefaultMode' ) or
               addrFamily in alwaysNedModeAfs )

   addrOrPgName = peerKey.stringValue

   peer = bgpConfig.neighborConfig.get( peerKey )
   if not peer and saveAll:
      peer = bgpConfig.defaultPeerConfig

   if peer:
      afConfigAttrName = BgpLib.peerConfigAttrsAfMap[ 'af' ].get( addrFamily )
      afConfig = getattr( peer, afConfigAttrName )

      if addrFamily == 'ipv6' and peerKey.type in [ 'peerIpv4', 'peerGroup' ]:
         savePeerSixPeActivation( peer, cmds, addrOrPgName, afConfig, addrFamily,
                                  saveAll, nedMode )
      elif addrFamily == 'mvpn-ipv4' and peerKey.type in [ 'peerIpv6',
                                                           'peerIpv6Ll' ]:
         t2( 'No IPv6 transport peer support for mvpn-ipv4' )
      else:
         savePeerAfActivation( peer, cmds, addrOrPgName, afConfig, addrFamily,
                               saveAll, nedMode )

      # Only neighbor activate and neighbor route-map in command are supported
      # in sr-te sub-mode
      if 'sr-te' in addrFamily:
         if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'routeMapIn' ]:
            savePeerPerAfRouteMap( peer, cmds, addrOrPgName, addrFamily, bgpConfig,
                                   saveAll, nedMode )
         if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'grHelperRouteMap' ]:
            savePeerAfGrHelperRouteMap( peer, cmds, addrOrPgName, addrFamily,
                                   saveAll, nedMode )
         savePeerMissingPolicy( peer, cmds, addrOrPgName, saveAll, nedMode,
                                addrFamily )
         return

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'orr' ]:
         savePeerAfOrr( peer, cmds, addrOrPgName, addrFamily, saveAll, nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'rfdPolicyAttachment' ]:
         savePeerAfRfd( peer, cmds, addrOrPgName, addrFamily, saveAll, nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'apRecv' ]:
         savePeerAfAddPathRecv( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'gracefulRestart' ]:
         savePeerAfGracefulRestart( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                    nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'grHelperRouteMap' ]:
         savePeerAfGrHelperRouteMap( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                     nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'routeMapIn' ] and \
         addrFamily in BgpLib.peerConfigAttrsAfMap[ 'routeMapOut' ]:
         savePeerPerAfRouteMap( peer, cmds, addrOrPgName, addrFamily, bgpConfig,
                                saveAll, nedMode )

      rcfAddrFamilyInPeerConfig = (
         any( addrFamily in BgpLib.peerConfigAttrsAfMap[ poa ]
            for poa in ( 'rcfIn', 'rcfOut' )
         )
      )

      rcfAddrFamilyEnabled = (
         addrFamily in [ 'ipv4', 'ipv6' ] or
         'labeled-unicast' in addrFamily or
         'evpn' in addrFamily or
         addrFamily in [ 'flow-spec ipv4', 'flow-spec ipv6' ] or
         ( addrFamily in [ 'flow-spec vpn-ipv4', 'flow-spec vpn-ipv6' ] and
           toggleRcfFlowspecVpnPoaEnabled ) or
         addrFamily in [ 'vpn-ipv4', 'vpn-ipv6' ] or
         'rt-membership' in addrFamily or
         'vpls' in addrFamily
      )

      if ( rcfAddrFamilyInPeerConfig and rcfAddrFamilyEnabled ):
         savePeerPerAfRcf( peer, cmds, addrOrPgName, addrFamily, bgpConfig, saveAll,
                           nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'weight' ]:
         savePeerAfWeight( peer, cmds, addrOrPgName, addrFamily, saveAll, nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'outDelay' ]:
         savePeerAfOutDelay( peer, cmds, addrOrPgName, addrFamily, saveAll, nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'prefixListIn' ] and \
         addrFamily in BgpLib.peerConfigAttrsAfMap[ 'prefixListOut' ] :
         savePeerPerAfPrefixList( peer, cmds, addrOrPgName, addrFamily, bgpConfig,
                                  saveAll, afCfgPresent, nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'defaultOriginate' ]:
         savePeerAfDefOrigin( peer, cmds, addrOrPgName, addrFamily, saveAll,
                              nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'apSend' ]:
         savePeerAfAddPathSend( peer, cmds, addrOrPgName, addrFamily, saveAll,
                              nedMode )

      if addrFamily == 'ipv4':
         savePeerAfNextHopAfV6( peer, cmds, addrOrPgName, saveAll, nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'nexthopUnchanged' ]:
         savePeerAfNexthopUnchanged( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                     nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'nextHopSelf' ]:
         savePeerAfNexthopSelf( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                     nedMode )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'nhSelfSrcIntf' ]:
         savePeerAfNhSelfSrcIntf( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                  nedMode )

      if BgpCommonToggleLib.toggleRejectAsSetConfedSetEnabled():
         if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'rejectAsSets' ]:
            savePeerAfRejectAsSets( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                    nedMode )

      savePeerMaxRoutes( peer, cmds, addrOrPgName, saveAll, nedMode, addrFamily )

      savePeerMaxAdvRoutes( peer, cmds, addrOrPgName, saveAll, nedMode, addrFamily )

      savePeerMissingPolicy( peer, cmds, addrOrPgName, saveAll, nedMode, addrFamily )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'aigpSession' ]:
         savePeerAfAigpSession( peer, cmds, addrOrPgName, addrFamily, saveAll,
                                nedMode )

      if addrFamily == 'evpn':
         saveEvpnEncapType( cmds, peer, saveAll, addrOrPgName )
         saveEvpnDomainRemote( cmds, peer, saveAll, addrOrPgName )
         saveEvpnRouteTypeNexthopSelf( cmds, peer, saveAll, addrOrPgName )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'nexthopLuOriginate' ]:
         savePeerAfNexthopLuOriginate( peer, cmds, addrOrPgName, saveAll, nedMode,
                                       addrFamily )
      if BgpCommonToggleLib.toggleArBgpNexthopLuPeerSetOriginateEnabled():
         if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'nexthopLuPeerSetOriginate' ]:
            savePeerAfNexthopLuPeerSetOriginate( peer, cmds, addrOrPgName, saveAll,
                                           nedMode, addrFamily )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'fwdFailoverTriggerSession' ]:
         savePeerFwdFailoverTriggerSession( peer, cmds, addrOrPgName, saveAll,
                                            nedMode, addrFamily )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'nhLuOrigLfibBackupIpFwd' ]:
         savePeerAfNhLuOrigLfibBackupIpFwd( peer, cmds, addrOrPgName, saveAll,
                                            addrFamily )

      attr = 'v4MappedV6NextHopTranslation'
      if addrFamily in BgpLib.peerConfigAttrsAfMap[ attr ]:
         savePeerAfV4MappedV6NextHopTranslation( peer, cmds, addrOrPgName,
                                                 addrFamily, saveAll )

      if addrFamily in BgpLib.peerConfigAttrsAfMap[ 'multipath' ]:
         savePeerMultipath( peer, cmds, addrOrPgName, addrFamily, saveAll )

      # If an additional per-af per-peer clisave callback is registered for
      # this address-family, call it to add the additional config.
      cb = saveAfPeerConfigCallbackDict.get( addrFamily )
      if cb:
         cb( cmds, peer, saveAll, addrOrPgName )

def evpnEncapTacValueToKeyword( encap ):
   if encap == 'encapVxlan':
      return 'vxlan'
   if encap == 'encapMpls':
      return 'mpls'
   if encap == 'encapSrv6':
      return 'segment-routing ipv6'
   if encap == 'encapDps':
      return 'path-selection'
   raise Exception( 'Unexpected encapsulation %s' % ( encap ) )

def saveEvpnEncapType( cmds, config, saveAll, peer='default' ):
   # 'neighbor default encapsulation ...' is saved only when encap is not default
   # 'neighbor PEER encapsulation ...' is saved whenever peer configuration exists
   defaultPeer = ( peer == 'default' )
   if config.afEvpnEncap != config.afEvpnEncapDefault or not defaultPeer:
      if not defaultPeer and not config.afEvpnEncapPresent:
         return
      additionalArgs = ''
      cliEncapType = evpnEncapTacValueToKeyword( config.afEvpnEncap )
      if ( cliEncapType == 'mpls' and
            config.afEvpnMplsNexthopSelfSrcIntf !=
            config.afEvpnMplsNexthopSelfSrcIntfDefault ):
         if peer == 'default' or config.afEvpnMplsNexthopSelfSrcIntfPresent:
            additionalArgs = 'next-hop-self source-interface %s' % (
                  config.afEvpnMplsNexthopSelfSrcIntf )
      if ( RoutingLibToggleLib.toggleSRv6BgpL3VpnEnabled() and
           cliEncapType == 'segment-routing ipv6' ):
         additionalArgs = 'locator %s' % config.afEvpnSrv6Locator
      cmds.append( 'neighbor %s encapsulation %s %s' % ( peer, cliEncapType,
         additionalArgs ) )
   elif saveAll:
      cliEncapType = evpnEncapTacValueToKeyword( config.afEvpnEncapDefault )
      cmds.append( 'neighbor %s encapsulation %s' % ( peer, cliEncapType ) )

def saveEvpnEncapTypeMulticast( cmds, config, saveAll, peer='default' ):
   # 'neighbor default encapsulation ...' is saved only when encap is not default
   if not MvpnLibToggleLib.toggleOismMplsTunnelEnabled():
      return
   cmd = 'neighbor default encapsulation mpls multicast protocol p2mp mldp'
   if config.afEvpnMplsMulticastPmsiTunnel != \
         config.afEvpnMplsMulticastPmsiTunnelDefault:
      cmds.append( cmd )
   elif saveAll:
      cmds.append( f'no {cmd}' )

def saveEvpnDomainRemote( cmds, config, saveAll, peer ):
   cmd = 'neighbor %s domain remote' % peer
   if config.afEvpnRemoteDomainPresent:
      cmds.append( cmd )
   elif saveAll:
      cmd = 'no ' + cmd
      cmds.append( cmd )

def saveEvpnRouteTypeNexthopSelf( cmds, config, saveAll, peer ):
   # pylint: disable-next=consider-using-dict-items
   for rType in evpnNexthopSelfSupportNlris:
      rTypeName = evpnNexthopSelfSupportNlris[ rType ][ 'keyword' ]
      cmd = 'neighbor %s next-hop-self received-evpn-routes route-type %s' % \
            ( peer, rTypeName )
      attrName = 'afEvpnNexthopSelfType%d' % rType
      present = getattr( config, attrName + 'Present' )
      if present:
         cmds.append( cmd )
      elif saveAll:
         cmds.append( 'no ' + cmd )

      interDomainAttrName = 'afEvpnNexthopSelfType%dInterDomain' % rType
      present = getattr( config, interDomainAttrName + 'Present' )
      if present:
         cmd += ' inter-domain'
         cmds.append( cmd )
      elif saveAll:
         cmds.append( 'no ' + cmd + ' inter-domain' )

def saveBgpVpnAfConfig( bgpConfig, defaultVrfBgpConfig, addrFamily, asdotConfigured,
                        saveAll ):
   """This method returns the VPN config for the given VRF and address-famliy"""
   cmds = saveRouteTargetConfig( bgpConfig, asdotConfigured,
         addrFamily=addrFamily )

   rcfImportAttrName = (
         BgpLib.bgpConfigAttrsAfMap[ 'rcfImport' ].get( addrFamily ) )
   rcfExportAttrName = (
         BgpLib.bgpConfigAttrsAfMap[ 'rcfExport' ].get( addrFamily ) )
   if rcfImportAttrName and rcfExportAttrName:
      for importExport, attrName in [ ( 'import', rcfImportAttrName ),
                                      ( 'export', rcfExportAttrName ) ]:
         # An explicit key is required for sort method here because vpnAf is of
         # type Routing::Bgp::AfiSafi but we need to sort them based on their cli
         # keyword here.
         afiSafiKwList = sorted( getattr( bgpConfig, attrName ).items(),
                                 key=lambda item: vpnAfTypeMap.get( item[ 0 ] ) )
         for vpnAf, rcfMainAndFilter in afiSafiKwList:
            if vpnAfTypeMap.get( vpnAf ):
               # Check if the optional vpn/vrf rcf function exists for the given
               # vpnAf and create the optional argument for the command
               direction = "vpn" if importExport == "import" else "vrf"
               filterArg = "" if not rcfMainAndFilter.filterFunction else (
                  " %s-route filter-rcf %s()" % ( direction,
                                                  rcfMainAndFilter.filterFunction ) )
               cmds.append( "route-target %s %s rcf %s()%s"
                  % ( importExport, vpnAfTypeMap[ vpnAf ],
                      rcfMainAndFilter.mainFunction, filterArg ) )

   routeMapImportAttrName = (
         BgpLib.bgpConfigAttrsAfMap[ 'routeMapImport' ].get( addrFamily ) )
   routeMapExportAttrName = (
         BgpLib.bgpConfigAttrsAfMap[ 'routeMapExport' ].get( addrFamily ) )
   if routeMapImportAttrName and routeMapExportAttrName:
      for importExport, attrName in [ ( 'import', routeMapImportAttrName ),
                                      ( 'export', routeMapExportAttrName ) ]:
         # An explicit key is required for sort method here because vpnAf is of
         # type Routing::Bgp::AfiSafi but we need to sort them based on thier
         # cli keywork here.
         for vpnAf, rmName in sorted(
               getattr( bgpConfig, attrName ).items(),
                        key=lambda item: vpnAfTypeMap.get( item[ 0 ] ) ):
            if vpnAfTypeMap.get( vpnAf ):
               cmds.append( "route-target %s %s route-map %s" % ( importExport,
                  vpnAfTypeMap[ vpnAf ], rmName ) )

   importedRouteExportAttrName = \
      BgpLib.bgpConfigAttrsAfMap[ 'allowImportedRouteToExport' ].get( addrFamily )
   if importedRouteExportAttrName:
      for importExport, attrName in [ ( 'export', importedRouteExportAttrName ) ]:
         for vpnAf, isImpRoSet in sorted(
               getattr( bgpConfig, attrName ).items(),
                        key=lambda item: vpnAfTypeMap.get( item[ 0 ] ) ):
            assert isImpRoSet
            if vpnAfTypeMap.get( vpnAf ):
               cmds.append( "route-target %s %s %s" % ( importExport,
                            vpnAfTypeMap[ vpnAf ], 'imported-route' ) )

   for peerKey in bgpConfig.neighborConfig:
      addrOrPgName = peerKey.stringValue
      peer = bgpConfig.neighborConfig.get( peerKey )
      savePeerRouteTargetExportFilterDisabled( peer, cmds, addrOrPgName,
                                               addrFamily, saveAll )

   return cmds

# saveAfConfigCallbackDict is the dictionary of callbacks keyed by address families.
# AfiSafi plugins can register and add their callbacks to it. It is used by
# doShowBgpConfig to display address family specific config in
# show bgp config active output, other than the configs shown by saveAfConfig
saveAfConfigCallbackDict = {}

# showBgpConfigInstanceCallbackDict is a list of callbacks for instance level
# submodes.  The callbacks take the bgpConfig as an arg, and return a dictionary
# of submode commands to a list of commands configured in that submode.
showBgpConfigInstanceCallbacks = []

def saveAfConfig( bgpConfig, defaultVrfBgpConfig, vrfName, addrFamily,
                  asdotConfigured=False, saveAll=False, saveAllDetail=False ):
   cmds = []
   afCfgPresent = hasAfConfig( bgpConfig )

   if vrfName:
      cmds.extend( saveBgpVpnAfConfig( bgpConfig, defaultVrfBgpConfig, addrFamily,
                                       asdotConfigured, saveAll ) )

   def maybeSetSkipRibBestpath( suffix ):
      if getattr( bgpConfig, 'skipRib%s' % suffix ):
         if getattr( bgpConfig, 'skipBestpath%s' % suffix ):
            cmds.append( "bgp skip rib-install bestpath-selection" )
         else:
            cmds.append( "bgp skip rib-install" )
      elif saveAll:
         cmds.append( "no bgp skip rib-install bestpath-selection" )

   if addrFamily == 'ipv4':
      if afCfgPresent:
         if bgpConfig.picIPv4Uni:
            if not bgpConfig.picEcmpPrimaryIPv4Uni:
               cmds.append( "bgp additional-paths install" )
            else:
               cmds.append( "bgp additional-paths install ecmp-primary" )
         elif saveAll:
            cmds.append( "no bgp additional-paths install" )
      # Print bgp next-hop address-family ipv6 in address family ipv4 mode.
      if bgpConfig.v6NextHopAfV4Uni != bgpConfig.v6NextHopAfV4UniDefault:
         if bgpConfig.v6NextHopAfV4Uni == \
                            ExtendedNextHopCapabilityEnum.isEnabledWithOriginate:
            cmds.append( "bgp next-hop address-family ipv6 originate" )
         else:
            cmds.append( "bgp next-hop address-family ipv6" )
      elif saveAll:
         cmds.append( "no bgp next-hop address-family ipv6" )
      maybeSetSkipRibBestpath( 'IPv4Uni' )
   elif addrFamily == 'ipv6':
      maybeSetSkipRibBestpath( 'IPv6Uni' )
      if bgpConfig.picIPv6Uni:
         if not bgpConfig.picEcmpPrimaryIPv6Uni:
            cmds.append( "bgp additional-paths install" )
         else:
            cmds.append( "bgp additional-paths install ecmp-primary" )
      elif saveAll:
         cmds.append( "no bgp additional-paths install" )
   elif addrFamily == 'flow-spec ipv4':
      maybeSetSkipRibBestpath( 'IPv4Flowspec' )
   elif addrFamily == 'flow-spec ipv6':
      maybeSetSkipRibBestpath( 'IPv6Flowspec' )
   elif addrFamily == 'ipv4 labeled-unicast':
      if bgpConfig.picIPv4LabeledUni:
         if not bgpConfig.picEcmpPrimaryIPv4LabeledUni:
            cmds.append( "bgp additional-paths install" )
         else:
            cmds.append( "bgp additional-paths install ecmp-primary" )
   elif addrFamily == 'ipv6 labeled-unicast':
      if bgpConfig.picIPv6LabeledUni:
         if not bgpConfig.picEcmpPrimaryIPv6LabeledUni:
            cmds.append( "bgp additional-paths install" )
         else:
            cmds.append( "bgp additional-paths install ecmp-primary" )
   # update wait-for-convergence command
   convergenceNoSyncAttrName = BgpLib.bgpConfigAttrsAfMap[ 'convergenceNoSync' ].\
                                                         get( addrFamily )
   if convergenceNoSyncAttrName:
      isConvergenceNoSyncSet, convergenceNoSyncVal = evaluateTristate( bgpConfig,
                                                     bgpConfig,
                                                     convergenceNoSyncAttrName )
      if isConvergenceNoSyncSet:
         if convergenceNoSyncVal:
            cmds.append( "no update wait-for-convergence" )
         else:
            cmds.append( "update wait-for-convergence" )
   # bgp missing-policy address family specific command
   # Some address families like rt-membership will not influence
   # missing policy, so do not CliSave the missing policy command
   skipMissingPolicyAf = [ 'rt-membership', 'path-selection', 'mvpn-ipv4', 'vpls',
                           'flow-spec vpn-ipv4', 'flow-spec vpn-ipv6' ]
   for direction in [ "In", "Out" ]:
      tok = [ 'bgp', 'missing-policy', 'direction', direction.lower(), 'action' ]
      actionAfAttr = 'missingPolicyActionAf' + direction
      actionIncludeSubAfAttr = 'missingPolicyIncludeSubRouteMapAf' + direction
      actionIncludePfxAfAttr = 'missingPolicyIncludePrefixListAf' + direction
      actionIncludeCommAfAttr = 'missingPolicyIncludeCommListAf' + direction
      actionAfConfig = getattr( bgpConfig, actionAfAttr )
      actionIncludeSubAfConfig = getattr( bgpConfig, actionIncludeSubAfAttr )
      actionIncludePfxAfConfig = getattr( bgpConfig, actionIncludePfxAfAttr )
      actionIncludeCommAfConfig = getattr( bgpConfig, actionIncludeCommAfAttr )
      af = afiSafiCache.getAfiSafi( addrFamily )
      if af in actionAfConfig:
         tok.append( policyActions[ actionAfConfig[ af ] ] )
         insertPos = 2
         if 'isTrue' in ( actionIncludeSubAfConfig.get( af ),
                          actionIncludePfxAfConfig.get( af ),
                          actionIncludeCommAfConfig.get( af ) ):
            # insert include keywords into the command token list alphabetically
            includeTokens = [ 'include' ]
            if actionIncludeCommAfConfig.get( af ) == 'isTrue':
               includeTokens.append( 'community-list' )
            if actionIncludePfxAfConfig.get( af ) == 'isTrue':
               includeTokens.append( 'prefix-list' )
            if actionIncludeSubAfConfig.get( af ) == 'isTrue':
               includeTokens.append( 'sub-route-map' )
            tok[ insertPos : insertPos ] = includeTokens
         missingPolicyCommand = ' '.join( tok )
         cmds.append( missingPolicyCommand )
      elif saveAll and addrFamily not in skipMissingPolicyAf:
         missingPolicyCommand = 'no ' + ' '.join( tok )
         cmds.append( missingPolicyCommand )
   # Add-path receive command
   apRecvAttrName = BgpLib.bgpConfigAttrsAfMap[ 'apRecv' ].get( addrFamily )
   if apRecvAttrName:
      isAPRecvSet, apRecvVal = evaluateTristate( bgpConfig, bgpConfig,
                                                 apRecvAttrName )
      if isAPRecvSet:
         if apRecvVal:
            cmds.append( 'bgp additional-paths receive' )
         else:
            cmds.append( 'no bgp additional-paths receive' )
      elif saveAll:
         # Add-path receive is enabled by default
         cmds.append( 'bgp additional-paths receive' )
   apSendAttrName = BgpLib.bgpConfigAttrsAfMap[ 'apSend' ].get( addrFamily )
   if apSendAttrName:
      apSendConfig = getattr( bgpConfig, apSendAttrName )
      cmd = ''
      for app in apSendConfig:
         appConfig = apSendConfig.get( app )
         if appConfig.enable:
            cmd = 'bgp '
         else:
            # app must be 'appNone' or 'appAny'
            cmd = 'no bgp '
         cmd += addPathSendCommon( app, appConfig=appConfig )
         # There can be only one app present
         break
      if saveAll and cmd == '':
         cmd = 'no bgp '
         cmd += addPathSendCommon( 'appNone', saveAll=saveAll )
      if cmd != '':
         cmds.append( cmd )

   # next-hop-unchanged command
   nhUnchangedAttrName = \
         BgpLib.bgpConfigAttrsAfMap[ 'nexthopUnchanged' ].get( addrFamily )
   if nhUnchangedAttrName:
      isNhUnchangedSet, nhUnchVal = evaluateTristate( bgpConfig, bgpConfig,
                                                      nhUnchangedAttrName )
      if isNhUnchangedSet:
         if nhUnchVal:
            cmds.append( 'bgp next-hop-unchanged' )
         else:
            cmds.append( 'no bgp next-hop-unchanged' )
      elif saveAll:
         cmds.append( 'no bgp next-hop-unchanged' )

   # next-hop-self command
   nhSelfAttrName = \
         BgpLib.bgpConfigAttrsAfMap[ 'nextHopSelf' ].get( addrFamily )

   if nhSelfAttrName:
      isNhSelfSet, nhSelfVal = evaluateTristate( bgpConfig, bgpConfig,
                                                 nhSelfAttrName )
      if isNhSelfSet:
         if nhSelfVal:
            cmds.append( 'neighbor default next-hop-self' )
         else:
            cmds.append( 'no neighbor default next-hop-self' )
      elif saveAll:
         cmds.append( 'no neighbor default next-hop-self' )

   routeInstallMapAttrName = \
         BgpLib.bgpConfigAttrsAfMap[ 'routeInstallMap' ].get( addrFamily )
   if routeInstallMapAttrName:
      val = getattr( bgpConfig, routeInstallMapAttrName )
      defVal = getattr( bgpConfig, routeInstallMapAttrName + 'Default' )
      if val != defVal:
         cmds.append( 'bgp route install-map %s' % val )
      elif saveAll:
         cmds.append( 'no bgp route install-map' )

   if addrFamily == 'evpn':
      saveEvpnEncapType( cmds, bgpConfig, saveAll )
      saveEvpnEncapTypeMulticast( cmds, bgpConfig, saveAll )

   if ( addrFamily == 'ipv4' or addrFamily == 'ipv6' or
        addrFamily == 'ipv4 labeled-unicast' or
        addrFamily == 'ipv6 labeled-unicast' or
        addrFamily == 'evpn' or
        addrFamily == 'vpn-ipv4' or addrFamily == 'vpn-ipv6' ):
      if addrFamily == 'ipv4':
         if bgpConfig.afV4NexthopResolutionDisabled:
            cmds.append( 'next-hop resolution disabled' )
         elif saveAll or saveAllDetail:
            cmds.append( 'no next-hop resolution disabled' )
      elif addrFamily == 'ipv6':
         if bgpConfig.afV6NexthopResolutionDisabled:
            cmds.append( 'next-hop resolution disabled' )
         elif saveAll or saveAllDetail:
            cmds.append( 'no next-hop resolution disabled' )

      pnhts = []
      pathAfiSafiNhType = Tac.Type( 'Routing::Bgp::PathAfiSafiNexthopType' )
      udpTunnel = False
      if addrFamily == 'ipv4':
         pnht = pathAfiSafiNhType.ipv4UniNexthopsAll
         pnhts = [ pnht ]
         udpTunnel = pnht in bgpConfig.nhResolutionRibUdpProfileTunnelConfig
      elif addrFamily == 'ipv4 labeled-unicast':
         pnht = pathAfiSafiNhType.ipv4MplsNexthopsAll
         pnhts = [ pnht ]
      elif addrFamily == 'ipv6':
         pnht1 = pathAfiSafiNhType.ipv6UniNexthopsAll
         # 6pe should be last since its a specialization of ipv6
         pnht2 = pathAfiSafiNhType.sixPeNexthopsAll
         pnhts = [ pnht1, pnht2 ]
         udpTunnel = pnht1 in bgpConfig.nhResolutionRibUdpProfileTunnelConfig
      elif addrFamily == 'ipv6 labeled-unicast':
         pnht = pathAfiSafiNhType.ipv6MplsNexthopsAll
         pnhts = [ pnht ]
      elif addrFamily == 'evpn':
         pnht1 = pathAfiSafiNhType.l2vpnEvpnNexthopsEncapMpls
         pnht2 = pathAfiSafiNhType.l2vpnEvpnNexthopsEncapVxlan
         pnhts = [ pnht1, pnht2 ]
      elif addrFamily == 'vpn-ipv4':
         pnht = pathAfiSafiNhType.ipv4MplsVpnNexthopsEncapMpls
         pnhts = [ pnht ]
      elif addrFamily == 'vpn-ipv6':
         pnht = pathAfiSafiNhType.ipv6MplsVpnNexthopsEncapMpls
         pnhts = [ pnht ]

      if not BgpCommonToggleLib.toggleIpOverDynamicGueTunnelsEnabled():
         udpTunnel = False

      for pnht in pnhts:
         rmName = bgpConfig.nhResolutionRibProfileRouteMapConfig.get( pnht )
         if rmName:
            cmd = "next-hop resolution ribs route-map %s" % rmName
            cmds.append( cmd )
            continue

         # entries are expected to always remain in the collection
         profile = bgpConfig.nhResolutionRibProfileConfig.get( pnht )
         defaultProfile = bgpConfig.nhResolutionRibProfileConfigDefault.get( pnht )
         if not profile and not udpTunnel:
            continue
         if not saveAll and profile == defaultProfile:
            continue

         if pnht == pathAfiSafiNhType.sixPeNexthopsAll:
            # SPECIAL CASE:
            # vrf-ipv6 does not support 6pe (only ipv6 does)
            # but because of two things:
            #   1. nhResolutionRibProfileConfig defaults are set in BgpConfig
            #      constructor
            #   2. an instance of BgpConfig is constructed per vrf
            # this means the vrf's will have a default value for 6pe which is
            # undesirable.
            # So :( we'll just prevent it from ever being displayed here...
            if vrfName:
               continue
            cmd = "next-hop 6pe resolution ribs"
         elif pnht == pathAfiSafiNhType.l2vpnEvpnNexthopsEncapMpls:
            cmd = "next-hop mpls resolution ribs"
         elif pnht == pathAfiSafiNhType.l2vpnEvpnNexthopsEncapVxlan:
            cmd = "next-hop vxlan resolution ribs"
         elif udpTunnel:
            cmd = "next-hop resolution ribs udp-tunnel"
            if profile.useDefault:
               cmd += ' system-default'
            else:
               cmd += ' ' + profile.stringValue()
            cmds.append( cmd )
            continue
         else:
            cmd = "next-hop resolution ribs"
         cmd += ' ' + profile.stringValue()
         cmds.append( cmd )

      if ( not vrfName and
           ( addrFamily == 'ipv4' or addrFamily == 'ipv6' ) and
           BgpCommonToggleLib.toggleIpOverDynamicGueTunnelsEnabled() ):
         if addrFamily == 'ipv4':
            if bgpConfig.afV4NexthopOriginateTunnelConfig.encapType == 'udp':
               policyName = bgpConfig.afV4NexthopOriginateTunnelConfig.policyName
               cmds.append(
                  f'next-hop originate tunnel type udp policy {policyName}' )
            elif saveAll or saveAllDetail:
               cmds.append( 'no next-hop originate tunnel type udp' )
         else:
            if bgpConfig.afV6NexthopOriginateTunnelConfig.encapType == 'udp':
               policyName = bgpConfig.afV6NexthopOriginateTunnelConfig.policyName
               cmds.append(
                  f'next-hop originate tunnel type udp policy {policyName}' )
            elif saveAll or saveAllDetail:
               cmds.append( 'no next-hop originate tunnel type udp' )

   # Iterate through the ordered collection bgpConfig.neighborConfig
   # and call savePeerAfConfig on every element.
   for peerKey in bgpConfig.neighborConfig:
      savePeerAfConfig( cmds, bgpConfig, defaultVrfBgpConfig, peerKey,
                        saveAll, addrFamily, afCfgPresent )

   # Only print networks in the address-family block if we have
   # unicast address-family specific config
   if afCfgPresent and ( addrFamily in [ 'ipv4', 'ipv6' ] ):
      saveNetworkList( cmds, bgpConfig, filterNetworkAf=( addrFamily ) )
   else:
      networkListAttrName = \
            BgpLib.bgpConfigAttrsAfMap[ 'networkList' ].get( addrFamily )
      if networkListAttrName:
         saveNetworkList( cmds, bgpConfig, addrFamily=addrFamily )

   saveEpeNexthopOriginateConfig( cmds, bgpConfig, addrFamily )
   saveEpeNexthopOriginateLfibBackupConfig( cmds, bgpConfig, addrFamily )

   saveLfibEntryInstallationSkippedCmd( cmds, bgpConfig, addrFamily, saveAll )

   saveLabelLocalTerminationExplicitNullCmd(
         cmds, bgpConfig, addrFamily, saveAll )

   # Print '[no] bgp redistribute-internal' in addrFamily mode if it was
   # configured in either of the addrFamily modes and if saving all then
   # we print the default value when bgp redistribute-internal is configured
   # for the other addrFamily mode.
   redistInternalAttrName = \
         BgpLib.bgpConfigAttrsAfMap[ 'bgpRedistInternal' ].get( addrFamily )
   if redistInternalAttrName:
      if bgpConfig.bgpRedistInternal == redistInternalStateEnum.notConfigured:
         state = getattr( bgpConfig, redistInternalAttrName )
         if state == redistInternalStateEnum.disableRedistInt:
            cmds.append( 'no bgp redistribute-internal' )
         elif saveAll:
            cmds.append( 'bgp redistribute-internal' )

   gracefulRestartAttrName = \
         BgpLib.bgpConfigAttrsAfMap[ 'gracefulRestart' ].get( addrFamily )
   if gracefulRestartAttrName:
      isGrSet, grVal = evaluateTristate( bgpConfig, bgpConfig,
                                         gracefulRestartAttrName )
      if isGrSet:
         if grVal:
            cmds.append( "graceful-restart" )
         else:
            cmds.append( "no graceful-restart" )

   redistributeAttrName = \
         BgpLib.bgpConfigAttrsAfMap[ 'redistributeConfig' ].get( addrFamily )
   if redistributeAttrName:
      # Print 'redistribute <proto>' in addrFamily mode
      redistributeConfig = getattr( bgpConfig, redistributeAttrName )
      for protocol in sorted( redistributeConfig.values(),
                              key=lambda p: p.proto ):
         cmd = redistributeCommand( protocol, addrFamily=addrFamily )
         if protocol.routeMap:
            cmd += ' route-map %s' % protocol.routeMap
         if protocol.rcf:
            cmd += ' rcf %s()' % protocol.rcf
         cmds.append( cmd )

   # aigp-session for ibgp peers (enabled by default)
   attr = 'aigpSessionIbgpDisabled'
   attrName = BgpLib.bgpConfigAttrsAfMap[ attr ].get( addrFamily )
   if attrName:
      isAttrSet, attrVal = evaluateTristate( bgpConfig, bgpConfig, attrName )
      if isAttrSet:
         if attrVal:
            cmds.append( 'no aigp-session ibgp' )
         else:
            cmds.append( 'aigp-session ibgp' )
      elif saveAll:
         cmds.append( 'aigp-session ibgp' )

   # aigp-session for confed peers (enabled by default)
   attr = 'aigpSessionConfedDisabled'
   attrName = BgpLib.bgpConfigAttrsAfMap[ attr ].get( addrFamily )
   if attrName:
      isAttrSet, attrVal = evaluateTristate( bgpConfig, bgpConfig, attrName )
      if isAttrSet:
         if attrVal:
            cmds.append( 'no aigp-session confederation' )
         else:
            cmds.append( 'aigp-session confederation' )
      elif saveAll:
         cmds.append( 'aigp-session confederation' )

   # aigp-session for ebgp peers (disabled by default)
   attr = 'aigpSessionEbgpEnabled'
   attrName = BgpLib.bgpConfigAttrsAfMap[ attr ].get( addrFamily )
   if attrName:
      isAttrSet, attrVal = evaluateTristate( bgpConfig, bgpConfig, attrName )
      if isAttrSet:
         if attrVal:
            cmds.append( 'aigp-session ebgp' )
         else:
            cmds.append( 'no aigp-session ebgp' )
      elif saveAll:
         cmds.append( 'no aigp-session ebgp' )

   # multicast aggregate-list config
   if addrFamily == 'ipv4 multicast':
      cmds.extend( saveAggregates(
         list( bgpConfig.aggregateAfV4MulticastList.values() ) ) )
   if addrFamily == 'ipv6 multicast':
      cmds.extend( saveAggregates(
         list( bgpConfig.aggregateAfV6MulticastList.values() ) ) )

   # peer-set config
   saveEpePeerSetConfig( cmds, bgpConfig, addrFamily )

   return cmds

@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config',
      requireMounts = ( 'routing/bgp/asn/config', ) )
def saveBaseAfConfig( bgpConfig, root, requireMounts, options ):
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )
   for af in BgpLib.bgpConfigAttrAfList():
      cmds = saveAfConfig( bgpConfig, bgpConfig, "", af, asdotConfigured,
                           saveAll=options.saveAll,
                           saveAllDetail=options.saveAllDetail )
      if cmds:
         bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
               ].getOrCreateModeInstance( af )
         for cmd in cmds:
            bgpAfMode[ 'Bgp.afConfig.%s' % af ].addCommand( cmd )

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts=( 'routing/bgp/config',
                                  'routing/bgp/asn/config', ) )
def saveVrfAfConfig( vrfConfigDir, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )

   for vrfName in vrfConfigDir.vrfConfig:
      vrfBgpConfig = vrfConfigDir.vrfConfig[ vrfName ]
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode ].getOrCreateModeInstance(
            vrfName )
      for af in BgpLib.bgpConfigAttrAfList( vrfName ):
         cmds = saveAfConfig( vrfBgpConfig, bgpConfig, vrfName, af,
                              asdotConfigured, saveAll=options.saveAll,
                              saveAllDetail=options.saveAllDetail )
         if cmds:
            bgpVrfAfMode = bgpVrfMode[ RouterBgpVrfAfConfigMode
                  ].getOrCreateModeInstance( ( af, vrfName ) )
            for cmd in cmds:
               bgpVrfAfMode[ 'Bgp.vrf.afConfig.%s' % af ].addCommand( cmd )

@CliSave.saver( 'Routing::AsnConfig', 'routing/bgp/asn/config',
                requireMounts = ( 'routing/bgp/config', ) )
def saveAsnConfig( asnConfig, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )

   cmds = bgpMode[ 'Bgp.config' ]

   def asnNotationString( asnNotation ):
      if asnNotation == Tac.Type( 'Routing::AsnNotation' ).asnNotationAsdot:
         return "asdot"
      elif asnNotation == Tac.Type( 'Routing::AsnNotation' ).asnNotationAsplain:
         return "asplain"
      else:
         raise Exception( 'Invalid AS Number Notation Configured' )

   asnNotation = asnConfig.asnNotation
   msg = "bgp asn notation %s" % asnNotationString( asnNotation )
   if asnNotation == asnConfig.asnNotationDefault and not options.saveAll:
      return
   cmds.addCommand( msg )

@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config' )
def saveServiceBgpConfig( bgpConfig, root, requireMounts, options ):
   cmd = None
   if bgpConfig.noEqualsDefaultMode != bgpConfig.noEqualsDefaultModeDefault:
      cmd = 'service routing configuration bgp no-equals-default'
   elif options.saveAll:
      cmd = 'no service routing configuration bgp no-equals-default'
   if cmd:
      root[ 'Bgp.service' ].addCommand( cmd )

def saveFieldSetMappingsConfig( vrfNames, defaultVrfBgpConfig, vrfConfigDir,
                                bgpMode, mapCtx, asdotConfigured, saveAll ):
   assert mapCtx in [ 'tp', 'vsp' ]
   mapStr = 'traffic-policy' if mapCtx == 'tp' else 'vrf-selection-policy'
   for vrfName in vrfNames:
      if vrfName == DEFAULT_VRF:
         vrfBgpConfig = defaultVrfBgpConfig
      else:
         vrfBgpConfig = vrfConfigDir.vrfConfig[ vrfName ]

      if getattr( vrfBgpConfig, mapCtx + 'FieldSetMappingDisabled' ) == 'isInvalid':
         # Nothing has been added to this mode.
         continue

      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( vrfName )
      mode = TrafficPolicyFieldSetMappingsVrfConfigMode if mapCtx == 'tp' else \
         VrfSelectionPolicyFieldSetMappingsVrfConfigMode
      bgpMappingsMode = bgpVrfMode[ mode ].getOrCreateModeInstance( vrfName )

      # Save configuration in the mappings mode.
      cmds = []
      isDisabled = getattr( vrfBgpConfig, mapCtx + 'FieldSetMappingDisabled' ) \
         == 'isTrue'
      if isDisabled or saveAll:
         cmds.append( '%sdisabled' % ( 'no ' if not isDisabled else '' ) )

      for cmd in cmds:
         bgpMappingsMode[ f'Bgp.vrf.{mapStr}.config' ].addCommand( cmd )

      # Save configuration in the field set submodes, if any.
      if not ( getattr( vrfBgpConfig, mapCtx + 'FieldSetMappingBindingsIpv4' ) or
               getattr( vrfBgpConfig, mapCtx + 'FieldSetMappingBindingsIpv6' ) ):
         continue

      for ipv6 in [ False, True ]:
         if ipv6:
            coll = getattr( vrfBgpConfig, mapCtx + 'FieldSetMappingBindingsIpv6' )
            mode = TpFieldSetIpv6ConfigMode if mapCtx == 'tp' else \
               VspFieldSetIpv6ConfigMode
            ver = 6
         else:
            coll = getattr( vrfBgpConfig, mapCtx + 'FieldSetMappingBindingsIpv4' )
            mode = TpFieldSetIpv4ConfigMode if mapCtx == 'tp' else \
               VspFieldSetIpv4ConfigMode
            ver = 4
         tag = f'Bgp.vrf.{mapStr}.ipv{ver}.config'
         for fsName in sorted( coll ):
            bgpFsMode = bgpMappingsMode[ mode ].getOrCreateModeInstance(
               ( fsName, vrfName ) )
            bindings = coll[ fsName ]

            if bindings.communityPresent:
               cmd = 'community %s' % commValueToPrint( bindings.community )
               bgpFsMode[ tag ].addCommand( cmd )
            if mapCtx == 'tp':
               if bindings.peerType != PeerFieldSetFilter.inactive:
                  bgpFsMode[ tag ].addCommand( 'peers ' + bindings.peerType )
               if bindings.groupFilter:
                  groupsLine = " ".join( sorted( bindings.groupFilter ) )
                  bgpFsMode[ tag ].addCommand( 'peers group ' + groupsLine )

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts=( 'routing/bgp/config',
                                'routing/bgp/asn/config', ) )
def saveTrafficPolicyFieldSetBindingsVrfConfig( vrfConfigDir, root,
                                                requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )
   vrfList = [ DEFAULT_VRF ] + list( vrfConfigDir.vrfConfig )
   mapCtx = 'tp'

   saveFieldSetMappingsConfig( vrfList, bgpConfig, vrfConfigDir, bgpMode,
                               mapCtx, asdotConfigured, options.saveAll )

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts=( 'routing/bgp/config',
                                'routing/bgp/asn/config', ) )
def saveVrfSelectionPolicyFieldSetBindingsVrfConfig( vrfConfigDir, root,
                                                     requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if options.intfFilter:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )
   vrfList = [ DEFAULT_VRF ]
   mapCtx = 'vsp'

   saveFieldSetMappingsConfig( vrfList, bgpConfig, vrfConfigDir, bgpMode,
                               mapCtx, asdotConfigured, options.saveAll )
