# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import Toggles.RsvpToggleLib
from CliMode.Rsvp import (
      RsvpMode,
      RsvpP2mpMode,
      HitlessRestartMode,
      GrHelperMode,
      GrSpeakerMode,
      )
import CliSave
from RsvpLibCliSave import bandwidthHumanReadable
import IpLibConsts
from CliSavePlugin.EthIntfCliSave import IntfConfigMode
from CliSavePlugin.Security import mgmtSecurityConfigPath
from MultiRangeRule import bitMaskCollToStr
import ReversibleSecretCli
from RoutingIntfUtils import allRoutingProtocolIntfNames
from TypeFuture import TacLazyType

EthType = TacLazyType( "Arnet::EthType" )
IpGenAddr = TacLazyType( "Arnet::IpGenAddr" )

FrrMode = TacLazyType( 'Rsvp::Cli::FrrMode' )
LabelLocalTerminationMode = TacLazyType( 'Rsvp::Cli::LabelLocalTerminationMode' )
P2mpMode = TacLazyType( 'Rsvp::Cli::P2mpMode' )
SrlgMode = TacLazyType( 'Rsvp::Cli::SrlgMode' )

RsvpCliConfig = TacLazyType( 'Rsvp::RsvpCliConfig' )
RsvpLspRequestId = TacLazyType( 'Rsvp::RsvpLspRequestId' )
RsvpP2mpCliConfig = TacLazyType( 'Rsvp::RsvpP2mpCliConfig' )
RsvpLerAgIndex = TacLazyType( 'Rsvp::RsvpLerAdminGroupIndex' )

IntfConfigMode.addCommandSequence( 'Rsvp.intf', after=[ 'Ira.ipIntf' ] )

class RsvpConfigMode( RsvpMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpMode.__init__( self )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RsvpConfigMode, after=[ 'Mpls.config' ] )
RsvpConfigMode.addCommandSequence( 'Rsvp.config' )

class RsvpHitlessRestartConfigMode( HitlessRestartMode, CliSave.Mode ):
   def __init__( self, param ):
      HitlessRestartMode.__init__( self )
      CliSave.Mode.__init__( self, param )

RsvpConfigMode.addChildMode( RsvpHitlessRestartConfigMode, after=[ 'Rsvp.config' ] )
RsvpHitlessRestartConfigMode.addCommandSequence( 'RsvpHr.config' )

class RsvpGrHelperConfigMode( GrHelperMode, CliSave.Mode ):
   def __init__( self, param ):
      GrHelperMode.__init__( self )
      CliSave.Mode.__init__( self, param )

RsvpConfigMode.addChildMode( RsvpGrHelperConfigMode,
                             after=[ RsvpHitlessRestartConfigMode ] )
RsvpGrHelperConfigMode.addCommandSequence( 'RsvpGrHelper.config' )

class RsvpGrSpeakerConfigMode( GrSpeakerMode, CliSave.Mode ):
   def __init__( self, param ):
      GrSpeakerMode.__init__( self )
      CliSave.Mode.__init__( self, param )

RsvpConfigMode.addChildMode( RsvpGrSpeakerConfigMode,
                             after=[ RsvpGrHelperConfigMode ] )
RsvpGrSpeakerConfigMode.addCommandSequence( 'RsvpGrSpeaker.config' )

class RsvpP2mpConfigMode( RsvpP2mpMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpP2mpMode.__init__( self )
      CliSave.Mode.__init__( self, param )

RsvpConfigMode.addChildMode( RsvpP2mpConfigMode, after=[ RsvpGrSpeakerConfigMode ], )
RsvpP2mpConfigMode.addCommandSequence( 'RsvpP2mp.config' )

_cliSavers = []

def rsvpCliSaver( func ):
   _cliSavers.append( func )

@rsvpCliSaver
def saveRefreshInterval( config, cmds, saveDefaultConfig, **kwargs ):
   if config.refreshInterval != config.refreshIntervalDefault or saveDefaultConfig:
      cmds.addCommand( 'refresh interval %d' % config.refreshInterval )

@rsvpCliSaver
def saveRefreshMethod( config, cmds, saveDefaultConfig, **kwargs ):
   if config.refreshMethod == 'refreshExplicit':
      cmds.addCommand( 'refresh method explicit' )
   elif saveDefaultConfig:
      cmds.addCommand( 'refresh method bundled' )

@rsvpCliSaver
def saveHelloIntervalMultiplier( config, cmds, saveDefaultConfig, **kwargs ):
   if config.helloInterval == config.helloIntervalDisabled:
      cmds.addCommand( 'no hello interval' )
   elif ( config.helloInterval != config.helloIntervalDefault or
        config.helloMultiplier != config.helloMultiplierDefault or
        saveDefaultConfig ):
      cmds.addCommand( 'hello interval %d multiplier %d' % (
                           config.helloInterval, config.helloMultiplier ) )

@rsvpCliSaver
def saveAuthenticationType( config, cmds, saveDefaultConfig, **kwargs ):
   if config.authType == 'md5':
      cmds.addCommand( 'authentication type md5' )
   elif saveDefaultConfig:
      cmds.addCommand( 'authentication type none' )

@rsvpCliSaver
def saveAuthenticationWindow( config, cmds, saveDefaultConfig, **kwargs ):
   if config.authSeqWindowSize != config.authSeqWindowSizeDefault \
         or saveDefaultConfig:
      cmds.addCommand( 'authentication sequence-number window %d' %
                       config.authSeqWindowSize )

@rsvpCliSaver
def saveAuthenticationPassword( config, cmds, saveDefaultConfig, **kwargs ):
   requireMounts = kwargs[ 'requireMounts' ]
   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   for index, passwd in config.encodedPassword.items():
      cmd = ReversibleSecretCli.getCliSaveCommand(
         f'authentication index {index} password {{}}',
         securityConfig,
         passwd )
      cmds.addCommand( cmd )

   if config.activePasswordIndex != 0:
      cmds.addCommand( 'authentication index %d active' %
                       config.activePasswordIndex )

@rsvpCliSaver
def saveNeighborAuthentication( config, cmds, saveDefaultConfig, **kwargs ):
   allNeighbors = set( list( config.authTypeForNeighbor ) +
                       list( config.activePasswordIndexForNeighbor ) )
   for neighborIp in sorted( allNeighbors ):
      if neighborIp in config.authTypeForNeighbor:
         authType = config.authTypeForNeighbor[ neighborIp ]
         formatStr = 'neighbor {NEIGHBOR_IP} authentication type {TYPE}'
         cmds.addCommand( formatStr.format( NEIGHBOR_IP=neighborIp, TYPE=authType ) )
      if neighborIp in config.activePasswordIndexForNeighbor:
         index = config.activePasswordIndexForNeighbor[ neighborIp ]
         formatStr = 'neighbor {NEIGHBOR_IP} authentication index {INDEX} active'
         cmds.addCommand( formatStr.format( NEIGHBOR_IP=neighborIp, INDEX=index ) )
   # TODO BUG883684: add a way to remove all neighbor authentication

@rsvpCliSaver
def saveServiceAclConfig( config, cmds, saveDefaultConfig, requireMounts, **kwargs ):
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   for aclType in [ 'ip', 'ipv6' ]:
      aclCpConfigForType = aclCpConfig.cpConfig.get( aclType )
      if aclCpConfigForType is None:
         continue
      serviceAclVrfConfig = \
         aclCpConfigForType.serviceAcl.get( IpLibConsts.DEFAULT_VRF )
      if ( serviceAclVrfConfig and
           serviceAclVrfConfig.service.get( 'rsvp' ) and
           serviceAclVrfConfig.service.get( 'rsvp' ).aclName ):
         cmds.addCommand( '{} access-group {}'.format( aclType,
                           serviceAclVrfConfig.service[ 'rsvp' ].aclName ) )
      elif saveDefaultConfig:
         cmds.addCommand( 'no %s access-group' % aclType )

@rsvpCliSaver
def saveFastRerouteMode( config, cmds, saveDefaultConfig, **kwargs ):
   if config.frrMode == FrrMode.frrLinkProtection:
      cmds.addCommand( 'fast-reroute mode link-protection' )
   elif config.frrMode == FrrMode.frrNodeProtection:
      cmds.addCommand( 'fast-reroute mode node-protection' )
   elif saveDefaultConfig:
      cmds.addCommand( 'fast-reroute mode none' )

@rsvpCliSaver
def saveFastRerouteReversionMode( config, cmds, saveDefaultConfig, **kwargs ):
   if config.reversionMode == 'reversionLocal':
      cmds.addCommand( 'fast-reroute reversion local' )
   elif saveDefaultConfig:
      cmds.addCommand( 'fast-reroute reversion global' )

@rsvpCliSaver
def saveBypassOptimizationInterval( config, cmds, saveDefaultConfig, **kwargs ):
   if config.bypassOptimizationInterval != config.bypassOptimizationIntervalDefault \
         or saveDefaultConfig:
      cmds.addCommand(
            'fast-reroute bypass tunnel optimization interval %d seconds' %
            config.bypassOptimizationInterval )

@rsvpCliSaver
def saveSrlgMode( config, cmds, saveDefaultConfig, **kwargs ):
   if config.srlgMode == SrlgMode.srlgStrict:
      cmds.addCommand( 'srlg strict' )
   elif config.srlgMode == SrlgMode.srlgLoose:
      cmds.addCommand( 'srlg' )
   elif saveDefaultConfig:
      cmds.addCommand( 'no srlg' )

@rsvpCliSaver
def saveLabelLocalTermination( config, cmds, saveDefaultConfig, **kwargs ):
   if config.labelLocalTerminationMode == LabelLocalTerminationMode.explicitNull:
      cmds.addCommand( 'label local-termination explicit-null' )
   elif saveDefaultConfig:
      cmds.addCommand( 'label local-termination implicit-null' )

@rsvpCliSaver
def savePreemptionTimer( config, cmds, saveDefaultConfig, **kwargs ):
   if config.preemptionTimer == config.preemptionTimerDisabled:
      cmds.addCommand( 'preemption method hard' )
   elif config.preemptionTimer != config.preemptionTimerDefault or saveDefaultConfig:
      cmds.addCommand( 'preemption method soft timer %d' %
                       config.preemptionTimer )

@rsvpCliSaver
def saveMtuSignalingEnabled( config, cmds, saveDefaultConfig, **kwargs ):
   if config.mtuSignalingEnabled:
      cmds.addCommand( 'mtu signaling' )
   elif saveDefaultConfig:
      cmds.addCommand( 'no mtu signaling' )

@rsvpCliSaver
def saveDebugTunnels( config, cmds, saveDefaultConfig, **kwargs ):
   for req in config.debugLspRequestColl.debugLspReq.values():
      lspReqId = req.lspReqId
      cmdParts = [ 'debug', 'tunnel' ]
      cmdParts.append( 'id %d' % lspReqId.reqId )
      cmdParts.append( 'sub-id %d' % lspReqId.reqSubId )
      cmdParts.append( 'src %s' % str( req.srcIp ) )
      cmdParts.append( 'dst %s' % str( req.dstIp ) )
      cliEthTypeToStr = { EthType.ethTypeIp : 'ip',
                          EthType.ethTypeIp6 : 'ip6',
                          EthType.ethTypeMpls : 'mpls' }
      cmdParts.append( 'ether-type %s' % cliEthTypeToStr.get( req.ethType, 'ip' ) )
      lpdStr = 'off'
      if req.nodeProtectionDesired:
         lpdStr = 'node'
      elif req.localProtectionDesired:
         lpdStr = 'link'
      cmdParts.append( 'lpd %s' % lpdStr )
      cmdParts.append( 'session-name "%s"' % req.sessionName )
      if req.bandwidth > 0.0:
         bw, unit = bandwidthHumanReadable( req.bandwidth * 8 )
         cmdParts.append( f'bandwidth {bw:.2f} {unit}' )
      if req.softPreemptionDesired:
         preemptStr = 'soft'
      else:
         preemptStr = 'hard'
      cmdParts.append( 'preemption %s' % preemptStr )
      if req.setupPriority < 7:
         cmdParts.append( 'setup-prio %d' % ( req.setupPriority ) )
      if req.holdPriority > 0:
         cmdParts.append( 'hold-prio %d' % ( req.holdPriority ) )
      cmdParts.append( 'hops' )
      for hop in req.hop.values():
         cmdParts.append( str( hop ) )
      cmds.addCommand( ' '.join( cmdParts ) )

@rsvpCliSaver
def saveHitlessRestart( config, cmds, saveDefaultConfig, **kwargs ):
   if config.hitlessRestartEnabled:
      rsvpMode = kwargs[ 'rsvpMode' ]
      rsvpHrMode = rsvpMode[ RsvpHitlessRestartConfigMode ].getSingletonInstance()
      recoveryTimeInSeconds = int( config.hitlessRestartRecoveryTime )
      if config.hitlessRestartRecoveryTime != \
         config.hitlessRestartRecoveryTimeDefault or saveDefaultConfig:
         rsvpHrMode[ 'RsvpHr.config' ].addCommand(
               'timer recovery %d seconds' % recoveryTimeInSeconds )
   elif saveDefaultConfig:
      cmds.addCommand( 'no hitless-restart' )

@rsvpCliSaver
def saveGracefulRestartHelper( config, cmds, saveDefaultConfig, **kwargs ):
   if config.grHelperModeEnabled:
      rsvpMode = kwargs[ 'rsvpMode' ]
      rsvpGrHelperMode = rsvpMode[ RsvpGrHelperConfigMode ].getSingletonInstance()
      if config.grHelperMaximumRestart is not None:
         rsvpGrHelperMode[ 'RsvpGrHelper.config' ].addCommand(
               'timer restart maximum %d seconds' % config.grHelperMaximumRestart )
      elif saveDefaultConfig:
         rsvpGrHelperMode[ 'RsvpGrHelper.config' ].addCommand(
               'no timer restart maximum' )

      if config.grHelperMaximumRecovery is not None:
         rsvpGrHelperMode[ 'RsvpGrHelper.config' ].addCommand(
               'timer recovery maximum %d seconds' % config.grHelperMaximumRecovery )
      elif saveDefaultConfig:
         rsvpGrHelperMode[ 'RsvpGrHelper.config' ].addCommand(
               'no timer recovery maximum' )

   elif saveDefaultConfig:
      cmds.addCommand( 'no graceful-restart role helper' )

@rsvpCliSaver
def saveGracefulRestartSpeaker( config, cmds, saveDefaultConfig, **kwargs ):
   if config.grSpeakerModeEnabled:
      rsvpMode = kwargs[ 'rsvpMode' ]
      rsvpGrSpeakerMode = rsvpMode[ RsvpGrSpeakerConfigMode ].getSingletonInstance()
      mode = rsvpGrSpeakerMode[ 'RsvpGrSpeaker.config' ]
      if config.grSpeakerRestart != config.grSpeakerRestartDefault or\
         saveDefaultConfig:
         mode.addCommand( 'timer restart %d seconds' % config.grSpeakerRestart )
      if config.grSpeakerRecovery != config.grSpeakerRecoveryDefault or\
         saveDefaultConfig:
         mode.addCommand( 'timer recovery %d seconds' % config.grSpeakerRecovery )

   elif saveDefaultConfig:
      cmds.addCommand( 'no graceful-restart role speaker' )

@rsvpCliSaver
def saveP2mpMode( config, cmds, saveDefaultConfig, **kwargs ):
   def getP2mpMode( rsvpMode ):
      return rsvpMode[ RsvpP2mpConfigMode ].\
         getSingletonInstance()[ 'RsvpP2mp.config' ]

   def saveSubGroupMaxSizeCmd( p2mpMode ):
      if not Toggles.RsvpToggleLib.toggleRsvpP2mpLerEnabled():
         return
      subGroupMaxSize = config.p2mpConfig.subGroupMaxSize
      if ( subGroupMaxSize != RsvpP2mpCliConfig.subGroupMaxSizeDefault or
           saveDefaultConfig ):
         p2mpMode.addCommand( f'sub-group limit {subGroupMaxSize} sub-lsps' )

   def saveDisabledCmd( p2mpMode ):
      assert config.p2mpConfig.mode != P2mpMode.p2mpModeNone
      if config.p2mpConfig.mode == P2mpMode.p2mpModeExplicitDisabled:
         p2mpMode.addCommand( 'disabled' )
      elif saveDefaultConfig:
         p2mpMode.addCommand( 'no disabled' )

   if config.p2mpConfig.mode != P2mpMode.p2mpModeNone:
      p2mpMode = getP2mpMode( kwargs[ 'rsvpMode' ] )
      saveSubGroupMaxSizeCmd( p2mpMode )
      saveDisabledCmd( p2mpMode )
   elif saveDefaultConfig:
      cmds.addCommand( 'no p2mp' )

@rsvpCliSaver
def saveBypassAdminGroupConstraints( config, cmds, saveDefaultConfig, **kwargs ):
   root = kwargs[ 'root' ]
   requireMounts = kwargs[ 'requireMounts' ]

   def constraintsToStr( values, names ):
      if len( values ) + len( names ) == 0:
         return ''
      valuesStr = bitMaskCollToStr( values )
      namesStr = ','.join( sorted( names ) )
      if valuesStr and namesStr:
         return valuesStr + ',' + namesStr
      else:
         return valuesStr + namesStr

   if saveDefaultConfig:
      intfs = allRoutingProtocolIntfNames( requireMounts )
   else:
      intfs = config.bypassAdminGroupConstraintsForIntf

   for intf in intfs:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      intfCmds = mode[ 'Rsvp.intf' ]
      if intf in config.bypassAdminGroupConstraintsForIntf:
         constraints = config.bypassAdminGroupConstraintsForIntf[ intf ]
         groups = ''
         includeAll = { k.value : v for k, v in constraints.includeAll.items() }
         constraintsStr = constraintsToStr( includeAll,
                                            constraints.includeAllByName )
         if constraintsStr:
            groups += f' include all {constraintsStr}'

         includeAny = { k.value : v for k, v in constraints.includeAny.items() }
         constraintsStr = constraintsToStr( includeAny,
                                            constraints.includeAnyByName )
         if constraintsStr:
            groups += f' include any {constraintsStr}'

         exclude = { k.value : v for k, v in constraints.exclude.items() }
         constraintsStr = constraintsToStr( exclude, constraints.excludeByName )

         if constraintsStr:
            groups += f' exclude {constraintsStr}'
         if groups:
            intfCmds.addCommand( 'rsvp bypass administrative-group' + groups )
         elif saveDefaultConfig:
            intfCmds.addCommand( 'no rsvp bypass administrative-group' )
      elif saveDefaultConfig:
         intfCmds.addCommand( 'no rsvp bypass administrative-group' )

# Always save this at end
RsvpConfigMode.addCommandSequence( 'Rsvp.shutdown',
                                   after=[ RsvpGrHelperConfigMode,
                                           RsvpGrSpeakerConfigMode,
                                           RsvpP2mpConfigMode ] )
@rsvpCliSaver
def saveShutdown( config, cmds, saveDefaultConfig, **kwargs ):
   rsvpMode = kwargs[ 'rsvpMode' ]
   shutdownCmdSeq = rsvpMode[ 'Rsvp.shutdown' ]
   if config.enabled:
      shutdownCmdSeq.addCommand( 'no shutdown' )
   elif saveDefaultConfig:
      shutdownCmdSeq.addCommand( 'shutdown' )

@CliSave.saver( 'Rsvp::RsvpCliConfig', RsvpCliConfig.mountPath,
                requireMounts=( 'routing/hardware/mpls/capability',
                                'acl/cpconfig/cli', 'interface/status/all',
                                'interface/config/all', mgmtSecurityConfigPath ) )
def saveRsvpCliConfig( rsvpCliConfig, root, requireMounts, options ):
   hwCapability = requireMounts[ 'routing/hardware/mpls/capability' ]

   # Save the default configs when user specifies saveAll and cmds are not guarded.
   # If user specifies saveAll but cmds are guarded, then we only show
   # non-default-configs, i.e., when hwCapability.mplsSupported is False,
   # then show active all only shows non-default configs.
   saveDefaultConfig = options.saveAll and hwCapability.mplsSupported

   mode = root[ RsvpConfigMode ].getSingletonInstance()
   cmds = mode[ 'Rsvp.config' ]

   for saver in _cliSavers:
      saver( rsvpCliConfig, cmds, saveDefaultConfig, requireMounts=requireMounts,
             options=options, rsvpMode=mode, root=root )
