#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from QosTypes import tacRateUnit
from SfeUtils import flowCacheWalkerThreadsDefault
from Arnet import IntfId
from MultiRangeRule import multiRangeToCanonicalString
from CliMode.SfePort import PortConfigModeBase, PortProfileModeBase, PortIntfModeBase
from Toggles.EosInitToggleLib import toggleSfeRssCaravanEnabled
import Tac
import Toggles.SfeToggleLib as Toggle
veosConfigDirPath = 'hardware/sfe/veosConfig'
cpuUtilDirPath = "dp/cpu/util"
PortIntfModeTypeEnum = Tac.Type( 'Sfe::ProfileIntfModeType' )

CliSave.GlobalConfigMode.addCommandSequence( 'Sfe.config' )

@CliSave.saver( "Sfe::BessdCliConfig", "bess/cli/config",
                requireMounts=( veosConfigDirPath, cpuUtilDirPath, ) )
def saveBessDebugConfig( entity, root, requireMounts, options ):
   cmds = root[ "Sfe.config" ]
   veosConfig = requireMounts[ veosConfigDirPath ]
   if Toggle.toggleSfeFctWalkerEnabled():
      cpuUtil = requireMounts[ cpuUtilDirPath ].cpuUtil
      # if dataplane core information is available, then infer number of control
      # plane cores since we assume cpCores are continuous from from 0 - firstDpCore.
      # If not available pass 0 so we can choose the minimum config
      nCpCores = min( cpuUtil ) if cpuUtil else 0

      # If the value of flowCacheWalkerThreads is set to 0, the default
      # number of walker threads is utilized, determined by the platform.
      threadCountDefault = flowCacheWalkerThreadsDefault( veosConfig.platformRuby,
                                                          nCpCores ).value
      if entity.flowCacheWalkerThreads:
         threadCount = entity.flowCacheWalkerThreads
      else:
         threadCount = threadCountDefault

      if threadCount != threadCountDefault or options.saveAll:
         cmds.addCommand( 'agent sfe threads flow cache scan '
                        f'{threadCount}' )

   if entity.traceConfig:
      cmds.addCommand( f"platform sfe debug trace {entity.traceConfig}" )

   if entity.vlogConfig:
      cmds.addCommand( f"platform sfe debug vlog {entity.vlogConfig}" )

   if entity.hashFuncId:
      cmds.addCommand( f"ip load-sharing sfe hash {entity.hashFuncId}" )

   cpuReceiveLimitDefault = Tac.Value( "Sfe::CpuReceiveLimitConfig" )
   if cpuReceiveLimitDefault != entity.cpuReceiveLimit or options.saveAll:
      if entity.cpuReceiveLimit.unit == tacRateUnit.rateUnitKbps:
         cliRateUnit = 'kbps'
      else:
         cliRateUnit = 'mbps'
      cmds.addCommand( "platform sfe control-plane receive-limit "
                       f"{entity.cpuReceiveLimit.value} {cliRateUnit}" )
   if entity.maxDpCores != 0:
      cmds.addCommand(
         f"platform sfe data-plane cpu allocation maximum {entity.maxDpCores}" )

   if entity.natCpuDecapPktTrace:
      cmds.addCommand(
         f"platform sfe nat trace cpu-decap {entity.natCpuDecapPktTrace}" )
   if entity.natCpuEncapPktTrace:
      cmds.addCommand(
         f"platform sfe nat trace cpu-encap {entity.natCpuEncapPktTrace}" )
   if entity.natFlowExportPktTrace:
      cmds.addCommand(
         f"platform sfe nat trace flow-export {entity.natFlowExportPktTrace}" )
   if entity.natFlowExportLearningRate:
      cmds.addCommand(
         f"platform sfe nat learning-rate {entity.natFlowExportLearningRate}" )
   if not entity.ipHwChecksumOffload:
      cmds.addCommand( "platform sfe ip checksum offload disabled" )

#-----------------------------------------------------------------
# Platform Sfe Port CliSave commands start
#-----------------------------------------------------------------
class PortConfigMode( PortConfigModeBase, CliSave.Mode ):
   def __init__( self, param ):
      PortConfigModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class PortProfileConfigMode( PortProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      PortProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class PortIntfConfigMode( PortIntfModeBase, CliSave.Mode ):
   def __init__( self, param ):
      profileName, intName = param
      PortIntfModeBase.__init__( self, profileName, intName )
      CliSave.Mode.__init__( self, self.longModeKey )

@CliSave.saver( 'Sfe::SfePortConfig', 'sfe/port/config' )
def savePortConfig( entity, root, requireMounts, options ):
   if not toggleSfeRssCaravanEnabled():
      return
   portConfigMode = root[ PortConfigMode ].getSingletonInstance()
   for profileName, portProfileConfig in sorted( entity.portProfileConfig.items() ):
      portProfileMode = portConfigMode[ PortProfileConfigMode ].\
                        getOrCreateModeInstance( profileName )
      for intf in sorted( portProfileConfig.profileIntfConfig.keys() ):
         mode = portProfileMode[ PortIntfConfigMode ].\
                        getOrCreateModeInstance( ( profileName, intf ) )
         cmds = mode[ 'PortProfile.Intf' ]
         profileIntf = portProfileConfig.profileIntfConfig[ IntfId( intf ) ]
         if profileIntf.numRxQs != profileIntf.numRxQsDefault or options.saveAll:
            cmd = f"rx-queue count {profileIntf.numRxQs}"
            cmds.addCommand( cmd )
         if profileIntf.rxWorkers:
            workerValuesStr = multiRangeToCanonicalString(
                     list( profileIntf.rxWorkers ) )
            cmd = f"rx-queue worker {workerValuesStr}"
            cmds.addCommand( cmd )
         if profileIntf.rxIntfMode != profileIntf.rxIntfModeDefault or\
               options.saveAll:
            cmd = f"rx-queue mode {profileIntf.rxIntfMode}"
            cmds.addCommand( cmd )

@CliSave.saver( 'Sfe::SfePortConfig', 'sfe/port/config' )
def savePortProfileApply( entity, root, requireMounts, options ):
   if not toggleSfeRssCaravanEnabled():
      return
   portConfigMode = root[ PortConfigMode ].getSingletonInstance()
   cmds = portConfigMode[ 'PortProfileApplied.config' ]
   if entity.portProfileApplied:
      cmds.addCommand( f'port profile {entity.portProfileApplied}' )

def registerPortCommands():
   if not toggleSfeRssCaravanEnabled():
      return
   CliSave.GlobalConfigMode.addCommandSequence( 'Port.config' )
   CliSave.GlobalConfigMode.addChildMode( PortConfigMode )
   PortConfigMode.addChildMode( PortProfileConfigMode )
   PortProfileConfigMode.addCommandSequence( 'Port.profile' )
   PortProfileConfigMode.addChildMode( PortIntfConfigMode,
                                    after=[ 'Port.profile' ] )
   PortIntfConfigMode.addCommandSequence( 'PortProfile.Intf' )
   PortConfigMode.addCommandSequence( 'PortProfileApplied.config' )

registerPortCommands()
#-----------------------------------------------------------------
# Platform Sfe Port CliSave commands end
#-----------------------------------------------------------------
