# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from TypeFuture import TacLazyType
from CliMode.SfeInternetExit import (
        RouterInternetExitModeBase,
        RouterIeExitGroupModeBase,
        RouterIePolicyModeBase,
        )
from Assert import assertNotEqual

ExitType = TacLazyType( 'SfeInternetExit::ExitType' )
PolicyConfig = TacLazyType( 'SfeInternetExit::PolicyConfig' )
ExitGroupConfig = TacLazyType( 'SfeInternetExit::ExitGroupConfig' )

class RouterInternetExitSaveMode( RouterInternetExitModeBase,
                                  CliSave.Mode ):
   def __init__( self, param ):
      RouterInternetExitModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

class RouterIePolicySaveMode( RouterIePolicyModeBase, CliSave.Mode ):
   def __init__( self, param ):
      self.policyName = param
      RouterIePolicyModeBase.__init__( self, self.policyName )
      CliSave.Mode.__init__( self, param )

class RouterIeExitGroupSaveMode( RouterIeExitGroupModeBase,
                                 CliSave.Mode ):
   def __init__( self, param ):
      self.exitGroupName = param
      RouterIeExitGroupModeBase.__init__( self, self.exitGroupName )
      CliSave.Mode.__init__( self, param )

def registerAllCommands():
   CliSave.GlobalConfigMode.addChildMode( RouterInternetExitSaveMode )
   RouterInternetExitSaveMode.addCommandSequence(
                                          'SfeInternetExit.RouterInternetExit' )

   RouterInternetExitSaveMode.addChildMode( RouterIePolicySaveMode )
   RouterIePolicySaveMode.addCommandSequence( 'SfeInternetExit.IePolicy' )

   # exit-group configs should come before policy configs. This is because the
   # keyword 'exit-group' is valid within the policy as well as in parent mode.
   RouterInternetExitSaveMode.addChildMode( RouterIeExitGroupSaveMode,
                        before=[ RouterIePolicySaveMode ] )
   RouterIeExitGroupSaveMode.addCommandSequence( 'SfeInternetExit.IeExitGroup' )

registerAllCommands()

def getIeExitOptionStr( exitType ):
   exitTypeMapper = {
         ExitType.localExit : 'local connection ',
         ExitType.remoteExit : 'remote service group ',
         ExitType.fibDefault : 'fib-default',
         }
   return exitTypeMapper.get( exitType, 'none' )

@CliSave.saver( 'SfeInternetExit::InternetExitCliConfig', 'ie/cli/config' )
def saveConfig( entity, root, requireMounts, options ):
   if not entity.enabled:
      return

   ieMode = root[ RouterInternetExitSaveMode ].getOrCreateModeInstance(
                                             'SfeInternetExit.RouterInternetExit' )
   saveAll = options.saveAll
   # handle the exit-groups
   for egName, egCfg in entity.exitGroupConfig.items():
      # We are not printing default exit-group config, if saveAll option not set
      if ( not saveAll and egName == ExitGroupConfig.defaultName ):
         continue
      egMode = ieMode[ RouterIeExitGroupSaveMode ].getOrCreateModeInstance(
                                                     egName )
      egCmds = egMode[ 'SfeInternetExit.IeExitGroup' ]
      # requires manual sort as TACC does not support ordered sets
      for exitOption in sorted( egCfg.exitOption ):
         typeStr = getIeExitOptionStr( exitOption.type )
         assertNotEqual( typeStr, 'none', msg='Invalid ExitType' )
         egCmds.addCommand( f'{typeStr}{exitOption.name}' )

   # handle policy configs
   for policyName, policy in entity.policyConfig.items():
      # We are not printing default policy config, if saveAll option not set
      if ( not saveAll and policyName == PolicyConfig.defaultName ):
         continue
      polMode = ieMode[ RouterIePolicySaveMode ].getOrCreateModeInstance(
                                                   policyName )
      polCmds = polMode[ 'SfeInternetExit.IePolicy' ]
      for exitGroup in policy.exitGroup.values():
         polCmds.addCommand( f'exit-group {exitGroup.exitGroupName}' )
