#!/usr/bin/env python3
# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from binascii import hexlify
from CliMode.SnmpOverride import SnmpOverrideMode, SnmpOverrideContextMode
import CliSave
from CliSave import escapeFormatString
from CliSaveBlock import SensitiveCommand
import Tracing
import Url
import Tac
from IpLibConsts import DEFAULT_VRF
from SnmpCliUtil import trapToToken
from CliSavePlugin.NetworkCliSave import networkConfigCmdSeq

__defaultTraceHandle__ = Tracing.Handle( 'SnmpCliSave' )

# I want the Snmp commands to go after Network.config because some of the Snmp
# commands take hostname-or-ip-addrs and attempt to resolve hostnames, so it's
# better if the nameserver config is done first.
CliSave.GlobalConfigMode.addCommandSequence( 'Snmp.global',
   after=[ networkConfigCmdSeq ] )

class SnmpOverrideConfigMode( SnmpOverrideMode, CliSave.Mode ):
   def __init__( self, param ):
      SnmpOverrideMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class SnmpOverrideContextConfigMode( SnmpOverrideContextMode, CliSave.Mode ):
   def __init__( self, param ):
      SnmpOverrideContextMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( SnmpOverrideConfigMode,
                                       after=[ 'Snmp.global' ] )
# Since there are not actually any commands in this mode, we do not create
# a command sequence for it.  (Comments still work without a command
# sequence.)
SnmpOverrideConfigMode.addChildMode( SnmpOverrideContextConfigMode )
SnmpOverrideContextConfigMode.addCommandSequence( 'Snmp.override.context' )

_authTypes = { 'authMd5': 'md5', 'authSha': 'sha', 'authSha224': 'sha224',
      'authSha256': 'sha256', 'authSha384': 'sha384', 'authSha512': 'sha512' }
_authLevels = { 'levelNoAuth': 'noauth', 'levelAuth': 'auth',
                'levelAuthAndPriv': 'priv' }
_privTypes = { 'privacyAes': 'aes', 'privacyDes': 'des',
      'privacy3Des': '3des', 'privacyAes192': 'aes192',
      'privacyAes256': 'aes256' }
_defaultRemotePort = 162

# When someone types "show running-config", the Cli code walks the sysdb
# object tree, calling all the functions which registered with CliSave.saver
# to be called for a particular object. Below, I register
# saveSnmpConfig to be called for 'snmp/config'. When I'm called, I walk our
# entire Snmp config object tree, generating all non-default config.
@CliSave.saver( 'Snmp::Config', 'snmp/config',
                requireMounts=( 'hardware/entmib', ) )
def saveSnmpConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Snmp.global' ]
   saveAll = options.saveAll

   if entity.engineId != "":
      cmds.addCommand( 'snmp-server engineID local %s' % entity.engineId )
   else:
      # Intentionally making it so the generator is completely transient and a
      # reference can never be saved to it
      def getEngineId():
         Tracing.trace0( "Instantiating transient EngineIdGenerator" )
         entityMibStatus = requireMounts[ 'hardware/entmib' ]
         return Tac.newInstance( "Snmp::EngineIdGenerator",
                                 entityMibStatus, entity ).engineId

      # If there are any SNMPv3 users we emit the default engineID in the
      # running-config just to be safe, since the SNMPv3 users have localized
      # keys based on the engineID.
      engineIdDisplayed = False
      for u in entity.user.values():
         if u.protocolVersion == 'v3':
            cmds.addCommand( 'snmp-server engineID local %s' % getEngineId() )
            engineIdDisplayed = True
            break
      if not engineIdDisplayed and saveAll:
         cmds.addCommand( 'no snmp-server engineID local' )

   def _addCmdIfNotEmpty( attr, cliName ):
      val = getattr( entity, attr )
      if val != "":
         cmds.addCommand( 'snmp-server %s %s' % ( cliName, val ) )
      elif saveAll:
         cmds.addCommand( 'no snmp-server %s' % cliName )

   stringAttrInfo = [
      ( 'chassisId', 'chassis-id' ),
      ( 'contact', 'contact' ),
      ( 'location', 'location' ) ]

   for attr in stringAttrInfo:
      _addCmdIfNotEmpty( attr[ 0 ], attr[ 1 ] )

   if entity.tcpTransport == False: # pylint: disable=singleton-comparison
      cmds.addCommand( "no snmp-server transport tcp" )
   elif saveAll:
      cmds.addCommand( "default snmp-server transport tcp" )

   srcIntfs = entity.notificationSourceIntf
   for vrf in srcIntfs:
      n = srcIntfs[ vrf ]
      if vrf == DEFAULT_VRF and not saveAll:
         cmds.addCommand( "snmp-server local-interface %s" % n )
      else:
         cmds.addCommand( "snmp-server vrf %s local-interface %s" % ( vrf, n ) )
   if not srcIntfs and saveAll:
      cmds.addCommand( "no snmp-server vrf default local-interface" )

   for v in entity.view.values():
      subtrees = v.subtree
      for st in subtrees.values():
         cmds.addCommand( "snmp-server view %s %s %s" % ( v.viewName, st.root,
                                                         st.viewType ) )

   for c in entity.community.values():
      cmd = "snmp-server community {}" # place holder for communityString
      if c.view != "":
         cmd += " view " + escapeFormatString( c.view )
      cmd += " " + escapeFormatString( c.access )
      if c.acl6 != "":
         cmd += " ipv6 " + escapeFormatString( c.acl6 )
      if c.acl != "":
         cmd += " " + escapeFormatString( c.acl )
      if c.context:
         cmd += " default-context " + escapeFormatString( c.context )
      cmds.addCommand( SensitiveCommand( cmd, c.communityString ) )

   for g in entity.group.values():
      cmd = "snmp-server group %s %s" % ( g.groupName, g.protocolVersion )
      if g.protocolVersion == 'v3':
         cmd += " " + _authLevels[ g.authLevel ]
      if g.context != "":
         cmd += " context " + g.context
      if g.readView != "":
         cmd += " read " + g.readView
      if g.writeView != "":
         cmd += " write " + g.writeView
      if g.notifyView != "":
         cmd += " notify " + g.notifyView
      cmds.addCommand( cmd )

   for u in entity.user.values():
      cmd = f"snmp-server user {u.userName} {u.group} {u.protocolVersion}"
      formatStr = None
      tokens = []
      if u.protocolVersion == 'v3':
         if u.authType != 'authNone' and u.authLocalizedKey != "":
            formatStr = escapeFormatString( cmd + f" localized {u.engineId} auth "
                                            f"{_authTypes[ u.authType ]} " ) + "{}"
            tokens.append( u.authLocalizedKey )
         if u.privacyType != 'privacyNone' and u.privacyLocalizedKey != "":
            formatStr += \
               f" priv {escapeFormatString( _privTypes[ u.privacyType ] )} {{}}"
            tokens.append( u.privacyLocalizedKey )
      if formatStr:
         cmd = SensitiveCommand( formatStr, *tuple( tokens ) )
      cmds.addCommand( cmd )

   for e in entity.remoteEngineId.values():
      cmd = "snmp-server engineID remote %s" % ( e.spec.hostname )
      if e.spec.port != _defaultRemotePort or saveAll:
         cmd += " udp-port %d" % ( e.spec.port )
      cmd += " " + e.engineId
      cmds.addCommand( cmd )

   for u in entity.remoteUser.values():
      cmd = "snmp-server user {user} {group} remote {host}"
      cmd = cmd.format( user=u.userName, group=u.group, host=u.spec.hostname )
      if u.spec.port != _defaultRemotePort or saveAll:
         cmd += f" udp-port {u.spec.port}"
      cmd += " v3"
      if u.authType != 'authNone' and u.authLocalizedKey != "":
         authType = _authTypes[ u.authType ]
         cmd += f" localized {u.engineId} auth {authType} {u.authLocalizedKey}"
      if u.privacyType != 'privacyNone' and u.privacyLocalizedKey != "":
         privType = _privTypes[ u.privacyType ]
         cmd += f" priv {privType} {u.privacyLocalizedKey}"
      cmds.addCommand( cmd )

   for h in entity.notificationSink.values():
      cmd = "snmp-server host %s" % ( h.hostname )
      if h.vrf:
         cmd += " vrf " + h.vrf
      elif saveAll:
         cmd += " vrf " + DEFAULT_VRF
      if h.notificationType != 'trap' or saveAll:
         # turn "inform" into "informs" and "trap" into "traps"
         cmd += " " + h.notificationType + "s"
      if h.refresh:
         cmd += " refresh"
      cmd += " version " + h.protocolVersion[ 1 : ] # drop the leading 'v'
      if h.protocolVersion == 'v3':
         cmd += " " + _authLevels[ h.authLevel ]

      cmd = escapeFormatString( cmd ) + " {}" # place holder for securityName
      if h.port != _defaultRemotePort or saveAll:
         cmd += f" udp-port {h.port}"

      cmds.addCommand( SensitiveCommand( cmd, h.securityName ) )

   if entity.notificationsGlobalEnabled == 'notifEnabled':
      cmds.addCommand( "snmp-server enable traps" )
   elif entity.notificationsGlobalEnabled == 'notifDisabled':
      cmds.addCommand( "no snmp-server enable traps" )
   elif saveAll:
      cmds.addCommand( "default snmp-server enable traps" )

   def addAllNotifCmds( notificationTypes, parentPath ):
      for name, n in sorted( notificationTypes.items() ):
         path = ' '.join( parentPath + [ name ] )
         if n.enabled == 'notifEnabled':
            cmds.addCommand( "snmp-server enable traps %s" % path )
         elif n.enabled == 'notifDisabled':
            cmds.addCommand( "no snmp-server enable traps %s" % path )
         elif saveAll:
            cmds.addCommand( "default snmp-server enable traps %s" % path )

         for trap, t in sorted( n.notification.items() ):
            if t.enabled == 'notifEnabled':
               cmds.addCommand( "snmp-server enable traps %s %s" %
                     ( path, trapToToken( trap, n.strip ) ) )
            elif t.enabled == 'notifDisabled':
               cmds.addCommand( "no snmp-server enable traps %s %s" %
                     ( path, trapToToken( trap, n.strip ) ) )
            elif saveAll:
               cmds.addCommand( "default snmp-server enable traps %s %s" %
                     ( path, trapToToken( trap, n.strip ) ) )
         addAllNotifCmds( n.subtype, parentPath + [ name ] )

   addAllNotifCmds( entity.notificationConfig, [] )

   for objectName, disabledConfig in entity.disableObjectsConfig.items():
      if not disabledConfig.enabled:
         cmds.addCommand( "snmp-server objects %s disable" % ( objectName, ) )
      elif saveAll:
         cmds.addCommand( "no snmp-server objects %s disable" % ( objectName, ) )

   for oid, e in entity.extension.items():
      cmds.addCommand( "snmp-server extension %s %s%s" % ( oid,
         Url.filenameToUrl( e.handler, simpleFile=False ),
         " one-shot" if e.oneShot else "" ) )

   # See BUG1720.
   if not DEFAULT_VRF in entity.vrf:
      cmds.addCommand( "no snmp-server vrf %s" % DEFAULT_VRF )
   for vrf in entity.vrf:
      if vrf != DEFAULT_VRF or saveAll:
         cmds.addCommand( "snmp-server vrf %s" % vrf )

   notifLogEntryLimit = entity.notificationLogEntryLimit
   if notifLogEntryLimit != entity.notificationLogEntryLimitDefault or saveAll:
      cmds.addCommand( 'snmp-server notification log entry limit %s' %
                       notifLogEntryLimit )

   # snmp-server qos dscp <dscpValue>
   if entity.dscpValue != entity.dscpValueDefault:
      cmds.addCommand( 'snmp-server qos dscp %s' % entity.dscpValue )
   elif saveAll:
      cmds.addCommand( 'snmp-server qos dscp %s' % entity.dscpValueDefault )

   if entity.transmitMsgSize != entity.transmitMsgSizeDefault or saveAll:
      cmds.addCommand( 'snmp-server transmit max-size %u' %
                       entity.transmitMsgSize )

   top = root[ SnmpOverrideConfigMode ].getOrCreateModeInstance( None )
   for context in sorted( entity.contextOverride ):
      mode = top[ SnmpOverrideContextConfigMode ].getOrCreateModeInstance( context )
      cmds = mode[ 'Snmp.override.context' ]
      for oid in sorted( entity.contextOverride[ context ].objectOverride ):
         cmd = 'oid ' + oid
         obj = entity.contextOverride[ context ].objectOverride[ oid ]
         if obj.tableInstance:
            cmd += ' table-instance'
         if obj.asnType == 'asnInteger' and obj.integerValue is not None:
            cmd += ' integer %d' % obj.integerValue
         elif obj.asnType == 'asnOctetStr' and obj.octetStringValue is not None:
            if obj.octetStringHex:
               cmd += ' octet-string 0x%s' % hexlify( obj.octetStringValue ). \
                                                                decode( 'utf-8' )
            else:
               cmd += ' octet-string %s' % obj.octetStringValue.decode( 'utf-8' )
         elif obj.asnType == 'asnObjectId' and obj.oidValue is not None:
            cmd += ' object-id %s' % obj.oidValue
         elif obj.asnType == 'asnCounter' and obj.unsignedValue is not None:
            cmd += ' counter %u' % obj.unsignedValue
         elif obj.asnType == 'asnUnsigned' and obj.unsignedValue is not None:
            cmd += ' unsigned %u' % obj.unsignedValue
         elif obj.asnType == 'asnTimeTicks' and obj.unsignedValue is not None:
            cmd += ' timeticks %u' % obj.unsignedValue
         cmds.addCommand( cmd )

@CliSave.saver( 'Snmp::NetSnmpConfig', 'snmp/netSnmpConfig' )
def saveSnmpAgentConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Snmp.global' ]
   if entity.clientRecvBuf > 0:
      cmds.addCommand( "snmp-server net-snmp buffer client receive %s" %
                       entity.clientRecvBuf )
   if entity.clientSendBuf > 0:
      cmds.addCommand( "snmp-server net-snmp buffer client send %s" %
                       entity.clientSendBuf )
   if entity.serverRecvBuf > 0:
      cmds.addCommand( "snmp-server net-snmp buffer server receive %s" %
                       entity.serverRecvBuf )
   if entity.serverSendBuf > 0:
      cmds.addCommand( "snmp-server net-snmp buffer server send %s" %
                       entity.serverSendBuf )

@CliSave.saver( 'Acl::Input::CpConfig', 'acl/cpconfig/cli',
                requireMounts=( 'acl/config/cli', ) )
def saveSnmpAclConfig( aclCpConfig, root, requireMounts, options ):
   cmds = root[ 'Snmp.global' ]
   # save for snmp-server service acl
   for aclType in ( 'ip', 'ipv6' ):
      serviceAcl = aclCpConfig.cpConfig[ aclType ].serviceAcl
      for vrfName, serviceAclVrfConfig in sorted( serviceAcl.items() ):
         # since the configuration for snmp-udp and snmp-tcp is always
         # the same and is not known to the user, only snmp-udp is used here
         serviceAclConfig = serviceAclVrfConfig.service.get( 'snmp-udp' )
         if serviceAclConfig:
            cmd = 'snmp-server '
            cmd += 'ipv4' if aclType == 'ip' else aclType
            cmd += ' access-list ' + serviceAclConfig.aclName
            if vrfName != DEFAULT_VRF:
               cmd += " vrf " + vrfName
            cmds.addCommand( cmd )
