#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.Srv6 import ( RoutingSrv6Mode,
                           RoutingSrv6VrfMode,
                           RoutingSrv6VrfUsidDomainMode,
                           RoutingSrv6VrfLocatorMode )
from CliSavePlugin.RouterGeneralCliSave import RouterGeneralCliSaveMode
from Toggles import Srv6LibToggleLib

class RouterSrv6CliSaveMode( RoutingSrv6Mode, CliSave.Mode ):
   def __init__( self, param=None ):
      RoutingSrv6Mode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterSrv6VrfCliSaveMode( RoutingSrv6VrfMode, CliSave.Mode ):
   def __init__( self, param=None ):
      RoutingSrv6VrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterSrv6VrfUsidDomainCliSaveMode( RoutingSrv6VrfUsidDomainMode,
                                          CliSave.Mode ):
   def __init__( self, param=None ):
      RoutingSrv6VrfUsidDomainMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterSrv6VrfLocatorCliSaveMode( RoutingSrv6VrfLocatorMode, CliSave.Mode ):
   def __init__( self, param=None ):
      RoutingSrv6VrfLocatorMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterSrv6CliSaveMode,
                                       after=[ RouterGeneralCliSaveMode ] )
RouterSrv6CliSaveMode.addCommandSequence( 'config' )

RouterSrv6CliSaveMode.addChildMode( RouterSrv6VrfCliSaveMode )
RouterSrv6VrfCliSaveMode.addCommandSequence( 'vrf.config' )

RouterSrv6VrfCliSaveMode.addChildMode( RouterSrv6VrfUsidDomainCliSaveMode,
                                       after=[ 'vrf.config' ] )
RouterSrv6VrfUsidDomainCliSaveMode.addCommandSequence( 'vrf.domain.config' )

RouterSrv6VrfCliSaveMode.addChildMode( RouterSrv6VrfLocatorCliSaveMode,
                                       after=[ RouterSrv6VrfUsidDomainCliSaveMode ] )
RouterSrv6VrfLocatorCliSaveMode.addCommandSequence( 'vrf.locator.config' )

def saveUsidDomainConfig( usidDomainConfig, usidDomainMode, saveAll ):
   cmds = usidDomainMode[ 'vrf.domain.config' ]
   if usidDomainConfig.block != usidDomainConfig.blockDefault:
      cmds.addCommand( 'block %s' % usidDomainConfig.block )
   elif saveAll:
      cmds.addCommand( 'no block' )

   if usidDomainConfig.usidLength != usidDomainConfig.usidLengthDefault or saveAll:
      cmds.addCommand( 'usid length %s' % usidDomainConfig.usidLength )

   if usidDomainConfig.gibRange != usidDomainConfig.gibRangeDefault or saveAll:
      cmds.addCommand( 'gib range %d - %d' % ( usidDomainConfig.gibRange.min,
                                               usidDomainConfig.gibRange.max ) )

def saveLocatorConfig( locatorConfig, locatorMode, saveAll ):
   cmds = locatorMode[ 'vrf.locator.config' ]
   if locatorConfig.functionLength != locatorConfig.functionLengthDefault or saveAll:
      cmds.addCommand( 'function length %s' % locatorConfig.functionLength )

   usidPrefixCfg = locatorConfig.usidPrefixConfig
   if locatorConfig.prefix != locatorConfig.prefixDefault:
      cmds.addCommand( 'prefix %s' % locatorConfig.prefix )
   elif usidPrefixCfg != locatorConfig.usidPrefixConfigDefault:
      cmds.addCommand( 'prefix micro-segment domain %s end usid %d' % (
         usidPrefixCfg.domain, usidPrefixCfg.globalUsid ) )
   elif saveAll:
      cmds.addCommand( 'no prefix' )

   if Srv6LibToggleLib.toggleSrv6EndSidEnabled():
      if locatorConfig.endSid != locatorConfig.endSidDefault:
         endSid = locatorConfig.endSid
         cmds.addCommand( f'sid end function {endSid.function}' )
      elif saveAll:
         cmds.addCommand( 'no sid end function' )

def saveSrv6VrfConfig( vrfConfig, vrfMode, saveAll ):
   vrfName = vrfConfig.vrfName
   cmds = vrfMode[ 'vrf.config' ]
   if vrfConfig.enabled != vrfConfig.enabledDefault:
      cmds.addCommand( 'disabled' )
   elif saveAll:
      cmds.addCommand( 'no disabled' )

   if vrfConfig.sourceIp6Addr != vrfConfig.sourceIp6AddrDefault:
      cmds.addCommand( 'local address %s' % vrfConfig.sourceIp6Addr )
   elif saveAll:
      cmds.addCommand( 'no local address' )

   for domainName in vrfConfig.usidDomainConfig:
      usidDomainConfig = vrfConfig.usidDomainConfig[ domainName ]
      usidDomainMode = vrfMode[ RouterSrv6VrfUsidDomainCliSaveMode
                               ].getOrCreateModeInstance( ( vrfName, domainName ) )
      saveUsidDomainConfig( usidDomainConfig, usidDomainMode, saveAll )

   for locatorName in vrfConfig.locatorConfig:
      locatorConfig = vrfConfig.locatorConfig[ locatorName ]
      locatorMode = vrfMode[ RouterSrv6VrfLocatorCliSaveMode
                            ].getOrCreateModeInstance( ( vrfName, locatorName ) )
      saveLocatorConfig( locatorConfig, locatorMode, saveAll )

@CliSave.saver( 'Srv6::Config', 'routing/srv6/config' )
def saveConfig( srv6Config, root, requireMounts, options ):
   saveAll = options.saveAll

   if not Srv6LibToggleLib.toggleSrv6CoreEnabled():
      return

   if not srv6Config.created:
      return

   mode = root[ RouterSrv6CliSaveMode ].getSingletonInstance()
   if srv6Config.defaultVrfConfig:
      vrfConfig = srv6Config.defaultVrfConfig
      vrfMode = mode[ RouterSrv6VrfCliSaveMode ].getOrCreateModeInstance(
         vrfConfig.name )
      saveSrv6VrfConfig( vrfConfig, vrfMode, saveAll )
