#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.StunCliMode import StunBaseMode, StunServerBaseMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from TypeFuture import TacLazyType

SslConnectionLifetimeUnit = TacLazyType( "Stun::SslConnectionLifetimeUnit" )
StunTransportType = TacLazyType( "Stun::TransportType" )
TurnserverTracingLevel = TacLazyType( "Stun::TurnserverTracingLevel" )

class StunCliSaveMode( StunBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      StunBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( StunCliSaveMode,
                                       after=[ IntfConfigMode ] )

class StunServerCliSaveMode( StunServerBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      StunServerBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

StunCliSaveMode.addChildMode( StunServerCliSaveMode )

def isServerConfigDefault( stunServerConfig ):
   return (
      stunServerConfig.disabled == stunServerConfig.disabledDefault and
      not stunServerConfig.localIntfIds and
      stunServerConfig.port == stunServerConfig.portDefault and
      stunServerConfig.passwordProfile == stunServerConfig.passwordProfileDefault and
      stunServerConfig.sslProfile == stunServerConfig.sslProfileDefault and
      stunServerConfig.transportType == stunServerConfig.transportTypeDefault and
      ( stunServerConfig.sslConnectionLifetime.lifetime ==
        stunServerConfig.sslConnectionLifetimeDefault and
        stunServerConfig.sslConnectionLifetime.unit ==
        SslConnectionLifetimeUnit.minutes ) )

@CliSave.saver( 'Stun::ServerConfig', 'stun/server/config' )
def saveStunServerConfig( stunServerConfig, root, requireMounts, options ):
   # We shouldn't save any command if all the attributes in the config are set to the
   # default values and 'show running-config' wasn't called with saveAll option
   if ( not options.saveAll and isServerConfigDefault( stunServerConfig ) ):
      return

   stunMode = root[ StunCliSaveMode ].getSingletonInstance()
   stunServerMode = stunMode[ StunServerCliSaveMode ].getSingletonInstance()
   cmds = stunServerMode[ 'Stun.serverConfig' ]

   if stunServerConfig.disabled:
      cmds.addCommand( "disabled" )
   elif options.saveAll:
      cmds.addCommand( "no disabled" )

   # need to sort manually until TACC supports ordered sets
   for intf in sorted( stunServerConfig.localIntfIds ):
      cmds.addCommand( "local-interface %s" % intf )

   if stunServerConfig.port != stunServerConfig.portDefault:
      cmds.addCommand( "port %s" % stunServerConfig.port )
   elif options.saveAll:
      cmds.addCommand( "no port" )

   if stunServerConfig.passwordProfile != stunServerConfig.passwordProfileDefault:
      cmds.addCommand( "password profile " + stunServerConfig.passwordProfile )
   elif options.saveAll:
      cmds.addCommand( "no password profile" )

   if stunServerConfig.sslProfile != stunServerConfig.sslProfileDefault:
      cmds.addCommand( "ssl profile " + stunServerConfig.sslProfile )
   elif options.saveAll:
      cmds.addCommand( "no ssl profile" )

   if stunServerConfig.transportType == StunTransportType.tcp:
      cmds.addCommand( "transport protocol tcp" )
   # Do not save hidden commands in saveAll
   # elif options.saveAll:
   #    cmds.addCommand( "no transport protocol" )

   if ( stunServerConfig.bindingResponseTimeout !=
        stunServerConfig.bindingResponseTimeoutDefault ):
      tm = stunServerConfig.bindingResponseTimeout
      cmds.addCommand( f"binding timeout {int( tm )} seconds" )
   elif options.saveAll:
      cmds.addCommand( "no binding timeout" )

   if ( stunServerConfig.turnTracing == TurnserverTracingLevel.errors and
        options.saveAll ):
      cmds.addCommand( "trace level errors" )
   elif stunServerConfig.turnTracing == TurnserverTracingLevel.info:
      cmds.addCommand( "trace level info" )
   elif stunServerConfig.turnTracing == TurnserverTracingLevel.verbose:
      cmds.addCommand( "trace level verbose" )

   if stunServerConfig.forceRestartOnConfigChange:
      cmds.addCommand( "config change force-restart" )
   # Do not save hidden commands in saveAll
   # elif options.saveAll:
   #    cmds.addCommand( "no config change force-restart" )

   if stunServerConfig.udpdtls:
      cmds.addCommand( "config change security migration" )
   # Do not save hidden commands in saveAll
   # elif options.saveAll:
   #    cmds.addCommand( "no config change security migration" )

   if ( stunServerConfig.sslConnectionLifetime.lifetime !=
        stunServerConfig.sslConnectionLifetimeDefault ):
      divisor = 60              # if time units is minutes
      units = "minutes"
      if ( stunServerConfig.sslConnectionLifetime.unit ==
           SslConnectionLifetimeUnit.hours ):
         divisor = 60 * 60
         units = "hours"
      lt = int( stunServerConfig.sslConnectionLifetime.lifetime / divisor )
      cmds.addCommand( f"ssl connection lifetime {lt} {units}" )
   elif options.saveAll:
      cmds.addCommand( "no ssl connection lifetime" )

StunServerCliSaveMode.addCommandSequence( 'Stun.serverConfig' )
