#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.Swag import (
   SwagMode,
   SwagTopoMode,
   SwagTopoMemberMode,
)
from CliSavePlugin.IntfCliSave import IntfConfigMode
from Toggles.SwagCoreToggleLib import toggleSwagPhase1Enabled
from TypeFuture import TacLazyType
import CliSave
import Tracing

traceHandle = Tracing.Handle( 'Swag::SwagCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

FabricChannelIntfId = TacLazyType( "Arnet::FabricChannelIntfId" )

class SwagSaveMode( SwagMode, CliSave.Mode ):
   def __init__( self, param ):
      SwagMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class SwagTopoSaveMode( SwagTopoMode, CliSave.Mode ):
   def __init__( self, param ):
      SwagTopoMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class SwagTopoMemberSaveMode( SwagTopoMemberMode, CliSave.Mode ):
   def __init__( self, param ):
      SwagTopoMemberMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return False

if toggleSwagPhase1Enabled():
   IntfConfigMode.addCommandSequence( 'Swag.fcIntf' )
   CliSave.GlobalConfigMode.addChildMode( SwagSaveMode,
                                          after=[ IntfConfigMode ] )
   SwagSaveMode.addCommandSequence( 'Swag.config' )
   SwagSaveMode.addChildMode( SwagTopoSaveMode,
                              after=[ 'Swag.config' ] )
   SwagTopoSaveMode.addCommandSequence( 'Swag.topo' )
   SwagTopoSaveMode.addChildMode( SwagTopoMemberSaveMode,
                                  after=[ 'Swag.topo' ] )
   SwagTopoMemberSaveMode.addCommandSequence( 'Swag.topoMember' )

@CliSave.saver( 'SwagAgent::Config', 'swag/config',
                requireMounts=( 'interface/config/eth/lag', ) )
def saveSwagConfig( swagConfig, root, requireMounts, options ):
   # If toggle is disabled or Swag is not supported in hardware, bail
   # out.
   if not toggleSwagPhase1Enabled():
      t0( 'toggle SwagPhase1 disabled' )
      return
   toRun = []
   saveAll = options.saveAll or options.saveAllDetail
   if swagConfig.memberId:
      toRun.append( f'member id {swagConfig.memberId}' )
   elif saveAll:
      toRun.append( 'no member id' )

   if swagConfig.vlan:
      toRun.append( f'control network vlan {swagConfig.vlan}' )
   elif saveAll:
      toRun.append( 'no control network vlan' )

   mode = root[ SwagSaveMode ].getOrCreateModeInstance( None )
   cmds = mode[ 'Swag.config' ]
   if toRun:
      for cmd in toRun:
         cmds.addCommand( cmd )

   topoMode = mode[ SwagTopoSaveMode ].getOrCreateModeInstance( None )
   if swagConfig.topologyConfig.topoConfig:
      for memberId, topoConfig in swagConfig.topologyConfig.topoConfig.items():
         topoMemberMode = (
            topoMode[ SwagTopoMemberSaveMode ].getOrCreateModeInstance( memberId ) )
         topoMemberCmds = topoMemberMode[ 'Swag.topoMember' ]

         for neighborMemberId, fcIntf in topoConfig.neighbor.items():
            topoMemberCmds.addCommand(
               f'interface {fcIntf} member {neighborMemberId}' )

   elif saveAll:
      cmds.addCommand( 'no topology' )

   ethLagIntfConfigDir = requireMounts[ 'interface/config/eth/lag' ]
   for intf in ethLagIntfConfigDir.intfConfig:
      if not FabricChannelIntfId.isFabricChannelIntfId( intf ):
         continue
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Swag.fcIntf' ]
      memberId = FabricChannelIntfId.memberId( intf )
      topoConfig = swagConfig.topologyConfig.topoConfig.get( memberId, None )
      if topoConfig:
         found = False
         for peerId, fcIntf in topoConfig.neighbor.items():
            if fcIntf == intf:
               cmds.addCommand( f'peer member id {peerId}' )
               found = True
               break
         if not found and saveAll:
            cmds.addCommand( 'no peer member id' )
      elif saveAll:
         cmds.addCommand( 'no peer member id' )
