# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import (
   absolute_import,
   division,
   print_function,
)

from CliSave import (
   GlobalConfigMode,
   Mode,
   saver,
)
from SynceTypesFuture import Synce
import six

from CliMode.Synce import (
   SynceIntfMode,
   SynceMode,
)
from CliSavePlugin.IntfCliSave import IntfConfigMode

# sync-e [ instance NAME ]
#    network option ( 1 | 2 )
class SynceConfigSaveMode( SynceMode, Mode ):
   def __init__( self, param ):
      SynceMode.__init__( self )
      Mode.__init__( self, param )

GlobalConfigMode.addChildMode( SynceConfigSaveMode )
SynceConfigSaveMode.addCommandSequence( 'Synce.config' )

@saver( 'Synce::CliConfig', 'synce/config/instance' )
def saveSynceConfig( entity, root, requireMounts, options ):
   if not entity.enable:
      return
   mode = root[ SynceConfigSaveMode ].getOrCreateModeInstance( None )
   cmds = mode[ 'Synce.config' ]
   # Network Option
   if entity.networkOption == Synce.NetworkOption.invalid:
      if options.saveAll:
         cmds.addCommand( 'default network option' )
   else:
      optionCmd = Synce.CliHelper.networkOptionCapiValue( entity.networkOption )
      cmds.addCommand( 'network option {}'.format( optionCmd ) )
   # Wait-to-restore is a hidden command, so we only display it if it's configured
   if entity.waitToRestore != Synce.WaitToRestore():
      cmds.addCommand( 'timer wait-to-restore {} seconds'.format(
         entity.waitToRestore ) )

# interface EthernetXX
#    sync-e [ instance NAME ]
#       disabled
#       priority PRIORITY
#       quality-level rx network option ( 1 OPTION1QL | 2 OPTION2QL )
class SynceIntfConfigSaveMode( SynceIntfMode, Mode ):
   def __init__( self, intfName ):
      SynceIntfMode.__init__( self, intfName )
      Mode.__init__( self, intfName )

IntfConfigMode.addChildMode( SynceIntfConfigSaveMode, after=[ 'EthIntf.ethIntf' ] )
SynceIntfConfigSaveMode.addCommandSequence( 'Synce.intfConfig' )

def qlCmd( direction, ql ):
   # direction { 'rx', 'tx' }
   qlStr = Synce.QualityLevelHelper.toStr( ql )
   if Synce.QualityLevelHelper.isInternal( ql ):
      # for non-default internal values, they're errors that we silently ignore
      return None
   else:
      option = ( Synce.NetworkOption.option1
                 if Synce.QualityLevelHelper.isOption1( ql ) else
                 Synce.NetworkOption.option2 )
      optionStr = Synce.CliHelper.networkOptionCapiValue( option )
      return 'quality-level {direction} network option {option} {ql}'.format(
         direction=direction, option=optionStr, ql=qlStr )

@saver( 'Synce::AllCliClockConfig', 'synce/config/clocks' )
def saveSynceIntfConfig( entity, root, requireMounts, options ):
   for intfId, intfClockConfig in six.iteritems( entity.intfClockConfig ):
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
      synceMode = mode[ SynceIntfConfigSaveMode ].getOrCreateModeInstance( intfId )
      cmds = synceMode[ 'Synce.intfConfig' ]
      # Priority
      if intfClockConfig.priority == Synce.Priority.defaultValue:
         if options.saveAll:
            cmds.addCommand( 'default priority' )
      else:
         cmds.addCommand( 'priority {}'.format(
            intfClockConfig.getRawAttribute( 'priority' ).stringValue() ) )
      # Both synchronous mode and quality-level are hidden commands, so we only
      # display the running configuration if it's configured

      # Synchronous-mode
      if intfClockConfig.synchronousMode == Synce.SynchronousMode.nonsynchronous:
         cmds.addCommand( 'disabled' )
      # Rx quality level
      if not Synce.QualityLevelHelper.isDefaultValue(
            intfClockConfig.rxQualityLevel ):
         cmd = qlCmd( 'rx', intfClockConfig.rxQualityLevel )
         if cmd is not None:
            cmds.addCommand( cmd )
