#!/usr/bin/env python3
# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
import TacacsConstants
import Tac
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.NetworkCliSave import networkConfigCmdSeq
from CliSavePlugin.Security import mgmtSecurityConfigPath
from CliSavePlugin.Security import SecurityConfigMode
from CliMode.Tacacs import TacacsServerGroupMode
from IpLibConsts import DEFAULT_VRF
import ReversibleSecretCli

#-------------------------------------------------------------------------------
# Object used for saving commands in "config-sg-tacacs+" mode.
#-------------------------------------------------------------------------------
class AaaTacacsServerGroupConfigMode( TacacsServerGroupMode, CliSave.Mode ):
   def __init__( self, param ):
      TacacsServerGroupMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( AaaTacacsServerGroupConfigMode,
                                       before=[ 'Aaa.global' ] )
AaaTacacsServerGroupConfigMode.addCommandSequence( 'Tacacs.serverGroup' )

# I want the Tacacs commands to go after Network.config because the
# "tacacs-server host" command takes hostname-or-ip-addrs and attempt to
# resolve hostnames, so it's better if the nameserver config is done first.
CliSave.GlobalConfigMode.addCommandSequence( 'AaaTacacs.global',
                                             before=[ 
                                                'Aaa.global', 
                                                AaaTacacsServerGroupConfigMode ],
                                             after=[ networkConfigCmdSeq,
                                                     SecurityConfigMode ] )

# When someone types "show running-config", the Cli code walks the sysdb object
# tree, calling all the functions which registered with CliSave.saver to be
# called for a particular object. Below, I register saveTacacsConfig to be
# called for 'security/aaa/tacacs/config'. When I'm called, I walk our entire
# Tacacs config object tree, generating all non-default config.

@CliSave.saver( 'Aaa::HostGroup', 'security/aaa/config' )
def saveHostGroup( entity, root, requireMounts, options ):
   if entity.groupType != 'tacacs':
      return
   mode = root[ AaaTacacsServerGroupConfigMode ].getOrCreateModeInstance( 
                                                            entity.name )
   cmds = mode[ 'Tacacs.serverGroup' ]
   for m in entity.member.values():
      cmd = "server %s" %( m.spec.hostname )
      assert m.spec.vrf != ''
      if m.spec.vrf != DEFAULT_VRF:
         cmd += " vrf %s" %( m.spec.vrf )
      if m.spec.port != TacacsConstants.defaultPort or options.saveAll:
         cmd += " port %d" %( m.spec.port ) 
      cmds.addCommand( cmd )

@CliSave.saver( 'Tacacs::Config', 'security/aaa/tacacs/config',
                requireMounts=( mgmtSecurityConfigPath, ) )
def saveTacacsConfig( cfg, root, requireMounts, options ):

   cmds = root[ 'AaaTacacs.global' ]

   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   if cfg.key:
      # <encryptionAlgo> <cipherText>
      cmd = ReversibleSecretCli.getCliSaveCommand( 'tacacs-server key {}',
                                                   securityConfig,
                                                   cfg.key )
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no tacacs-server key' )

   if cfg.timeout != cfg.defaultTimeout or options.saveAll:
      cmd = 'tacacs-server timeout %d' % ( int( cfg.timeout ) )
      cmds.addCommand( cmd )

   if cfg.maxUsernameLength != cfg.maxUsernameLengthDefault or options.saveAll:
      cmd = 'tacacs-server username max-length %d' % cfg.maxUsernameLength
      cmds.addCommand( cmd )

   if cfg.ignoreUnknownMandatoryAttr:
      cmd = 'tacacs-server policy unknown-mandatory-attribute ignore'
      cmds.addCommand( cmd )

   hosts = cfg.host
   for h in sorted( hosts.values(), key=lambda host: host.index ):
      cmd = "tacacs-server host " + h.hostname
      if h.connType == 'singleConnection':
         cmd += " single-connection"
      # Can't look at h.spec.defaultPort because constAttr on TAC value types
      # are not available to Python.
      assert h.vrf != ''
      if h.vrf != DEFAULT_VRF:
         cmd += ' vrf %s' % h.vrf
      if h.port != TacacsConstants.defaultPort or options.saveAll:
         cmd += " port %d" %( h.port )
      if h.useTimeout:
         cmd += " timeout %d" %( h.timeout )
      if h.useKey:
         # cmd could contain curly braces, maybe? escape to make sure
         cmd = ReversibleSecretCli.getCliSaveCommand(
            CliSave.escapeFormatString( cmd ) + ' key {}',
            securityConfig, h.key )
      cmds.addCommand( cmd )

   # tacacs-server qos dscp <dscpValue>
   if cfg.dscpValue != cfg.dscpValueDefault:
      cmd ='tacacs-server qos dscp %s' % cfg.dscpValue
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmd = 'tacacs-server qos dscp %s' % cfg.dscpValueDefault
      cmds.addCommand( cmd )

CliSave.GlobalConfigMode.addCommandSequence( 'Tacacs.srcIntf', 
                                             after=[ IntfConfigMode ] )
@CliSave.saver( 'Tacacs::Config', 'security/aaa/tacacs/config' )
def saveTacacsSrcIntfGlobal( entity, root, requireMounts, options ):

   if entity.srcIntfName:
      for k in entity.srcIntfName:
         if k == DEFAULT_VRF:
            root[ 'Tacacs.srcIntf' ].addCommand( 'ip tacacs source-interface %s' % \
                                            entity.srcIntfName[ k ] )
         else:
            root[ 'Tacacs.srcIntf' ].addCommand(
                                       'ip tacacs vrf %s source-interface %s' % \
                                            ( k, entity.srcIntfName[ k ] ))
   elif options.saveAll:
      root[ 'Tacacs.srcIntf' ].addCommand( 'no ip tacacs source-interface' )
      for k in entity.srcIntfName:
         if k == DEFAULT_VRF:
            root[ 'Tacacs.srcIntf' ].addCommand( 'no ip tacacs source-interface' )
         else:
            root[ 'Tacacs.srcIntf' ].addCommand(
                                     'no ip tacacs vrf %s source-interface' % k )

