# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import ReversibleDict
import CliSave, Tac
from RoutingIntfUtils import allRoutingProtocolIntfNames
from CliMode.Te import GlobalTeMode, FlexAlgoModeBase, FlexAlgoDefModeBase
from CliSavePlugin.IntfCliSave import IntfConfigMode
from MultiRangeRule import bitMaskToStr, bitMaskCollToStr
from Toggles import TeToggleLib

#-------------------------------------------------------------------------------
# Object used for saving commands in router-traffic-engineering mode
#-------------------------------------------------------------------------------
class RouterGlobalTeMode( GlobalTeMode, CliSave.Mode ):
   def __init__( self, param ):
      GlobalTeMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterGlobalTeMode,
                                      after=[ IntfConfigMode ] )
RouterGlobalTeMode.addCommandSequence( 'TrafficEngineering.global' )

class FlexAlgoMode( FlexAlgoModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FlexAlgoModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterGlobalTeMode.addChildMode( FlexAlgoMode,
                                 after=[ 'TrafficEngineering.global' ] )
FlexAlgoMode.addCommandSequence( 'FlexAlgo.config' )

class FlexAlgoDefMode( FlexAlgoDefModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FlexAlgoDefModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

FlexAlgoMode.addChildMode( FlexAlgoDefMode )
FlexAlgoDefMode.addCommandSequence( 'FlexAlgoDef.config' )

IntfConfigMode.addCommandSequence( 'TrafficEngineering.intf' )

def buildAdminGroupListStr( adminGroup, adminGroupNameSet=None ):
   adminGroupValueStr = ''
   if isinstance( adminGroup, int ):
      adminGroupValueStr = bitMaskToStr( adminGroup )
   else:
      adminGroupValueStr = bitMaskCollToStr( dict( adminGroup ) )
   adminGroupNamesStr = ''
   if adminGroupNameSet:
      adminGroupNamesStr = ','.join( list( sorted( adminGroupNameSet ) ) )
   listStr = ''
   if adminGroupValueStr and adminGroupNamesStr:
      listStr = adminGroupValueStr + ',' + adminGroupNamesStr
   elif adminGroupValueStr or adminGroupNamesStr:
      listStr = adminGroupValueStr + adminGroupNamesStr
   return listStr

def saveIntfConfig( teConfig, entity, root, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'TrafficEngineering.intf' ]

   if entity.enabled != entity.enabledDefault:
      cmds.addCommand( 'traffic-engineering' )
   elif options.saveAll:
      cmds.addCommand( 'no traffic-engineering' )

   bwUnit = Tac.Type( "TrafficEngineering::BwUnitType" )
   if entity.maxReservableBw != entity.maxReservableBwDefault:
      if entity.maxReservableBwUnit == bwUnit.percent:
         unitStr = "percent"
      elif entity.maxReservableBwUnit == bwUnit.mbps:
         unitStr = "mbps"
      elif entity.maxReservableBwUnit == bwUnit.gbps:
         unitStr = "gbps"
      cmds.addCommand( 'traffic-engineering bandwidth %d %s' %
                        ( entity.maxReservableBw, unitStr ) )
   elif options.saveAll:
      cmds.addCommand( 'no traffic-engineering bandwidth' )

   if entity.adminGroup or entity.adminGroupName:
      cmd = ''
      adminGroupListStr = buildAdminGroupListStr( entity.adminGroup,
                                                  entity.adminGroupName )
      cmd = f'traffic-engineering administrative-group {adminGroupListStr}'
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no traffic-engineering administrative-group' )

   if entity.srlgIdList:
      for srlg in sorted( entity.srlgIdList ):
         cmds.addCommand( 'traffic-engineering srlg %d' % srlg )
   if entity.srlgNameList:
      for srlg in sorted( entity.srlgNameList ):
         cmds.addCommand( 'traffic-engineering srlg %s' % srlg )

   if entity.metric != entity.metricDefault:
      cmds.addCommand( 'traffic-engineering metric %d' % ( entity.metric ) )
   elif options.saveAll:
      cmds.addCommand( 'no traffic-engineering metric' )

   if entity.minDelay != entity.minDelayDefault:
      cmds.addCommand( 'traffic-engineering min-delay static %d %s' %
                       ( entity.minDelay, entity.minDelayUnit ) )
   elif entity.dynMinDelayFallback != entity.dynMinDelayFallbackDefault:
      dynMinDelayFallback = entity.dynMinDelayFallback
      if entity.dynMinDelayFallbackUnit == 'milliseconds':
         dynMinDelayFallback = entity.dynMinDelayFallback / 1000
      cmds.addCommand( 'traffic-engineering min-delay dynamic twamp-light' \
                          ' fallback %d %s' % ( dynMinDelayFallback,
                                           entity.dynMinDelayFallbackUnit ) )
   elif options.saveAll:
      cmds.addCommand( 'no traffic-engineering min-delay ' )

   if TeToggleLib.toggleIsisIntfTeSenderProfileEnabled():
      if entity.senderProfile != entity.senderProfileDefault :
         cmds.addCommand( 'traffic-engineering twamp-light sender profile %s' %
                      ( entity.senderProfile, ) )
      elif options.saveAll:
         cmds.addCommand( 'no traffic-engineering twamp-light sender profile ' )

   

CliSave.GlobalConfigMode.addCommandSequence( 'TrafficEngineering.config',
                                             after=[ RouterGlobalTeMode ] )

def saveGlobalConfig( teConfig, root, options, flexAlgoEnabled ):

   if teConfig.enabled == teConfig.enabledDefault:
      return

   mode = root [ RouterGlobalTeMode ].getSingletonInstance()
   cmds = mode [ 'TrafficEngineering.global' ]

   if teConfig.routerId != teConfig.routerIdDefault:
      cmds.addCommand( 'router-id ipv4 %s' % teConfig.routerId )

   if teConfig.routerIdV6 != teConfig.routerIdV6Default:
      cmds.addCommand( 'router-id ipv6 %s' % teConfig.routerIdV6 )

   for srlgNameToIdMap in sorted( teConfig.srlgMap.values() ):
      cost = ""
      srlgId = srlgNameToIdMap.srlgId 
      costValue = teConfig.srlgCost[ srlgId ].cost
      defaultCost = teConfig.srlgCost[ srlgId ].defaultValue
      if costValue != defaultCost or options.saveAll:
         cost = " cost %d" % costValue
      cmds.addCommand( 'srlg %s %d%s' % ( srlgNameToIdMap.srlgName,
                                        srlgNameToIdMap.srlgId, cost ) )
   
   sortedNames = sorted( teConfig.adminGroupMap.keys() )
   for name in sortedNames:
      adminGroupValueStr = str( teConfig.adminGroupMap[ name ] )
      cmds.addCommand( 'administrative-group alias %s group %s' % ( name,
                                                        adminGroupValueStr ) )

   if teConfig.enabled != teConfig.enabledDefault:
      if teConfig.teBwFloodThreshold != teConfig.teBwFloodThresholdDefault:
         cmds.addCommand( 'igp bandwidth update threshold unreserved %d percent' %
            ( teConfig.teBwFloodThreshold ) )
      elif options.saveAll:
         cmds.addCommand( 'no igp bandwidth update threshold' )

   if teConfig.enabled != teConfig.enabledDefault:
      if teConfig.senderProfile != teConfig.senderProfileDefault:
         cmds.addCommand( 'twamp-light sender profile' +
                          ' %s' % ( teConfig.senderProfile ) )
      elif options.saveAll:
         cmds.addCommand( 'no twamp-light sender profile' )

   if flexAlgoEnabled:
      if teConfig.aslaFallback:
         flexMode = mode[ FlexAlgoMode ].getSingletonInstance()
         flexCmds = flexMode[ 'FlexAlgo.config' ]
         flexCmds.addCommand( 'link-attributes asla fallback' )
      elif options.saveAll:
         flexMode = mode[ FlexAlgoMode ].getSingletonInstance()
         flexCmds = flexMode[ 'FlexAlgo.config' ]
         flexCmds.addCommand( 'no link-attributes asla fallback' )

@CliSave.saver( 'TrafficEngineering::Config', 'te/config',
                requireMounts=( 'interface/config/all',
                                'interface/status/all',
                                'te/flexalgo/config' ) )
def saveConfig( teConfig, root, requireMounts, options ):   
   # Save traffic-engieering global config
   saveGlobalConfig( teConfig, root, options,
                     requireMounts[ 'te/flexalgo/config' ].enabled )

   # Check if any traffic-engieerning interface config is made
   teIntfConfigEnabled = len( teConfig.intfConfig ) > 0

   if not teIntfConfigEnabled and not options.saveAllDetail:
      return

   if options.saveAllDetail:
      rpIntfs = allRoutingProtocolIntfNames( requireMounts, includeEligible=True )
      intfs = set( rpIntfs )
      intfs.update( teConfig.intfConfig )
   elif options.saveAll:
      # Display TE configuration on routing protocol interfaces as well as
      # switchports present in intfConfig collection.
      rpIntfs = allRoutingProtocolIntfNames( requireMounts )
      intfs = set( rpIntfs )
      intfs.update( teConfig.intfConfig )
   else:
      intfs = teConfig.intfConfig

   # Save config for each interface
   for intfName in intfs:
      intfConfig = teConfig.intfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'TrafficEngineering::IntfConfig',
                                          intfName )
         else:
            continue
      saveIntfConfig( teConfig, intfConfig, root, options )

flexAlgoMetricMap = ReversibleDict( {
   'igp-metric' : 0,
   'min-delay' : 1,
   'te-metric' : 2,
   } )
flexAlgoReverseMetricMap = flexAlgoMetricMap.reverse()

@CliSave.saver( 'FlexAlgo::Config', 'te/flexalgo/config' )
def saveFlexAlgoConfig( flexAlgoConfig, root, requireMounts, options ):

   def savePrio():
      if defin.prio != defin.defaultPrio:
         subcmd.addCommand( 'priority %d' % defin.prio )
      elif options.saveAll:
         subcmd.addCommand( 'priority %d' % defin.defaultPrio )

   def saveAdminGroup():
      cmd = ""
      if defin.includeAllAdminGroup or defin.includeAllAdminGroupName:
         cmd += ' include all %s' % (
               buildAdminGroupListStr( defin.includeAllAdminGroup,
                                       defin.includeAllAdminGroupName ) )
      if defin.includeAnyAdminGroup or defin.includeAnyAdminGroupName:
         cmd += ' include any %s' % (
               buildAdminGroupListStr( defin.includeAnyAdminGroup,
                                       defin.includeAnyAdminGroupName ) )
      if defin.excludeAdminGroup or defin.excludeAdminGroupName:
         cmd += ' exclude %s' % (
               buildAdminGroupListStr( defin.excludeAdminGroup,
                                       defin.excludeAdminGroupName ) )
      if cmd:
         subcmd.addCommand( 'administrative-group' + cmd )

   def saveMetric():
      if defin.metric != defin.defaultMetric:
         metric = flexAlgoReverseMetricMap.get( defin.metric,
                                                str( defin.metric ) )
         subcmd.addCommand( 'metric ' + metric )
      elif options.saveAll:
         metric = flexAlgoReverseMetricMap.get( defin.defaultMetric,
                                                str( defin.defaultMetric ) )
         subcmd.addCommand( 'metric ' + metric )

   def saveSrlg():
      values = ""
      names = ""
      for val in sorted( defin.srlgValueSet ):
         values += ' %d' % val
      for name in sorted( defin.srlgNameSet ):
         names += ' %s' % name
      if values or names:
         subcmd.addCommand( 'srlg exclude' + values + names )

   def saveColor():
      if defin.colorDefined:
         subcmd.addCommand( 'color ' + str( defin.color ) )

   if flexAlgoConfig.enabled != flexAlgoConfig.enabledDefault:
      teMode = root [ RouterGlobalTeMode ].getSingletonInstance()
      faMode = teMode[ FlexAlgoMode ].getSingletonInstance()

      # Save the definitions
      for defId in sorted( flexAlgoConfig.definition ):
         defin = flexAlgoConfig.definition[ defId ]
         fadMode = faMode[ FlexAlgoDefMode ].getOrCreateModeInstance(
            ( defId, defin.name ) )
         subcmd = fadMode[ 'FlexAlgoDef.config' ]

         savePrio()
         saveAdminGroup()
         saveMetric()
         saveSrlg()
         saveColor()
