# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements saving the TunnelIntf CLI.
#-------------------------------------------------------------------------------
import CliSave
import TunnelIntfUtil
from CliSavePlugin import IntfCliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
import Arnet
import Intf.IntfRange
from IpLibConsts import DEFAULT_VRF
from Toggles.TunnelIntfToggleLib import (
   toggleTapAggGreTerminationRoutedPortEnabled
)

IntfConfigMode.addCommandSequence(
   'TunnelIntf.config', after=[ 'Arnet.intf' ] )

#-------------------------------------------------------------------------------
# Saves the state of an Interface::EthIntfConfig object.
#-------------------------------------------------------------------------------

@CliSave.saver( 'Interface::TunnelIntfConfig', 'interface/config/tunnel/intf',
                requireMounts = ( 'interface/status/all',
                                  'interface/config/all' ) )
def saveTunnelIntfConfig( entity, root, requireMounts, options ):

   # Handle base class (Arnet::IntfConfig) first
   IntfCliSave.saveIntfConfig( entity, root, requireMounts, options )

   cliMode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = cliMode[ 'TunnelIntf.config' ]

   if entity.mode == 'tunnelIntfModeGre':
      cmds.addCommand( 'tunnel mode gre' )
   elif entity.mode == 'tunnelIntfModeIpip':
      cmds.addCommand( 'tunnel mode ipip' )
   elif entity.mode == 'tunnelIntfModeIpsec':
      cmds.addCommand( 'tunnel mode ipsec' )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel mode' )

   if not entity.srcAddr.isUnspecified:
      cmds.addCommand( 'tunnel source %s' % entity.srcAddr )
   elif entity.srcIntf:
      cmds.addCommand( 'tunnel source interface %s' % entity.srcIntf )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel source' )

   if not entity.dstAddr.isUnspecified:
      cmds.addCommand( 'tunnel destination %s' % entity.dstAddr )
   elif entity.dstHost:
      cmds.addCommand( f'tunnel destination {entity.dstHost}' )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel destination' )

   if entity.pathMtuDiscovery:
      cmds.addCommand( 'tunnel path-mtu-discovery' )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel path-mtu-discovery' )

   if entity.decapDipOnlyMatch:
      cmds.addCommand( 'tunnel decap source any' )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel decap source any' )

   if entity.ttl != TunnelIntfUtil.tunnelDefaultTtl:
      cmds.addCommand( 'tunnel ttl %u' % entity.ttl )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel ttl' )

   if entity.tos != TunnelIntfUtil.tunnelDefaultTos:
      cmds.addCommand( 'tunnel tos %u' % entity.tos )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel tos' )

   if entity.key != TunnelIntfUtil.tunnelDefaultKey:
      cmds.addCommand( 'tunnel key %u' % entity.key )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel key' )

   if entity.maxMss != TunnelIntfUtil.tunnelDefaultMaxMss:
      cmds.addCommand( 'tunnel mss ceiling %u' % entity.maxMss )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel mss ceiling' )

   if entity.ipsec == True: # pylint: disable=singleton-comparison
      cmds.addCommand( "tunnel ipsec profile %s" 
                       % ( entity.ipsecProfile ) )
   elif options.saveAll:
      cmds.addCommand( "no tunnel ipsec profile" ) 

   if entity.underlayVrfName != DEFAULT_VRF:
      cmds.addCommand( 'tunnel underlay vrf %s' % entity.underlayVrfName )
   elif options.saveAll:
      cmds.addCommand( 'no tunnel underlay vrf' )

   # For tapAgg GRE tunnel termination
   tapGroups = list( entity.tapGroup )
   if tapGroups:
      cmds.addCommand( 'tap default group %s'
                       % ' group '.join( sorted( tapGroups ) ) )
   elif options.saveAll:
      cmds.addCommand( 'no tap default group' )

   tapRawIntfs = list( entity.tapRawIntf )
   phyIntfList = [ i for i in tapRawIntfs if not i.startswith( 'Po' ) ]
   lagIntfList = [ i for i in tapRawIntfs if i.startswith( 'Po' ) ]
   if tapRawIntfs:
      if phyIntfList:
         # Due to our parser, we can not specify range for physical interfaces
         # in the startup config. For lag interfaces, it works fine
         for intf in Arnet.sortIntf( phyIntfList ):
            cmds.addCommand( 'tap default interface %s' % intf )
      if lagIntfList:
         printLagList = Intf.IntfRange.intfListToCanonical( lagIntfList )
         cmds.addCommand( 'tap default interface %s'
                          % printLagList[ 0 ] )
   elif options.saveAll:
      cmds.addCommand( 'no tap default interface' )

   if entity.tapEncapGrePreserve:
      cmds.addCommand( 'tap encapsulation gre preserve' )
   elif options.saveAll:
      cmds.addCommand( 'no tap encapsulation gre preserve' )

   if entity.tapGreProtocolTunnel:
      for protocol, value in \
            sorted( entity.tapGreProtocolTunnel.items() ):
         cmd = 'tap encapsulation gre protocol 0x%x' % protocol
         if value.featureHeaderLength:
            cmd += ' feature header length %d' % value.featureHeaderLength
         if value.ethEncapEnabled:
            cmd += ' re-encapsulation ethernet'
         cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no tap encapsulation gre protocol' )

   tapIntfs = list( entity.tapMemberIntf )
   if tapIntfs:
      for intf in Arnet.sortIntf( tapIntfs ):
         if not entity.tapMemberIntf[ intf ].tapGroup and \
               not entity.tapMemberIntf[ intf ].tapRawIntf:
            cmds.addCommand( 'tap interface %s' % intf )
         if entity.tapMemberIntf[ intf ].tapGroup:
            tapGroups = list( entity.tapMemberIntf[ intf ].tapGroup )
            cmds.addCommand( 'tap interface %s tool group %s' % (
               intf, ' group '.join( sorted( tapGroups ) ) ) )
         if entity.tapMemberIntf[ intf ].tapRawIntf:
            tapRawIntfs = list( entity.tapMemberIntf[ intf ].tapRawIntf )
            phyIntfList = [ i for i in tapRawIntfs if not i.startswith( 'Po' ) ]
            lagIntfList = [ i for i in tapRawIntfs if i.startswith( 'Po' ) ]
            if phyIntfList:
               # Due to our parser, we can not specify range for physical interfaces
               # in the startup config. For lag interfaces, it works fine
               for toolIntf in Arnet.sortIntf( phyIntfList ):
                  cmds.addCommand( 'tap interface %s tool interface %s' %
                        ( intf, toolIntf ) )
            if lagIntfList:
               printLagList = Intf.IntfRange.intfListToCanonical( lagIntfList )
               cmds.addCommand( 'tap interface %s tool interface %s' %
                     ( intf, printLagList[ 0 ] ) )
   elif options.saveAll and toggleTapAggGreTerminationRoutedPortEnabled():
      cmds.addCommand( 'no tap interface' )
