# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.TrafficPolicyCliSave import ( TrafficPolicySaver,
                                                 FieldSetSaver,
                                                 ContentsSource,
                                                 FSPrefixSaveMode )
from CliMode.VrfSelectionPolicyMode import ( VrfSelectionMode,
                                             VrfSelectionConfigMode,
                                             VrfSelectionPolicyVrfMode )
from CliMode.TrafficPolicy import ( ActionsModeBase,
                                    DefaultActionModeBase,
                                    MatchRuleModeBase,
                                    TrafficPolicyModeBase )
from CliMode.VrfSelectionPolicyMode import FEATURE, VrfSelectionActionRuleConfigMode

class VrfSelConfigCliSaveMode( VrfSelectionMode, CliSave.Mode ):
   def __init__( self, param ):
      VrfSelectionMode.__init__( self )
      CliSave.Mode.__init__( self, None )

   def skipIfEmpty( self ):
      return True

class VrfSelPolicyCliSaveMode( TrafficPolicyModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TrafficPolicyModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, None )

   @property
   def configPolicyKeyword( self ):
      return 'policy'

   def instanceKey( self ):
      return self.trafficPolicyName

class VrfSelPolicyMatchRuleCliSaveMode( MatchRuleModeBase, CliSave.Mode ):
   def __init__( self, param ):
      MatchRuleModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, None )

   def instanceKey( self ):
      return self.prio

   @classmethod
   def useInsertionOrder( cls ):
      # because `instanceKey` is overridden with prio
      return True

class VrfSelPolicyActionRuleCliSaveMode( ActionsModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ActionsModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, None )

   def skipIfEmpty( self ):
      return True

class VrfSelPolicySetVrfActionCliSaveMode( VrfSelectionActionRuleConfigMode,
                                           CliSave.Mode ):
   def __init__( self, param ):
      # pylint: disable-next=no-value-for-parameter
      VrfSelectionActionRuleConfigMode.__init__( self, param )
      CliSave.Mode.__init__( self, None )

class VrfSelPolicyVrfCliSaveMode( VrfSelectionPolicyVrfMode, CliSave.Mode ):
   def __init__( self, param ):
      VrfSelectionPolicyVrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class VrfSelPolicyDefaultActionsCliSaveMode( DefaultActionModeBase, CliSave.Mode ):
   def __init__( self, param ):
      DefaultActionModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, None )

   def instanceKey( self ):
      return self.matchType

VrfSelConfigCliSaveMode.addCommandSequence( 'VrfSelPolicy.global' )
CliSave.GlobalConfigMode.addChildMode( VrfSelConfigCliSaveMode )

VrfSelPolicyCliSaveMode.addCommandSequence( 'VrfSelPolicy.policyConfig' )
VrfSelConfigCliSaveMode.addChildMode( VrfSelPolicyCliSaveMode,
      after=[ 'VrfSelPolicy.global' ] )

VrfSelPolicyDefaultActionsCliSaveMode.addCommandSequence(
      'VrfSelPolicy.actionDefaultConfig' )
VrfSelPolicyCliSaveMode.addChildMode( VrfSelPolicyDefaultActionsCliSaveMode,
                                      after=[ 'VrfSelPolicy.policyConfig' ] )

VrfSelPolicyMatchRuleCliSaveMode.addCommandSequence( 'VrfSelPolicy.matchConfig' )
VrfSelPolicyCliSaveMode.addChildMode( VrfSelPolicyMatchRuleCliSaveMode,
      after=[ 'VrfSelPolicy.policyConfig' ] )

VrfSelPolicyActionRuleCliSaveMode.addCommandSequence( 'VrfSelPolicy.actionConfig' )
VrfSelPolicyMatchRuleCliSaveMode.addChildMode( VrfSelPolicyActionRuleCliSaveMode,
      after=[ 'VrfSelPolicy.matchConfig' ] )

VrfSelPolicySetVrfActionCliSaveMode.addCommandSequence( 'VrfSelPolicy.setVrf' )
VrfSelPolicyActionRuleCliSaveMode.addChildMode( VrfSelPolicySetVrfActionCliSaveMode,
      after=[ 'VrfSelPolicy.actionConfig' ] )

VrfSelConfigCliSaveMode.addChildMode( FSPrefixSaveMode,
                                     after=[ 'VrfSelPolicy.global' ] )
FSPrefixSaveMode.addCommandSequence( 'VrfSelPolicy.FieldSetPrefix' )

VrfSelPolicyVrfCliSaveMode.addCommandSequence( 'VrfSelPolicy.vrfConfig' )
VrfSelConfigCliSaveMode.addChildMode( VrfSelPolicyVrfCliSaveMode,
      after=[ 'VrfSelPolicy.global' ] )

IntfConfigMode.addCommandSequence( 'VrfSelPolicy.intfConfig' )

class VrfSelPolicySaver( TrafficPolicySaver ):
   trafficPoliciesMode = VrfSelConfigCliSaveMode
   trafficPolicyMode = VrfSelPolicyCliSaveMode
   defaultActionsMode = VrfSelPolicyDefaultActionsCliSaveMode
   matchRuleMode = VrfSelPolicyMatchRuleCliSaveMode
   actionsRuleMode = VrfSelPolicyActionRuleCliSaveMode
   packetTypeMatchMode = None
   l4PortSaveAll = False

   def trafficPolicyModeCmds( self, policyMode ):
      return policyMode[ 'VrfSelPolicy.policyConfig' ]

   def defaultActionsModeCmds( self, defaultActionsMode ):
      return defaultActionsMode[ 'VrfSelPolicy.actionDefaultConfig' ]

   def matchModeCmds( self, matchMode ):
      return matchMode[ 'VrfSelPolicy.matchConfig' ]

   def actionModeCmds( self, actionsMode ):
      return actionsMode[ 'VrfSelPolicy.actionConfig' ]

   def saveFallbackVrfConfig( self ):
      '''Function to save the fallback vrf configuration'''
      vrfSelVrfconfig = self.requireMounts[ 'vrfSelectionPolicy/fallbackVrf' ]
      if not vrfSelVrfconfig.fallbackVrf:
         # no vrf fallbacks configured
         return

      vrfSelMode = self.root[
            VrfSelConfigCliSaveMode ].getOrCreateModeInstance( None )
      for vrfName, fallbackVrfName in vrfSelVrfconfig.fallbackVrf.items():
         vrfSelVrfMode = vrfSelMode[
               VrfSelPolicyVrfCliSaveMode ].getOrCreateModeInstance( vrfName )
         cmds = vrfSelVrfMode[ 'VrfSelPolicy.vrfConfig' ]
         cmds.addCommand( f'fallback vrf {fallbackVrfName}' )

   def saveDecapNhFallbackEnabledConfig( self ):
      if self.entity.decapNhFallbackEnabled:
         vrfSelMode = self.root[
               VrfSelConfigCliSaveMode ].getOrCreateModeInstance( None )
         cmds = vrfSelMode[ 'VrfSelPolicy.global' ]
         cmds.addCommand( 'next-hop fallback decapsulation vrf' )

   def saveDecapNhEnabledConfig( self ):
      if self.entity.decapNhEnabled:
         vrfSelMode = self.root[
               VrfSelConfigCliSaveMode ].getOrCreateModeInstance( None )
         cmds = vrfSelMode[ 'VrfSelPolicy.global' ]
         cmds.addCommand( 'next-hop decapsulation vrf' )

   def save( self ):
      TrafficPolicySaver.save( self )
      self.saveFallbackVrfConfig()
      self.saveDecapNhFallbackEnabledConfig()
      self.saveDecapNhEnabledConfig()
      intfConfig = self.requireMounts[ 'vrfSelectionPolicy/intf/config' ]
      for intfName, pmap in intfConfig.intf.items():
         intfMode = self.root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
         cmds = intfMode[ 'VrfSelPolicy.intfConfig' ]
         cmds.addCommand( f'vrf selection policy {pmap}' )

class VrfSelFieldSetSaver( FieldSetSaver ):
   trafficPoliciesMode = VrfSelConfigCliSaveMode
   trafficPoliciesConfigMode = VrfSelectionConfigMode
   feature = FEATURE
   fieldSetPrefixCmdSeq = 'VrfSelPolicy.FieldSetPrefix'

   def maybeGetExternalFieldSetSource( self, currCfg ):
      src = ''
      if self.options.saveAll and currCfg.source == ContentsSource.cli:
         # No need to add source of 'static' when using CLI non-saveAll
         src = 'static'
      elif currCfg.source == ContentsSource.bgp:
         src = 'bgp'
      return src

@CliSave.saver( 'VrfSelectionPolicy::VrfSelectionPolicyConfig',
                'vrfSelectionPolicy/config',
                requireMounts=( 'vrfSelectionPolicy/intf/config',
                                'vrfSelectionPolicy/fallbackVrf', ) )
def saveVrfSelectionPolicyCli( entity, root, requireMounts, options ):
   cliDumper = VrfSelPolicySaver( entity, root, requireMounts, options,
         FEATURE )
   cliDumper.save()

@CliSave.saver( 'Classification::FieldSetConfig',
                'vrfSelectionPolicy/fieldset/input/cli' )
def saveVrfSelectionFieldSetConfig( entity, root, requireMounts, options ):
   cliDumper = VrfSelFieldSetSaver( entity, root, requireMounts, options )
   cliDumper.save()
