# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliSave import escapeFormatString
from CliSaveBlock import SensitiveCommand
import Tac
from TypeFuture import TacLazyType
from CliSavePlugin.ServiceCliSave import serviceConfigCmdSeqName
from CliSavePlugin.IntfCliSave import IntfConfigMode
import XcvrLib
import Url
from XcvrMediaTypeLib import mediaTypeEnumToClis
from CliMode.Layer1Monitor import Layer1MonitorBaseMode
from MultiRangeRule import multiRangeToCanonicalString
import Toggles.XcvrToggleLib
import XcvrPrbsLib
import itertools
IntfConfigMode.addCommandSequence( 'Xcvr.config' )

lpoConfigEnable = Toggles.XcvrToggleLib.toggleXcvrLpoEnabled()
sfpHighPowerEnable = Toggles.XcvrToggleLib.toggleSfpHighPowerEnabled()
rxAutoSquelchEnable = Toggles.XcvrToggleLib.toggleRxAutoSquelchEnabled()
smbusErrLogEnable = Toggles.XcvrToggleLib.toggleSmbusErrLogsEnabled()
coherentFecDegradeEnable = Toggles.XcvrToggleLib.toggleCoherentFecDegradeEnabled()
cmisLoopbackEnabled = Toggles.XcvrToggleLib.toggleCmisLoopbackEnabled()
defaultTuningExceptionEnabled =\
   Toggles.XcvrToggleLib.toggleDefaultTuningExceptionEnabled()
xcvrRecoveryEnabled = Toggles.XcvrToggleLib.toggleXcvrRecoveryEnabled()

CmisLoopbackMode = TacLazyType( 'Xcvr::CmisLoopback::Mode' )
XcvrElectricalConfigType = TacLazyType( 'Xcvr::XcvrElectricalConfigType' )
XcvrLpoType = TacLazyType( 'Xcvr::LpoType' )

transceiverElectricalSignalBwCmd = "transceiver electrical signal bandwidth"

class Layer1MonitorConfigMode( Layer1MonitorBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      Layer1MonitorBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( Layer1MonitorConfigMode )
Layer1MonitorConfigMode.addCommandSequence( 'Layer1Monitor.config' )

@CliSave.saver( 'Xcvr::Xgc', 'hardware/xcvr/xgc' )
def saveXcvrXgcConfig( xgc, root, requireMounts, options ):
   cmds = root[ serviceConfigCmdSeqName ]

   # NOTE: we ignore "options.saveAll" and "options.saveAllDetail"
   #       here, because the "service unsupported-transceiver" command
   #       is hidden.
   #
   #       We DO make sure to use the sanitized version of the key
   #       when the user requests a sanitized config to be shown. This
   #       is implemented quite nicely by CliSave.sanitizedOutput().
   if xgc.licensee:
      # We avoid formatting the key using "hex( xgc.key )" because
      # "hex()" generates a leading "0x", which I want to avoid in
      # order to simplify the CliParser.PatternRule ever so slightly.
      # This lets us avoid one extra test case in the CliSaveTest.
      cmd = SensitiveCommand(
         f'service unsupported-transceiver {escapeFormatString( xgc.licensee )} '
         f'{{}}',
         f'{xgc.key:x}' )
      cmds.addCommand( cmd )

   if xgc.ethComplianceBit7IsPsm4:
      cmds.addCommand( 'transceiver qsfp ethernet-compliance bit7-is-psm4' )

   if defaultTuningExceptionEnabled:
      if xgc.overrideTuningExceptionSrc != "":
         cmds.addCommand( 'transceiver electrical tuning file %s' %
                          Url.filenameToUrl( xgc.overrideTuningExceptionSrc ) )
      elif options.saveAll:
         cmds.addCommand( 'no transceiver electrical tuning file' )

   pm = Tac.Value( 'Xcvr::Sff8436PerformanceMonitoring' )
   if xgc.performanceMonitoringPeriod.valueInSeconds != pm.defaultPMPeriod:
      # Convert period to the orginally configured units
      val = xgc.performanceMonitoringPeriod.valueInSeconds
      if xgc.performanceMonitoringPeriod.unit == 'm':
         val = val / 60         # minutes
      elif xgc.performanceMonitoringPeriod.unit == 'h':
         val = val / ( 60 * 60 )        # hours
      elif xgc.performanceMonitoringPeriod.unit == 'd':
         val = val / ( 60 * 60 * 24 )   # days
      cmds.addCommand( 'performance-monitoring period %d%s' %
            ( val, xgc.performanceMonitoringPeriod.unit ) )
   elif options.saveAll:
      cmds.addCommand( 'no performance-monitoring period' )

   if xgc.performanceMonitoringGlobal:
      cmds.addCommand( 'performance-monitoring transceiver default' )
   elif options.saveAll:
      cmds.addCommand( 'no performance-monitoring transceiver default' )

   defaultTargetTemp = Tac.Value( 'Units::Celsius' )
   for mediaType, token in mediaTypeEnumToClis.items():
      targetTemperature = \
         xgc.targetTemperatureForMediaType.get( mediaType, defaultTargetTemp.value )
      if targetTemperature != defaultTargetTemp.value:
         cmds.addCommand(
            'transceiver media %s temperature cooling-target %d celsius' %
            ( token, targetTemperature ) )
      elif options.saveAll:
         cmds.addCommand(
            'no transceiver media %s temperature cooling-target' % token )

   for config in xgc.tuningConfig.values():
      intf = config.name
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      tuningConstants = Tac.Value( 'Xcvr::TuningConstants' )
      if config.frequency == 0:
         channelStr = 'transceiver channel %d' % config.channel
         gridStr = ''
         if config.grid != tuningConstants.defaultGrid:
            if config.grid % 1e3:
               gridStr = ' grid-spacing %.2f' % ( float( config.grid ) / 1e3 )
            else:
               gridStr = ' grid-spacing %d' % ( config.grid / 1e3 )
         cmds.addCommand( channelStr + gridStr )
      else:
         cmds.addCommand( 'transceiver frequency %.3f%s' %
                          ( ( float( config.frequency ) / 1e3 ),
                            ' ghz' if config.displayUnits else '' ) )
   for rxConfig in xgc.rxTuningConfig.values():
      intf = rxConfig.name
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      if rxConfig.frequency:
         cmds.addCommand( 'transceiver receiver frequency %.3f ghz' %
                          ( float( rxConfig.frequency ) / 1e3 ) )
      if rxConfig.fineFrequency:
         cmds.addCommand( 'transceiver receiver frequency fine %.3f ghz' %
                          ( float( rxConfig.fineFrequency ) / 1e3 ) )

   for config in xgc.txPowerConfig.values():
      intf = config.name
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      powerConstants = Tac.Value( 'Xcvr::PowerConstants' )
      if config.signalPower != powerConstants.defaultTxPower:
         cmds.addCommand( 'transceiver transmitter signal-power %.2f'
                          % config.signalPower )
      elif options.saveAll:
         cmds.addCommand( 'no transceiver transmitter signal-power' )

   for config in xgc.rxPowerConfig.values():
      intf = config.name
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      powerConstants = Tac.Value( 'Xcvr::PowerConstants' )
      if config.signalPower != powerConstants.defaultRxPower:
         cmds.addCommand( 'transceiver receiver signal-power %.2f'
                          % config.signalPower )
      elif options.saveAll:
         cmds.addCommand( 'no transceiver receiver signal-power' )

   for intf in xgc.txDisabled:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      cmds.addCommand( 'transceiver transmitter disabled' )

   for config in xgc.performanceMonitoringConfig.values():
      intf = config.name
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ 'Xcvr.config' ]
      if config.performanceMonitoringEnabled:
         cmds.addCommand( 'performance-monitoring transceiver' )
      elif options.saveAll:
         cmds.addCommand( 'no performance-monitoring transceiver' )

   for intfId in xgc.tributary:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
      cmds = mode[ 'Xcvr.config' ]
      cmds.addCommand( 'transceiver media tributary %d' % xgc.tributary[ intfId ] )

   if xgc.domThresholdOverrideEnabled:
      cmds.addCommand( 'transceiver dom-threshold file %s' %
                       xgc.domThresholdOverrideSrc )
   elif options.saveAll:
      cmds.addCommand( 'no transceiver dom-threshold file' )

   if xcvrRecoveryEnabled:
      if xgc.recoveryEnabled:
         cmds.addCommand( 'transceiver power-cycle on-boot' )
      elif options.saveAll:
         cmds.addCommand( 'no transceiver power-cycle on-boot' )

   if smbusErrLogEnable and ( xgc.domThresholdLogging or xgc.smbusFailureLogging ):
      mode = root[ Layer1MonitorConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'Layer1Monitor.config' ]
      if xgc.smbusFailureLogging and xgc.domThresholdLogging:
         cmds.addCommand( 'logging transceiver dom' )
         cmds.addCommand( 'logging transceiver communication' )
      elif xgc.domThresholdLogging and not xgc.smbusFailureLogging:
         cmds.addCommand( 'logging transceiver dom' )
      elif xgc.smbusFailureLogging and not xgc.domThresholdLogging:
         cmds.addCommand( 'logging transceiver communication' )
      elif options.saveAll:
         cmds.addCommand( 'no logging transceiver dom' )
         cmds.addCommand( 'no logging transceiver communication' )
   elif xgc.domThresholdLogging or options.saveAll:
      mode = root[ Layer1MonitorConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'Layer1Monitor.config' ]
      if xgc.domThresholdLogging:
         cmds.addCommand( 'logging transceiver' )
      elif options.saveAll:
         cmds.addCommand( 'no logging transceiver' )

def createTestPatternConfigCommands( testPatternConfig ):
   """
   Creates a list of test pattern config commands to recreate
   the state of an existing testPatternConfig object. The order in which
   you execute these commnands is irrelevant.

   Parameters:
   -----------
   testPatternConfig : Xcvr::TestPatternConfig : An existing config object
      holding all of the information we wish to replicate/recreate

   Return:
   -------
   testPatternCommands : List[ String ] : A list of cli commands that will
      recreate an existing config exactly
   """
   testPatternCommands = []
   for side in Tac.Type( "Xcvr::TestPatternSide" ).attributes:
      if ( testPatternConfig.sideConfig( side ) and
           testPatternConfig.sideConfig( side ).enabled ):
         sideConfig = testPatternConfig.sideConfig( side )
         fecOn = "fec" if sideConfig.fec else ""
         cmdTemplate = 'transceiver diag test pattern {} {} {}'
         cliSideName = XcvrPrbsLib.prbsTacSideToCliSide( side ).value
         cmd = cmdTemplate.format( sideConfig.pattern,
                                   cliSideName, fecOn )
         testPatternCommands.append( cmd )
   return testPatternCommands

def baseSaveCmdStr( source: str, side: str ):
   return f"traffic-loopback source {source} device transceiver {side}"

def addSavedCmd( cmds, side, lbType, laneSpec='' ):
   # NOTE: this should not be inserted into the collection
   # normally, just filtering it out for safety
   if lbType != CmisLoopbackMode.NoLoopback:
      source = XcvrLib.ConfigToModelMapping[ side ][ lbType ]
      cmds.addCommand( f"{baseSaveCmdStr(source,side)}{laneSpec}" )

def saveConfigForSide( cmds, cmisLoopbackConfig, side: str ):
   coll = XcvrLib.selectCollection( cmisLoopbackConfig, side )
   if not coll:
      return
   allCfg = getattr( cmisLoopbackConfig, f'all{side.capitalize()}LanesConfigured' )
   if not allCfg:
      laneCfgs = sorted( [ ( lbType, lane ) for lane, lbType in coll.items()
                           if lbType != CmisLoopbackMode.NoLoopback ] )
      for lbType, group in itertools.groupby( laneCfgs, lambda e: e[ 0 ] ):
         lanes = tuple( XcvrLib.internalLaneIdToCli( lane ) for _, lane in group )
         laneSpec = multiRangeToCanonicalString( lanes )
         addSavedCmd( cmds, side, lbType, f" lane {laneSpec}" )
   else:
      # NOTE: we won't handle inconsistencies here
      # the invariant is that if all lanes configured flag is set
      # then all entries match in the collections
      addSavedCmd( cmds, side, coll.values()[ 0 ] )

def CmisLoopbackConfigSaveHandler( cmds, xcvrConfigCli ):
   if not xcvrConfigCli:
      return
   cmisLoopbackConfig = getattr( xcvrConfigCli, "cmisLoopbackConfig", None )
   if not cmisLoopbackConfig:
      return
   saveConfigForSide( cmds, cmisLoopbackConfig, 'host' )
   saveConfigForSide( cmds, cmisLoopbackConfig, 'media' )

@CliSave.saver( 'Tac::Dir', 'hardware/xcvr/cli/config/slice' )
def saveXcvrConfigCli( baseDir, root, requireMounts, options ):
   for xcvrConfigCliDir in baseDir.values():
      for xcvrConfigCli in xcvrConfigCliDir.xcvrConfigCli.values():
         intf = xcvrConfigCli.name
         mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
         cmds = mode[ 'Xcvr.config' ]
         if xcvrConfigCli.overrideMediaType != "xcvrUnknown":
            mediaType = mediaTypeEnumToClis[ xcvrConfigCli.overrideMediaType ]
            cmds.addCommand( 'transceiver media override %s' % mediaType )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver media override' )

         if xcvrConfigCli.forceModuleRemoved:
            cmds.addCommand( 'transceiver diag simulate removed' )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver diag simulate removed' )

         if sfpHighPowerEnable:
            if xcvrConfigCli.preventSfpHighPowerEnable:
               cmds.addCommand( 'transceiver sfp power low' )
            elif options.saveAll:
               cmds.addCommand( 'no transceiver sfp power' )

         if xcvrConfigCli.forceQsfpManagementMode:
            cmds.addCommand( 'transceiver diag management' )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver diag management' )

         for laneId in xcvrConfigCli.slotTuningParams.keys():
            cliTuningParam = ''
            laneTuningParam = xcvrConfigCli.slotTuningParams[ laneId ]
            if laneTuningParam.txInputEqualization.valid:
               cliTuningParam += ' tx-input-equalization ' + \
                  str( laneTuningParam.txInputEqualization.val )
            elif laneTuningParam.txInputEqualization.moduleDefault:
               cliTuningParam += ' tx-input-equalization module-default'
            elif laneTuningParam.txInputAdaptiveEqEnable.valid:
               cliTuningParam += ' tx-input-equalization auto'

            if laneTuningParam.rxOutputPreEmphasis.valid:
               cliTuningParam += ' rx-output-pre-emphasis ' + \
                  str( laneTuningParam.rxOutputPreEmphasis.val )
            elif laneTuningParam.rxOutputPreEmphasis.moduleDefault:
               cliTuningParam += ' rx-output-pre-emphasis module-default'

            if laneTuningParam.rxOutputPostEmphasis.valid:
               cliTuningParam += ' rx-output-post-emphasis ' + \
                  str( laneTuningParam.rxOutputPostEmphasis.val )

            if laneTuningParam.rxOutputAmplitude.valid:
               cliTuningParam += ' rx-output-amplitude ' + \
                  str( laneTuningParam.rxOutputAmplitude.val )
            elif laneTuningParam.rxOutputAmplitude.moduleDefault:
               cliTuningParam += ' rx-output-amplitude module-default'

            if cliTuningParam:
               cmds.addCommand( 'transceiver electrical lane %s%s' % (
                                laneId + 1, cliTuningParam ) )
            elif options.saveAll:
               cmds.addCommand( 'no transceiver electrical lane %s' % (
                                laneId + 1 ) )

         if lpoConfigEnable:
            if xcvrConfigCli.overrideLpoType != XcvrElectricalConfigType.unset:
               if xcvrConfigCli.overrideLpoType == XcvrElectricalConfigType.linear:
                  cmds.addCommand( 'transceiver electrical drive linear' )
               else:
                  cmds.addCommand( 'transceiver electrical drive limiting' )
            elif options.saveAll:
               cmds.addCommand( 'no transceiver electrical drive' )

         if lpoConfigEnable:
            if xcvrConfigCli.signalBwConfig != XcvrLpoType.none:
               token = ( "high" if xcvrConfigCli.signalBwConfig == XcvrLpoType.highBw
                        else "low" )
               cmds.addCommand( f'{transceiverElectricalSignalBwCmd} {token}' )
            elif options.saveAll:
               cmds.addCommand( f'default {transceiverElectricalSignalBwCmd}' )

         if xcvrConfigCli.overrideTuningParams:
            cmds.addCommand( 'transceiver electrical tuning override' )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver electrical tuning override' )

         if xcvrConfigCli.modulePowerIgnore:
            cmds.addCommand( 'transceiver power ignore' )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver power ignore' )

         if xcvrConfigCli.legacySRBDCompatibilityConfig:
            cmds.addCommand( 'transceiver application override 100gbase-srbd' )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver application override 100gbase-srbd' )

         if rxAutoSquelchEnable:
            if xcvrConfigCli.rxAutoSquelch.valid:
               if xcvrConfigCli.rxAutoSquelch.moduleDefault:
                  # Don't print the default:
                  #    'default transceiver receiver squelch auto'
                  pass
               elif xcvrConfigCli.rxAutoSquelch.enabled:
                  cmds.addCommand( 'transceiver receiver squelch auto' )
               else:
                  cmds.addCommand( 'transceiver receiver squelch auto disabled' )
            elif options.saveAll:
               cmds.addCommand( 'default transceiver receiver squelch auto' )

         if xcvrConfigCli.cmisOverrideApplication:
            for startLane, appOver in xcvrConfigCli.cmisOverrideApplication.items():
               apSel = appOver.apSel
               laneCount = appOver.laneCount
               if laneCount != 0:
                  # A laneCount different from the maximum number of lanes means
                  # that we need the 'end' token.
                  cmds.addCommand( 'transceiver application override %s lanes start'
                                   ' %s end %s' % ( apSel, startLane,
                                                    startLane + laneCount - 1 ) )
               elif startLane != 0:
                  # If we need to set the override for any lane other than the first
                  # then we need to specify the 'lanes start' tokens.
                  cmds.addCommand( 'transceiver application override %s '
                                   'lanes start %s' % ( apSel, startLane ) )
               else:
                  # If we want to apply the override for all lanes starting with the
                  # first, then we don't need the optional tokens.
                  cmds.addCommand( 'transceiver application override %s' % apSel )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver application override' )

         if ( xcvrConfigCli.testPatternConfig and
              xcvrConfigCli.testPatternConfig.isActiveConfig() ):
            prbsConfig = xcvrConfigCli.testPatternConfig
            for cmd in createTestPatternConfigCommands( prbsConfig ):
               cmds.addCommand( cmd )
         elif options.saveAll:
            cmds.addCommand( 'no transceiver diag test pattern' )

         if coherentFecDegradeEnable:
            detectedDegrade = xcvrConfigCli.cmisFecDegradeConfig.detectedDegrade
            excessiveDegrade = xcvrConfigCli.cmisFecDegradeConfig.excessiveDegrade
            # Save coherent error-correction degrade thresholds
            if detectedDegrade.enabled:
               cmds.addCommand(
                  'transceiver error-correction coherent degrade detected '
                  'raise {} '
                  'clear {} '.format(
                     detectedDegrade.raiseThreshold,
                     detectedDegrade.clearThreshold ) )
            elif options.saveAll:
               cmds.addCommand(
                  'no transceiver error-correction coherent degrade detected' )
            if excessiveDegrade.enabled:
               cmds.addCommand(
                  'transceiver error-correction coherent degrade excessive '
                  'raise {} '
                  'clear {} '.format(
                     excessiveDegrade.raiseThreshold,
                     excessiveDegrade.clearThreshold ) )

            elif options.saveAll:
               cmds.addCommand(
                  'no transceiver error-correction coherent degrade excessive' )

         if cmisLoopbackEnabled:
            CmisLoopbackConfigSaveHandler(
               cmds, xcvrConfigCli )
