# Copyright (c) 2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from array import array
from socket import gethostname
from datetime import datetime, timedelta
import re, os, time
import Logging
import Tac

SchedCliType = Tac.Type( "System::CliScheduler::ScheduledCli" )
intervalMin = 2
# Maximum scheduling interval is 1 day. That transalates to 24X3600 seconds
intervalMax = ( 24 * 60 )

# Timeout in minutes
timeoutMin = SchedCliType.timeoutMin / 60
timeoutMax = SchedCliType.timeoutMax / 60
timeoutDefault = SchedCliType.timeoutDefault / 60

compressAlgoEnum = { "gzip": "Compression algorithm gzip",
                      "bzip2": "Compression algorithm bzip2",
                      "xz": "Compression algorithm xz" }
compressAlgoDefault = "gzip"
decompressAlgoBySuffix = { ".gz": "gunzip", ".bz2": "bunzip2", ".xz": "unxz" }
decompressAlgoByCompressAlgo = { "gzip": "gunzip", "bzip2": "bunzip2", "xz": "unxz" }

maxLogFilesMin = 0
maxLogFilesMax = 10000

oneKb = 1024
oneMeg = 1024 * oneKb
oneGig = 1024 * oneMeg

jobsInProgressMin = 1
jobsInProgressMax = 4

# pkgdeps: import CliPlugin.CliCli
eosCliShell = os.getenv( 'CLI_SCHED_EOS_CLISHELL', "/usr/bin/CliShell" )
showTechJobName = os.getenv( 'CLI_SCHED_DEFAULT_JOBNAME', "tech-support" )
showTechCliCommand = os.getenv( 'CLI_SCHED_DEFAULT_COMMAND', "show tech-support" )
showTechIntervalDefault = 60
showTechMaxLogFilesDefault = 100
showTechMaxTotalSizeDefault = 0
showTechVerboseDefault = 'CLI_SCHED_DEFAULT_JOBNAME' in os.environ

scheduleNow = 0
scheduleNowStr = "now"

scheduleOnce = 0
scheduleOnceStr = "once"

DATEFMT_ERROR = { "INVALID": -1, "PAST": -2 }

logPrefixDefault = "flash:schedule/"

SYS_CLI_SCHEDULER_JOB_COMPLETED = Logging.LogHandle(
   "SYS_CLI_SCHEDULER_JOB_COMPLETED",
   severity=Logging.logDebug,
   fmt="The scheduled CLI execution job \'%s\' completed successfully.%s",
   explanation="A scheduled CLI command successfully executed. The output "\
      "of that command is stored in a log file if max-log-files > 0. The default "\
      "log file location is flash:/schedule/..., but may be in another location "\
      "if the \'loglocation\' option was specified.",
   recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CLI_SCHEDULER_ABORT = Logging.LogHandle(
   "SYS_CLI_SCHEDULER_ABORT",
   severity=Logging.logWarning,
   fmt="Execution of scheduled CLI execution job \'%s\' was aborted due "\
          "to an error: %s",
   explanation="Scheduled execution of a CLI command was aborted. The output "\
               "of that command has been stored in a log file if "\
               "max-log-files > 0. The default log file location is flash, "\
               "but may be in another location if the \'loglocation\' option was "\
               "specified.",
   recommendedAction="Please try to execute the CLI command interactively to "\
         "make sure it works fine." )

SYS_CLI_SCHEDULER_FILESYSTEM_FULL = Logging.LogHandle(
   "SYS_CLI_SCHEDULER_FILESYSTEM_FULL",
   severity=Logging.logWarning,
   fmt="Execution of scheduled CLI execution job \'%s\' was aborted due "\
         "to target filesystem being full",
   explanation="Scheduled execution of a CLI command was aborted due to lack "\
         "of space in target filesystem.",
   recommendedAction="Please delete unused files to free up space." )

SYS_CLI_SCHEDULER_SKIP = Logging.LogHandle(
   "SYS_CLI_SCHEDULER_SKIP",
   severity=Logging.logNotice,
   fmt="Execution of scheduled CLI execution job \'%s\' was skipped",
   explanation="Scheduled execution of a CLI command was skipped "\
         "since previous execution for this job is yet to complete.",
   recommendedAction="This may be due to CLI command taking long to complete. "\
         "Please try to increase execution interval to a larger value." )

SYS_MEMORY_EXHAUSTION_CLI_SCHEDULER_DISABLED = Logging.LogHandle(
   "SYS_MEMORY_EXHAUSTION_CLI_SCHEDULER_DISABLED",
   severity=Logging.logNotice,
   fmt="CliScheduler is disabled.  All subsequent scheduled CLI execution jobs "\
         "will be skipped.",
   explanation="CliScheduler is disabled due to memory exhaustion on the system.",
   recommendedAction="Please try to disable features, then run "\
         "'reset system memory exhaustion'." ) 

SYS_CLI_SCHEDULER_DISABLED_SKIP = Logging.LogHandle(
   "SYS_CLI_SCHEDULER_DISABLED_SKIP",
   severity=Logging.logNotice,
   fmt="Execution of scheduled CLI execution job \'%s\' was skipped because "\
         "CliScheduler is currently disabled.",
   explanation="Scheduled execution of a CLI command was skipped "\
         "since CliScheduler is currently disabled.",
   recommendedAction="See the previous log why CliScheduler is disabled." )

SYS_CLI_SCHEDULER_ENABLED = Logging.LogHandle(
   "SYS_CLI_SCHEDULER_ENABLED",
   severity=Logging.logNotice,
   fmt="CliScheduler is enabled, continuing its execution of scheduled CLI jobs.",
   explanation="CliScheduler is enabled, resuming its execution of scheduled "\
         "CLI jobs.",
   recommendedAction=Logging.NO_ACTION_REQUIRED )

def replaceToken( tokenString ):
   ''' This function takes a string as input and replaces the following
   tokens with their respective values :
   1)  %h           :       Hostname until the first '.' 
   2)  %H           :       Hostname
   3)  %D           :       Date and Time ( YYYY-MM-DD.HH-MM )
   4)  %D{format}   :       Date and Time using strftime() utility'''

   fqdnHostName = gethostname()
   localtime = time.localtime()

   # The following block searches for '%D{formatString}' type of tokens and 
   # replaces them by the date-time in the desired format. 
   regexObj = re.compile( '(%D{[^}]*})' )
   match = regexObj.search( tokenString )
   while match:
      matchedToken = match.group()
      formatString = matchedToken[ 3:-1 ]
      formatTime = time.strftime( formatString, localtime )
      tokenString = tokenString.replace( matchedToken, formatTime )
      match = regexObj.search( tokenString )

   # Replace %h and %H
   hostname = fqdnHostName.split( ".", 1 )[ 0 ]
   tokenString = tokenString.replace( '%h', hostname )
   tokenString = tokenString.replace( '%H', fqdnHostName )

   # Replace %D
   formatString = '%Y-%m-%d.%H%M'
   tokenString = tokenString.replace( '%D',
   time.strftime( formatString, localtime ) )

   return tokenString

def extractTsFromFilename( fname ):
   '''
   Filename will be 
      <jobName>_YYYY-MM-DD.HHMM.log if its yet to be compressed or
      <jobName>_YYYY-MM-DD.HHMM.log.gz or
      <jobName>_YYYY-MM-DD.HHMM.log.bz2 or
      <jobName>_YYYY-MM-DD.HHMM.log.xz
   e.g. tech-support_2011-02-28.0027.log or
        tech-support_2011-02-28.0027.log.gz or
        tech-support_2011-02-28.0027.log.bz2 or
        tech-support_2011-02-28.0027.log.xz
   If filename is invalid return None
   '''
   try:
      ts = re.findall( r"\d{4}-\d{2}-\d{2}.\d{4}", fname )[ 0 ]
   except IndexError:
      return None
   try:
      return time.mktime( time.strptime( ts, "%Y-%m-%d.%H%M" ) )
   except ValueError:
      return None
   except OverflowError:
      return time.mktime( time.localtime() )

def extractAtFromDateTime( dateTime ):
   if dateTime == scheduleNow:
      return scheduleNowStr
   else:
      atTime = time.localtime( dateTime )
      atStr = time.strftime( "%H:%M:%S %m/%d/%Y", atTime )
      return atStr

# Returns the count of "Job under progress" from "show schedule summary" output      
def jobUnderProgressCountInSummary( summary ):
   return len( re.findall( 'under.+?progress', summary, flags=re.DOTALL ) )

# Returns the number of seconds since epoch or 0 (scheduleNow) or -1 (on error)
# Input format is {"now" | 0 | [hh, mm, ss] | [hh, mm, ss, MM, dd, yyyy]}
def createDateTimefromAt( at, interval ):   
   result = array( 'd' )
   if at == scheduleNowStr:
      result.append( scheduleNow )
   elif at == scheduleNow:
      result.append( at )
   else:
      strAt = " ".join( map( str, at ) )
      try:
         if len( at ) == 3:
            curTime = datetime.now()
            # Check if we have already crossed that time for today
            if ( curTime.hour > at[ 0 ] or ( curTime.hour == at[ 0 ] and
                                             ( curTime.minute > at[ 1 ] or
                                               ( curTime.minute == at[ 1 ] and
                                                 curTime.second > at[ 2 ] ) ) ) ):
               # Schedule it on the next day
               curTime = curTime + timedelta( days=1 )
            strAt = strAt + time.strftime( " %m %d %Y", curTime.timetuple() )
         epochSched = time.mktime( time.strptime( strAt, "%H %M %S %m %d %Y" ) )
         epochCur = time.time()
         result.append( epochSched )
         if epochCur > epochSched and interval:
            nextSched = datetime.strptime( strAt, "%H %M %S %m %d %Y" )
            nextSched = nextSched + timedelta( minutes = interval )
            now = datetime.now()
            if nextSched < now:
               diff = now - nextSched
               toAdd = divmod( diff.total_seconds(), interval * 60 )
               nextSched = nextSched + \
                           timedelta( minutes = toAdd[0] * interval )
               if toAdd[1]:
                  nextSched = nextSched + timedelta( minutes = interval )
            result.append( time.mktime( nextSched.timetuple() ) )
      except ValueError:
         result.append( DATEFMT_ERROR[ "INVALID" ] )
   return result

def inPast( epoch ):
   # scheduleNow (0) should not be treated as past
   return ( epoch != scheduleNow ) and ( time.time() > epoch )

def diskUseFmt( diskUse ):
   if diskUse == showTechMaxTotalSizeDefault:
      return '-'
   if diskUse >= oneGig and diskUse % oneGig == 0:
      diskUseText = str( int( diskUse / oneGig ) ) + ' GB'
   elif diskUse >= oneMeg and diskUse % oneMeg == 0:
      diskUseText = str( int ( diskUse / oneMeg ) ) + ' MB'
   elif diskUse >= oneKb and diskUse % oneKb == 0:
      diskUseText = str( int( diskUse / oneKb ) ) + ' KB'
   else:
      diskUseText = str( int( diskUse ) ) + ' B'
   return diskUseText

class CliSchedLogfileList:
   def __init__( self, root ):
      self.root = root
      self.fileList = []
      self.refresh()

   def refresh( self ):
      self.fileList = []
      for root, _, files in os.walk( self.root ):
         self.fileList += [ os.path.join( root, f ) for f in files \
                          if extractTsFromFilename( f ) ]
      self.fileList.sort( key=extractTsFromFilename )

   def deleteFiles( self, flist ):
      for fname in flist:
         os.unlink( fname )
      self.refresh()

   def __getitem__( self, idx ):
      return self.fileList[ idx ]

   def getDiskSpace( self ):
      currentSpace = 0
      for fileName in self.fileList:
         currentSpace += os.path.getsize( fileName )
      return currentSpace

class CliSchedLogMgr:
   def __init__( self, root, maxLogFiles, maxTotalSize ):
      self.root = root
      self.maxLogFiles = maxLogFiles
      self.maxTotalSize = maxTotalSize
      self.fileset = CliSchedLogfileList( self.root )

   def getSnapshots( self ):
      return self.fileset[:]

   def rotateSnapshots( self ):
      files = self.fileset[:]
      files.reverse()
      files = files[ self.maxLogFiles: ]
      self.fileset.deleteFiles( files )

   def purgeFiles( self ):
      if self.maxTotalSize == showTechMaxTotalSizeDefault:
         return False
      fileCount = len( self.fileset.fileList )
      if fileCount <= 1:
         return False
      currentSpace = 0
      # Start at the last index (newest file) and work backwards...
      for fileIndex in reversed ( range ( fileCount ) ):
         currentSpace += os.path.getsize( self.fileset[ fileIndex ] )
         if currentSpace > self.maxTotalSize:
            deleteIndex = fileIndex + 1
            # Make sure we don't delete the last file.
            if fileCount == deleteIndex:
               continue
            self.fileset.deleteFiles( self.fileset[ :deleteIndex ] )
            return True
      return False

   def diskSpaceUsed( self ):
      return self.fileset.getDiskSpace()
