# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Assert
import re

from ArnetLib import U16_MAX_VALUE
from CliCommand import CliExpression, Node
from CliParserCommon import Completion, noMatch, MatchResult
from CliParser import safeInt
from CliMatcher import (
      Matcher,
      EnumMatcher,
      IntegerMatcher,
   )

COMMUNITY_BASE10_MAX = 4294967040

class CommunityAannMatcher( Matcher ):
   """Matches community in aa:nn notation"""
   def __init__( self, helpdesc, **kargs ):
      super().__init__( helpdesc=helpdesc, **kargs )
      self.completion_ = Completion( 'aa:nn', helpdesc, False )
      self.commFormatRe_ = re.compile( r'^(\d+):(\d+)$' )
      self.commFormatCompletionRe_ = re.compile(
         r'^(\d+)?:?(\d+)?$' )

   def match( self, mode, context, token ):
      m = self.commFormatRe_.match( token )
      if m is None:
         return noMatch
      for group in m.groups():
         if group is not None and ":" not in group and int( group ) > U16_MAX_VALUE:
            return noMatch
      return MatchResult( token, token )

   def completions( self, mode, context, token ):
      m = self.commFormatCompletionRe_.match( token )
      if m is None:
         return []
      for group in m.groups( '0' ):
         n = safeInt( group )
         if n is None or n > U16_MAX_VALUE:
            return []
      return [ self.completion_ ]

   def __str__( self ):
      return '<aa:nn>'

communityEnumMatcher = EnumMatcher( {
   'GSHUT' : 'Graceful Shutdown (well-known community)',
   'internet' : 'Internet (well-known community)',
   'local-as' : 'Do not send outside local AS',
   'no-advertise' : 'Do not advertise to any peer',
   'no-export' : 'Do not export to next AS',
   } )

class CommunityConstExpr( CliExpression ):
   expression = "AANN_FORMAT | COMMUNITY_ENUM | COMMUNITY_U32"
   data = {
         "AANN_FORMAT" : Node(
            matcher=CommunityAannMatcher( "Community number" ),
            alias="communityValue" ),
         "COMMUNITY_ENUM" : Node(
            matcher=communityEnumMatcher,
            alias="communityValue" ),
         "COMMUNITY_U32" : Node(
            matcher=IntegerMatcher( 1, COMMUNITY_BASE10_MAX,
               helpdesc='Community number' ),
            alias="communityValue" ),
      }

commValueEnum = {
   'no-export' : 4294967041,
   'no-advertise' : 4294967042,
   'local-as' : 4294967043,
   'internet' : 0,
   'GSHUT' : 4294901760,
}

def getCommIntValue( value ):
   '''
   This function assumes the value we try to convert is santized beforehand, this
   means that the value passed must fit in a 32 bit number. If the value is a string
   in the aa:nn format, both numbers (aa and nn) should fit in a 16 bit numbers.
   '''
   if isinstance( value, int ):
      commValue = value
   elif ':' in str( value ):
      valueSplit = value.split( ':' )
      Assert.assertEqual( len( valueSplit ), 2 )
      high = int( valueSplit[ 0 ] )
      low = int( valueSplit[ 1 ] )
      Assert.assertTrue( 0 <= low <= 0xFFFF )
      Assert.assertTrue( 0 <= high <= 0xFFFF )
      commValue = ( high << 16 ) | ( low & 0xFFFF )
   elif str( value ) in commValueEnum:
      commValue = commValueEnum[ str( value ) ]
   else:
      commValue = int( value )
   Assert.assertTrue( 0 <= commValue <= 0xFFFFFFFF )
   return commValue
