# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from CliCommand import CliExpression
from CliParserCommon import Completion, noMatch, MatchResult
from CliMatcher import (
      Matcher,
      KeywordMatcher,
      FloatMatcher
   )

#-------------------------------------------------------------------------
# Common Tokens
# In the future tokens in RouteMapCli.CommonTokens can be merged here.
#-------------------------------------------------------------------------
class CommonTokens:
   '''
   This class is intended to keep common tokens that may be shared across several
   commands with the new CLI parser.
   '''
   additive = KeywordMatcher( 'additive',
                              helpdesc='Add communities to those already present' )
   delete = KeywordMatcher( 'delete', helpdesc='Delete matching communities' )
   # To be used by protocols at route-map application points.
   routeMapApplication = KeywordMatcher( 'route-map',
                                         helpdesc='Specify which route map to use' )
   nextHop = KeywordMatcher( 'next-hop', helpdesc='Route Next-hop' )

# shared by both CliPlugin and CliSavePlugins
statementConfigToken = 'statement'

#-------------------------------------------------------------------------
# Extended Community Expressions
#-------------------------------------------------------------------------
class ExtCommActionExpression( CliExpression ):
   expression = '( additive | delete )'
   data = {
      'additive': CommonTokens.additive,
      'delete': CommonTokens.delete
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      action = args.get( 'additive' )
      if not action:
         action = args.get( 'delete' )
      if action:
         args[ 'ACTION' ] = [ action ]

#-------------------------------------------------------------------------
# Set Metric Matchers
#-------------------------------------------------------------------------
class MetricModifierAndNumberMatcher( Matcher ):
   def __init__( self, modifier, matcher, helpdesc=None, helpname=None ):
      self.modifier_ = modifier
      self.matcher_ = matcher
      Matcher.__init__( self, helpdesc=helpdesc, helpname=helpname )

   def match( self, mode, context, token ):
      if token.startswith( self.modifier_ ):
         numberMatch = self.matcher_.match( mode, context,
                                            token[ len( self.modifier_ ) : ] )
         if numberMatch is not noMatch:
            return MatchResult( numberMatch.result, str( numberMatch.result ) )

      return noMatch

   def completions( self, mode, context, token ):
      if ( token == '' or token == self.modifier_ or
           self.match( mode, context, token ) is not noMatch ):
         return [ Completion( self.helpname_, self.helpdesc_, False ) ]
      return []

class InputRestrictedFloatMatcher( FloatMatcher ):
   # This matcher extends FloatMatcher in order to prevent user input of
   # a float with more than N number of decimal places of precision.
   def __init__( self, lbound, ubound, precisionString, maxPrecision ):
      self.maxPrecision_ = maxPrecision
      FloatMatcher.__init__( self, lbound=lbound, ubound=ubound,
                             precisionString=precisionString )

   def precisionOutOfBounds( self, strFloat ):
      decimalPointIndex = strFloat.find( '.' )
      if ( decimalPointIndex != -1 and
           len( strFloat[ decimalPointIndex : ] ) > self.maxPrecision_ + 1 ):
         return True
      return False

   def match( self, mode, context, token ):
      if self.precisionOutOfBounds( token ):
         # pylint: disable-next=consider-using-f-string
         mode.addErrorAndStop( 'Precision of floating point value must be {} '
                               'decimal places or less'.format(
                                  self.maxPrecision_ ) )
         return noMatch
      return FloatMatcher.match( self, mode, context, token )

class MetricModifierAndFloatMatcher( MetricModifierAndNumberMatcher ):
   def completions( self, mode, context, token ):
      if ( token == '' or token == self.modifier_ or
           token == self.modifier_ + '.' or
           self.match( mode, context, token ) is not noMatch ):
         return [ Completion( self.helpname_, self.helpdesc_, False ) ]
      return []
