# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Tac
from CliMatcher import (
      EnumMatcher,
      FloatMatcher,
      IntegerMatcher,
      KeywordMatcher,
      PatternMatcher,
      StringMatcher,
)
from CliCommand import (
   CliExpression,
   hiddenKeyword,
   Node,
)
from ArnetLib import asnStrToNum
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.RouteMapCli import (
      asplainNumMatcher,
      asdotNumMatcher,
      peerFilterNameMatcher,
      mapNameMatcher,
)
from CliPlugin.RcfCliLib import rcfFunctionMatcher
from CliPlugin.MplsCli import evpnMplsSupportedGuard, mplsSupportedGuard
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.WanTEShowCli import pathSelectionSupportedGuard
from CliToken.RoutingBgpConstants import evpnNexthopSelfSupportNlris
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import MultiRangeRule

U16_MAX_VALUE = 0xFFFF
I32_MAX_VALUE = 0x7FFFFFFF
U32_MAX_VALUE = 0xFFFFFFFF
U64_MAX_VALUE = 0xFFFFFFFFFFFFFFFF
MAX_PORT_VALUE = U16_MAX_VALUE

# Keyword matchers: please keep these alphabetically sorted
action = KeywordMatcher( "action", helpdesc="Missing policy action options" )
activate = KeywordMatcher( "activate",
      helpdesc="Activate neighbor in the address family" )
addPath = KeywordMatcher( "additional-paths",
      helpdesc="BGP additional-paths commands" )
addpathInstall = KeywordMatcher( "install", helpdesc="Install BGP backup path" )
addpathInstallEcmpPrimary = KeywordMatcher( "ecmp-primary",
      helpdesc="allow additional path with ECMP primary path" )
addpathReceive = KeywordMatcher( "receive", helpdesc="Receive multiple paths" )
addpathSend = KeywordMatcher( "send", helpdesc="Send multiple paths" )
addrFamily = KeywordMatcher( "address-family",
      helpdesc="Address-family configuration options" )
advertiseInactive = KeywordMatcher( "advertise-inactive",
      helpdesc="Advertise BGP routes even if they are inactive in RIB" )
advertiseOnly = KeywordMatcher( "advertise-only",
      helpdesc="Advertise without installing the generated blackhole route in FIB" )
aggAttributes = KeywordMatcher( "aggregate-attributes",
      helpdesc="Name of the route map used to set the attributes of automatic "
               "aggregate routes" )
age = hiddenKeyword( 'age' )
aggregate = KeywordMatcher( "aggregate-address",
      helpdesc="Configure aggregate address" )
aigpPeerType = EnumMatcher( {
      'ibgp': 'iBGP peers',
      'ebgp': 'eBGP peers',
      'confederation': 'Confederation peers' } )
aigpSession = KeywordMatcher( 'aigp-session',
      helpdesc='AIGP session' )
allowAsIn = KeywordMatcher( 'allowas-in',
      helpdesc='Allow local-as in updates' )
always = KeywordMatcher( "always",
      helpdesc="Always preserve route attributes, overwriting route map changes" )
alwaysCompareMed = KeywordMatcher( "always-compare-med",
      helpdesc="BGP Always Compare MED" )
appAny = KeywordMatcher( 'any', helpdesc='Any eligible path' )
appBackup = KeywordMatcher( 'backup',
      helpdesc='Best path and installed backup path' )
appEcmp = KeywordMatcher( 'ecmp', helpdesc='All paths in best path ECMP group' )
appLimit = KeywordMatcher( 'limit', helpdesc='Limit to <n> eligible paths' )
apSendPrefixList = KeywordMatcher( 'prefix-list',
      helpdesc='Apply the configuration only to routes matching the prefix list' )
asMatch = KeywordMatcher( "match", helpdesc="Match first <n> ASNs" )
asPath = KeywordMatcher( "as-path", helpdesc="AS path commands" )
aspathCmpIncludeNh = KeywordMatcher( 'aspath-cmp-include-nexthop',
      helpdesc='Include nexthop in as-path comparison' )
asSet = KeywordMatcher( "as-set",
      helpdesc="Generate autonomous system set path information" )
asn = KeywordMatcher( "asn", helpdesc="Autonomous system number" )
asnNotation = KeywordMatcher( "notation", helpdesc="AS number format notation" )
asnNotationAsdot = KeywordMatcher( "asdot",
      helpdesc="AS number representation in asdot format" )
asnNotationAsplain = KeywordMatcher( "asplain",
      helpdesc="AS number representation in asplain format" )
attribute = KeywordMatcher( 'attribute',
      helpdesc='Path attributes manipulation commands' )
attributeTunneling = KeywordMatcher( 'tunneling',
      helpdesc="Tunnel customer's path attributes over VPN" )
attributeRcf = KeywordMatcher( "attribute",
      helpdesc="Configuration to set attributes on aggregate route" )
attrMap = KeywordMatcher( "attribute-map",
      helpdesc="Name of the route map used to set the attribute of the aggregate "
               "route" )
autoLocalAddr = KeywordMatcher( "auto-local-addr",
      helpdesc="Automatically determine the local address to be used for the "
               "non-transport AF" )
autoAggDomain = KeywordMatcher( 'auto-aggregation-domain',
      helpdesc='Configure an automatic aggregation domain' )
batchSize = KeywordMatcher( "batch-size",
      helpdesc="batch size for FIB route acknowledgements" )
bestPath = KeywordMatcher( "bestpath",
      helpdesc="Select the bestpath selection algorithm for BGP routes" )
bestPathEcmp = KeywordMatcher( 'ecmp',
      helpdesc='ECMP' )
bestPathEcmpDisabled = KeywordMatcher( 'disabled',
      helpdesc='Disable ECMP' )
bestPathNexthop = KeywordMatcher( 'next-hop',
      helpdesc='next-hop resolution domain' )
bestPathNexthopResolution = KeywordMatcher( 'resolution',
      helpdesc='next-hop resolution domain' )
bestPathNexthopTunnel = KeywordMatcher( 'tunnel',
      helpdesc='tunnel resolution domain ' )
bestPathNhg = KeywordMatcher( 'nexthop-group',
      helpdesc='static nexthop-group tunnel' )
bfd = KeywordMatcher( "bfd", helpdesc="Configure BFD fallover for this peer" )
bgp = KeywordMatcher( "bgp", helpdesc="Border Gateway Protocol" )
bgpRedistInternal = KeywordMatcher( "redistribute-internal",
      helpdesc="Redistribute internal BGP routes" )
bgpTimers = KeywordMatcher( "bgp", helpdesc="Timers for the BGP protocol" )
bothImportExport = KeywordMatcher( "both", helpdesc="Both Route Import and Export" )
cluster = KeywordMatcher( "cluster-id",
      helpdesc="Cluster ID of this router acting as a route reflector" )
community = KeywordMatcher( "community", helpdesc="BGP community attribute" )
compatibleMatcher = KeywordMatcher( 'compatible',
      helpdesc='treat as compatible in bestpath selection' )
confed = KeywordMatcher( "confederation", helpdesc="Confederation" )
confedId = KeywordMatcher( "identifier", helpdesc="Confederation Identifier" )
confedPeers = KeywordMatcher( "peers", helpdesc="Confederation Peers" )
connectionModeDeprecated = Node( KeywordMatcher( "connection-mode",
      helpdesc="Configure connection-mode for TCP session" ),
      deprecatedByCmd='neighbor passive' )
controlPlane = KeywordMatcher( "control-plane-filter",
      helpdesc="Control plane filter for BGP" )
convergence = KeywordMatcher( "convergence", helpdesc="Bgp convergence parameters" )
convergenceSlowPeer = KeywordMatcher( "slow-peer",
      helpdesc="Peers that do not establish session within a reasonable time" )
convergenceTime = KeywordMatcher( "time",
      helpdesc="Maximum amount of time to wait before declaring initial BGP "
               "convergence" )
counters = KeywordMatcher( "counters", helpdesc="ACL Counters" )
deactivate = KeywordMatcher( 'deactivate',
      helpdesc='Deactivate neighbor in the address family' )
domain = KeywordMatcher( 'domain', helpdesc="Configure domain for EVPN peer" )
domainPath = KeywordMatcher( 'd-path',
      helpdesc="Domain path length in bestpath route selection" )
domainRemote = KeywordMatcher( 'remote', helpdesc="Remote domain" )
domainAll = KeywordMatcher( 'all', helpdesc='Apply config for all domains' )
defOrigin = KeywordMatcher( "default-originate",
      helpdesc="Advertise a default route to this peer" )
defOriginAlways = KeywordMatcher( "always",
      helpdesc="Always advertise a default route to this peer" )
default = KeywordMatcher( "default",
      helpdesc="Default neighbor configuration commands" )
defaultAllow = KeywordMatcher( "default-allow",
      helpdesc="Set the default rule as ACCEPT at the end of the Bgp Service Chain" )
defaultEncap = KeywordMatcher( 'default',
      helpdesc="Default transport encapsulation for neighbor" )
defaultMetric = KeywordMatcher( "default-metric",
      helpdesc="BGP multi-exit discriminator default-metric" )
defaultRoute = KeywordMatcher( "default-route",
      helpdesc="Advertise a default route to VPN neighbors" )
defaultRouteExportAlways = KeywordMatcher( "always",
      helpdesc="Always advertise a default route to VPN neighbors" )
defaultRouteTarget = KeywordMatcher( 'default-route-target',
      helpdesc='Default route target prefix advertisement' )
description = KeywordMatcher( 'description',
      helpdesc='Assign a description to a neighbor' )
dhcp = KeywordMatcher( "dhcp", helpdesc="Redistribute DHCPv6 routes" )
directInstall = KeywordMatcher( "direct-install",
      helpdesc="Install routes directly" )
direction = KeywordMatcher( "direction",
      helpdesc="Missing policy direction options" )
directionIn = KeywordMatcher( "in", helpdesc="Missing policy inbound direction" )
directionOut = KeywordMatcher( "out", helpdesc="Missing policy outbound direction" )
disabled = KeywordMatcher( 'disabled', helpdesc='Disable show command indexing' )
disabledKwMatcherForTpFieldSetMappings = KeywordMatcher( 'disabled',
      helpdesc='Disable traffic policy field set updates' )
disabledKwMatcherForVspFieldSetMappings = KeywordMatcher( 'disabled',
      helpdesc='Disable vrf selection policy field set updates' )
disabledKwMatcherForNhResolution = KeywordMatcher( 'disabled',
      helpdesc='Disable next-hop resolution' )
discardPathAttr = KeywordMatcher( 'discard', helpdesc='Discard path attributes' )
distance = KeywordMatcher( "distance",
      helpdesc="Administrative distance configuration" )
distanceBgp = KeywordMatcher( "bgp",
      helpdesc="Administrative distance for the BGP protocol" )
dontCapabilityNegotiate = KeywordMatcher(
      'dont-capability-negotiate',
      helpdesc="Don't perform Capability Negotiation with this neighbor" )
drainUndrain = KeywordMatcher( "drain-undrain",
      helpdesc="In-place FEC update for drain-undrain events" )
drainUndrainHidden = Node( matcher=drainUndrain, hidden=True )
dscp = KeywordMatcher( "dscp", helpdesc="Set DSCP value for BGP traffic" )
dynamic = KeywordMatcher( "dynamic", helpdesc="Set the BGP dynamic properties" )
ecmp = KeywordMatcher( "ecmp", helpdesc="Maximum number of installed ECMP routes" )
ecmpFast = KeywordMatcher( "ecmp-fast",
      helpdesc="Tie-break BGP paths in a ECMP group based on the order of arrival" )
eBgpMatcher = KeywordMatcher( 'ebgp',
      helpdesc='paths received from eBGP neighbor' )
ebgpMultiHop = KeywordMatcher( 'ebgp-multihop',
      helpdesc="Allow BGP connections to indirectly connected external peers" )
eligibleRoutes = KeywordMatcher( "eligible-routes",
      helpdesc="Name of the route map used to filter the contributors to automatic "
               "aggregate routes" )
encap = KeywordMatcher( "encapsulation",
      helpdesc="Default transport encapsulation for neighbor" )
enforceFirstAs = KeywordMatcher( "enforce-first-as",
      helpdesc="Enforce the First AS for EBGP routes(default)" )
evpn = KeywordMatcher( "evpn", helpdesc="EVPN address family" )
evpnRouteTypeNhSelf = KeywordMatcher( "route-type",
      helpdesc="Enable next hop self for EVPN route type" )
export = KeywordMatcher( "export", helpdesc="Route Export" )
exportLocalPref = KeywordMatcher( 'export-localpref',
      helpdesc="Override localpref when exporting to an internal peer" )
exportMethodMatcher = EnumMatcher( {
      "all": "All eligible paths",
      "best": "Only the best path" } )
extendedCommunity = KeywordMatcher( 'extended',
      helpdesc='Send extended community attribute' )
external = KeywordMatcher( "external",
      helpdesc="OSPF routes learned from external sources" )
fallbackAs = KeywordMatcher( 'fallback',
      helpdesc="Prefer router AS number over local AS number" )
fallOverDeprecated = Node( KeywordMatcher( "fall-over",
      helpdesc="Configure BFD protocol options for this peer" ),
      deprecatedByCmd='neighbor bfd' )
fec = KeywordMatcher( "fec", helpdesc="Configure BGP FEC parameters" )
fib = KeywordMatcher( "fib", helpdesc="Host routes to FIB" )
fieldSet = KeywordMatcher( "field-set", helpdesc="Field Set" )
flowspecVpnIpv4 = KeywordMatcher( "flow-spec-vpn-ipv4",
      helpdesc="Flowspec VPN IPv4 unicast address family" )
flowspecVpnIpv6 = KeywordMatcher( "flow-spec-vpn-ipv6",
      helpdesc="Flowspec VPN IPv6 unicast address family" )
forwarding = KeywordMatcher( "forwarding", helpdesc="Forwarding" )
fwdFailover = KeywordMatcher( "failover", helpdesc="Failover" )
fwdFailoverTrigger = KeywordMatcher( "trigger", helpdesc="Trigger" )
fwdFailoverTriggerSession = KeywordMatcher( "session", helpdesc="Session" )
grHelper = KeywordMatcher( "graceful-restart-helper",
      helpdesc="Enable graceful restart helper mode" )
llgrHelper = KeywordMatcher( "long-lived",
            helpdesc="Enable long lived graceful restart helper mode" )
grHelperStaleRoute = KeywordMatcher( "stale-route",
            helpdesc="Apply policy on stale route retention" )
sessionFailure = KeywordMatcher( "session-failure",
      helpdesc="Retain routes on session failures including hold-time expiry" )
gracefulRestart = KeywordMatcher( "graceful-restart",
      helpdesc="Enable graceful restart mode" )
group = KeywordMatcher( "group", helpdesc="Name of the peer-group" )
groupBy = KeywordMatcher( "group-by",
      helpdesc="List of contributor path attributes that are required to match" )
groupByAsPath = KeywordMatcher( "as-path", helpdesc="Group by AS path" )
groupByCommunity = KeywordMatcher( "community", helpdesc="Group by community" )
groupByExtCommunity = KeywordMatcher( "ext-community",
      helpdesc="Group by extended community" )
groupByLocPref = KeywordMatcher( "local-preference",
      helpdesc="Group by local preference" )
groupByMed = KeywordMatcher( "med", helpdesc="Group by metric" )
halfLifeSecondsKw = KeywordMatcher( "seconds",
      helpdesc="Half-life in seconds" )
halfLifeKw = KeywordMatcher( "half-life",
      helpdesc="Configure decay half-life for penalty" )
halfLifeMinutesKw = KeywordMatcher( "minutes",
      helpdesc="Half-life in minutes" )
holdForever = KeywordMatcher( "0", helpdesc="Hold forever: don't send keep-alives" )
hostRoutes = KeywordMatcher( "host-routes",
      helpdesc="BGP host routes configuration" )
interDomain = KeywordMatcher( "inter-domain",
      helpdesc="Enable nexthop self across local and remote EVPN domains" )
iPv4Unicast = KeywordMatcher( "ipv4-unicast",
      helpdesc="Enable IPv4 unicast address family on all IPv4 neighbors" )
iPv6AfterTransport = KeywordMatcher( "ipv6",
      helpdesc="Enable IPv4 unicast address family on all IPv6 neighbors" )
iPv6Unicast = KeywordMatcher( "ipv6-unicast",
      helpdesc="Enable IPv6 unicast address family on all neighbors" )
ignore = KeywordMatcher( "ignore",
      helpdesc="Ignore the AS path length in the bestpath route selection." )
inbound = KeywordMatcher( 'inbound', helpdesc='configure inbound' )
routeImport = KeywordMatcher( "import", helpdesc="Route Import" )
iarEvent = KeywordMatcher( "event",
      helpdesc="In-place FEC update operation for specified events" )
iarEventAll = KeywordMatcher( "all",
      helpdesc="In-place FEC update for all events" )
iarUpdate = KeywordMatcher( "update",
      helpdesc="In-place FEC update operation" )
iarTimeout = KeywordMatcher( "timeout",
      helpdesc="In-place FEC update tracking timeout" )
redistInclude = KeywordMatcher( 'include',
      helpdesc='Include following routes while redistributing' )
iBgpMatcher = KeywordMatcher( 'ibgp',
      helpdesc='paths received from iBGP neighbor' )
idleRestartTimer = KeywordMatcher( 'idle-restart-timer',
      helpdesc="Neighbor's idle restart timer" )
bgpMinHoldTime = KeywordMatcher( 'min-hold-time',
      helpdesc="Neighbor's minimum hold time constraint" )
bgpSendFailure = KeywordMatcher( 'send-failure',
      helpdesc="Neighbor's send failure related constraints" )
bgpSendFailureHoldTime = KeywordMatcher( 'hold-time',
      helpdesc="Neighbor's send failure hold time constraint" )
importLocalPref = KeywordMatcher( 'import-localpref',
      helpdesc="Override localpref when importing from an external peer" )
index = KeywordMatcher( 'index', helpdesc='Configure show command indexing' )
inPlace = KeywordMatcher( "in-place",
      helpdesc="In-place FEC operation" )
installMap = KeywordMatcher( "install-map",
      helpdesc="Name of the route map to allow routes" )
interface = KeywordMatcher( "interface",
      helpdesc="Interface range to be used for BGP session establishment" )
internal = KeywordMatcher( "internal",
      helpdesc="OSPF routes learned from internal sources" )
intraCluster = KeywordMatcher( 'intra-cluster',
      helpdesc='Intra-cluster route reflection' )
ip = KeywordMatcher( "ip", helpdesc="Install LU routes into IP RIB" )
ipForwarding = KeywordMatcher( "ip-forwarding", helpdesc="IP forwarding" )
ipv4 = KeywordMatcher( "ipv4", helpdesc="IPv4 related" )
ipv6 = KeywordMatcher( "ipv6", helpdesc="IPv6 related" )
ipv6Encaps = KeywordMatcher( 'ipv6', 'SRv6 transport' )
isLevel1 = KeywordMatcher( "level-1", helpdesc="Redistribute IS-IS level-1 routes" )
isLevel12 = KeywordMatcher( "level-1-2",
      helpdesc="Redistribute IS-IS level-1 and level-2 routes" )
isLevel2 = KeywordMatcher( "level-2", helpdesc="Redistribute IS-IS level-2 routes" )
isis = KeywordMatcher( "isis", helpdesc="IS-IS routes" )
largeCommunity = KeywordMatcher( 'large',
      helpdesc='Send large community attribute' )
layer2 = KeywordMatcher( "layer-2", helpdesc="Exchange L2 MAC/IP EVPN NLRI" )
layer3 = KeywordMatcher( "layer-3", helpdesc="Exchange L3 IP Prefix EVPN NLRI" )
layerMatcher = EnumMatcher( {
      # When both layer2 and layer3 are supported uncomment the following line
      # Bug# 499696: Add support for exporting best path only for VPN layer-2 paths
      # 'layer-2': 'L2 VPN NLRI',
      'layer-3': 'L3 VPN NLRI' } )
leaked = KeywordMatcher( 'leaked',
      helpdesc='Include leaked routes of this protocol while redistributing' )
lfibBackup = KeywordMatcher( 'lfib-backup',
      helpdesc="Backup configuration for next hop labeled unicast origination "
               "LFIB entry" )
limit = KeywordMatcher( "limit",
      helpdesc="Set limit on the number of dynamic BGP peers allowed" )
limitDeprecated = Node( KeywordMatcher( "limit",
      helpdesc="Set limit on the number of dynamic BGP peers allowed" ),
      deprecatedByCmd='dynamic peer max' )
linkBw = KeywordMatcher( "link-bandwidth",
      helpdesc="Enable link bandwidth community for routes from this peer" )
linkBwAdjust = KeywordMatcher( "adjust",
      helpdesc="Adjust link bandwidth for routes from this peer" )
linkBwAuto = KeywordMatcher( "auto",
      helpdesc="Enable link bandwidth auto generation for routes from this peer" )
linkBwAutoAfterAdjust = KeywordMatcher( "auto",
      helpdesc="Add the peering link's speed" )
linkBwDefault = KeywordMatcher( "default",
      helpdesc="Enable link bandwidth default generation for routes from this peer" )
linkBwPercent = KeywordMatcher( "percent", helpdesc="Add a percentage of the speed" )
linkBwSend = KeywordMatcher( 'link-bandwidth',
      helpdesc='Enable link bandwidth community for routes to this peer' )
linkState = KeywordMatcher( 'link-state',
      helpdesc='Link State address family' )
linkStateIdentifier = KeywordMatcher( 'identifier',
      helpdesc='Configure BGP-LS identifier' )
listen = KeywordMatcher( "listen", helpdesc="BGP listen" )
listenPort = KeywordMatcher( "listen-port",
      helpdesc="Configure the BGP listen port" )
localAs = KeywordMatcher( "local-as",
      helpdesc="Configure local AS number advertised to peer" )
localIp4Addr = KeywordMatcher( 'local-v4-addr',
   helpdesc='Configure a local IPv4 address for the neighbor' )
localIp6Addr = KeywordMatcher( 'local-v6-addr',
   helpdesc='Configure a local IPv6 address for the neighbor' )
logNeighborChanges = KeywordMatcher( "log-neighbor-changes",
      helpdesc="Log neighbor up/down events" )
lu = Node( matcher=KeywordMatcher( "labeled-unicast", helpdesc="Labeled Unicast" ),
           guard=mplsSupportedGuard )
mappings = KeywordMatcher( "mappings", helpdesc="Grouping parameters" )
match = KeywordMatcher( "match", helpdesc="Routes learned by the OSPF protocol" )
matchMap = KeywordMatcher( "match-map",
      helpdesc="Name of the route map used to filter the contributors of the "
               "aggregate route" )
maxAcceptedRoutes = KeywordMatcher( "maximum-accepted-routes",
      helpdesc="Maximum number of routes accepted from this peer" )
maxAdvRoutes = KeywordMatcher( "maximum-advertised-routes",
      helpdesc="Maximum number of routes advertised to this peer" )
maxAfterDynamicPeer = KeywordMatcher( "max",
      helpdesc="Set limit on the number of dynamic BGP peers allowed" )
maxHops = KeywordMatcher( 'maximum-hops', helpdesc="Maximum IP hops" )
maxNumPaths = KeywordMatcher( "maximum-paths",
      helpdesc="Maximum number of equal cost paths" )
maxRoutes = KeywordMatcher( "maximum-routes",
      helpdesc="Maximum number of routes received from this peer" )
maxSegSize = KeywordMatcher( "mss",
      helpdesc="Configure the BGP TCP socket maximum segment size" )
med = KeywordMatcher( "med", helpdesc="MED attribute" )
medConfed = KeywordMatcher( "confed", helpdesc="MED Confed" )
meshed = KeywordMatcher( 'meshed', helpdesc='Meshed route reflector' )
metricOut = KeywordMatcher( 'metric-out',
      helpdesc="Configure MED value advertised to peer" )
missingAsWorst = KeywordMatcher( "missing-as-worst",
      helpdesc="MED missing-as-worst" )
missingPolicy = KeywordMatcher( "missing-policy",
      helpdesc="Missing policy override configuration commands" )
mpAddrFamily = KeywordMatcher( 'address-family',
      helpdesc='Address family for missing policy configuration' )
mpAddrFamilyAll = KeywordMatcher( 'all',
      helpdesc='Include all address families for missing policy configuration' )
mpInclude = KeywordMatcher( 'include',
      helpdesc='Include route map references in missing policy decision' )
mpCommList = KeywordMatcher( 'community-list',
      helpdesc='Examine community list references' )
mpPrefixList = KeywordMatcher( 'prefix-list',
      helpdesc='Examine prefix list references' )
mpSubRouteMap = KeywordMatcher( 'sub-route-map',
      helpdesc='Examine sub route map references' )
mpDirection = KeywordMatcher( 'direction',
      helpdesc='Missing policy direction options' )
mpAction = KeywordMatcher( 'action',
      helpdesc='Missing policy action options' )
mpls = KeywordMatcher( "mpls", helpdesc="MPLS transport" )
mplsForEvpn = Node( matcher=KeywordMatcher( 'mpls', helpdesc='MPLS transport' ),
      guard=evpnMplsSupportedGuard )
multiPathRelax = KeywordMatcher( "multipath-relax",
      helpdesc="Allow ECMP with different AS paths" )
multicast = KeywordMatcher( "multicast", helpdesc="Multicast sub-address family" )
neighbor = KeywordMatcher( "neighbor",
      helpdesc='Neighbor configuration' )
neighborDefault = KeywordMatcher( "default",
      helpdesc="Default transport encapsulation for neighbor" )
neighborDefaultGlobal = KeywordMatcher( 'default',
      helpdesc='Apply to all neighbors' )
network = KeywordMatcher( "network", helpdesc="Configure routing for a network" )
neighborNextHop = KeywordMatcher( "next-hop",
      helpdesc="Peer next hop related configuration" )
neighborPeerSet = KeywordMatcher( "peer-set",
      helpdesc="A peer set name" )
newRoute = KeywordMatcher( "new-route",
      helpdesc="Delay new route programming for in-place FEC update" )
nextHop = KeywordMatcher( "next-hop",
      helpdesc="Next-hop address-family configuration" )
nextHopAf = KeywordMatcher( "address-family",
      helpdesc="Next-hop address-family configuration" )
nextHopKw = KeywordMatcher( 'next-hop',
      helpdesc='Next hop configuration' )
nextHopLuOriginateKwMatcher = KeywordMatcher( "originate",
      helpdesc="Originate BGP LU route for peer next hop" )
nextHopEpeLuOriginate = KeywordMatcher( 'originate',
      helpdesc="Originate BGP LU route for next hop" )
nextHopTunnelOriginate = KeywordMatcher( 'originate',
      helpdesc="Originate tunnel for next hop" )
nextHopPeer = KeywordMatcher( 'next-hop-peer',
      helpdesc="Use peer address as next hop address for routes from the peer" )
nextHopSelf = KeywordMatcher( "next-hop-self",
      helpdesc="Always advertise this router's address as the BGP next hop" )
thirdParty = KeywordMatcher( "third-party",
      helpdesc="Originate BGP LU routes for third party next hops received"
               " from the peer" )
nexthopUnchanged = KeywordMatcher( "next-hop-unchanged",
      helpdesc="Preserve original nexthop while advertising routes to eBGP peers" )
nextHopV6Addr = KeywordMatcher( 'next-hop-v6-addr',
      helpdesc='Configure a IPv6 next-hop address' )
nextHopV6AddrIn = KeywordMatcher( 'in',
      helpdesc='Configure IPv6 next-hop address for the inbound v6 routes' )
noEqualsDefaultForNeighbor = KeywordMatcher( "no-equals-default",
      helpdesc="Interpret all 'no neighbor' commands as 'default neighbor' "
               "commands" )
noPrepend = KeywordMatcher( 'no-prepend',
      helpdesc="Do not prepend the local AS number in AS path advertisements" )
nssaExternal = KeywordMatcher( "nssa-external",
      helpdesc="OSPF routes learned from external NSSA sources" )
orr = KeywordMatcher( 'optimal-route-reflection',
                      helpdesc='BGP optimal route reflection (ORR)' )
orrIgpPrefix = KeywordMatcher( 'igp-prefix',
                               helpdesc='Identify position by originator of prefix' +
                               ' in IGP' )
orrPosition = KeywordMatcher( 'position',
      helpdesc='Position in IGP topology for BGP optimal route reflection' )
orrPositionPeer = KeywordMatcher( 'peer-address',
      helpdesc='Use host route corresponding to peer address as ORR position' )
originAsBestPath = KeywordMatcher( "origin-as",
      helpdesc="Origin AS in bestpath route selection" )
originAsValidityBestPath = KeywordMatcher( "validity",
      helpdesc="RPKI origin AS validity in bestpath route selection" )
originate = KeywordMatcher( "originate",
      helpdesc="Changing next hop originate ability for v6 next-hops" )
rip = KeywordMatcher( "rip", helpdesc="RIP routes" )
dynamicPolicy = KeywordMatcher( "dynamic", helpdesc="Dynamic policy routes" )
attachedHost = KeywordMatcher( "attached-host",
      helpdesc="ARP generated host routes" )
ospf = KeywordMatcher( "ospf", helpdesc="OSPF protocol" )
ospf3 = KeywordMatcher( "ospfv3", helpdesc="Open Shortest Path First (OSPFv3)" )
ospf3Old = KeywordMatcher( "ospf3", helpdesc="Open Shortest Path First (OSPFv3)" )
out = KeywordMatcher( 'out', helpdesc="Direction to apply" )
outDelayChangesMatcherForConfig = KeywordMatcher( 'changes',
      helpdesc='Out delay applied to all route changes' )
outDelayMatcherForConfig = KeywordMatcher( 'out-delay',
      helpdesc='Delay outbound route updates' )
outDelayWithdrawalsMatcherForConfig = KeywordMatcher( 'withdrawals',
      helpdesc='Out delay applied to all route changes and withdrawals' )
passive = KeywordMatcher( 'passive',
      helpdesc='Configure passive TCP connection' )
paths = KeywordMatcher( "paths",
      helpdesc="Bgp paths" )
pathMtuDiscovery = KeywordMatcher( "pmtud",
      helpdesc="Configure the TCP socket path MTU discovery" )
buffer = KeywordMatcher( "buffer",
      helpdesc="Configure the TCP socket send and receive buffers" )
kbytes = KeywordMatcher( "kbytes",
      helpdesc="Size in Kilobytes" )
peer = KeywordMatcher( "peer", helpdesc="Bgp peer configuration" )
peerAfterDynamic = KeywordMatcher( "peer",
      helpdesc="Set the BGP dynamic peer properties" )
peerBestpathMatcher = KeywordMatcher( 'peer', helpdesc='type of peer' )
peerFilter = KeywordMatcher( "peer-filter",
      helpdesc="Name of the peer-filter to allow ASNs" )
peerGroup = KeywordMatcher( "peer-group", helpdesc="Name of the peer-group" )
peerGroupDeprecated = Node( KeywordMatcher( "peer-group",
      helpdesc="Name of the peer-group" ),
      deprecatedByCmd='neighbor peer group' )
peerInit = KeywordMatcher( "peer-init",
      helpdesc="In-place FEC update for peer-init events" )
peerMacResolutionTimeout = KeywordMatcher( "peer-mac-resolution-timeout",
      helpdesc='Time to wait for proactive resolution' )
percent = KeywordMatcher( "percent",
      helpdesc="Warning limit to be specified as a percentage" )
penaltyAdvertisementKw = KeywordMatcher( "advertisement",
      helpdesc="Configure penalty for path advertisment" )
penaltyChangeKw = KeywordMatcher( "change",
      helpdesc="Configure penalty for path attribute change" )
penaltyDecayKw = KeywordMatcher( "decay",
      helpdesc="Configure decay rate for penalty" )
penaltyKw = KeywordMatcher( "penalty",
      helpdesc="Route flap damping penalty configuration" )
penaltyThresholdKw = KeywordMatcher( "threshold",
      helpdesc="Route flap damping threshold configuration" )
penaltyThresholdMaximumKw = KeywordMatcher( "maximum",
      helpdesc="Maximum value of penalty for a path" )
penaltyThresholdReuseKw = KeywordMatcher( "reuse",
      "Value of penalty below which suppressed path would be reused" )
penaltyThresholdSuppressedKw = KeywordMatcher( "suppression",
      "Value of penalty above which path would be suppressed" )
penaltyWithdrawalTrackingKw = KeywordMatcher( "withdrawal-tracking",
      helpdesc="Configure penalty for path readvertisement" )
penaltyWithdrawalKw = KeywordMatcher( "withdrawal",
      helpdesc="Configure penalty for path withdrawal" )
suppressionConfigKw = KeywordMatcher( "suppression",
      helpdesc="Route flap damping suppression configuration" )
suppressionIgnoredKw = KeywordMatcher( "ignored",
      helpdesc="Configure parameters to ignore suppression" )
suppressionIgnoredPrefixListKw = KeywordMatcher( "prefix-list",
      helpdesc="Configure prefix list to ignore suppression" )
ipv4PrefixListKw = KeywordMatcher( "ipv4", helpdesc="IPv4 unicast prefix list" )
ipv6PrefixListKw = KeywordMatcher( "ipv6", helpdesc="IPv6 unicast prefix list" )
pathId = KeywordMatcher( "path-id", helpdesc="Specify the add path ID of the path" )
penaltyZero = KeywordMatcher( "0", helpdesc="Do not apply any penalty" )
prefixList = KeywordMatcher( "prefix-list", helpdesc="Prefix list reference" )
prefixListIn = KeywordMatcher( 'in', 'Configure an inbound prefix-list' )
prefixListOut = KeywordMatcher( 'out', 'Configure an outbound prefix-list' )
prependOwn = KeywordMatcher( "prepend-own",
      helpdesc="Prepend own AS number to AS path" )
prependOwnDisabled = KeywordMatcher( 'disabled',
      helpdesc="Disable prepending own AS number to AS Path" )
prePolicy = KeywordMatcher( 'pre-policy',
      helpdesc='configure how to handle paths that fail import' )
preserveAttrs = KeywordMatcher( 'preserve-attributes',
      helpdesc="Preserve refected routes' attributes" )
protocolInstance = KeywordMatcher( 'instance',
      helpdesc='Protocol instance' )
peerSetKw = KeywordMatcher( "peer-set",
      helpdesc="EPE Peer Set configuration" )
qos = KeywordMatcher( "qos",
      helpdesc="Set QoS parameters for CPU generated BGP traffic" )
subnetRange = KeywordMatcher( "range",
      helpdesc="Subnet Range to be associated with the peer-group" )
rcf = KeywordMatcher( 'rcf',
      helpdesc="Routing control functions configuration to "
               "filter and modify paths" )
rcfIn = KeywordMatcher( 'in',
      helpdesc="Inbound RCF function" )
rcfOut = KeywordMatcher( 'out',
      helpdesc="Outbound RCF function" )
rcfAttrSet = KeywordMatcher( 'rcf',
      helpdesc="RCF function used to set attributes on route" )
rcfFilter = KeywordMatcher( 'filter-rcf',
      helpdesc="RCF filter routes" )
rcfVpnRoute = KeywordMatcher( 'vpn-route',
      helpdesc="RCF filter VPN routes" )
rcfVrfRoute = KeywordMatcher( 'vrf-route',
      helpdesc="RCF filter VRF routes" )
rd = KeywordMatcher( "rd", helpdesc="BGP route distinguisher" )
rdAssignment = KeywordMatcher( 'assignment',
      helpdesc='route distinguisher assignment configuration' )
rdAuto = KeywordMatcher( 'auto',
      helpdesc='route distinguisher auto assignment configuration' )
reason = KeywordMatcher( 'reason',
      helpdesc='Shut down reason to be sent to the peer(s)' )
receivedEvpnRoutes = KeywordMatcher( "received-evpn-routes",
      helpdesc="Use local next hop for advertising received EVPN routes" )
receivedPathAttr = KeywordMatcher( "received",
      helpdesc="Received path attribute configuration" )
redistAttachedHost = KeywordMatcher( 'attached-host',
      helpdesc='Multicast source routes' )
redistMcastConnected = KeywordMatcher( 'connected',
      helpdesc='Connected interface routes' )
redistribute = KeywordMatcher( "redistribute",
      helpdesc="Redistribute routes in to BGP" )
redistMcastStatic = KeywordMatcher( 'static', helpdesc='Static multicast routes' )
reflection = KeywordMatcher( 'reflection', helpdesc='Route reflection control' )
remoteAs = KeywordMatcher( "remote-as", helpdesc="Neighbor's Autonomous System" )
remotePort = KeywordMatcher( 'remote-port',
      helpdesc='Configure BGP peer TCP port to connect to' )
removePrivateAs = KeywordMatcher( 'remove-private-as',
      helpdesc="Remove private AS numbers in outbound AS path if only private "
               "AS numbers are present" )
removePrivateAsAll = KeywordMatcher( 'all',
      helpdesc="Always remove private AS numbers in outbound AS path" )
removePrivateAsIngress = KeywordMatcher( 'ingress',
      helpdesc="Always remove private AS numbers in inbound AS path" )
replaceLocalAs = KeywordMatcher( 'replace-as',
      helpdesc="Replace the routing process AS number with the local AS "
               "number in AS path advertisements" )
replacePrivateAs = KeywordMatcher( 'replace-as',
      helpdesc="Replace private AS numbers with local AS number in outbound "
               "AS path" )
replacePrivateAsIngress = KeywordMatcher( 'replace-as',
      helpdesc="Replace private AS numbers with the neighbor's AS number in "
               "inbound AS path" )
replaceRemoteAs = KeywordMatcher( 'replace',
      helpdesc="Replace AS number with local AS number" )
resolution = KeywordMatcher( "resolution",
      helpdesc="Resolution related configuration" )
resolutionRouteMap = KeywordMatcher( 'route-map',
      helpdesc='route map' )
restartTime = KeywordMatcher( "restart-time",
      helpdesc="Set the max time needed to restart and come back up" )
retain = KeywordMatcher( 'retain',
      helpdesc='keep the paths that fail import' )
rfdDisabledKw = KeywordMatcher( "disabled",
      helpdesc="Disable route flap damping for neighbor" )
rfdKw = KeywordMatcher( "route-flap-damping",
      helpdesc="Route flap damping configuration" )
rfdMonitorOnlyKw = KeywordMatcher( "monitor-only",
      "Configure route flap damping in monitor only mode" )
rfdLog = KeywordMatcher( "log",
      "Configure logging for route flap damping" )
rfdLogConfigEvents = KeywordMatcher( "config-events",
      "Log route flap damping configuration change events" )
rfdLogPathEvents = KeywordMatcher( "path-events",
      "Log path suppression/reuse events" )
rib = KeywordMatcher( "rib", helpdesc="Which rib to install LU routes" )
ribRouteMap = KeywordMatcher( 'route-map',
      helpdesc='Route map with which to filter given RIB' )
ribs = KeywordMatcher( 'ribs', helpdesc='Resolving RIBs in order of preference' )
ribIn = KeywordMatcher( 'rib-in', helpdesc='configure peer rib-in policy' )
ribInDelay = KeywordMatcher( 'delay',
      helpdesc='Delay processing initial inbound route updates' )
ribInDelayEvent = KeywordMatcher( 'event',
      helpdesc='Delay inbound route updates for specified events' )
ribInDelayPeerInit = KeywordMatcher( 'peer-init',
      helpdesc='Start the timer when the peer is established' )
rmIn = KeywordMatcher( "in", helpdesc="Configure an inbound route map" )
rmOut = KeywordMatcher( "out", helpdesc="Configure an outbound route map" )
routes = KeywordMatcher( 'routes', helpdesc='Advertised and received routes' )
route = KeywordMatcher( "route",
      helpdesc="Configure route map for route installation" )
routeChanges = KeywordMatcher( "route-changes",
      helpdesc="In-place FEC update for route-change events" )
routeMap = KeywordMatcher( "route-map", helpdesc="Name a route map" )
routeTarget = KeywordMatcher( "route-target", helpdesc="Route Target" )
importedRoute = KeywordMatcher( "imported-route",
      helpdesc="Export routes imported from the same Afi/Safi" )
routeReflector = KeywordMatcher( 'route-reflector',
      helpdesc='Configure route reflector options' )
routeReflectorClient = KeywordMatcher( 'route-reflector-client',
      helpdesc='Configure peer as a route reflector client' )
routeToPeer = KeywordMatcher( 'route-to-peer',
      helpdesc="Use routing table information to reach the peer" )
routerId = KeywordMatcher( "router-id",
      helpdesc="Configure the router ID for the BGP process" )
secondsKwMatcher = KeywordMatcher( "seconds",
      helpdesc="Second(s)" )
segmentRoutingForEvpn = KeywordMatcher( 'segment-routing',
                                        'Segment Routing transport' )
sendCommunity = KeywordMatcher( 'send-community',
      helpdesc='Enable sending communities' )
sendCommunityAdd = KeywordMatcher( 'add',
      helpdesc='Add community attributes to send to this neighbor' )
sendCommunityRemove = KeywordMatcher( 'remove',
      helpdesc='Remove community attributes to send to this neighbor' )
sendExtCommunity = KeywordMatcher( 'extended',
      helpdesc='Send extended community attribute to this neighbor' )
sendLargeCommunity = KeywordMatcher( 'large',
      helpdesc='Send large community attribute to this neighbor' )
sendLbwAggregate = KeywordMatcher( 'aggregate',
      helpdesc='Send link-bandwidth cumulative attribute to this neighbor' )
sendLbwDivide = KeywordMatcher( 'divide',
      helpdesc='Send link-bandwidth divided attribute to this neighbor' )
sendLinkBandwidthCommunity = KeywordMatcher( 'link-bandwidth',
      helpdesc='Send link-bandwidth community attribute to this neighbor' )
sendStandardCommunity = KeywordMatcher( 'standard',
      helpdesc='Send standard community attribute to this neighbor' )
setForBestPathUse = KeywordMatcher( 'set',
      helpdesc='Set attribute for best path selection use' )
setNhAttrForBestPath = KeywordMatcher( 'next-hop',
      helpdesc="Use underlay next hop attribute for setting" )
setNhIgpCost = KeywordMatcher( "igp-cost",
      helpdesc="Set IGP cost in the best path selection to"
               " underlay next hop IGP cost" )
sixPe = KeywordMatcher( "6pe", helpdesc="Activate 6pe for the neighbor" )
skip = KeywordMatcher( "skip", helpdesc="FEC Skip operation" )
skipAigp = KeywordMatcher( 'aigp-metric',
      helpdesc='skip the AIGP metric rule in the bestpath selection' )
skipMatcher = KeywordMatcher( 'skip',
      helpdesc='skip one of the tie breaking rules in the bestpath selection' )
skipNextHop = KeywordMatcher( "next-hop",
      helpdesc="skip the next-hop igp-cost rule in the bestpath selection" )
skipNhIgpCost = KeywordMatcher( "igp-cost",
      helpdesc="skip the next-hop igp-cost rule in the bestpath selection" )
softReconfigAll = KeywordMatcher( 'all', helpdesc='configure inbound to keep all' )
softReconfigDeprecated = Node( KeywordMatcher( 'soft-reconfiguration',
      helpdesc='Configure how to handle routes that fail import' ),
      deprecatedByCmd='neighbor rib-in pre-policy retain' )
sourceInterface = KeywordMatcher( "source-interface",
      helpdesc="Source interface to update BGP next hop address" )
srTe = KeywordMatcher( "sr-te",
      helpdesc="Segment Routing Traffic Engineering Policy sub-address family" )
srv6Locator = KeywordMatcher( 'locator', 'SRv6 locator' )
dps = KeywordMatcher( "path-selection",
      helpdesc="Dynamic Path Selection sub-address family" )
sockFaultHidden = hiddenKeyword( 'sockfault' )
sockFaultGreedy = KeywordMatcher( 'greedyrecvestabfail',
      helpdesc='SockFault with a greedy receive established read failure' )
stalepathTime = KeywordMatcher( "stalepath-time",
      helpdesc="Set the max time to hold onto restarting peer's stale paths" )
stallHidden = hiddenKeyword( 'stall' )
stallFail = KeywordMatcher( 'fail',
      helpdesc='Stall with a write error' )
standardCommunity = KeywordMatcher( 'standard',
      helpdesc='Send standard community attribute' )
summaryOnly = KeywordMatcher( "summary-only",
      helpdesc="Filters all more-specific routes from updates" )
threshold = KeywordMatcher( "warning-limit",
      helpdesc="Percentage of maximum-routes at which warning is to be issued" )
tieBreak = KeywordMatcher( "tie-break",
      helpdesc="Configure the tie-break option for BGP bestpath selection" )
timers = KeywordMatcher( "timers", helpdesc="Timers" )
trafficPolicy = KeywordMatcher( "traffic-policy", helpdesc="Traffic Policy" )
transport = KeywordMatcher( "transport",
      helpdesc="Configure transport options for TCP session" )
transportForIpv4Unicast = KeywordMatcher( 'transport',
      helpdesc='Enable IPv4 unicast address family on neighbors of one AF' )
ttl = KeywordMatcher( 'ttl', helpdesc="BGP TTL security check" )
tunnel = KeywordMatcher( "tunnel", helpdesc="Install LU routes into Tunnel RIB" )
tunnelProtocol = KeywordMatcher( "tunnel", helpdesc="Tunnel protocols" )
tunnelSrcProtocol = KeywordMatcher( "source-protocol",
      helpdesc="Configure the tunnel source" )
typeBestpathMatcher = KeywordMatcher( 'type', helpdesc='type of peer' )
ucmpClear = KeywordMatcher( "clear",
      helpdesc="Configure UCMP fec utilization normal threshold" )
ucmpEncodingWeighted = KeywordMatcher( "encoding-weighted",
      helpdesc="Configure UCMP Link Bandwidth Weighted Mode" )
ucmpFec = KeywordMatcher( "fec",
      helpdesc="Configure UCMP fec utilization threshold" )
ucmpLinkbwDelay = KeywordMatcher( "update-delay",
      helpdesc="Link Bandwidth Advertisement delay" )
ucmpLinkbw = KeywordMatcher( "link-bandwidth",
      helpdesc="Configure UCMP Link Bandwidth" )
ucmpMode = KeywordMatcher( "mode", helpdesc="Configure UCMP mode" )
ucmpRecursive = KeywordMatcher( "recursive",
      helpdesc="UCMP Link Bandwidth Resolution From Nexthops" )
ucmpThreshold = KeywordMatcher( "threshold",
      helpdesc="Configure UCMP fec utilization threshold" )
ucmpTrigger = KeywordMatcher( "trigger",
      helpdesc="Configure UCMP fec utilization too high threshold" )
ucmpWarnOnly = KeywordMatcher( "warning-only",
      helpdesc="Configure UCMP fec utilization threshold action as warning-only" )
udpTunnel = KeywordMatcher( "udp-tunnel",
                            helpdesc="Resolve the next-hop using UDP tunnels" )
unicast = KeywordMatcher( "unicast",
      helpdesc="Unicast sub-address family" )
update = KeywordMatcher( "update", helpdesc="Configure BGP update generation" )
updateSrc = KeywordMatcher( 'update-source',
      helpdesc="Specify the local source interface for peer BGP sessions" )
v4MappedV6 = KeywordMatcher( "v4-mapped-v6",
      helpdesc="IPv4-mapped IPv6 address" )
v4MappedV6Translation = KeywordMatcher( "translation",
      helpdesc="Treat IPv4-mapped IPv6 address as IPv4 address" )
vpn = KeywordMatcher( "vpn",
      helpdesc="Any VPN address family" )
vpnClient = KeywordMatcher( 'client', helpdesc='VPN client configuration' )
vpnClientIbgp = KeywordMatcher( 'ibgp', helpdesc='iBGP VPN client configuration' )
vpnForAfContext = KeywordMatcher( 'vpn',
      helpdesc='VPN address family configuration' )
vpnIpv4 = KeywordMatcher( "vpn-ipv4",
      helpdesc="MPLS L3 VPN IPv4 unicast address family" )
vpnIpv6 = KeywordMatcher( "vpn-ipv6",
      helpdesc="MPLS L3 VPN IPv6 unicast address family" )
vpnMatcher = EnumMatcher( {
      'evpn': 'EVPN neighbors',
      'vpn-ipv4': 'MPLS L3 VPN IPv4 neighbors',
      'vpn-ipv6': 'MPLS L3 VPN IPv6 neighbors' } )
vrf = KeywordMatcher( "vrf", helpdesc="Configure BGP in a VRF" )
vrfLocalLabel = KeywordMatcher( 'vrf-label',
   helpdesc="Configure a local IP lookup MPLS label for a VRF" )
vxlan = KeywordMatcher( "vxlan", helpdesc="Vxlan transport" )
waitForConvergence = KeywordMatcher( "wait-for-convergence",
      helpdesc="Wait for BGP to converge before sending out any route updates" )
waitInstall = KeywordMatcher( "wait-install",
      helpdesc="Synchronize update generation with route programming in the FIB" )
warning = KeywordMatcher( "warning-only",
      helpdesc="Only warn, no restart, if max route limit exceeded" )
weight = KeywordMatcher( "weight",
      helpdesc="Assign weight for routes learnt from this peer" )
warningLimit = KeywordMatcher( "warning-limit",
      helpdesc="Keyword to preface threshold arg" )

# Following Vsp tokens are for VRF selection policy which can be matched under
# router bgp/vrf default or globally
vrfVsp = KeywordMatcher( 'vrf', helpdesc='Configure VRFs' )
selectionVsp = KeywordMatcher( 'selection', helpdesc='VRF selection policy' )
policyVsp = KeywordMatcher( 'policy', helpdesc='VRF selection policy' )

# Integer matchers: please keep these alphabetically sorted
allowAsRangeMatcher = IntegerMatcher(
      1,
      10,
      helpdesc='Number of local ASNs allowed in a BGP update'
)
appLimitRangeMatcher = IntegerMatcher(
      2,
      64,
      helpdesc='Number of additional-paths to advertise'
)
dscpRangeMatcher = IntegerMatcher(
      0,
      63,
      helpdesc='DSCP value'
)

halfLifeValueMatcher = IntegerMatcher(
      10,
      1000,
      helpdesc="Decay half-life value for penalty"
)
holdTimeRangeMatcher = IntegerMatcher(
      3,
      7200,
      helpdesc='Hold time in seconds'
)
sendFailureHoldTimeRangeMatcher = IntegerMatcher(
      60,
      U16_MAX_VALUE,
      helpdesc='Send failure hold time in seconds'
)
idleRestartTimerRangeMatcher = IntegerMatcher(
      60,
      U32_MAX_VALUE,
      helpdesc="Time (in seconds) before restarting, after going to idle state"
)
keepaliveTimeRangeMatcher = IntegerMatcher(
      0,
      3600,
      helpdesc='Time between BGP keepalive messages in seconds'
)
lfibDelayRangeMatcher = IntegerMatcher(
   1,
   U16_MAX_VALUE,
   helpdesc='Next hop labeled unicast origination LFIB uninstall delay in seconds'
)
limitMatcher = IntegerMatcher(
      1,
      I32_MAX_VALUE,
      helpdesc='Maximum number of dynamic BGP peers allowed'
)
lsIdentifierValueRangeMatcher = IntegerMatcher(
      0,
      U64_MAX_VALUE,
      helpdesc='BGP-LS identifier value'
)
localPrefMatcher = IntegerMatcher(
      1,
      U32_MAX_VALUE,
      helpdesc='Local Pref value for BGP Update'
)
metricRangeMatcher = IntegerMatcher(
      0,
      U32_MAX_VALUE,
      helpdesc='MED value to advertise to peer' )
mssV4RangeMatcher = IntegerMatcher(
      536,
      16344,
      helpdesc='Configure the maximum segment size to this value'
)
mssV6RangeMatcher = IntegerMatcher(
      516,
      16324,
      helpdesc='Configure the maximum segment size to this value'
)
numHopsRangeMatcher = IntegerMatcher(
      0,
      254,
      helpdesc="Maximum number of hops"
)
numMaxRoutesRangeMatcher = IntegerMatcher( 0, 4294967294,
      helpdesc='Maximum number of routes (0 means unlimited)' )
pathIdRangeMatcher = IntegerMatcher(
      0,
      U32_MAX_VALUE,
      helpdesc='additional-path id'
)
penaltyReuseThresholdMatcher = IntegerMatcher(
      500,
      1000000,
      helpdesc="Penalty reuse threshold value"
)
penaltySuppressionThresholdMatcher = IntegerMatcher(
      1000,
      1000000,
      helpdesc="Penalty suppression threshold value"
)
penaltyValueMatcher = IntegerMatcher(
      500,
      5000,
      helpdesc="Penalty value for flap"
)
portNumRangeMatcher = IntegerMatcher(
      1,
      MAX_PORT_VALUE,
      helpdesc='Configure the BGP listen port to this value'
)
remotePortNumRangeMatcher = IntegerMatcher(
      1,
      MAX_PORT_VALUE,
      helpdesc='Configure the remote port to this value'
)
ribInDelayTimeMatcher = IntegerMatcher(
      0,
      3600,
      helpdesc='Time in seconds by which inbound route updates are delayed'
)
ttlRangeMatcher = IntegerMatcher(
      1,
      255,
      helpdesc="Time-to-live in the range 1-255 hops"
)
ucmpDelayRangeMatcher = IntegerMatcher(
      0,
      600,
      helpdesc='Value for Link Bandwidth Advertisement delay'
)
ucmpModeRangeMatcher = IntegerMatcher( 1, 1, helpdesc='UCMP Mode' )

#
# The following matchers are defined in CliPlugin/RoutingBgpInstance.py in order to
# avoid a dependency cycle:
#  - maxPathsRangeMatcher
#  - maxPathsEcmpRangeMatcher
asPathLenMatcher = IntegerMatcher( 1, 65535, helpdesc='Number of ASNs to match' )
linkBwPercentRangeMatcher = FloatMatcher(
      1.00,
      100.00,
      helpdesc='Percentage',
      precisionString='%.2f'
)
numAcceptedRoutesRangeMatcher = IntegerMatcher( 0, 4294967294,
      helpdesc='Maximum number of accepted routes (0 means unlimited)' )
numSeconds3600RangeMatcher = IntegerMatcher( 1, 3600,
      helpdesc='Number of seconds' )
secondsRangeMatcher = IntegerMatcher( 1, 3600, helpdesc='Number of seconds' )
thresholdPercentageMatcher = IntegerMatcher( 1, 100,
      helpdesc="Percentage of maximum number of routes at which to warn ( 1-100 )" )
thresholdRangeMatcher = IntegerMatcher( 0, 4294967294,
      helpdesc='Maximum number of routes after which a '
               'warning is issued (0 means never warn)' )
ucmpFecPercentTrigMatcher = IntegerMatcher(
      0,
      100,
      helpdesc='Value for UCMP FEC utilization Trigger thresholds'
)
ucmpFecPercentClearMatcher = IntegerMatcher(
      0,
      100,
      helpdesc='Value for UCMP FEC utilization Clear thresholds'
)

# String matchers: please keep these alphabetically sorted
descriptionMatcher = StringMatcher( helpname='LINE',
      helpdesc='Text describing this neighbor' )
# The instance name pattern should be the same as that of the ISIS/OSPF name
# patterns
instanceNamePattern = '.{1,100}'
instanceNameRule = Node( matcher=PatternMatcher(
   instanceNamePattern, helpname='NAME',
   helpdesc='Name of the IS-IS protocol instance' ) )
messageMatcher = StringMatcher( helpname='MESSAGE',
      helpdesc='Shut down communication message' )

ipGenAddrMatcher = IpGenAddrMatcher( 'IP address',
      helpdesc4="IP address", helpdesc6="IPv6 address" )

# Expressions: please keep these alphabetically sorted
def mplsOrMulticastGuard( mode, token ):
   if token == 'multicast':
      return None
   if token == 'sr-te':
      if ( pathSelectionSupportedGuard( mode, token ) is None and
           getEffectiveProtocolModel( mode ) == ProtoAgentModel.multiAgent ):
         return None
   return mplsSupportedGuard( mode, token )

class AddrFamilyExpr( CliExpression ):
   expression = 'evpn | ( AF [ SUFFIX ] )'
   data = {
         'evpn': KeywordMatcher( 'evpn', helpdesc='EVPN address family' ),
         'AF': Node( matcher=EnumMatcher( {
                     'ipv4': 'IPv4 related',
                     'ipv6': 'IPv6 related',
               } )
         ),
         'SUFFIX': Node( matcher=EnumMatcher( {
                     'labeled-unicast': 'Labeled-unicast sub-address family',
                     'multicast': 'Multicast sub-address family',
                     'sr-te': ( 'Segment Routing Traffic Engineering Policy '
                                'sub-address family' )
               } ),
               guard=mplsOrMulticastGuard,
         ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      af = args.get( 'evpn', args.get( 'AF' ) )
      if 'SUFFIX' in args:
         af += ' %s' % args[ 'SUFFIX' ] # pylint: disable=consider-using-f-string

      args[ 'af' ] = af

class Ipv4UnicastAfExpr( CliExpression ):
   expression = "ipv4 unicast"
   data = {
      "ipv4": ipv4,
      "unicast": unicast,
   }

class Ipv6UnicastAfExpr( CliExpression ):
   expression = "ipv6 unicast"
   data = {
      "ipv6": ipv6,
      "unicast": unicast,
   }

class AsNumCliExpr( CliExpression ):
   expression = 'AS_PLAIN_NUM | AS_DOT_NUM'
   data = { 'AS_PLAIN_NUM': asplainNumMatcher,
            'AS_DOT_NUM': asdotNumMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'AS_DOT_NUM' in args:
         args[ 'AS_NUM' ] = asnStrToNum( args[ 'AS_DOT_NUM' ] )
      elif 'AS_PLAIN_NUM' in args:
         args[ 'AS_NUM' ] = args[ 'AS_PLAIN_NUM' ]

class AsNumOptCliExpr( CliExpression ):
   expression = f'none | {AsNumCliExpr.expression}'
   data = dict( AsNumCliExpr.data, none='AS Number is omitted' )

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'none' in args:
         args[ 'AS_NUM' ] = 'none'
      else:
         AsNumCliExpr.adapter( mode, args, argsList )

class AsNumMultiRangeExpr( CliExpression ):
   expression = 'AS_PLAIN_NUMS | AS_DOT_NUMS'
   data = {
         'AS_PLAIN_NUMS': MultiRangeRule.MultiRangeMatcher(
               lambda: ( 1, U32_MAX_VALUE ),
               False,
               'AS number or range(s) of AS numbers' ),
         'AS_DOT_NUMS': MultiRangeRule.MultiRangeMatcher(
               lambda: ( 65536, U32_MAX_VALUE ),
               False,
               'AS number or range(s) of AS numbers in asdot notation',
               numberFormat=MultiRangeRule.NumberFormat.DOTTED ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      asNums = set()
      asnList = args.get( 'AS_PLAIN_NUMS' ) or args.get( 'AS_DOT_NUMS' )
      for subList in asnList:
         asNums = asNums | set( subList.values() )
      args[ 'AS_NUMS' ] = asNums

class ConnectedStaticExpression( CliExpression ):
   expression = '( connected | static )'
   data = { 'connected': 'Connected interface routes',
            'static': 'Static routes'
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'connected' in args:
         args[ 'proto' ] = 'protoDirect'
      elif 'static' in args:
         args[ 'proto' ] = 'protoStatic'

class EvpnRouteTypeExpression( CliExpression ):
   expression = 'RTYPE_KW'
   data = {
      'RTYPE_KW': EnumMatcher( { t[ 'keyword' ]: t[ 'helpdesc' ]
                     for t in evpnNexthopSelfSupportNlris.values() } ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      keywordToTypeMap = { evpnNexthopSelfSupportNlris[ k ][ 'keyword' ]: k
            # pylint: disable-next=consider-using-dict-items
            for k in evpnNexthopSelfSupportNlris
      }
      types = set( [ keywordToTypeMap[ args[ 'RTYPE_KW' ] ] ] )
      args[ 'ROUTE_TYPES' ] = types

class GrHelperRestartTimeExp( CliExpression ):
   expression = 'restart-time RESTART_TIME [ UNITS ]'
   data = {
      'restart-time': "Override restart time received from peer",
      'RESTART_TIME': IntegerMatcher( 1, 100000000,
                                      helpdesc='Value ( default: seconds )' ),
      'UNITS': EnumMatcher( {
         'minutes': "Value in minutes",
         'hours': "Value in hours",
         'days': "Value in days", } )
   }

class GrHelperRmGrOptionalExp( CliExpression ):
   expression = 'graceful-restart-negotiation optional'
   data = {
         'graceful-restart-negotiation': 'Retain routes with optional graceful '
                                         'restart negotiation',
         'optional': 'Retain routes with graceful restart negotiation optional'
   }

class IarEventTypeExpression( CliExpression ):
   expression = '( route-changes | drain-undrain | all )'
   data = {
      'route-changes': routeChanges,
      'drain-undrain': drainUndrainHidden,
      'all': iarEventAll,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'route-changes' in args:
         args[ 'IAR_EVENT' ] = 'route-changes'
      elif 'drain-undrain' in args:
         args[ 'IAR_EVENT' ] = 'drain-undrain'
      elif 'all' in args:
         args[ 'IAR_EVENT' ] = 'all'

class IsisLevelExpr( CliExpression ):
   expression = '( level-1 | level-2 | level-1-2 )'
   data = { 'level-1': isLevel1,
            'level-2': isLevel2,
            'level-1-2': isLevel12,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'level-1' in args:
         args[ 'isisLevel' ] = 'level-1'
      elif 'level-2' in args:
         args[ 'isisLevel' ] = 'level-2'
      elif 'level-1-2' in args:
         args[ 'isisLevel' ] = 'level-1-2'

class McastRedistExpr( CliExpression ):
   expression = '( connected | static | attached-host )'
   data = {
         'connected': redistMcastConnected,
         'static': redistMcastStatic,
         'attached-host': redistAttachedHost,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'connected' in args:
         proto = 'protoDirect'
      elif 'static' in args:
         proto = 'protoStatic'
      else:
         proto = 'protoAttachedHost'
      args[ 'proto' ] = proto

class McastRedist6Expr( CliExpression ):
   expression = '( connected | static )'
   data = {
         'connected': redistMcastConnected,
         'static': redistMcastStatic,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'connected' in args:
         proto = 'protoDirect'
      else:
         proto = 'protoStatic'
      args[ 'proto' ] = proto

class OspfExpr( CliExpression ):
   expression = 'ospf'
   data = { 'ospf': ospf }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ospf' in args:
         args[ 'proto' ] = 'protoOspf'

class OspfMatchIntExtExpr( CliExpression ):
   expression = 'match ( internal | external )'
   data = {
      'match': match,
      'internal': internal,
      'external': external,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'internal' in args:
         args[ 'intext' ] = 'internal'
      elif 'external' in args:
         args[ 'intext' ] = 'external'

class OspfMatchNssaExtExpr( CliExpression ):
   expression = 'match nssa-external [ NSSA_TYPE ]'
   data = {
         'match': match,
         'nssa-external': nssaExternal,
         'NSSA_TYPE': IntegerMatcher(
               1,
               2,
               helpdesc='NSSA External Type Number'
         ),
   }

class Ospf3IncludingHiddenExpr( CliExpression ):
   expression = '( ospfv3 | ospf3 )'
   data = { 'ospfv3': ospf3,
            'ospf3': Node( ospf3Old, hidden=True )
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ospfv3' in args or 'ospf3' in args:
         args[ 'proto' ] = 'protoOspf3'

class Ospf3OrOspfExpr( CliExpression ):
   expression = '( OSPF | OSPF3 )'
   data = {
         'OSPF': OspfExpr,
         'OSPF3': Ospf3IncludingHiddenExpr,
   }

class RipExpr( CliExpression ):
   expression = 'rip'
   data = { 'rip': rip }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'rip' in args:
         args[ 'proto' ] = 'protoRip'

class DynamicPolicyExpr( CliExpression ):
   expression = 'dynamic'
   data = { 'dynamic': dynamicPolicy }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'dynamic' in args:
         args[ 'proto' ] = 'protoDynamic'

class AttachedHostExpr( CliExpression ):
   expression = 'attached-host'
   data = { 'attached-host': attachedHost }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'attached-host' in args:
         args[ 'proto' ] = 'protoAttachedHost'

class ProtocolExpression( CliExpression ):
   expression = '( rip | aggregate | attached-host )'
   data = { 'rip': RipExpr,
            'aggregate': Node( KeywordMatcher( 'aggregate',
                                             helpdesc='Aggregate routes' ),
                                          hidden=True ),
            'attached-host': AttachedHostExpr
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'aggregate' in args:
         args[ 'proto' ] = 'protoBgpAggregate'

class TunnelProtocolExpression( CliExpression ):
   expression = '( ldp | ( isis segment-routing ) )'
   data = { 'ldp': 'LDP tunnel',
            'isis': 'IS-IS tunnel',
            'segment-routing': 'IS-IS segment routing tunnel',
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ldp' in args:
         args[ 'proto' ] = 'protoLdp'
      elif 'isis' in args and 'segment-routing' in args:
         args[ 'proto' ] = 'protoIsisSr'

class PeerSpecExpression( CliExpression ):
   expression = '( ( remote-as AS_NUM ) | ( peer-filter PEER_FILTER ) )'
   data = {
         'remote-as': remoteAs,
         'AS_NUM': AsNumCliExpr,
         'peer-filter': peerFilter,
         'PEER_FILTER': peerFilterNameMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'PEER_SPEC' ] = args.get( 'AS_NUM', args.get( 'PEER_FILTER' ) )

class RouteMapOrRcfFunction( CliExpression ):
   expression = '( route-map MAP_NAME ) | ( rcf FUNCTION )'
   data = {
         'route-map': routeMap,
         'MAP_NAME': mapNameMatcher,
         'rcf': rcf,
         'FUNCTION': rcfFunctionMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'POLICY_TYPE' ] = (
            args.get( 'MAP_NAME' ) or
            args.get( 'FUNCTION' )
      )

class RouteTargetRcfMplsVpnTypeExpr( CliExpression ):
   expression = '( vpn-ipv4 | vpn-ipv6 )'
   data = {
         'vpn-ipv4': vpnIpv4,
         'vpn-ipv6': vpnIpv6,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'VPN_TYPE' ] = (
            args.get( 'vpn-ipv4' ) or
            args.get( 'vpn-ipv6' )
      )

class RouteTargetVpnTypeExpr( CliExpression ):
   expression = '( evpn | vpn-ipv4 | vpn-ipv6 )'
   data = {
         'evpn': evpn,
         'vpn-ipv4': vpnIpv4,
         'vpn-ipv6': vpnIpv6,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'VPN_TYPE' ] = (
            args.get( 'evpn' ) or
            args.get( 'vpn-ipv4' ) or
            args.get( 'vpn-ipv6' )
      )

class RouteTargetFlowspecVpnTypeExpr( CliExpression ):
   expression = '( flow-spec-vpn-ipv4 | flow-spec-vpn-ipv6 )'
   data = {
         'flow-spec-vpn-ipv4': flowspecVpnIpv4,
         'flow-spec-vpn-ipv6': flowspecVpnIpv6,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'VPN_TYPE' ] = (
            args.get( 'flow-spec-vpn-ipv4' ) or
            args.get( 'flow-spec-vpn-ipv6' )
      )

class VpnNlriTypeExpr( CliExpression ):
   expression = '( ( evpn ( ipv4 | ipv6 ) ) | vpn-ipv4 | vpn-ipv6 )'
   data = { 'evpn': 'EVPN address family',
            'ipv4': 'EVPN type-5 IPv4',
            'ipv6': 'EVPN type-5 IPv6',
            'vpn-ipv4': 'MPLS L3 VPN IPv4',
            'vpn-ipv6': 'MPLS L3 VPN IPv6' }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ipv4' in args:
         vpnNlriType = 'evpnType5Ipv4'
      elif 'ipv6' in args:
         vpnNlriType = 'evpnType5Ipv6'
      elif 'vpn-ipv4' in args:
         vpnNlriType = 'mplsVpnIpv4Unicast'
      elif 'vpn-ipv6' in args:
         vpnNlriType = 'mplsVpnIpv6Unicast'
      args[ 'VPN_NLRI_TYPE' ] = Tac.Value( 'Routing::Bgp::NlriType', vpnNlriType )

class BgpAttributeMultiRangeExpr( CliExpression ):
   expression = 'BGP_ATTRIBUTES'
   data = {
         'BGP_ATTRIBUTES': MultiRangeRule.MultiRangeMatcher(
               lambda: ( 4, 255 ),
               False,
               'BGP attribute type code(s) and/or range(s) of BGP attribute type '
               'codes' ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      bgpAttr = set()
      # pylint: disable-next=use-list-literal
      bgpAttrList = args.get( 'BGP_ATTRIBUTES', list() )
      for subList in bgpAttrList:
         bgpAttr = bgpAttr | set( subList.values() )
      args[ 'BGP_ATTRIBUTES_SET' ] = bgpAttr

vpnNlriTypeKeywordDir = { 'evpnType5Ipv4': 'evpn ipv4',
                          'evpnType5Ipv6': 'evpn ipv6',
                          'mplsVpnIpv4Unicast': 'vpn-ipv4',
                          'mplsVpnIpv6Unicast': 'vpn-ipv6' }

def vpnNlriTypeKeyword( vpnNlriType ):
   return vpnNlriTypeKeywordDir[ vpnNlriType.toStrep() ]

class UserExpression( CliExpression ):
   expression = 'user'
   data = { 'user': 'EosSdk routes'
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'user' in args:
         args[ 'proto' ] = 'protoEosSdk'

class AutoRdAdressFamilyExpression( CliExpression ):
   expression = 'address-family l2-evpn'
   data = {
         'address-family': addrFamily,
         'l2-evpn': KeywordMatcher( 'l2-evpn',
            helpdesc="Enable auto rd assignment for l2-evpn" )
   }
