#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in
# pylint: disable=consider-using-f-string

from MplsPingHandlers import (
   handleLspPingBgpLu,
   handleLspPingRaw,
   handleLspPingLdpMldpSr,
   handleLspPingRsvp,
   handleLspPingNhg,
   handleLspPingStatic,
   handleLspPingSrTe,
   handleLspPingPwLdp,
   handleLspPingNhgTunnel,
   handleLspPingVpn,
)

from MplsTracerouteHandlers import (
   handleLspTracerouteBgpLu,
   handleLspTracerouteGeneric,
   handleLspTracerouteRaw,
   handleLspTracerouteLdp,
   handleLspTracerouteMldp,
   handleLspTracerouteSr,
   handleLspTracerouteRsvp,
   handleLspTracerouteNhg,
   handleLspTracerouteStatic,
   handleLspTracerouteSrTe,
   handleLspTracerouteNhgTunnel
)

from ClientCommonLib import (
   getClientSocket,
   LspPing,
   LspTraceroute, 
   LspPingTypeBgpLu,
   LspPingTypeGeneric,
   LspPingTypeRaw,
   LspPingTypeLdp,
   LspPingTypeMldp, 
   LspPingTypeRsvp,
   LspPingTypeSr,
   LspPingTypeStatic,
   LspPingTypeNhg,
   LspPingTypeSrTe,
   LspPingTypePwLdp,
   LspPingTypeNhgTunnel,
   LspPingTypeVpn,
)

from ClientState import ( 
   sessionIdIncr, 
   getGlobalState, 
   setGlobalClientIdBaseOverride 
)

import datetime
import Cell
import errno
import os
import Tac
import Tracing
import BothTrace
import SharedMem
import Shark
import Smash
import SmashLazyMount
import socket
from ForwardingHelper import ( forwardingHelperKwFactory )
from TypeFuture import TacLazyType
import Intf.AllIntfLib
from IpLibConsts import DEFAULT_VRF
from Arnet.NsLib import DEFAULT_NS
from SrTePolicyCommonLib import srTePolicyStatusPath

#--------------------------------
# BothTrace Short hand variables
#--------------------------------
__defaultTraceHandle__ = Tracing.Handle( "ClientDispatcher" )
bv = BothTrace.Var
bt8 = BothTrace.trace8

TunnelTableMounter = TacLazyType( "Tunnel::TunnelTable::TunnelTableMounter" )
TunnelTableIdentifier = TacLazyType( "Tunnel::TunnelTable::TunnelTableIdentifier" )

class LspUtilMount:
   def __init__( self, lspPingType, entityManager, vrf=None ):
      vrf = vrf or DEFAULT_VRF
      self.vrf = vrf
      self.entityManager = entityManager
      mg = self.entityManager.mountGroup()
      # Local Entities must be created inside the enclosing mount group
      localEntityTypes = [
         ( 'interface/status/all', 'Interface::AllIntfStatusDir' ),
         ( Cell.path( 'interface/status/local' ),
           'Interface::AllIntfStatusLocalDir' )
      ]
      Intf.AllIntfLib.createLocalEntities( entityManager, localEntityTypes )

      self.ipStatus = mg.mount( 'ip/status', 'Ip::Status', 'r' )
      self.ip6Status = mg.mount( 'ip6/status', 'Ip6::Status', 'r' )
      self.bridgingConfig = mg.mount( "bridging/config", "Bridging::Config", "r" )
      self.traceConfig = mg.mount( "agent/config",
                                   "Agent::GlobalConfigDir", "ri" )
      self.allIntfStatusDir = mg.mount( 'interface/status/all',
                                        'Interface::AllIntfStatusDir', 'r' )
      self.allIntfStatusLocalDir = mg.mount(
         ( Cell.path( 'interface/status/local' ) ),
         'Interface::AllIntfStatusLocalDir', 'r' )

      self.config = mg.mount( 'mplsutils/config',
                              'MplsUtils::Config', 'r' )
      self.vrfNameStatus = mg.mount( Cell.path( 'vrf/vrfNameStatus' ),
                                     'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )
      self.intfConfigDir = mg.mount( 'l3/intf/config', 'L3::Intf::ConfigDir', 'r' )

      startNexthopGroupSm = False
      # Smash, Shark mount
      shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )
      if lspPingType == LspPingTypeNhg or lspPingType == LspPingTypeStatic or \
         lspPingType == LspPingTypeNhgTunnel:
         self.routingHwNexthopGroupStatus = mg.mount( 
            'routing/hardware/nexthopgroup/status',
            'Routing::Hardware::NexthopGroupStatus', 'r' )
         self._routingNhgConfig = Tac.newInstance( 'Routing::NexthopGroup::Config' )
         self.nexthopGroupCliConfig = mg.mount( 'routing/nexthopgroup/input/cli',
                                                'Routing::NexthopGroup::ConfigInput',
                                                'r' )
         self.nexthopGroupConfigDir = mg.mount( 'routing/nexthopgroup/input/config',
                                                'Tac::Dir', 'ri' )
         startNexthopGroupSm = True
         nexthopEntryTableInfo = Tac.Value( 'NexthopGroup::TableInfo' )
         nexthopEntryStatusMountInfo = nexthopEntryTableInfo.entryStatus( 'shadow' )
         self.smashNhgStatus = shmemEm.doMount(
               "routing/nexthopgroup/entrystatus",
               "NexthopGroup::EntryStatus", 
               nexthopEntryStatusMountInfo )
      self.tunnelFib = SmashLazyMount.mount( self.entityManager,
                                             'tunnel/tunnelFib',
                                             'Tunnel::TunnelFib::TunnelFib',
                                             Smash.mountInfo( 'reader' ) )
      self.arpSmash = shmemEm.doMount( 'arp/status', 'Arp::Table::Status',
                                       Smash.mountInfo( 'reader' ) )
      self.arpSmashVrfIdMap = shmemEm.doMount( 'vrf/vrfIdMapStatus',
                                               'Vrf::VrfIdMap::Status',
                                               Smash.mountInfo( 'reader' ) )
      self.arpSmashVrfNameToIdMap = Tac.newInstance( 'Vrf::VrfIdMap::NameToIdMap' )
      self.arpSmashVrfIdReverseMapper = Tac.newInstance(
                                                'Vrf::VrfIdMap::ReverseMapper',
                                                self.arpSmashVrfIdMap,
                                                self.arpSmashVrfNameToIdMap )
      self.routingVrfInfoDir = mg.mount( 'routing/vrf/routingInfo/status',
                                         'Tac::Dir', 'ri' )
      self.routing6VrfInfoDir = mg.mount( 'routing6/vrf/routingInfo/status',
                                          'Tac::Dir', 'ri' )
   
      fibInfo = Tac.Value( 'Smash::Fib::TableInfo' )
      routeMountInfo = fibInfo.routeInfo( 'reader' )
      forwardingMountInfo = fibInfo.forwardingInfo( 'reader' )

      if vrf == DEFAULT_VRF:
         routeStatusPath = 'routing/status'
         route6StatusPath = 'routing6/status'
      else:
         routeStatusPath = 'routing/vrf/status/%s' % vrf
         route6StatusPath = 'routing6/vrf/status/%s' % vrf

      self.l3Config_ = mg.mount( "l3/config", "L3::Config", "r" )
      self.routeStatusDefaultVrf = (
         shmemEm.doMount( 'routing/status',
                          'Smash::Fib::RouteStatus',
                          routeMountInfo ) )
      self.routeStatus = shmemEm.doMount( routeStatusPath,
                                          'Smash::Fib::RouteStatus',
                                          routeMountInfo )
      if lspPingType in [ LspPingTypeSrTe, LspPingTypePwLdp ]:
         self.srTeForwardingStatus = \
                                 shmemEm.doMount( 'forwarding/srte/status',
                                                  'Smash::Fib::ForwardingStatus',
                                                  forwardingMountInfo )

      self.mplsTunnelConfig = mg.mount( 'routing/mpls/tunnel/config',
                                        'Tunnel::MplsTunnelConfig', 'r' )

      # BGP LU specific mounts. Mount BGP LU TunnelRib to  get fec to tunnelId
      # mapping
      if lspPingType == LspPingTypeBgpLu:
         self.bgpLuTunnelRib = shmemEm.doMount(
            'tunnel/protoTunnelRib/bgpLu', 'Tunnel::TunnelTable::ColoredTunnelRib',
            Smash.mountInfo( 'keyshadow' ) )

      # Nexthop group tunnel specific mounts. Mount nexthop group TunnelRib to
      # get fec to tunnelId mapping
      if lspPingType == LspPingTypeNhgTunnel:
         self.nhgTunnelRib = shmemEm.doMount(
            'tunnel/protoTunnelRib/nexthopGroup',
            'Tunnel::TunnelTable::ColoredTunnelRib',
            Smash.mountInfo( 'keyshadow' ) )

      # Ldp specific mounts. Mount ldp TunnelRib to get fec to
      # tunnelId mapping and tunnel table to get tunnelId to via mapping
      if lspPingType == LspPingTypeLdp or lspPingType == LspPingTypeGeneric:
         self.ldpTunnelRib = shmemEm.doMount(
            'tunnel/protoTunnelRib/ldp', 'Tunnel::TunnelTable::ColoredTunnelRib',
            Smash.mountInfo( 'reader' ) )
         tableInfo = TunnelTableMounter.getMountInfo(
                     TunnelTableIdentifier.ldpTunnelTable ).tableInfo
         self.ldpTunnelTable = SmashLazyMount.mount(
                self.entityManager, tableInfo.mountPath, tableInfo.tableType,
                Smash.mountInfo( 'reader' ) )
         RsvpLerSysdbStatus = Tac.Type( "Rsvp::RsvpLerSysdbStatus" )
         self.rsvpLerStatus = mg.mount( RsvpLerSysdbStatus.mountPath,
                                        'Rsvp::RsvpLerSysdbStatus', 'rS' )

      # Mldp specific mounts. Mounting protoLfib mldp to get nexthop and
      # label info.
      if lspPingType == LspPingTypeMldp:
         self.mldpLfib = shmemEm.doMount(
            "mpls/protoLfibInputDir/mldp", "Mpls::LfibStatus",
            Smash.mountInfo( 'reader' ) )

         self.mldpOpaqueValueTable = mg.mount(
            "mpls/ldp/mldpOpaqueValueTable", "Mpls::MldpOpaqueValueTable", "r" )

      # Sr specific mounts.
      # sr TunnelRib provides: FEC --> tunnelId
      # SrTunnelTable provides: tunnelId --> via
      # Acheive FEC --> via
      if lspPingType == LspPingTypeSr or lspPingType == LspPingTypeGeneric:
         self.srTunnelRib = shmemEm.doMount(
            'tunnel/protoTunnelRib/sr', 'Tunnel::TunnelTable::ColoredTunnelRib',
            Smash.mountInfo( 'reader' ) )
         self.ospfSrTunnelRib = shmemEm.doMount(
            'tunnel/protoTunnelRib/ospfSr', 'Tunnel::TunnelTable::ColoredTunnelRib',
            Smash.mountInfo( 'reader' ) )
         self.isisFlexAlgoTunnelRib = shmemEm.doMount(
            'tunnel/protoTunnelRib/isisFlexAlgoTunnel',
            'Tunnel::TunnelTable::ColoredTunnelRib',
            Smash.mountInfo( 'reader' ) )

         tableInfo = TunnelTableMounter.getMountInfo(
             TunnelTableIdentifier.srTunnelTable ).tableInfo
         self.srTunnelTable = SmashLazyMount.mount(
            self.entityManager, tableInfo.mountPath, tableInfo.tableType,
            Smash.mountInfo( 'reader' ) )
         tableInfo = TunnelTableMounter.getMountInfo(
             TunnelTableIdentifier.ospfSrTunnelTable ).tableInfo
         self.ospfSrTunnelTable = SmashLazyMount.mount(
            self.entityManager, tableInfo.mountPath, tableInfo.tableType,
            Smash.mountInfo( 'reader' ) )
         tableInfo = TunnelTableMounter.getMountInfo(
            TunnelTableIdentifier.tiLfaTunnelTable ).tableInfo
         self.tiLfaTunnelTable = SmashLazyMount.mount(
            self.entityManager, tableInfo.mountPath, tableInfo.tableType,
            Smash.mountInfo( "reader" ) )
         tableInfo = TunnelTableMounter.getMountInfo(
            TunnelTableIdentifier.isisFlexAlgoTunnelTable ).tableInfo
         self.isisFlexAlgoTunnelTable = SmashLazyMount.mount(
            self.entityManager, tableInfo.mountPath, tableInfo.tableType,
            Smash.mountInfo( "reader" ) )
         RsvpLerSysdbStatus = Tac.Type( "Rsvp::RsvpLerSysdbStatus" )
         self.rsvpLerStatus = mg.mount( RsvpLerSysdbStatus.mountPath,
                                        'Rsvp::RsvpLerSysdbStatus', 'rS' )
         self.flexAlgoConfig = mg.mount( 'te/flexalgo/config',
                                         'FlexAlgo::Config', 'r' )

      if lspPingType == LspPingTypeRsvp:
         RsvpSysdbStatus = Tac.Type( "Rsvp::RsvpSysdbStatus" )
         self.rsvpStatus = mg.mount( RsvpSysdbStatus.mountPath,
                                     'Rsvp::RsvpSysdbStatus', 'rS' )
         shmemMg = shmemEm.getMountGroup()
         RsvpSharkStatus = Tac.Type( "Rsvp::RsvpSharkStatus" )
         self.rsvpSharkStatus = shmemMg.doMount( RsvpSharkStatus.mountPath,
                                                 'Rsvp::RsvpSharkStatus',
                                                 Shark.mountInfo( 'shadow' ) )
         shmemMg.doClose()
         RsvpLerSysdbStatus = Tac.Type( "Rsvp::RsvpLerSysdbStatus" )
         self.rsvpLerStatus = mg.mount( RsvpLerSysdbStatus.mountPath,
                                        'Rsvp::RsvpLerSysdbStatus', 'rS' )

      # SR-TE policy specific mounts.
      if lspPingType in [ LspPingTypePwLdp, LspPingTypeSrTe ]:
         tableInfo = TunnelTableMounter.getMountInfo(
             TunnelTableIdentifier.srTeSegmentListTunnelTable ).tableInfo
         self.srTeSegmentListTunnelTable = SmashLazyMount.mount(
            self.entityManager, tableInfo.mountPath, tableInfo.tableType,
            Smash.mountInfo( 'reader' ) )
         self.policyStatus = SmashLazyMount.mount(
               self.entityManager, srTePolicyStatusPath(),
               "SrTePolicy::PolicyStatus", Smash.mountInfo( "reader" ) )
      # PW specific mounts
      if lspPingType == LspPingTypePwLdp:
         self.pwConfig = mg.mount( 'pseudowire/config', 'Pseudowire::Config', 'r' )
         self.pwAttr = mg.mount( 'pseudowire/agent/pseudowireAttributeStatusColl',
                                'Pseudowire::PseudowireAttributeStatusColl',
                                'r' )
         self.ldpProtoConfig = mg.mount( 'mpls/ldp/ldpProtoConfigColl/',
                                         'Ldp::LdpProtoConfigColl', 'r' )

      self.kniStatus = shmemEm.doMount( 'kni/ns/%s/status' % DEFAULT_NS,
                                        'KernelNetInfo::Status',
                                        Smash.mountInfo( 'reader' ) )

      mg.close( blocking=True )

      self.fecModeStatus_ = Tac.newInstance( "Smash::Fib::FecModeStatus",
                                             "fecModeStatus" )
      self.fecModeSm_ = Tac.newInstance( "Ira::FecModeSm", self.l3Config_,
                                         self.fecModeStatus_ )
      self.fecModeRestartSm_ = Tac.newInstance( "Ira::FecModeRestartSm",
                                                self.fecModeStatus_ )
      if self.fecModeStatus_.fecMode == 'fecModeUnified':
         forwardingStatusPath = "forwarding/unifiedStatus"
         forwarding6StatusPath = "forwarding6/unifiedStatus"
         forwardingGenStatusPath = "forwardingGen/unifiedStatus"
         forwardingGenStatusDefaultVrfPath = "forwardingGen/unifiedStatus"
      else:
         forwardingGenStatusDefaultVrfPath = "forwardingGen/status"
         if vrf == DEFAULT_VRF:
            forwardingStatusPath = 'forwarding/status'
            forwarding6StatusPath = 'forwarding6/status'
            forwardingGenStatusPath = "forwardingGen/status"
         else:
            forwardingStatusPath = 'forwarding/vrf/status/%s' % vrf
            forwardingGenStatusPath = "forwardingGen/vrf/status/%s" % vrf
            forwarding6StatusPath = 'forwarding6/vrf/status/%s' % vrf

      self.forwardingStatusDefaultVrf = (
         shmemEm.doMount( 'forwarding/status',
                          'Smash::Fib::ForwardingStatus',
                          forwardingMountInfo ) )
      self.forwardingStatus = shmemEm.doMount( forwardingStatusPath,
                                               'Smash::Fib::ForwardingStatus',
                                               forwardingMountInfo )
      self.forwardingGenStatus = shmemEm.doMount( forwardingGenStatusPath,
                                                  'Smash::FibGen::ForwardingStatus',
                                                  forwardingMountInfo )
      self.forwardingGenStatusDefaultVrf = (
            shmemEm.doMount( forwardingGenStatusDefaultVrfPath,
                             'Smash::FibGen::ForwardingStatus',
                             forwardingMountInfo ) )

      if route6StatusPath and forwarding6StatusPath:
         self.route6StatusDefaultVrf = (
            shmemEm.doMount( 'routing6/status',
                             'Smash::Fib6::RouteStatus',
                             routeMountInfo ) )
         self.route6Status = shmemEm.doMount( route6StatusPath,
                                              'Smash::Fib6::RouteStatus',
                                              routeMountInfo )
         self.forwarding6StatusDefaultVrf = (
            shmemEm.doMount( 'forwarding6/status',
                             'Smash::Fib6::ForwardingStatus',
                             forwardingMountInfo ) )
         self.forwarding6Status = shmemEm.doMount( forwarding6StatusPath,
                                                   'Smash::Fib6::ForwardingStatus',
                                                   forwardingMountInfo )
      else:
         self.route6Status = self.forwarding6Status = None

      self.vrfIpIntfStatusDefaultVrf = (
         self.ipStatus.vrfIpIntfStatus.get( DEFAULT_VRF ) )
      self.vrfIpIntfStatus = self.ipStatus.vrfIpIntfStatus.get( vrf )
      self.vrfIp6IntfStatusDefaultVrf = (
         self.ip6Status.vrfIp6IntfStatus.get( DEFAULT_VRF ) )
      self.vrfIp6IntfStatus = self.ip6Status.vrfIp6IntfStatus.get( vrf )

      if startNexthopGroupSm:
         Typename = 'NexthopGroup::CliDynamicConfigMergeSmWrapper'
         self.nexthopGroupConfigMergeSm = \
               Tac.newInstance( Typename,
                                self._routingNhgConfig,
                                self.nexthopGroupCliConfig,
                                self.nexthopGroupConfigDir )

      self.trie = Tac.newInstance( "Routing::Trie", "trie" )
      self.trieBuilder = Tac.newInstance( "Routing::TrieBuilder", self.routeStatus,
                                          self.trie )
      self.v6trie = Tac.newInstance( "Routing6::Trie", "v6trie" )
      self.v6trieBuilder = Tac.newInstance( "Routing6::TrieBuilder",
                                            self.route6Status, self.v6trie )

      # Instantiate the Etba Forwarding helper class.
      vrfRoutingStatus = { vrf : self.routeStatus }
      vrfRouting6Status = { vrf : self.route6Status }
      vrfTrie = { vrf : self.trie }
      vrfTrie6 = { vrf : self.v6trie }
      resolveHelperKwargs = {
         'vrfRoutingStatus' : vrfRoutingStatus,
         'vrfRouting6Status' : vrfRouting6Status,
         'forwardingStatus' : self.forwardingStatus,
         'forwarding6Status' :self.forwarding6Status,
         'forwardingGenStatus' : self.forwardingGenStatus,
         'arpSmash' : self.arpSmash,
         'trie4' : vrfTrie,
         'trie6' : vrfTrie6,
         'vrfNameStatus' : self.vrfNameStatus,
         'intfConfigDir' : self.intfConfigDir,
         'tunnelFib' : self.tunnelFib,
      }
      if lspPingType in [ LspPingTypeSrTe, LspPingTypePwLdp ]:
         resolveHelperKwargs[ 'srTeForwardingStatus' ] = self.srTeForwardingStatus
         resolveHelperKwargs[ 'srTeSegmentListTunnelTable' ] = \
                                                   self.srTeSegmentListTunnelTable
      self.fwdingHelper = forwardingHelperKwFactory( **resolveHelperKwargs )

   @property
   def routingNhgConfig( self ):
      if self._routingNhgConfig is None:
         return None
      # The NexthopGroupConfigMergeSm uses async mounts, so wait for the
      # SM's configReady flag to be set before returning a reference to the merged
      # nexthop group config.
      Tac.waitFor( lambda: self._routingNhgConfig.configReady,
            description='NexthopGroupConfigMergeSm configReady flag to be True' )
      return self._routingNhgConfig

def createTraceFileLog( clientTraceConfig, tracePath, traceFileName ):
   if clientTraceConfig:
      os.environ[ 'TRACEFILE' ] = os.environ.get( 'TRACEFILE',
                                                  tracePath + '/' + traceFileName )
      Tracing.traceSettingIs( clientTraceConfig.trace if clientTraceConfig else "" )

def createOrReplaceTraceFile( util, agentName, qtPath, tracePath, clientTraceConfig,
                              qtTraceFileName, traceFileName, maxNumLogs=10 ):
   # cleanup oldest qt file and trace file if max files is reached
   listOfQTFiles = [ name for name in os.listdir( qtPath )
                     if name.startswith( agentName + "-" + util ) ]
   listOfTraceFiles = [ name for name in os.listdir( tracePath )
                        if name.startswith( agentName + "-" + util ) ]
   qtraceFileExceeded = False
   traceFileExceeded = False
   if len( listOfQTFiles ) >= maxNumLogs:
      qtraceFileExceeded = True
   if len( listOfTraceFiles ) >= maxNumLogs:
      traceFileExceeded = True

   if qtraceFileExceeded:
      listOfFilesAbsPath = [ qtPath + '/' + name for name in listOfQTFiles ]
      timeSortedQtFiles = sorted( listOfFilesAbsPath, key=os.path.getctime )
      # Handle qt files
      for qtFile in timeSortedQtFiles:
         try:
            os.remove( os.path.abspath( qtFile ) )
         except OSError:
            pass
         else:
            BothTrace.initialize( qtTraceFileName )
            break
      else:
         # OS deletion errored for all existing trace files, hence do not keep
         # adding new files, instead overwrite traces to one file in /var/log/qt.
         # This way qtraces doesn't fail because of OS error and
         # we don't keep adding new files.
         BothTrace.initialize( agentName + "-" + "LspPingTraceroute.qt" )

   if traceFileExceeded:
      listOfFilesAbsPath = [ tracePath + '/' + name for name in listOfTraceFiles ]
      timeSortedFiles = sorted( listOfFilesAbsPath, key=os.path.getctime )

      # Handle trace files
      for traceFile in timeSortedFiles:
         try:
            os.remove( os.path.abspath( traceFile ) )
         except OSError:
            pass
         else:
            createTraceFileLog( clientTraceConfig, tracePath, traceFileName )
            break
      else:
         # OS deletion errored for all existing trace files, hence do not keep
         # adding new files, instead overwrite traces to one file in /var/log/agents.
         # This way traces doesn't fail because of OS error and
         # we don't keep adding new files.
         createTraceFileLog( clientTraceConfig, tracePath,
                             agentName + "-" + "LspPingTraceroute" )
   if not qtraceFileExceeded:
      BothTrace.initialize( qtTraceFileName )
   if not traceFileExceeded:
      createTraceFileLog( clientTraceConfig, tracePath, traceFileName )

def allocateTraceFile( util, dstType, traceConfig ):
   timeStamp = str( datetime.datetime.now().time() ).replace( ".", ":" )
   agentName = "MplsUtilLspClient"
   traceFileName = "-".join( [ agentName, util, dstType, timeStamp ] )
   tracePath = "/var/log/agents"

   # tracefile name same as qt except .qt at end
   clientTraceConfig = traceConfig.agentGlobalConfig.get( agentName )
   os.environ[ 'QUICKTRACEDIR' ] = "/var/log/qt"
   qtTraceFileName = traceFileName + ".qt"
   qtPath = os.environ[ 'QUICKTRACEDIR' ]
   if not os.path.exists( qtPath ):
      os.makedirs( qtPath, 0o755 )
   if not os.path.exists( tracePath ):
      tracePath = "/var/log"
   createOrReplaceTraceFile( util, agentName, qtPath, tracePath, clientTraceConfig,
                             qtTraceFileName, traceFileName )

def useCapi( util, dstType ):
   kws = set()
   if util == LspPing:
      kws.update( [ 'vpn', 'ldp', 'mldp', 'segment-routing', 'generic', 'bgpLu',
                    'rsvp', 'pwLdp' ] )
   if util == LspTraceroute:
      kws.update( [ 'ldp', 'segment-routing', 'generic', 'rsvp', 'mldp' ] )
   return dstType in kws

lspUtilHandlerMap = {
   LspPing : {       LspPingTypeBgpLu : handleLspPingBgpLu,
                     LspPingTypeGeneric : handleLspPingLdpMldpSr,
                     LspPingTypeRaw : handleLspPingRaw,
                     LspPingTypeLdp : handleLspPingLdpMldpSr,
                     LspPingTypeMldp : handleLspPingLdpMldpSr,
                     LspPingTypeRsvp : handleLspPingRsvp,
                     LspPingTypeSr : handleLspPingLdpMldpSr,
                     LspPingTypeNhg : handleLspPingNhg,
                     LspPingTypeStatic : handleLspPingStatic,
                     LspPingTypeSrTe : handleLspPingSrTe,
                     LspPingTypePwLdp : handleLspPingPwLdp,
                     LspPingTypeNhgTunnel : handleLspPingNhgTunnel,
                     LspPingTypeVpn : handleLspPingVpn },
   LspTraceroute : { LspPingTypeBgpLu : handleLspTracerouteBgpLu,
                     LspPingTypeGeneric : handleLspTracerouteGeneric,
                     LspPingTypeRaw : handleLspTracerouteRaw,
                     LspPingTypeLdp : handleLspTracerouteLdp,
                     LspPingTypeMldp : handleLspTracerouteMldp,
                     LspPingTypeRsvp : handleLspTracerouteRsvp,
                     LspPingTypeSr : handleLspTracerouteSr,
                     LspPingTypeNhg : handleLspTracerouteNhg,
                     LspPingTypeStatic : handleLspTracerouteStatic,
                     LspPingTypeSrTe : handleLspTracerouteSrTe,
                     LspPingTypeNhgTunnel : handleLspTracerouteNhgTunnel }
}

# ------------------------------------------------------------------------
#       lspUtilHandler object used by ping/traceroute utilities 
# ------------------------------------------------------------------------

def lspUtilHandler( util, entityManager, args ):
   if not util in [ LspPing, LspTraceroute ]:
      print( 'Util not supported: %s' % util )
      return errno.EINVAL

   if not args or not isinstance( args, dict ):
      print( 'Wrong type of arguments' )
      return errno.EINVAL

   sessionIdIncr()
   currState = getGlobalState()

   # Ping scale test only: set ClientIdBase and request src port
   if 'cidbase' in args:
      clientIdBaseOverride = args.pop( 'cidbase' )
      if clientIdBaseOverride is not None:
         setGlobalClientIdBaseOverride( clientIdBaseOverride )
         currState.clientIdBase = int( clientIdBaseOverride )
   if util == LspPing:
      reqSrcPort = args.pop( 'sport' )
      if reqSrcPort:
         currState.clientRootUdpPamSrcPort = int( reqSrcPort )

   retCode = 0
   dst = args.pop( 'destination' )
   dstType = args[ 'type' ]
   handler = lspUtilHandlerMap[ util ].get( dstType )

   if args.get( 'session_name' ):
      session = args.pop( 'session_name' )
      args[ 'session' ] = session
   # session_id could be 0, so explicitly check is not None
   if args.get( 'session_id' ) is not None:
      session = args.pop( 'session_id' )
      args[ 'session' ] = session

   if handler :
      if args.get( 'label' ):
         args[ 'label' ] = [ int( l ) for l in args[ 'label' ].split( ',' ) ]
      mount = LspUtilMount( dstType, entityManager, args[ 'vrf' ] )
      allocateTraceFile( util, dstType, mount.traceConfig )
      if useCapi( util, dstType ):
         servSockPort = args.pop( 'servSockPort' )
         sock = getClientSocket( servSockPort )
         args[ 'sock' ] = sock
      retCode = handler( dst, mount, **args )
      if useCapi( util, dstType ):
         bt8( "Clean up client socket", bv( sock.getsockname() ) )
         sock.shutdown( socket.SHUT_RDWR )
         sock.close()
   else:
      print( f'{util} type not supported: {dstType}' )
      retCode = errno.EINVAL
   return retCode

